/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.mongodb;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.mongodb.DBRef;
import com.mongodb.client.MongoClient;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.Collation;
import com.mongodb.client.model.CollationCaseFirst;
import com.mongodb.client.model.CollationStrength;
import com.mongodb.client.model.CreateCollectionOptions;
import io.trino.Session;
import io.trino.execution.QueryStats;
import io.trino.plugin.mongodb.MongoQueryRunner;
import io.trino.plugin.mongodb.MongoServer;
import io.trino.plugin.mongodb.TypeUtils;
import io.trino.spi.type.Type;
import io.trino.sql.planner.plan.FilterNode;
import io.trino.sql.planner.plan.LimitNode;
import io.trino.sql.planner.plan.ProjectNode;
import io.trino.sql.query.QueryAssertions;
import io.trino.testing.BaseConnectorTest;
import io.trino.testing.MaterializedResult;
import io.trino.testing.MaterializedRow;
import io.trino.testing.QueryRunner;
import io.trino.testing.TestingConnectorBehavior;
import io.trino.testing.TestingNames;
import io.trino.testing.sql.TestTable;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Set;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.AssertProvider;
import org.assertj.core.api.Assertions;
import org.bson.Document;
import org.bson.types.Decimal128;
import org.bson.types.ObjectId;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInstance;

@TestInstance(value=TestInstance.Lifecycle.PER_CLASS)
public class TestMongoConnectorTest
extends BaseConnectorTest {
    protected MongoServer server;
    protected MongoClient client;

    protected QueryRunner createQueryRunner() throws Exception {
        this.server = new MongoServer();
        this.client = MongoQueryRunner.createMongoClient(this.server);
        return MongoQueryRunner.builder(this.server).setInitialTables(REQUIRED_TPCH_TABLES).build();
    }

    @BeforeAll
    public void initTestSchema() {
        this.assertUpdate("CREATE SCHEMA IF NOT EXISTS test");
    }

    @AfterAll
    public final void destroy() {
        this.server.close();
        this.server = null;
        this.client.close();
        this.client = null;
    }

    protected boolean hasBehavior(TestingConnectorBehavior connectorBehavior) {
        return switch (connectorBehavior) {
            case TestingConnectorBehavior.SUPPORTS_ADD_COLUMN_WITH_POSITION, TestingConnectorBehavior.SUPPORTS_ADD_FIELD, TestingConnectorBehavior.SUPPORTS_CREATE_MATERIALIZED_VIEW, TestingConnectorBehavior.SUPPORTS_CREATE_VIEW, TestingConnectorBehavior.SUPPORTS_DROP_FIELD, TestingConnectorBehavior.SUPPORTS_MERGE, TestingConnectorBehavior.SUPPORTS_NOT_NULL_CONSTRAINT, TestingConnectorBehavior.SUPPORTS_RENAME_FIELD, TestingConnectorBehavior.SUPPORTS_RENAME_SCHEMA, TestingConnectorBehavior.SUPPORTS_SET_FIELD_TYPE, TestingConnectorBehavior.SUPPORTS_TRUNCATE, TestingConnectorBehavior.SUPPORTS_UPDATE -> false;
            default -> super.hasBehavior(connectorBehavior);
        };
    }

    protected TestTable createTableWithDefaultColumns() {
        return (TestTable)Assumptions.abort((String)"MongoDB connector does not support column default values");
    }

    @Test
    public void testColumnName() {
        for (String columnName : this.testColumnNameDataProvider()) {
            if (columnName.equals("a.dot")) {
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.testColumnName(columnName, TestMongoConnectorTest.requiresDelimiting((String)columnName))).isInstanceOf(RuntimeException.class)).hasMessage("Column name must not contain '$' or '.' for INSERT: " + columnName);
                continue;
            }
            this.testColumnName(columnName, TestMongoConnectorTest.requiresDelimiting((String)columnName));
        }
    }

    @Test
    public void testSortItemsReflectedInExplain() {
        this.assertExplain("EXPLAIN SELECT name FROM nation ORDER BY nationkey DESC NULLS LAST LIMIT 5", new String[]{"TopNPartial\\[count = 5, orderBy = \\[nationkey DESC"});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testNonLowercaseCollection() {
        String suffix = TestingNames.randomNameSuffix();
        String schema = "test_db_" + suffix;
        String table = "test_collection_" + suffix;
        String mixedTable = "Test_Collection_" + suffix;
        try {
            MongoDatabase db = this.client.getDatabase(schema);
            db.createCollection(table);
            db.getCollection(table).insertOne((Object)new Document("lowercase", (Object)1));
            db.createCollection(mixedTable);
            db.getCollection(mixedTable).insertOne((Object)new Document("mixed", (Object)2));
            Assertions.assertThatThrownBy(() -> this.client.getDatabase(schema.toUpperCase(Locale.ENGLISH)).createCollection(table)).hasMessageContaining("db already exists with different case");
            db.createCollection(table.toUpperCase(Locale.ENGLISH));
            db.getCollection(table.toUpperCase(Locale.ENGLISH)).insertOne((Object)new Document("uppercase", (Object)3));
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM information_schema.tables WHERE table_catalog = 'mongodb' AND table_schema = '" + schema + "'"))).matches("VALUES (VARCHAR 'mongodb', VARCHAR '" + schema + "', VARCHAR '" + table + "', VARCHAR 'BASE TABLE')");
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT table_name, column_name FROM information_schema.columns WHERE table_catalog = 'mongodb' AND table_schema = '" + schema + "'"))).matches("VALUES (VARCHAR '" + table + "', VARCHAR 'lowercase')");
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM " + schema + "." + table))).matches("VALUES BIGINT '1'");
        }
        finally {
            this.client.getDatabase(schema).drop();
        }
    }

    protected Optional<BaseConnectorTest.DataMappingTestSetup> filterDataMappingSmokeTestData(BaseConnectorTest.DataMappingTestSetup dataMappingTestSetup) {
        String typeName = dataMappingTestSetup.getTrinoTypeName();
        if (typeName.equals("time(6)") || typeName.equals("timestamp(6)") || typeName.equals("timestamp(6) with time zone")) {
            return Optional.of(dataMappingTestSetup.asUnsupported());
        }
        return Optional.of(dataMappingTestSetup);
    }

    @Test
    public void testGuessFieldTypes() {
        this.testGuessFieldTypes("true", "true");
        this.testGuessFieldTypes("2147483647", "bigint '2147483647'");
        this.testGuessFieldTypes("{\"$numberLong\": \"9223372036854775807\"}", "9223372036854775807");
        this.testGuessFieldTypes("1.23", "double '1.23'");
        this.testGuessFieldTypes("{\"$date\": \"1970-01-01T00:00:00.000Z\"}", "timestamp '1970-01-01 00:00:00.000'");
        this.testGuessFieldTypes("'String type'", "varchar 'String type'");
        this.testGuessFieldTypes("{$binary: \"\",\"$type\": \"0\"}", "to_utf8('')");
        this.testGuessFieldTypes("{\"$oid\": \"6216f0c6c432d45190f25e7c\"}", "ObjectId('6216f0c6c432d45190f25e7c')");
        this.testGuessFieldTypes("[1]", "array[bigint '1']");
        this.testGuessFieldTypes("{\"field\": \"object\"}", "CAST(row('object') AS row(field varchar))");
        this.testGuessFieldTypes("[9, \"test\"]", "CAST(row(9, 'test') AS row(_pos1 bigint, _pos2 varchar))");
        this.testGuessFieldTypes("{\"$ref\":\"test_ref\",\"$id\":ObjectId(\"4e3f33de6266b5845052c02c\"),\"$db\":\"test_db\"}", "CAST(row('test_db', 'test_ref', ObjectId('4e3f33de6266b5845052c02c')) AS row(databasename varchar, collectionname varchar, id ObjectId))");
    }

    private void testGuessFieldTypes(String mongoValue, String trinoValue) {
        String tableName = "test_guess_field_type_" + TestingNames.randomNameSuffix();
        Document document = Document.parse((String)String.format("{\"test\":%s}", mongoValue));
        this.assertUpdate("DROP TABLE IF EXISTS test." + tableName);
        this.client.getDatabase("test").getCollection(tableName).insertOne((Object)document);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT test FROM test." + tableName))).matches("SELECT " + trinoValue);
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void createTableWithEveryType() {
        String tableName = "test_types_table_" + TestingNames.randomNameSuffix();
        String query = "CREATE TABLE " + tableName + " AS SELECT 'foo' _varchar, cast('bar' as varbinary) _varbinary, cast(1 as bigint) _bigint, 3.14E0 _double, true _boolean, DATE '1980-05-07' _date, TIMESTAMP '1980-05-07 11:22:33.456' _timestamp, ObjectId('ffffffffffffffffffffffff') _objectid, JSON '{\"name\":\"alice\"}' _json, cast(12.3 as decimal(30, 5)) _long_decimal";
        this.assertUpdate(query, 1L);
        MaterializedResult results = this.getQueryRunner().execute(this.getSession(), "SELECT * FROM " + tableName).toTestTypes();
        Assertions.assertThat((int)results.getRowCount()).isEqualTo(1);
        MaterializedRow row = (MaterializedRow)results.getMaterializedRows().get(0);
        Assertions.assertThat((Object)row.getField(0)).isEqualTo((Object)"foo");
        Assertions.assertThat((Object)row.getField(1)).isEqualTo((Object)"bar".getBytes(StandardCharsets.UTF_8));
        Assertions.assertThat((Object)row.getField(2)).isEqualTo((Object)1L);
        Assertions.assertThat((Object)row.getField(3)).isEqualTo((Object)3.14);
        Assertions.assertThat((Object)row.getField(4)).isEqualTo((Object)true);
        Assertions.assertThat((Object)row.getField(5)).isEqualTo((Object)LocalDate.of(1980, 5, 7));
        Assertions.assertThat((Object)row.getField(6)).isEqualTo((Object)LocalDateTime.of(1980, 5, 7, 11, 22, 33, 456000000));
        Assertions.assertThat((Object)row.getField(8)).isEqualTo((Object)"{\"name\":\"alice\"}");
        Assertions.assertThat((Object)row.getField(9)).isEqualTo((Object)new BigDecimal("12.30000"));
        this.assertUpdate("DROP TABLE " + tableName);
        Assertions.assertThat((boolean)this.getQueryRunner().tableExists(this.getSession(), tableName)).isFalse();
    }

    @Test
    public void testInsertWithEveryType() {
        String tableName = "test_insert_types_table_" + TestingNames.randomNameSuffix();
        String createSql = "CREATE TABLE " + tableName + "(  vc varchar, vb varbinary, bi bigint, d double, b boolean, dt  date, ts  timestamp, objid objectid, _json json)";
        this.getQueryRunner().execute(this.getSession(), createSql);
        String insertSql = "INSERT INTO " + tableName + " SELECT 'foo' _varchar, cast('bar' as varbinary) _varbinary, cast(1 as bigint) _bigint, 3.14E0 _double, true _boolean, DATE '1980-05-07' _date, TIMESTAMP '1980-05-07 11:22:33.456' _timestamp, ObjectId('ffffffffffffffffffffffff') _objectid, JSON '{\"name\":\"alice\"}' _json";
        this.getQueryRunner().execute(this.getSession(), insertSql);
        MaterializedResult results = this.getQueryRunner().execute(this.getSession(), "SELECT * FROM " + tableName).toTestTypes();
        Assertions.assertThat((int)results.getRowCount()).isEqualTo(1);
        MaterializedRow row = (MaterializedRow)results.getMaterializedRows().get(0);
        Assertions.assertThat((Object)row.getField(0)).isEqualTo((Object)"foo");
        Assertions.assertThat((Object)row.getField(1)).isEqualTo((Object)"bar".getBytes(StandardCharsets.UTF_8));
        Assertions.assertThat((Object)row.getField(2)).isEqualTo((Object)1L);
        Assertions.assertThat((Object)row.getField(3)).isEqualTo((Object)3.14);
        Assertions.assertThat((Object)row.getField(4)).isEqualTo((Object)true);
        Assertions.assertThat((Object)row.getField(5)).isEqualTo((Object)LocalDate.of(1980, 5, 7));
        Assertions.assertThat((Object)row.getField(6)).isEqualTo((Object)LocalDateTime.of(1980, 5, 7, 11, 22, 33, 456000000));
        Assertions.assertThat((Object)row.getField(8)).isEqualTo((Object)"{\"name\":\"alice\"}");
        this.assertUpdate("DROP TABLE " + tableName);
        Assertions.assertThat((boolean)this.getQueryRunner().tableExists(this.getSession(), tableName)).isFalse();
    }

    @Test
    public void testDeleteWithComplexPredicate() {
        Assertions.assertThatThrownBy(() -> super.testDeleteWithComplexPredicate()).hasStackTraceContaining("TrinoException: This connector does not support modifying table rows");
    }

    @Test
    public void testDeleteWithLike() {
        Assertions.assertThatThrownBy(() -> super.testDeleteWithLike()).hasStackTraceContaining("TrinoException: This connector does not support modifying table rows");
    }

    @Test
    public void testDeleteWithSemiJoin() {
        Assertions.assertThatThrownBy(() -> super.testDeleteWithSemiJoin()).hasStackTraceContaining("TrinoException: This connector does not support modifying table rows");
    }

    @Test
    public void testDeleteWithSubquery() {
        Assertions.assertThatThrownBy(() -> super.testDeleteWithSubquery()).hasStackTraceContaining("TrinoException: This connector does not support modifying table rows");
    }

    @Test
    public void testExplainAnalyzeWithDeleteWithSubquery() {
        Assertions.assertThatThrownBy(() -> super.testExplainAnalyzeWithDeleteWithSubquery()).hasStackTraceContaining("TrinoException: This connector does not support modifying table rows");
    }

    @Test
    public void testPredicatePushdown() {
        this.testPredicatePushdown("true");
        this.testPredicatePushdown("tinyint '1'");
        this.testPredicatePushdown("smallint '2'");
        this.testPredicatePushdown("integer '3'");
        this.testPredicatePushdown("bigint '4'");
        this.testPredicatePushdown("decimal '3.14'");
        this.testPredicatePushdown("decimal '1234567890.123456789'");
        this.testPredicatePushdown("'test'");
        this.testPredicatePushdown("char 'test'");
        this.testPredicatePushdown("objectid('6216f0c6c432d45190f25e7c')");
        this.testPredicatePushdown("date '1970-01-01'");
        this.testPredicatePushdown("time '00:00:00.000'");
        this.testPredicatePushdown("timestamp '1970-01-01 00:00:00.000'");
        this.testPredicatePushdown("timestamp '1970-01-01 00:00:00.000 UTC'");
    }

    private void testPredicatePushdown(String value) {
        try (TestTable table = this.newTrinoTable("test_predicate_pushdown", "AS SELECT %s col".formatted(value));){
            this.testPredicatePushdown(table.getName(), "col = " + value);
            this.testPredicatePushdown(table.getName(), "col != " + value);
            this.testPredicatePushdown(table.getName(), "col < " + value);
            this.testPredicatePushdown(table.getName(), "col > " + value);
            this.testPredicatePushdown(table.getName(), "col <= " + value);
            this.testPredicatePushdown(table.getName(), "col >= " + value);
        }
    }

    @Test
    public void testPredicatePushdownRealType() {
        this.testPredicatePushdownFloatingPoint("real '1.234'");
    }

    @Test
    public void testPredicatePushdownDoubleType() {
        this.testPredicatePushdownFloatingPoint("double '5.678'");
    }

    private void testPredicatePushdownFloatingPoint(String value) {
        try (TestTable table = this.newTrinoTable("test_floating_point_pushdown", "AS SELECT %s col".formatted(value));){
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM " + table.getName() + " WHERE col = " + value))).isFullyPushedDown();
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM " + table.getName() + " WHERE col <= " + value))).isFullyPushedDown();
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM " + table.getName() + " WHERE col >= " + value))).isFullyPushedDown();
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM " + table.getName() + " WHERE col > " + value))).returnsEmptyResult().isFullyPushedDown();
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM " + table.getName() + " WHERE col < " + value))).returnsEmptyResult().isFullyPushedDown();
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM " + table.getName() + " WHERE col != " + value))).returnsEmptyResult().isNotFullyPushedDown(FilterNode.class, new Class[0]);
        }
    }

    @Test
    public void testPredicatePushdownCharWithPaddedSpace() {
        try (TestTable table = this.newTrinoTable("test_predicate_pushdown_char_with_padded_space", "(k, v) AS VALUES   (-1, CAST(NULL AS char(3))),    (0, CAST('' AS char(3))),   (1, CAST(' ' AS char(3))),    (2, CAST('  ' AS char(3))),    (3, CAST('   ' AS char(3))),   (4, CAST('x' AS char(3))),   (5, CAST('x ' AS char(3))),   (6, CAST('x  ' AS char(3))),   (7, CAST('\u0000' AS char(3))),   (8, CAST('\u0000 ' AS char(3))),   (9, CAST('\u0000  ' AS char(3)))");){
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT k FROM " + table.getName() + " WHERE v = ''"))).matches("VALUES 0, 1, 2, 3").isFullyPushedDown();
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT k FROM " + table.getName() + " WHERE v = 'x '"))).matches("VALUES 4, 5, 6").isFullyPushedDown();
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT k FROM " + table.getName() + " WHERE v = '\u0000  '"))).matches("VALUES 7, 8, 9").isFullyPushedDown();
        }
    }

    @Test
    public void testPredicatePushdownMultipleNotEquals() {
        try (TestTable table = this.newTrinoTable("test_predicate_pushdown_with_multiple_not_equals", "(id, value) AS VALUES (1, 10), (2, 20), (3, 30)");){
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM " + table.getName() + " WHERE id != 1 AND value != 20"))).matches("VALUES (3, 30)").isFullyPushedDown();
        }
    }

    @Test
    public void testHighPrecisionDecimalPredicate() {
        try (TestTable table = this.newTrinoTable("test_high_precision_decimal_predicate", "(col DECIMAL(34, 0))", Arrays.asList("decimal '3141592653589793238462643383279502'", null));){
            String predicateValue = "decimal '31415926535897932384626433832795028841'";
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM " + table.getName() + " WHERE col = " + predicateValue))).returnsEmptyResult();
            this.testPredicatePushdown(table.getName(), "col != " + predicateValue);
            this.testPredicatePushdown(table.getName(), "col < " + predicateValue);
            this.testPredicatePushdown(table.getName(), "col > " + predicateValue);
            this.testPredicatePushdown(table.getName(), "col <= " + predicateValue);
            this.testPredicatePushdown(table.getName(), "col >= " + predicateValue);
            predicateValue = "decimal '3141592653589793238462643383279502'";
            this.testPredicatePushdown(table.getName(), "col = " + predicateValue);
            this.testPredicatePushdown(table.getName(), "col != " + predicateValue);
            this.testPredicatePushdown(table.getName(), "col < " + predicateValue);
            this.testPredicatePushdown(table.getName(), "col > " + predicateValue);
            this.testPredicatePushdown(table.getName(), "col <= " + predicateValue);
            this.testPredicatePushdown(table.getName(), "col >= " + predicateValue);
        }
    }

    private void testPredicatePushdown(String tableName, String whereClause) {
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM " + tableName + " WHERE " + whereClause))).isFullyPushedDown();
    }

    @Test
    public void testJson() {
        String tableName = "test_json_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " (id INT, col JSON)");
        this.assertUpdate("INSERT INTO " + tableName + " VALUES (1, JSON '{\"name\":\"alice\"}')", 1L);
        this.assertQuery("SELECT json_extract_scalar(col, '$.name') FROM " + tableName + " WHERE id = 1", "SELECT 'alice'");
        this.assertUpdate("INSERT INTO " + tableName + " VALUES (2, JSON '{\"numbers\":[1, 2, 3]}')", 1L);
        this.assertQuery("SELECT json_extract(col, '$.numbers[0]') FROM " + tableName + " WHERE id = 2", "SELECT 1");
        this.assertUpdate("INSERT INTO " + tableName + " VALUES (3, NULL)", 1L);
        this.assertQuery("SELECT col FROM " + tableName + " WHERE id = 3", "SELECT NULL");
        this.assertQueryFails("CREATE TABLE test_json_scalar AS SELECT JSON '1' AS col", "Can't convert json to MongoDB Document.*");
        this.assertQueryFails("CREATE TABLE test_json_array AS SELECT JSON '[\"a\", \"b\", \"c\"]' AS col", "Can't convert json to MongoDB Document.*");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testArrays() {
        String arrayIntegerTable = "test_array_integer" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + arrayIntegerTable + " AS SELECT ARRAY[1, 2, NULL] AS col", 1L);
        this.assertQuery("SELECT col[2] FROM " + arrayIntegerTable, "SELECT 2");
        this.assertQuery("SELECT col[3] FROM " + arrayIntegerTable, "SELECT NULL");
        this.assertUpdate("DROP TABLE " + arrayIntegerTable);
        String arrayDoubleTable = "test_array_double" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + arrayDoubleTable + " AS SELECT ARRAY[1.0E0, 2.5E0, 3.5E0] AS col", 1L);
        this.assertQuery("SELECT col[2] FROM " + arrayDoubleTable, "SELECT 2.5");
        this.assertUpdate("DROP TABLE " + arrayDoubleTable);
        String arrayVarcharTable = "test_array_varchar" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + arrayVarcharTable + " AS SELECT ARRAY['puppies', 'kittens', NULL] AS col", 1L);
        this.assertQuery("SELECT col[2] FROM " + arrayVarcharTable, "SELECT 'kittens'");
        this.assertQuery("SELECT col[3] FROM " + arrayVarcharTable, "SELECT NULL");
        this.assertUpdate("DROP TABLE " + arrayVarcharTable);
        String arrayBooleanTable = "test_array_boolean" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + arrayBooleanTable + " AS SELECT ARRAY[TRUE, NULL] AS col", 1L);
        this.assertQuery("SELECT col[1] FROM " + arrayBooleanTable, "SELECT TRUE");
        this.assertQuery("SELECT col[2] FROM " + arrayBooleanTable, "SELECT NULL");
        this.assertUpdate("DROP TABLE " + arrayBooleanTable);
        String nestedArrayIntegerTable = "test_nested_array_integer" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + nestedArrayIntegerTable + " AS SELECT ARRAY[ARRAY[1, 2], NULL, ARRAY[3, 4]] AS col", 1L);
        this.assertQuery("SELECT col[1][2] FROM " + nestedArrayIntegerTable, "SELECT 2");
        this.assertUpdate("DROP TABLE " + nestedArrayIntegerTable);
        String nestedArrayVarcharTable = "test_nested_array_varchar" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + nestedArrayVarcharTable + " AS SELECT ARRAY[ARRAY['\"hi\"'], NULL, ARRAY['puppies']] AS col", 1L);
        this.assertQuery("SELECT col[1][1] FROM " + nestedArrayVarcharTable, "SELECT '\"hi\"'");
        this.assertQuery("SELECT col[3][1] FROM " + nestedArrayVarcharTable, "SELECT 'puppies'");
        this.assertUpdate("DROP TABLE " + nestedArrayVarcharTable);
    }

    @Test
    public void testTemporalArrays() {
        String dateArrayTable = "test_array_date" + TestingNames.randomNameSuffix();
        String timestampArrayTable = "test_array_timestamp" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + dateArrayTable + " AS SELECT ARRAY[DATE '2014-09-30'] AS col", 1L);
        this.assertOneNotNullResult("SELECT col[1] FROM " + dateArrayTable);
        this.assertUpdate("DROP TABLE " + dateArrayTable);
        this.assertUpdate("CREATE TABLE " + timestampArrayTable + " AS SELECT ARRAY[TIMESTAMP '2001-08-22 03:04:05.321'] AS col", 1L);
        this.assertOneNotNullResult("SELECT col[1] FROM " + timestampArrayTable);
        this.assertUpdate("DROP TABLE " + timestampArrayTable);
    }

    @Test
    public void testSkipUnknownTypes() {
        String unknownFieldTable = "test_unknown_field" + TestingNames.randomNameSuffix();
        Document document1 = new Document("col", (Object)Document.parse((String)"{\"key1\": \"value1\", \"key2\": null}"));
        this.client.getDatabase("test").getCollection(unknownFieldTable).insertOne((Object)document1);
        this.assertQuery("SHOW COLUMNS FROM test." + unknownFieldTable, "SELECT 'col', 'row(key1 varchar)', '', ''");
        this.assertQuery("SELECT col.key1 FROM test." + unknownFieldTable, "SELECT 'value1'");
        this.assertUpdate("DROP TABLE test." + unknownFieldTable);
        String allUnknownFieldTable = "test_all_unknown_field" + TestingNames.randomNameSuffix();
        Document document2 = new Document("col", (Object)new Document("key1", null));
        this.client.getDatabase("test").getCollection(allUnknownFieldTable).insertOne((Object)document2);
        this.assertQueryReturnsEmptyResult("SHOW COLUMNS FROM test." + allUnknownFieldTable);
        this.assertUpdate("DROP TABLE test." + allUnknownFieldTable);
    }

    @Test
    public void testSkipUnsupportedDecimal128() {
        String tableName = "test_unsupported_decimal128" + TestingNames.randomNameSuffix();
        Document document = new Document((Map)ImmutableMap.builder().put((Object)"col", (Object)1).put((Object)"nan", (Object)Decimal128.NaN).put((Object)"negative_nan", (Object)Decimal128.NEGATIVE_NaN).put((Object)"positive_infinity", (Object)Decimal128.POSITIVE_INFINITY).put((Object)"negative_infinity", (Object)Decimal128.NEGATIVE_INFINITY).put((Object)"negative_zero", (Object)Decimal128.NEGATIVE_ZERO).buildOrThrow());
        this.client.getDatabase("test").getCollection(tableName).insertOne((Object)document);
        this.assertQuery("SHOW COLUMNS FROM test." + tableName, "SELECT 'col', 'bigint', '', ''");
        this.assertQuery("SELECT col FROM test." + tableName, "SELECT 1");
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testNegativeZeroDecimal() {
        String tableName = "test_negative_zero" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + tableName + "(id int, short_decimal decimal(1), long_decimal decimal(38))");
        this.client.getDatabase("test").getCollection(tableName).insertOne((Object)new Document((Map)ImmutableMap.builder().put((Object)"id", (Object)1).put((Object)"short_decimal", (Object)Decimal128.NEGATIVE_ZERO).put((Object)"long_decimal", (Object)Decimal128.NEGATIVE_ZERO).buildOrThrow()));
        this.client.getDatabase("test").getCollection(tableName).insertOne((Object)new Document((Map)ImmutableMap.builder().put((Object)"id", (Object)2).put((Object)"short_decimal", (Object)Decimal128.parse((String)"-0.000")).put((Object)"long_decimal", (Object)Decimal128.parse((String)"-0.000")).buildOrThrow()));
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM test." + tableName))).matches("VALUES (1, CAST('0' AS decimal(1)), CAST('0' AS decimal(38))), (2, CAST('0' AS decimal(1)), CAST('0' AS decimal(38)))");
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT id FROM test." + tableName + " WHERE short_decimal = decimal '0'"))).matches("VALUES 1, 2");
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT id FROM test." + tableName + " WHERE long_decimal = decimal '0'"))).matches("VALUES 1, 2");
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testDBRef() {
        this.testDBRef("String type", "varchar 'String type'", "varchar");
        this.testDBRef("BinData".getBytes(StandardCharsets.UTF_8), "to_utf8('BinData')", "varbinary");
        this.testDBRef(1234567890, "bigint '1234567890'", "bigint");
        this.testDBRef(true, "true", "boolean");
        this.testDBRef(Float.valueOf(12.3f), "double '12.3'", "double");
        this.testDBRef(new Date(0L), "timestamp '1970-01-01 00:00:00.000'", "timestamp(3)");
        this.testDBRef(ImmutableList.of((Object)1), "array[bigint '1']", "array(bigint)");
        this.testDBRef(new ObjectId("5126bc054aed4daf9e2ab772"), "ObjectId('5126bc054aed4daf9e2ab772')", "ObjectId");
    }

    private void testDBRef(Object objectId, String expectedValue, String expectedType) {
        Document document = Document.parse((String)"{\"_id\":ObjectId(\"5126bbf64aed4daf9e2ab771\"),\"col1\":\"foo\"}");
        DBRef dbRef = new DBRef("test", "creators", objectId);
        document.append("creator", (Object)dbRef);
        String tableName = "test_dbref_" + TestingNames.randomNameSuffix();
        this.assertUpdate("DROP TABLE IF EXISTS test." + tableName);
        this.client.getDatabase("test").getCollection(tableName).insertOne((Object)document);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT creator.databaseName, creator.collectionName, creator.id FROM test." + tableName))).matches("SELECT varchar 'test', varchar 'creators', " + expectedValue);
        this.assertQuery("SELECT typeof(creator) FROM test." + tableName, "SELECT 'row(databaseName varchar, collectionName varchar, id " + expectedType + ")'");
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testDbRefFieldOrder() {
        String tableName = "test_dbref_field_order" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + tableName + "(x row(id int, \"collectionName\" varchar, \"databaseName\" varchar))");
        Document document = new Document().append("x", (Object)new DBRef("test_db", "test_collection", (Object)1));
        this.client.getDatabase("test").getCollection(tableName).insertOne((Object)document);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM test." + tableName))).matches("SELECT CAST(row(1, 'test_collection', 'test_db') AS row(id int, \"collectionName\" varchar, \"databaseName\" varchar))");
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testDbRefMissingField() {
        String tableName = "test_dbref_missing_field" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + tableName + "(x row(\"databaseName\" varchar, \"collectionName\" varchar))");
        Document document = new Document().append("x", (Object)new DBRef("test_db", "test_collection", (Object)1));
        this.client.getDatabase("test").getCollection(tableName).insertOne((Object)document);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM test." + tableName))).failure().hasMessageContaining("DBRef should have 3 fields : row(databaseName varchar, collectionName varchar)");
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testDbRefWrongFieldName() {
        String tableName = "test_dbref_wrong_field_name" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + tableName + "(x row(a varchar, b varchar, c int))");
        Document document = new Document().append("x", (Object)new DBRef("test_db", "test_collection", (Object)1));
        this.client.getDatabase("test").getCollection(tableName).insertOne((Object)document);
        this.assertQueryFails("SELECT * FROM test." + tableName, "Unexpected field name for DBRef: a");
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testDbRefWrongFieldType() {
        String tableName = "test_dbref_wrong_field_type" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + tableName + "(x row(\"databaseName\" int, \"collectionName\" int, id int))");
        Document document = new Document().append("x", (Object)new DBRef("test_db", "test_collection", (Object)"test_id"));
        this.client.getDatabase("test").getCollection(tableName).insertOne((Object)document);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM test." + tableName))).matches("SELECT CAST(row(NULL, NULL, NULL) AS row(\"databaseName\" int, \"collectionName\" int, id int))");
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testMaps() {
        String mapIntegerTable = "test_map_integer" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + mapIntegerTable + " AS SELECT MAP(ARRAY[0,1], ARRAY[2,NULL]) AS col", 1L);
        this.assertQuery("SELECT col[0] FROM " + mapIntegerTable, "SELECT 2");
        this.assertQuery("SELECT col[1] FROM " + mapIntegerTable, "SELECT NULL");
        this.assertUpdate("DROP TABLE " + mapIntegerTable);
        String mapDoubleTable = "test_map_double" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + mapDoubleTable + " AS SELECT MAP(ARRAY[1.0E0], ARRAY[2.5E0]) AS col", 1L);
        this.assertQuery("SELECT col[1.0] FROM " + mapDoubleTable, "SELECT 2.5");
        this.assertUpdate("DROP TABLE " + mapDoubleTable);
        String mapVarcharTable = "test_map_varchar" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + mapVarcharTable + " AS SELECT MAP(ARRAY['puppies'], ARRAY['kittens']) AS col", 1L);
        this.assertQuery("SELECT col['puppies'] FROM " + mapVarcharTable, "SELECT 'kittens'");
        this.assertUpdate("DROP TABLE " + mapVarcharTable);
        String mapBooleanTable = "test_map_boolean" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + mapBooleanTable + " AS SELECT MAP(ARRAY[TRUE], ARRAY[FALSE]) AS col", "SELECT 1");
        this.assertQuery("SELECT col[TRUE] FROM " + mapBooleanTable, "SELECT FALSE");
        this.assertUpdate("DROP TABLE " + mapBooleanTable);
        String mapDoubleNestedTable = "test_map_double_nested" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + mapDoubleNestedTable + " AS SELECT MAP(ARRAY[1.0E0], ARRAY[ARRAY[1, 2]]) AS col", 1L);
        this.assertQuery("SELECT col[1.0][2] FROM " + mapDoubleNestedTable, "SELECT 2");
        this.assertUpdate("DROP TABLE " + mapDoubleNestedTable);
        String mapDateTable = "test_map_date" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + mapDateTable + " AS SELECT MAP(ARRAY[DATE '2014-09-30'], ARRAY[DATE '2014-09-29']) AS col", 1L);
        this.assertOneNotNullResult("SELECT col[DATE '2014-09-30'] FROM " + mapDateTable);
        this.assertUpdate("DROP TABLE " + mapDateTable);
        String mapTimestampTable = "test_map_timestamp" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + mapTimestampTable + " AS SELECT MAP(ARRAY[TIMESTAMP '2001-08-22 03:04:05.321'], ARRAY[TIMESTAMP '2001-08-22 03:04:05.321']) AS col", 1L);
        this.assertOneNotNullResult("SELECT col[TIMESTAMP '2001-08-22 03:04:05.321'] FROM " + mapTimestampTable);
        this.assertUpdate("DROP TABLE " + mapTimestampTable);
        String mapTable = "test_map" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + mapTable + " (col MAP<VARCHAR, VARCHAR>)");
        this.client.getDatabase("test").getCollection(mapTable).insertOne((Object)new Document((Map)ImmutableMap.of((Object)"col", (Object)new Document((Map)ImmutableMap.of((Object)"key1", (Object)"value1", (Object)"key2", (Object)"value2")))));
        this.assertQuery("SELECT col['key1'] FROM test." + mapTable, "SELECT 'value1'");
        this.assertUpdate("DROP TABLE test." + mapTable);
        String simpleMapToVarcharTable = "test_simple_map_to_varchar" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + simpleMapToVarcharTable + " (col VARCHAR)");
        this.client.getDatabase("test").getCollection(simpleMapToVarcharTable).insertOne((Object)new Document((Map)ImmutableMap.of((Object)"col", (Object)new Document((Map)ImmutableMap.of((Object)"key1", (Object)"value1", (Object)"key2", (Object)"value2")))));
        this.assertQuery("SELECT col FROM test." + simpleMapToVarcharTable, "SELECT '{\"key1\": \"value1\", \"key2\": \"value2\"}'");
        this.assertUpdate("DROP TABLE test." + simpleMapToVarcharTable);
        String listMapToVarcharTable = "test_list_map_to_varchar" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + listMapToVarcharTable + " (col VARCHAR)");
        this.client.getDatabase("test").getCollection(listMapToVarcharTable).insertOne((Object)new Document((Map)ImmutableMap.of((Object)"col", (Object)ImmutableList.of((Object)new Document((Map)ImmutableMap.of((Object)"key1", (Object)"value1", (Object)"key2", (Object)"value2")), (Object)new Document((Map)ImmutableMap.of((Object)"key3", (Object)"value3", (Object)"key4", (Object)"value4"))))));
        this.assertQuery("SELECT col FROM test." + listMapToVarcharTable, "SELECT '[{\"key1\": \"value1\", \"key2\": \"value2\"}, {\"key3\": \"value3\", \"key4\": \"value4\"}]'");
        this.assertUpdate("DROP TABLE test." + listMapToVarcharTable);
        String integerToVarcharTable = "test_integer_to_varchar" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + integerToVarcharTable + " (col VARCHAR)");
        this.client.getDatabase("test").getCollection(integerToVarcharTable).insertOne((Object)new Document((Map)ImmutableMap.of((Object)"col", (Object)10)));
        this.assertQuery("SELECT col FROM test." + integerToVarcharTable, "SELECT '10'");
        this.assertUpdate("DROP TABLE test." + integerToVarcharTable);
        String arrayToVarcharTable = "test_array_to_varchar" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + arrayToVarcharTable + " (col VARCHAR)");
        this.client.getDatabase("test").getCollection(arrayToVarcharTable).insertOne((Object)new Document((Map)ImmutableMap.of((Object)"col", Arrays.asList(10, null, 11))));
        this.assertQuery("SELECT col FROM test." + arrayToVarcharTable, "SELECT '[10, null, 11]'");
        this.assertUpdate("DROP TABLE test." + arrayToVarcharTable);
    }

    @Test
    public void testCollectionNameContainsDots() {
        String tableName = "test.dot1_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE \"" + tableName + "\" AS SELECT 'foo' _varchar", 1L);
        this.assertQuery("SELECT _varchar FROM \"" + tableName + "\"", "SELECT 'foo'");
        this.assertUpdate("DROP TABLE \"" + tableName + "\"");
    }

    @Test
    public void testObjectIds() {
        String values = "VALUES  (10, NULL, NULL), (11, ObjectId('ffffffffffffffffffffffff'), ObjectId('ffffffffffffffffffffffff')), (12, ObjectId('ffffffffffffffffffffffff'), ObjectId('aaaaaaaaaaaaaaaaaaaaaaaa')), (13, ObjectId('000000000000000000000000'), ObjectId('000000000000000000000000')), (14, ObjectId('ffffffffffffffffffffffff'), NULL), (15, NULL, ObjectId('ffffffffffffffffffffffff'))";
        String inlineTable = String.format("(%s) AS t(i, one, two)", values);
        String tableName = "test_objectid_" + TestingNames.randomNameSuffix();
        this.assertUpdate("DROP TABLE IF EXISTS " + tableName);
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT * FROM " + inlineTable, 6L);
        this.assertQuery("SELECT i FROM " + inlineTable + " WHERE one IS NULL", "VALUES 10, 15");
        this.assertQuery("SELECT i FROM " + tableName + " WHERE one IS NULL", "SELECT 0 WHERE false");
        this.assertQuery("SELECT i, CAST(one AS varchar) FROM " + inlineTable + " WHERE i <= 13", "VALUES (10, NULL), (11, 'ffffffffffffffffffffffff'), (12, 'ffffffffffffffffffffffff'), (13, '000000000000000000000000')");
        this.assertQuery("SELECT i FROM " + tableName + " WHERE one = two", "VALUES 11, 13");
        this.assertQuery("SELECT i FROM " + tableName + " WHERE one = ObjectId('ffffffffffffffffffffffff')", "VALUES 11, 12, 14");
        this.assertQuery("SELECT i FROM " + inlineTable + " WHERE one IS DISTINCT FROM two", "VALUES 12, 14, 15");
        this.assertQuery("SELECT i FROM " + inlineTable + " WHERE one IS NOT DISTINCT FROM two", "VALUES 10, 11, 13");
        this.assertQuery("SELECT i FROM " + tableName + " WHERE one IS DISTINCT FROM two", "VALUES 10, 12, 14, 15");
        this.assertQuery("SELECT i FROM " + tableName + " WHERE one IS NOT DISTINCT FROM two", "VALUES 11, 13");
        this.assertQuery(String.format("SELECT l.i, r.i FROM (%1$s) AS l(i, one, two) JOIN (%1$s) AS r(i, one, two) ON l.one = r.two", values), "VALUES (11, 11), (14, 11), (11, 15), (12, 15), (12, 11), (14, 15), (13, 13)");
        this.assertQuery("SELECT array_agg(i ORDER BY i) FROM " + inlineTable + " GROUP BY one", "VALUES (ARRAY[10, 15]), (ARRAY[11, 12, 14]), (ARRAY[13])");
        this.assertQuery("SELECT i FROM " + inlineTable + " GROUP BY one, i", "VALUES 10, 11, 12, 13, 14, 15");
        this.assertQuery("SELECT r.i, count(*) FROM (SELECT CAST(row(one, i) AS row(one ObjectId, i bigint)) r FROM " + inlineTable + ") GROUP BY r", "VALUES (10, 1), (11, 1), (12, 1), (13, 1), (14, 1), (15, 1)");
        this.assertQuery("SELECT r.x, CAST(r.one AS varchar), count(*) FROM (SELECT CAST(row(one, i / 3 * 3) AS row(one ObjectId, x bigint)) r FROM " + inlineTable + ") GROUP BY r", "VALUES (9, NULL, 1), (9, 'ffffffffffffffffffffffff', 1), (12, 'ffffffffffffffffffffffff', 2), (12, '000000000000000000000000', 1), (15, NULL, 1)");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testSelectView() {
        String tableName = "test_view_base_" + TestingNames.randomNameSuffix();
        String viewName = "test_view_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + tableName + " AS SELECT 'foo' _varchar", 1L);
        this.client.getDatabase("test").createView(viewName, tableName, (List)ImmutableList.of());
        this.assertQuery("SELECT * FROM test." + viewName, "SELECT 'foo'");
        this.assertUpdate("DROP TABLE test." + viewName);
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testBooleanPredicates() {
        String tableName = "test_boolean_predicates_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + "(id integer, value boolean)");
        this.assertUpdate("INSERT INTO " + tableName + " VALUES(1, true)", 1L);
        this.assertUpdate("INSERT INTO " + tableName + " VALUES(2, false)", 1L);
        this.assertQuery("SELECT id FROM " + tableName + " WHERE value = true", "VALUES 1");
        this.assertQuery("SELECT id FROM " + tableName + " WHERE value = false", "VALUES 2");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testNullPredicates() {
        String tableName = "test_null_predicates_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + tableName + "(name varchar, value integer)");
        MongoCollection collection = this.client.getDatabase("test").getCollection(tableName);
        collection.insertOne((Object)new Document((Map)ImmutableMap.of((Object)"name", (Object)"abc", (Object)"value", (Object)1)));
        collection.insertOne((Object)new Document((Map)ImmutableMap.of((Object)"name", (Object)"abcd")));
        collection.insertOne((Object)new Document((Map)Document.parse((String)"{\"name\": \"abcde\", \"value\": null}")));
        this.assertQuery("SELECT count(*) FROM test." + tableName + " WHERE value IS NULL OR rand() = 42", "SELECT 2");
        this.assertQuery("SELECT count(*) FROM test." + tableName + " WHERE value IS NULL", "SELECT 2");
        this.assertQuery("SELECT count(*) FROM test." + tableName + " WHERE value IS NOT NULL", "SELECT 1");
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testLimitPushdown() {
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT name FROM nation LIMIT 30"))).isFullyPushedDown();
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT name FROM nation LIMIT 0"))).returnsEmptyResult();
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT name FROM nation LIMIT 2147483647"))).isFullyPushedDown();
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT name FROM nation LIMIT 2147483648"))).isNotFullyPushedDown(LimitNode.class, new Class[0]);
    }

    @Test
    public void testCollationAccent() {
        String tableName = "test_collation_accent" + TestingNames.randomNameSuffix();
        Collation collation = Collation.builder().locale("en_US").collationStrength(CollationStrength.PRIMARY).build();
        this.client.getDatabase("test").createCollection(tableName, new CreateCollectionOptions().collation(collation));
        this.client.getDatabase("test").getCollection(tableName).insertMany((List)ImmutableList.of((Object)new Document("text", (Object)"e"), (Object)new Document("text", (Object)"\u00e9")));
        this.assertQuery("SELECT * FROM test." + tableName + " WHERE text = 'e'", "VALUES 'e'");
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testCollationCaseSensitivity() {
        String tableName = "test_collation_case_sensitivity" + TestingNames.randomNameSuffix();
        Collation collation = Collation.builder().locale("en_US").collationCaseFirst(CollationCaseFirst.LOWER).build();
        this.client.getDatabase("test").createCollection(tableName, new CreateCollectionOptions().collation(collation));
        this.client.getDatabase("test").getCollection(tableName).insertMany((List)ImmutableList.of((Object)new Document("text", (Object)"abc"), (Object)new Document("text", (Object)"ABC")));
        this.assertQuery("SELECT * FROM test." + tableName + " WHERE text > 'ABC'", "VALUES 'abc'");
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testCollationNumericOrdering() {
        String tableName = "test_collation_numeric_ordering" + TestingNames.randomNameSuffix();
        Collation collation = Collation.builder().locale("en_US").numericOrdering(Boolean.valueOf(true)).build();
        this.client.getDatabase("test").createCollection(tableName, new CreateCollectionOptions().collation(collation));
        this.client.getDatabase("test").getCollection(tableName).insertMany((List)ImmutableList.of((Object)new Document("number", (Object)"-10"), (Object)new Document("number", (Object)"-2.1"), (Object)new Document("number", (Object)"1")));
        this.assertQuery("SELECT * FROM test." + tableName + " WHERE number > '-2.1'", "VALUES '1'");
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testAddColumnConcurrently() {
        Assumptions.abort((String)"TODO");
    }

    @Test
    public void testNativeQuerySimple() {
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM TABLE(mongodb.system.query(database => 'tpch', collection => 'region', filter => '{ regionkey: 1 }'))"))).matches("SELECT * FROM region WHERE regionkey = 1");
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM TABLE(mongodb.system.query(database => 'tpch', collection => 'region', filter => '{}'))"))).matches("SELECT * FROM region");
    }

    @Test
    public void testNativeQueryArray() {
        String tableName = "test_array" + TestingNames.randomNameSuffix();
        MongoCollection collection = this.client.getDatabase("tpch").getCollection(tableName);
        collection.insertOne((Object)new Document("array_field", (Object)ImmutableList.of((Object)"zero", (Object)"one", (Object)"two")));
        collection.insertOne((Object)new Document("array_field", (Object)ImmutableList.of((Object)"0", (Object)"1", (Object)"2")));
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT array_field FROM TABLE(mongodb.system.query(database => 'tpch', collection => '" + tableName + "', filter => '{ \"array_field.1\": \"one\" }'))"))).matches("VALUES CAST(ARRAY['zero', 'one', 'two'] AS array(varchar))");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testNativeQueryNestedRow() {
        String tableName = "test_nested_row" + TestingNames.randomNameSuffix();
        MongoCollection collection = this.client.getDatabase("tpch").getCollection(tableName);
        collection.insertOne((Object)new Document("row_field", (Object)new Document("first", (Object)new Document("second", (Object)1))));
        collection.insertOne((Object)new Document("row_field", (Object)new Document("first", (Object)new Document("second", (Object)2))));
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT row_field.first.second FROM TABLE(mongodb.system.query(database => 'tpch', collection => '" + tableName + "', filter => '{ \"row_field.first.second\": 1 }'))"))).matches("VALUES BIGINT '1'").isFullyPushedDown();
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testNativeQueryHelperFunction() {
        String tableName = "test_query_helper_function" + TestingNames.randomNameSuffix();
        MongoCollection collection = this.client.getDatabase("tpch").getCollection(tableName);
        collection.insertOne((Object)new Document((Map)ImmutableMap.of((Object)"id", (Object)1, (Object)"timestamp", (Object)LocalDateTime.of(2023, 3, 20, 1, 2, 3))));
        collection.insertOne((Object)new Document((Map)ImmutableMap.of((Object)"id", (Object)2, (Object)"timestamp", (Object)LocalDateTime.of(2024, 3, 20, 1, 2, 3))));
        this.assertQuery("SELECT id FROM TABLE(mongodb.system.query(database => 'tpch', collection => '" + tableName + "', filter => '{ timestamp: ISODate(\"2023-03-20T01:02:03.000Z\") }'))", "VALUES 1");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testNativeQueryFilterAndWhere() {
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM TABLE(mongodb.system.query(database => 'tpch', collection => 'nation', filter => '{ regionkey: 0 }')) WHERE name = 'ALGERIA'"))).matches("SELECT * FROM nation WHERE regionkey = 0 AND name = 'ALGERIA'");
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM TABLE(mongodb.system.query(database => 'tpch', collection => 'nation', filter => '{ regionkey: {$gte: 1} }')) WHERE regionkey = 4"))).matches("SELECT * FROM nation WHERE regionkey >= 1 AND regionkey = 4");
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM TABLE(mongodb.system.query(database => 'tpch', collection => 'nation', filter => '{ regionkey: {$gte: 1} }')) WHERE regionkey < 1"))).returnsEmptyResult();
    }

    @Test
    public void testNativeQueryEmptyResult() {
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM TABLE(mongodb.system.query(database => 'tpch', collection => 'region', filter => '{ regionkey: 999 }'))"))).returnsEmptyResult();
    }

    @Test
    public void testNativeQueryLimit() {
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM TABLE(mongodb.system.query(database => 'tpch', collection => 'region', filter => '{}')) LIMIT 30"))).isFullyPushedDown();
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM TABLE(mongodb.system.query(database => 'tpch', collection => 'region', filter => '{}')) LIMIT 0"))).returnsEmptyResult();
    }

    @Test
    public void testNativeQueryProjection() {
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT name FROM TABLE(mongodb.system.query(database => 'tpch', collection => 'region', filter => '{}'))"))).matches("SELECT name FROM region").isFullyPushedDown();
    }

    @Test
    public void testNativeQueryCaseNonLowercaseColumn() {
        String tableName = "test_non_lowercase_column" + TestingNames.randomNameSuffix();
        this.client.getDatabase("test").getCollection(tableName).insertOne((Object)new Document("TestColumn", (Object)1));
        this.assertQuery("SELECT * FROM TABLE(mongodb.system.query(database => 'test', collection => '" + tableName + "', filter => '{\"TestColumn\": 1}'))", "VALUES 1");
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testNativeQueryInvalidArgument() {
        this.assertQueryFails("SELECT * FROM TABLE(mongodb.system.query(database => 'invalid', collection => 'region', filter => '{}'))", "Table 'invalid.region' not found");
        this.assertQueryFails("SELECT * FROM TABLE(mongodb.system.query(database => 'tpch', collection => 'invalid', filter => '{}'))", "Table 'tpch.invalid' not found");
        this.assertQueryFails("SELECT * FROM TABLE(mongodb.system.query(database => 'TPCH', collection => 'region', filter => '{}'))", "Only lowercase database name is supported");
        this.assertQueryFails("SELECT * FROM TABLE(mongodb.system.query(database => 'tpch', collection => 'REGION', filter => '{}'))", "Only lowercase collection name is supported");
        this.assertQueryFails("SELECT * FROM TABLE(mongodb.system.query(database => 'tpch', collection => 'region', filter => '{ invalid }'))", "Can't parse 'filter' argument as json");
    }

    @Test
    public void testRenameTableTo120bytesTableName() {
        String sourceTableName = "test_rename_source_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + sourceTableName + " AS SELECT 123 x", 1L);
        String targetTableName = "a".repeat(120 - "tpch.".length() - 3) + "\u3042";
        Assertions.assertThat((int)targetTableName.length()).isLessThan(120);
        this.assertUpdate("ALTER TABLE " + sourceTableName + " RENAME TO \"" + targetTableName + "\"");
        this.assertQuery("SELECT x FROM \"" + targetTableName + "\"", "VALUES 123");
        this.assertUpdate("DROP TABLE \"" + targetTableName + "\"");
        targetTableName = targetTableName + "z";
        this.assertUpdate("CREATE TABLE " + sourceTableName + " AS SELECT 123 x", 1L);
        this.assertQueryFails("ALTER TABLE " + sourceTableName + " RENAME TO \"" + targetTableName + "\"", "Qualified identifier name must be shorter than or equal to '120' bytes: .*");
        this.assertUpdate("DROP TABLE \"" + sourceTableName + "\"");
    }

    @Test
    public void testListTablesFromSchemaWithBigAmountOfTables() {
        MongoDatabase database = this.client.getDatabase("huge_schema");
        for (int i = 0; i < 10000; ++i) {
            database.createCollection("table_" + i);
        }
        Assertions.assertThat((int)this.getQueryRunner().execute("SHOW TABLES FROM mongodb.huge_schema").getRowCount()).isEqualTo(10000);
    }

    @Test
    public void testSystemSchemas() {
        this.assertQueryReturnsEmptyResult("SHOW SCHEMAS IN mongodb LIKE 'admin'");
        this.assertQueryReturnsEmptyResult("SHOW SCHEMAS IN mongodb LIKE 'config'");
        this.assertQueryReturnsEmptyResult("SHOW SCHEMAS IN mongodb LIKE 'local'");
    }

    @Test
    public void testReadTopLevelDottedField() {
        String tableName = "test_read_top_level_dotted_field_" + TestingNames.randomNameSuffix();
        Document document = new Document().append("_id", (Object)new ObjectId("5126bbf64aed4daf9e2ab771")).append("dotted.field", (Object)"foo");
        this.client.getDatabase("test").getCollection(tableName).insertOne((Object)document);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT \"dotted.field\" FROM test." + tableName))).skippingTypesCheck().matches("SELECT NULL").isFullyPushedDown();
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testReadMiddleLevelDottedField() {
        String tableName = "test_read_middle_level_dotted_field_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + tableName + " (root ROW(\"dotted.field\" ROW(leaf VARCHAR)))");
        this.assertUpdate("INSERT INTO test." + tableName + " SELECT ROW(ROW('foo'))", 1L);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT root.\"dotted.field\" FROM test." + tableName))).skippingTypesCheck().matches("SELECT ROW(varchar 'foo')").isNotFullyPushedDown(ProjectNode.class, new Class[0]);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT root.\"dotted.field\".leaf FROM test." + tableName))).matches("SELECT varchar 'foo'").isNotFullyPushedDown(ProjectNode.class, new Class[0]);
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testReadLeafLevelDottedField() {
        String tableName = "test_read_leaf_level_dotted_field_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + tableName + " (root ROW(\"dotted.field\" VARCHAR, field VARCHAR))");
        this.assertUpdate("INSERT INTO test." + tableName + " SELECT ROW('foo', 'bar')", 1L);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT root.\"dotted.field\" FROM test." + tableName))).matches("SELECT varchar 'foo'").isNotFullyPushedDown(ProjectNode.class, new Class[0]);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT root.\"dotted.field\", root.field FROM test." + tableName))).matches("SELECT varchar 'foo', varchar 'bar'").isNotFullyPushedDown(ProjectNode.class, new Class[0]);
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testReadWithDollarPrefixedFieldName() {
        String tableName = "test_read_with_dollar_prefixed_field_name_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + tableName + " (root ROW(\"$field1\" VARCHAR, field2 VARCHAR))");
        this.assertUpdate("INSERT INTO test." + tableName + " SELECT ROW('foo', 'bar')", 1L);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT root.\"$field1\" FROM test." + tableName))).matches("SELECT varchar 'foo'").isNotFullyPushedDown(ProjectNode.class, new Class[0]);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT root.\"$field1\", root.field2 FROM test." + tableName))).matches("SELECT varchar 'foo', varchar 'bar'").isNotFullyPushedDown(ProjectNode.class, new Class[0]);
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testReadWithDollarInsideFieldName() {
        String tableName = "test_read_with_dollar_inside_field_name_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + tableName + " (root ROW(\"fi$ld1\" VARCHAR, field2 VARCHAR))");
        this.assertUpdate("INSERT INTO test." + tableName + " SELECT ROW('foo', 'bar')", 1L);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT root.\"fi$ld1\" FROM test." + tableName))).matches("SELECT varchar 'foo'").isNotFullyPushedDown(ProjectNode.class, new Class[0]);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT root.\"fi$ld1\", root.field2 FROM test." + tableName))).matches("SELECT varchar 'foo', varchar 'bar'").isNotFullyPushedDown(ProjectNode.class, new Class[0]);
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testReadDottedFieldInsideDollarPrefixedField() {
        String tableName = "test_read_dotted_field_inside_dollar_prefixed_field_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + tableName + " (root ROW(\"$field\" ROW(\"dotted.field\" VARCHAR)))");
        this.assertUpdate("INSERT INTO test." + tableName + " SELECT ROW(ROW('foo'))", 1L);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT root.\"$field\".\"dotted.field\" FROM test." + tableName))).matches("SELECT varchar 'foo'").isNotFullyPushedDown(ProjectNode.class, new Class[0]);
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testReadDollarPrefixedFieldInsideDottedField() {
        String tableName = "test_read_dollar_prefixed_field_inside_dotted_field_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + tableName + " (root ROW(\"dotted.field\" ROW(\"$field\" VARCHAR)))");
        this.assertUpdate("INSERT INTO test." + tableName + " SELECT ROW(ROW('foo'))", 1L);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT root.\"dotted.field\".\"$field\" FROM test." + tableName))).matches("SELECT varchar 'foo'").isNotFullyPushedDown(ProjectNode.class, new Class[0]);
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testPredicateOnDottedField() {
        String tableName = "test_predicate_on_dotted_field_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + tableName + " (root ROW(\"dotted.field\" VARCHAR))");
        this.assertUpdate("INSERT INTO test." + tableName + " SELECT ROW('foo')", 1L);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT root.\"dotted.field\" FROM test." + tableName + " WHERE root.\"dotted.field\" = 'foo'"))).matches("SELECT varchar 'foo'").isNotFullyPushedDown(FilterNode.class, new Class[0]);
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testPredicateOnDollarPrefixedField() {
        String tableName = "test_predicate_on_dollar_prefixed_field_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + tableName + " (root ROW(\"$field\" VARCHAR))");
        this.assertUpdate("INSERT INTO test." + tableName + " SELECT ROW('foo')", 1L);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT root.\"$field\" FROM test." + tableName + " WHERE root.\"$field\" = 'foo'"))).matches("SELECT varchar 'foo'").isNotFullyPushedDown(FilterNode.class, new Class[0]);
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testProjectionPushdownMixedWithUnsupportedFieldName() {
        String tableName = "test_projection_pushdown_mixed_with_unsupported_field_name_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + tableName + " (id INT, root1 ROW(field VARCHAR, \"dotted.field\" VARCHAR), root2 ROW(field VARCHAR, \"$field\" VARCHAR))");
        this.assertUpdate("INSERT INTO test." + tableName + " SELECT 1, ROW('foo1', 'bar1'), ROW('foo2', 'bar2')", 1L);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT root1.field, root2.\"$field\" FROM test." + tableName))).matches("SELECT varchar 'foo1', varchar 'bar2'").isNotFullyPushedDown(ProjectNode.class, new Class[0]);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT root1.\"dotted.field\", root2.field FROM test." + tableName))).matches("SELECT varchar 'bar1', varchar 'foo2'").isNotFullyPushedDown(ProjectNode.class, new Class[0]);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT root1.\"dotted.field\", root2.\"$field\" FROM test." + tableName))).matches("SELECT varchar 'bar1', varchar 'bar2'").isNotFullyPushedDown(ProjectNode.class, new Class[0]);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT root1.field, root2.field FROM test." + tableName))).matches("SELECT varchar 'foo1', varchar 'foo2'").isFullyPushedDown();
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testFiltersOnDereferenceColumnReadsLessData() {
        this.testFiltersOnDereferenceColumnReadsLessData("varchar 'String type'", "varchar");
        this.testFiltersOnDereferenceColumnReadsLessData("to_utf8('BinData')", "varbinary");
        this.testFiltersOnDereferenceColumnReadsLessData("bigint '1234567890'", "bigint");
        this.testFiltersOnDereferenceColumnReadsLessData("true", "boolean");
        this.testFiltersOnDereferenceColumnReadsLessData("double '12.3'", "double");
        this.testFiltersOnDereferenceColumnReadsLessData("timestamp '1970-01-01 00:00:00.000'", "timestamp(3)");
        this.testFiltersOnDereferenceColumnReadsLessData("array[bigint '1']", "array(bigint)");
        this.testFiltersOnDereferenceColumnReadsLessData("ObjectId('5126bc054aed4daf9e2ab772')", "ObjectId");
    }

    private void testFiltersOnDereferenceColumnReadsLessData(String expectedValue, String expectedType) {
        if (!TypeUtils.isPushdownSupportedType((Type)this.getQueryRunner().getPlannerContext().getTypeManager().fromSqlType(expectedType))) {
            return;
        }
        Session sessionWithoutPushdown = Session.builder((Session)this.getSession()).setCatalogSessionProperty((String)this.getSession().getCatalog().orElseThrow(), "projection_pushdown_enabled", "false").build();
        try (TestTable table = this.newTrinoTable("filter_on_projection_columns", String.format("(col_0 ROW(col_1 %1$s, col_2 ROW(col_3 %1$s, col_4 ROW(col_5 %1$s))))", expectedType));){
            this.assertUpdate(String.format("INSERT INTO %s VALUES NULL", table.getName()), 1L);
            this.assertUpdate(String.format("INSERT INTO %1$s SELECT ROW(%2$s, ROW(%2$s, ROW(%2$s)))", table.getName(), expectedValue), 1L);
            this.assertUpdate(String.format("INSERT INTO %1$s SELECT ROW(%2$s, ROW(NULL, ROW(%2$s)))", table.getName(), expectedValue), 1L);
            ImmutableSet expected = ImmutableSet.of((Object)1);
            this.assertQueryStats(this.getSession(), String.format("SELECT 1 FROM %s WHERE col_0.col_1 = %s", table.getName(), expectedValue), arg_0 -> this.lambda$testFiltersOnDereferenceColumnReadsLessData$0(sessionWithoutPushdown, table, expectedValue, (Set)expected, arg_0), arg_0 -> TestMongoConnectorTest.lambda$testFiltersOnDereferenceColumnReadsLessData$3((Set)expected, arg_0));
            this.assertQueryStats(this.getSession(), String.format("SELECT 1 FROM %s WHERE col_0.col_2.col_3 = %s", table.getName(), expectedValue), arg_0 -> this.lambda$testFiltersOnDereferenceColumnReadsLessData$4(sessionWithoutPushdown, table, expectedValue, (Set)expected, arg_0), arg_0 -> TestMongoConnectorTest.lambda$testFiltersOnDereferenceColumnReadsLessData$7((Set)expected, arg_0));
            this.assertQueryStats(this.getSession(), String.format("SELECT 1 FROM %s WHERE col_0.col_2.col_4.col_5 = %s", table.getName(), expectedValue), arg_0 -> this.lambda$testFiltersOnDereferenceColumnReadsLessData$8(sessionWithoutPushdown, table, expectedValue, (Set)expected, arg_0), arg_0 -> TestMongoConnectorTest.lambda$testFiltersOnDereferenceColumnReadsLessData$11((Set)expected, arg_0));
        }
    }

    @Test
    public void testFiltersOnDereferenceColumnReadsLessDataNativeQuery() {
        String tableName = "test_filter_on_dereference_column_reads_less_data_native_query_" + TestingNames.randomNameSuffix();
        MongoCollection collection = this.client.getDatabase("test").getCollection(tableName);
        collection.insertOne((Object)new Document("row_field", (Object)new Document("first", (Object)new Document("second", (Object)1))));
        collection.insertOne((Object)new Document("row_field", (Object)new Document("first", (Object)new Document("second", null))));
        collection.insertOne((Object)new Document("row_field", (Object)new Document("first", null)));
        this.assertQueryStats(this.getSession(), "SELECT row_field.first.second FROM TABLE(mongodb.system.query(database => 'test', collection => '" + tableName + "', filter => '{ \"row_field.first.second\": 1 }'))", stats -> Assertions.assertThat((long)stats.getProcessedInputPositions()).isEqualTo(1L), results -> Assertions.assertThat((Collection)results.getOnlyColumnAsSet()).isEqualTo((Object)ImmutableSet.of((Object)1L)));
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testFilterPushdownOnFieldInsideJson() {
        String tableName = "test_filter_pushdown_on_json_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + tableName + " (id INT, col JSON)");
        this.assertUpdate("INSERT INTO test." + tableName + " VALUES (1, JSON '{\"name\": { \"first\": \"Monika\", \"last\": \"Geller\" }}')", 1L);
        this.assertUpdate("INSERT INTO test." + tableName + " VALUES (2, JSON '{\"name\": { \"first\": \"Rachel\", \"last\": \"Green\" }}')", 1L);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT json_extract_scalar(col, '$.name.first') FROM test." + tableName + " WHERE json_extract_scalar(col, '$.name.last') = 'Geller'"))).matches("SELECT varchar 'Monika'").isNotFullyPushedDown(FilterNode.class, new Class[0]);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT 1 FROM test." + tableName + " WHERE json_extract_scalar(col, '$.name.last') = 'Geller'"))).matches("SELECT 1").isNotFullyPushedDown(FilterNode.class, new Class[0]);
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testProjectionPushdownWithDifferentTypeInDocuments() {
        String tableName = "test_projection_pushdown_with_different_type_in_document_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + tableName + " (col1 ROW(child VARCHAR))");
        MongoCollection collection = this.client.getDatabase("test").getCollection(tableName);
        collection.insertOne((Object)new Document("col1", (Object)100));
        collection.insertOne((Object)new Document("col1", (Object)new Document("child", (Object)"value1")));
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT col1.child FROM test." + tableName))).skippingTypesCheck().matches("VALUES ('value1'), (NULL)").isFullyPushedDown();
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testProjectionPushdownWithColumnMissingInDocument() {
        String tableName = "test_projection_pushdown_with_column_missing_in_document_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE test." + tableName + " (col1 ROW(child VARCHAR))");
        MongoCollection collection = this.client.getDatabase("test").getCollection(tableName);
        collection.insertOne((Object)new Document("col1", (Object)new Document("child1", (Object)"value1")));
        collection.insertOne((Object)new Document("col1", (Object)new Document("child", (Object)"value2")));
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT col1.child FROM test." + tableName))).skippingTypesCheck().matches("VALUES ('value2'), (NULL)").isFullyPushedDown();
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testProjectionPushdownWithDBRef() {
        this.testProjectionPushdownWithDBRef("String type", "varchar 'String type'", "varchar");
        this.testProjectionPushdownWithDBRef("BinData".getBytes(StandardCharsets.UTF_8), "to_utf8('BinData')", "varbinary");
        this.testProjectionPushdownWithDBRef(1234567890, "bigint '1234567890'", "bigint");
        this.testProjectionPushdownWithDBRef(true, "true", "boolean");
        this.testProjectionPushdownWithDBRef(Float.valueOf(12.3f), "double '12.3'", "double");
        this.testProjectionPushdownWithDBRef(new Date(0L), "timestamp '1970-01-01 00:00:00.000'", "timestamp(3)");
        this.testProjectionPushdownWithDBRef(ImmutableList.of((Object)1), "array[bigint '1']", "array(bigint)");
        this.testProjectionPushdownWithDBRef(new ObjectId("5126bc054aed4daf9e2ab772"), "ObjectId('5126bc054aed4daf9e2ab772')", "ObjectId");
    }

    private void testProjectionPushdownWithDBRef(Object objectId, String expectedValue, String expectedType) {
        String tableName = "test_projection_pushdown_with_dbref_" + TestingNames.randomNameSuffix();
        DBRef dbRef = new DBRef("test", "creators", objectId);
        Document document = new Document().append("_id", (Object)new ObjectId("5126bbf64aed4daf9e2ab771")).append("col1", (Object)"foo").append("creator", (Object)dbRef).append("parent", (Object)new Document("child", objectId));
        this.client.getDatabase("test").getCollection(tableName).insertOne((Object)document);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT parent.child, creator.databaseName, creator.collectionName, creator.id FROM test." + tableName))).matches("SELECT " + expectedValue + ", varchar 'test', varchar 'creators', " + expectedValue).isNotFullyPushedDown(ProjectNode.class, new Class[0]);
        this.assertQuery("SELECT typeof(creator) FROM test." + tableName, "SELECT 'row(databaseName varchar, collectionName varchar, id " + expectedType + ")'");
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testProjectionPushdownWithNestedDBRef() {
        this.testProjectionPushdownWithNestedDBRef("String type", "varchar 'String type'", "varchar");
        this.testProjectionPushdownWithNestedDBRef("BinData".getBytes(StandardCharsets.UTF_8), "to_utf8('BinData')", "varbinary");
        this.testProjectionPushdownWithNestedDBRef(1234567890, "bigint '1234567890'", "bigint");
        this.testProjectionPushdownWithNestedDBRef(true, "true", "boolean");
        this.testProjectionPushdownWithNestedDBRef(Float.valueOf(12.3f), "double '12.3'", "double");
        this.testProjectionPushdownWithNestedDBRef(new Date(0L), "timestamp '1970-01-01 00:00:00.000'", "timestamp(3)");
        this.testProjectionPushdownWithNestedDBRef(ImmutableList.of((Object)1), "array[bigint '1']", "array(bigint)");
        this.testProjectionPushdownWithNestedDBRef(new ObjectId("5126bc054aed4daf9e2ab772"), "ObjectId('5126bc054aed4daf9e2ab772')", "ObjectId");
    }

    private void testProjectionPushdownWithNestedDBRef(Object objectId, String expectedValue, String expectedType) {
        String tableName = "test_projection_pushdown_with_dbref_" + TestingNames.randomNameSuffix();
        DBRef dbRef = new DBRef("test", "creators", objectId);
        Document document = new Document().append("_id", (Object)new ObjectId("5126bbf64aed4daf9e2ab771")).append("col1", (Object)"foo").append("parent", (Object)new Document().append("creator", (Object)dbRef).append("child", objectId));
        this.client.getDatabase("test").getCollection(tableName).insertOne((Object)document);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT parent.child, parent.creator.databaseName, parent.creator.collectionName, parent.creator.id FROM test." + tableName))).matches("SELECT " + expectedValue + ", varchar 'test', varchar 'creators', " + expectedValue).isNotFullyPushedDown(ProjectNode.class, new Class[0]);
        this.assertQuery("SELECT typeof(parent.creator) FROM test." + tableName, "SELECT 'row(databaseName varchar, collectionName varchar, id " + expectedType + ")'");
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testProjectionPushdownWithPredefinedDBRefKeyword() {
        this.testProjectionPushdownWithPredefinedDBRefKeyword("String type", "varchar 'String type'", "varchar");
        this.testProjectionPushdownWithPredefinedDBRefKeyword("BinData".getBytes(StandardCharsets.UTF_8), "to_utf8('BinData')", "varbinary");
        this.testProjectionPushdownWithPredefinedDBRefKeyword(1234567890, "bigint '1234567890'", "bigint");
        this.testProjectionPushdownWithPredefinedDBRefKeyword(true, "true", "boolean");
        this.testProjectionPushdownWithPredefinedDBRefKeyword(Float.valueOf(12.3f), "double '12.3'", "double");
        this.testProjectionPushdownWithPredefinedDBRefKeyword(new Date(0L), "timestamp '1970-01-01 00:00:00.000'", "timestamp(3)");
        this.testProjectionPushdownWithPredefinedDBRefKeyword(ImmutableList.of((Object)1), "array[bigint '1']", "array(bigint)");
        this.testProjectionPushdownWithPredefinedDBRefKeyword(new ObjectId("5126bc054aed4daf9e2ab772"), "ObjectId('5126bc054aed4daf9e2ab772')", "ObjectId");
    }

    private void testProjectionPushdownWithPredefinedDBRefKeyword(Object objectId, String expectedValue, String expectedType) {
        String tableName = "test_projection_pushdown_with_predefined_dbref_keyword_" + TestingNames.randomNameSuffix();
        DBRef dbRef = new DBRef("test", "creators", objectId);
        Document document = new Document().append("_id", (Object)new ObjectId("5126bbf64aed4daf9e2ab771")).append("col1", (Object)"foo").append("parent", (Object)new Document("id", (Object)dbRef));
        this.client.getDatabase("test").getCollection(tableName).insertOne((Object)document);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT parent.id, parent.id.id FROM test." + tableName))).skippingTypesCheck().matches("SELECT row('test', 'creators', %1$s), %1$s".formatted(expectedValue)).isNotFullyPushedDown(ProjectNode.class, new Class[0]);
        this.assertQuery("SELECT typeof(parent.id), typeof(parent.id.id) FROM test." + tableName, "SELECT 'row(databaseName varchar, collectionName varchar, id %1$s)', '%1$s'".formatted(expectedType));
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testDBRefLikeDocument() {
        this.testDBRefLikeDocument("String type", "varchar 'String type'");
        this.testDBRefLikeDocument("BinData".getBytes(StandardCharsets.UTF_8), "to_utf8('BinData')");
        this.testDBRefLikeDocument(1234567890, "bigint '1234567890'");
        this.testDBRefLikeDocument(true, "true");
        this.testDBRefLikeDocument(Float.valueOf(12.3f), "double '12.3'");
        this.testDBRefLikeDocument(new Date(0L), "timestamp '1970-01-01 00:00:00.000'");
        this.testDBRefLikeDocument(ImmutableList.of((Object)1), "array[bigint '1']");
        this.testDBRefLikeDocument(new ObjectId("5126bc054aed4daf9e2ab772"), "ObjectId('5126bc054aed4daf9e2ab772')");
        this.testDBRefLikeDocument(TestMongoConnectorTest.dbRefDocument("String type"), TestMongoConnectorTest.documentWithSameDbRefFieldOrder("String type"), "varchar 'String type'");
        this.testDBRefLikeDocument(TestMongoConnectorTest.dbRefDocument("String type"), TestMongoConnectorTest.getDocumentWithDifferentDbRefFieldOrder("String type"), "varchar 'String type'");
        this.testDBRefLikeDocument(TestMongoConnectorTest.documentWithSameDbRefFieldOrder("String type"), TestMongoConnectorTest.dbRefDocument("String type"), "varchar 'String type'");
        this.testDBRefLikeDocument(TestMongoConnectorTest.dbRefDocument("BinData".getBytes(StandardCharsets.UTF_8)), TestMongoConnectorTest.documentWithSameDbRefFieldOrder("BinData".getBytes(StandardCharsets.UTF_8)), "to_utf8('BinData')");
        this.testDBRefLikeDocument(TestMongoConnectorTest.dbRefDocument("BinData".getBytes(StandardCharsets.UTF_8)), TestMongoConnectorTest.getDocumentWithDifferentDbRefFieldOrder("BinData".getBytes(StandardCharsets.UTF_8)), "to_utf8('BinData')");
        this.testDBRefLikeDocument(TestMongoConnectorTest.documentWithSameDbRefFieldOrder("BinData".getBytes(StandardCharsets.UTF_8)), TestMongoConnectorTest.dbRefDocument("BinData".getBytes(StandardCharsets.UTF_8)), "to_utf8('BinData')");
        this.testDBRefLikeDocument(TestMongoConnectorTest.dbRefDocument(1234567890), TestMongoConnectorTest.documentWithSameDbRefFieldOrder(1234567890), "bigint '1234567890'");
        this.testDBRefLikeDocument(TestMongoConnectorTest.dbRefDocument(1234567890), TestMongoConnectorTest.getDocumentWithDifferentDbRefFieldOrder(1234567890), "bigint '1234567890'");
        this.testDBRefLikeDocument(TestMongoConnectorTest.documentWithSameDbRefFieldOrder(1234567890), TestMongoConnectorTest.dbRefDocument(1234567890), "bigint '1234567890'");
        this.testDBRefLikeDocument(TestMongoConnectorTest.dbRefDocument(true), TestMongoConnectorTest.documentWithSameDbRefFieldOrder(true), "true");
        this.testDBRefLikeDocument(TestMongoConnectorTest.dbRefDocument(true), TestMongoConnectorTest.getDocumentWithDifferentDbRefFieldOrder(true), "true");
        this.testDBRefLikeDocument(TestMongoConnectorTest.documentWithSameDbRefFieldOrder(true), TestMongoConnectorTest.dbRefDocument(true), "true");
        this.testDBRefLikeDocument(TestMongoConnectorTest.dbRefDocument(Float.valueOf(12.3f)), TestMongoConnectorTest.documentWithSameDbRefFieldOrder(Float.valueOf(12.3f)), "double '12.3'");
        this.testDBRefLikeDocument(TestMongoConnectorTest.dbRefDocument(Float.valueOf(12.3f)), TestMongoConnectorTest.getDocumentWithDifferentDbRefFieldOrder(Float.valueOf(12.3f)), "double '12.3'");
        this.testDBRefLikeDocument(TestMongoConnectorTest.documentWithSameDbRefFieldOrder(Float.valueOf(12.3f)), TestMongoConnectorTest.dbRefDocument(Float.valueOf(12.3f)), "double '12.3'");
        this.testDBRefLikeDocument(TestMongoConnectorTest.dbRefDocument(new Date(0L)), TestMongoConnectorTest.documentWithSameDbRefFieldOrder(new Date(0L)), "timestamp '1970-01-01 00:00:00.000'");
        this.testDBRefLikeDocument(TestMongoConnectorTest.dbRefDocument(new Date(0L)), TestMongoConnectorTest.getDocumentWithDifferentDbRefFieldOrder(new Date(0L)), "timestamp '1970-01-01 00:00:00.000'");
        this.testDBRefLikeDocument(TestMongoConnectorTest.documentWithSameDbRefFieldOrder(new Date(0L)), TestMongoConnectorTest.dbRefDocument(new Date(0L)), "timestamp '1970-01-01 00:00:00.000'");
        this.testDBRefLikeDocument(TestMongoConnectorTest.dbRefDocument(ImmutableList.of((Object)1)), TestMongoConnectorTest.documentWithSameDbRefFieldOrder(ImmutableList.of((Object)1)), "array[bigint '1']");
        this.testDBRefLikeDocument(TestMongoConnectorTest.dbRefDocument(ImmutableList.of((Object)1)), TestMongoConnectorTest.getDocumentWithDifferentDbRefFieldOrder(ImmutableList.of((Object)1)), "array[bigint '1']");
        this.testDBRefLikeDocument(TestMongoConnectorTest.documentWithSameDbRefFieldOrder(ImmutableList.of((Object)1)), TestMongoConnectorTest.dbRefDocument(ImmutableList.of((Object)1)), "array[bigint '1']");
        this.testDBRefLikeDocument(TestMongoConnectorTest.dbRefDocument(new ObjectId("5126bc054aed4daf9e2ab772")), TestMongoConnectorTest.documentWithSameDbRefFieldOrder(new ObjectId("5126bc054aed4daf9e2ab772")), "ObjectId('5126bc054aed4daf9e2ab772')");
        this.testDBRefLikeDocument(TestMongoConnectorTest.dbRefDocument(new ObjectId("5126bc054aed4daf9e2ab772")), TestMongoConnectorTest.getDocumentWithDifferentDbRefFieldOrder(new ObjectId("5126bc054aed4daf9e2ab772")), "ObjectId('5126bc054aed4daf9e2ab772')");
        this.testDBRefLikeDocument(TestMongoConnectorTest.documentWithSameDbRefFieldOrder(new ObjectId("5126bc054aed4daf9e2ab772")), TestMongoConnectorTest.dbRefDocument(new ObjectId("5126bc054aed4daf9e2ab772")), "ObjectId('5126bc054aed4daf9e2ab772')");
    }

    private void testDBRefLikeDocument(Document document1, Document document2, String expectedValue) {
        String tableName = "test_dbref_like_document_" + TestingNames.randomNameSuffix();
        this.client.getDatabase("test").getCollection(tableName).insertOne((Object)document1);
        this.client.getDatabase("test").getCollection(tableName).insertOne((Object)document2);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM test." + tableName))).skippingTypesCheck().matches("VALUES ROW(ROW(varchar 'dbref_test', varchar 'dbref_creators', " + expectedValue + ")), ROW(ROW(varchar 'doc_test', varchar 'doc_creators', " + expectedValue + "))").isFullyPushedDown();
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT creator.id FROM test." + tableName))).skippingTypesCheck().matches("VALUES (%1$s), (%1$s)".formatted(expectedValue)).isNotFullyPushedDown(ProjectNode.class, new Class[0]);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT creator.databasename, creator.collectionname, creator.id FROM test." + tableName))).skippingTypesCheck().matches("VALUES ('doc_test', 'doc_creators', %1$s), ('dbref_test', 'dbref_creators', %1$s)".formatted(expectedValue)).isNotFullyPushedDown(ProjectNode.class, new Class[0]);
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    private static Document getDocumentWithDifferentDbRefFieldOrder(Object objectId) {
        return new Document().append("_id", (Object)new ObjectId("5126bbf64aed4daf9e2ab771")).append("creator", (Object)new Document().append("collectionName", (Object)"doc_creators").append("id", objectId).append("databaseName", (Object)"doc_test"));
    }

    private static Document documentWithSameDbRefFieldOrder(Object objectId) {
        return new Document().append("_id", (Object)new ObjectId("5126bbf64aed4daf9e2ab771")).append("creator", (Object)new Document().append("databaseName", (Object)"doc_test").append("collectionName", (Object)"doc_creators").append("id", objectId));
    }

    private static Document dbRefDocument(Object objectId) {
        return new Document().append("_id", (Object)new ObjectId("5126bbf64aed4daf9e2ab772")).append("creator", (Object)new DBRef("dbref_test", "dbref_creators", objectId));
    }

    private void testDBRefLikeDocument(Object objectId, String expectedValue) {
        String tableName = "test_dbref_like_document_fails_" + TestingNames.randomNameSuffix();
        Document documentWithDifferentDbRefFieldOrder = new Document().append("_id", (Object)new ObjectId("5126bbf64aed4daf9e2ab771")).append("creator", (Object)new Document().append("databaseName", (Object)"doc_test").append("collectionName", (Object)"doc_creators").append("id", objectId));
        Document dbRefDocument = new Document().append("_id", (Object)new ObjectId("5126bbf64aed4daf9e2ab772")).append("creator", (Object)new DBRef("dbref_test", "dbref_creators", objectId));
        this.client.getDatabase("test").getCollection(tableName).insertOne((Object)documentWithDifferentDbRefFieldOrder);
        this.client.getDatabase("test").getCollection(tableName).insertOne((Object)dbRefDocument);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM test." + tableName))).skippingTypesCheck().matches("VALUES  row(row('doc_test', 'doc_creators', " + expectedValue + ")), row(row('dbref_test', 'dbref_creators', " + expectedValue + "))");
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT creator.id FROM test." + tableName))).skippingTypesCheck().matches("VALUES " + "(%1$s), (%1$s)".formatted(expectedValue));
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT creator.databasename, creator.collectionname, creator.id FROM test." + tableName))).skippingTypesCheck().matches("VALUES " + "('doc_test', 'doc_creators', %1$s), ('dbref_test', 'dbref_creators', %1$s)".formatted(expectedValue));
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testPredicateOnDBRefField() {
        this.testPredicateOnDBRefField(true, "true");
        this.testPredicateOnDBRefField(4, "bigint '4'");
        this.testPredicateOnDBRefField("test", "'test'");
        this.testPredicateOnDBRefField(new ObjectId("6216f0c6c432d45190f25e7c"), "ObjectId('6216f0c6c432d45190f25e7c')");
        this.testPredicateOnDBRefField(new Date(0L), "timestamp '1970-01-01 00:00:00.000'");
    }

    private void testPredicateOnDBRefField(Object objectId, String expectedValue) {
        String tableName = "test_predicate_on_dbref_field_" + TestingNames.randomNameSuffix();
        Document document = new Document().append("_id", (Object)new ObjectId("5126bbf64aed4daf9e2ab771")).append("creator", (Object)new DBRef("test", "creators", objectId));
        this.client.getDatabase("test").getCollection(tableName).insertOne((Object)document);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM test." + tableName + " WHERE creator.id = " + expectedValue))).skippingTypesCheck().matches("SELECT ROW(varchar 'test', varchar 'creators', " + expectedValue + ")").isNotFullyPushedDown(FilterNode.class, new Class[0]);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT creator.id FROM test." + tableName + " WHERE creator.id = " + expectedValue))).skippingTypesCheck().matches("SELECT " + expectedValue).isNotFullyPushedDown(FilterNode.class, new Class[0]);
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testPredicateOnDBRefLikeDocument() {
        this.testPredicateOnDBRefLikeDocument(true, "true");
        this.testPredicateOnDBRefLikeDocument(4, "bigint '4'");
        this.testPredicateOnDBRefLikeDocument("test", "'test'");
        this.testPredicateOnDBRefLikeDocument(new ObjectId("6216f0c6c432d45190f25e7c"), "ObjectId('6216f0c6c432d45190f25e7c')");
        this.testPredicateOnDBRefLikeDocument(new Date(0L), "timestamp '1970-01-01 00:00:00.000'");
    }

    private void testPredicateOnDBRefLikeDocument(Object objectId, String expectedValue) {
        String tableName = "test_predicate_on_dbref_like_document_" + TestingNames.randomNameSuffix();
        Document document = new Document().append("_id", (Object)new ObjectId("5126bbf64aed4daf9e2ab771")).append("creator", (Object)new Document().append("databaseName", (Object)"test").append("collectionName", (Object)"creators").append("id", objectId));
        this.client.getDatabase("test").getCollection(tableName).insertOne((Object)document);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM test." + tableName + " WHERE creator.id = " + expectedValue))).skippingTypesCheck().matches("SELECT ROW(varchar 'test', varchar 'creators', " + expectedValue + ")").isNotFullyPushedDown(FilterNode.class, new Class[0]);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT creator.id FROM test." + tableName + " WHERE creator.id = " + expectedValue))).skippingTypesCheck().matches("SELECT " + expectedValue).isNotFullyPushedDown(FilterNode.class, new Class[0]);
        this.assertUpdate("DROP TABLE test." + tableName);
    }

    @Test
    public void testProjectionPushdownReadsLessData() {
        Assumptions.abort((String)"MongoDB connector does not calculate physical data input size");
    }

    @Test
    public void testProjectionPushdownPhysicalInputSize() {
        Assumptions.abort((String)"MongoDB connector does not calculate physical data input size");
    }

    protected OptionalInt maxSchemaNameLength() {
        return OptionalInt.of(63);
    }

    protected void verifySchemaNameLengthFailurePermissible(Throwable e) {
        Assertions.assertThat((Throwable)e).hasMessageContaining("Invalid database name");
    }

    protected OptionalInt maxTableNameLength() {
        return OptionalInt.of(120 - "tpch.".length());
    }

    protected void verifyTableNameLengthFailurePermissible(Throwable e) {
        Assertions.assertThat((Throwable)e).hasMessageMatching(".*fully qualified namespace .* is too long.*|Qualified identifier name must be shorter than or equal to '120'.*");
    }

    protected void verifySetColumnTypeFailurePermissible(Throwable e) {
        Assertions.assertThat((Throwable)e).hasMessageContaining("Cannot change type");
    }

    protected Optional<BaseConnectorTest.SetColumnTypeSetup> filterSetColumnTypesDataProvider(BaseConnectorTest.SetColumnTypeSetup setup) {
        switch ("%s -> %s".formatted(setup.sourceColumnType(), setup.newColumnType())) {
            case "bigint -> integer": 
            case "decimal(5,3) -> decimal(5,2)": 
            case "time(3) -> time(6)": 
            case "time(6) -> time(3)": 
            case "timestamp(3) -> timestamp(6)": 
            case "timestamp(6) -> timestamp(3)": 
            case "timestamp(3) with time zone -> timestamp(6) with time zone": 
            case "timestamp(6) with time zone -> timestamp(3) with time zone": {
                return Optional.of(setup.asUnsupported());
            }
        }
        return Optional.of(setup);
    }

    private void assertOneNotNullResult(String query) {
        MaterializedResult results = this.getQueryRunner().execute(this.getSession(), query).toTestTypes();
        Assertions.assertThat((int)results.getRowCount()).isEqualTo(1);
        Assertions.assertThat((int)((MaterializedRow)results.getMaterializedRows().get(0)).getFieldCount()).isEqualTo(1);
        Assertions.assertThat((Object)((MaterializedRow)results.getMaterializedRows().get(0)).getField(0)).isNotNull();
    }

    private static /* synthetic */ void lambda$testFiltersOnDereferenceColumnReadsLessData$11(Set expected, MaterializedResult results) {
        Assertions.assertThat((Collection)results.getOnlyColumnAsSet()).isEqualTo((Object)expected);
    }

    private /* synthetic */ void lambda$testFiltersOnDereferenceColumnReadsLessData$8(Session sessionWithoutPushdown, TestTable table, String expectedValue, Set expected, QueryStats statsWithPushdown) {
        long processedInputPositionWithPushdown = statsWithPushdown.getProcessedInputPositions();
        this.assertQueryStats(sessionWithoutPushdown, String.format("SELECT 1 FROM %s WHERE col_0.col_2.col_4.col_5 = %s", table.getName(), expectedValue), statsWithoutPushdown -> {
            Assertions.assertThat((long)statsWithoutPushdown.getProcessedInputPositions()).isEqualTo(3L);
            Assertions.assertThat((long)processedInputPositionWithPushdown).isEqualTo(2L);
            Assertions.assertThat((long)statsWithoutPushdown.getProcessedInputPositions()).isGreaterThan(processedInputPositionWithPushdown);
        }, results -> Assertions.assertThat((Collection)results.getOnlyColumnAsSet()).isEqualTo((Object)expected));
    }

    private static /* synthetic */ void lambda$testFiltersOnDereferenceColumnReadsLessData$7(Set expected, MaterializedResult results) {
        Assertions.assertThat((Collection)results.getOnlyColumnAsSet()).isEqualTo((Object)expected);
    }

    private /* synthetic */ void lambda$testFiltersOnDereferenceColumnReadsLessData$4(Session sessionWithoutPushdown, TestTable table, String expectedValue, Set expected, QueryStats statsWithPushdown) {
        long processedInputPositionWithPushdown = statsWithPushdown.getProcessedInputPositions();
        this.assertQueryStats(sessionWithoutPushdown, String.format("SELECT 1 FROM %s WHERE col_0.col_2.col_3 = %s", table.getName(), expectedValue), statsWithoutPushdown -> {
            Assertions.assertThat((long)statsWithoutPushdown.getProcessedInputPositions()).isEqualTo(3L);
            Assertions.assertThat((long)processedInputPositionWithPushdown).isEqualTo(1L);
            Assertions.assertThat((long)statsWithoutPushdown.getProcessedInputPositions()).isGreaterThan(processedInputPositionWithPushdown);
        }, results -> Assertions.assertThat((Collection)results.getOnlyColumnAsSet()).isEqualTo((Object)expected));
    }

    private static /* synthetic */ void lambda$testFiltersOnDereferenceColumnReadsLessData$3(Set expected, MaterializedResult results) {
        Assertions.assertThat((Collection)results.getOnlyColumnAsSet()).isEqualTo((Object)expected);
    }

    private /* synthetic */ void lambda$testFiltersOnDereferenceColumnReadsLessData$0(Session sessionWithoutPushdown, TestTable table, String expectedValue, Set expected, QueryStats statsWithPushdown) {
        long processedInputPositionWithPushdown = statsWithPushdown.getProcessedInputPositions();
        this.assertQueryStats(sessionWithoutPushdown, String.format("SELECT 1 FROM %s WHERE col_0.col_1 = %s", table.getName(), expectedValue), statsWithoutPushdown -> {
            Assertions.assertThat((long)statsWithoutPushdown.getProcessedInputPositions()).isEqualTo(3L);
            Assertions.assertThat((long)processedInputPositionWithPushdown).isEqualTo(2L);
            Assertions.assertThat((long)statsWithoutPushdown.getProcessedInputPositions()).isGreaterThan(processedInputPositionWithPushdown);
        }, results -> Assertions.assertThat((Collection)results.getOnlyColumnAsSet()).isEqualTo((Object)expected));
    }
}

