/*
 * Decompiled with CFR 0.152.
 */
package io.trino.orc.reader;

import com.google.common.base.MoreObjects;
import com.google.common.base.VerifyException;
import io.trino.memory.context.LocalMemoryContext;
import io.trino.orc.OrcColumn;
import io.trino.orc.OrcCorruptionException;
import io.trino.orc.metadata.ColumnEncoding;
import io.trino.orc.metadata.ColumnMetadata;
import io.trino.orc.metadata.OrcType;
import io.trino.orc.metadata.Stream;
import io.trino.orc.reader.ColumnReader;
import io.trino.orc.reader.ReaderUtils;
import io.trino.orc.stream.BooleanInputStream;
import io.trino.orc.stream.InputStreamSource;
import io.trino.orc.stream.InputStreamSources;
import io.trino.orc.stream.LongInputStream;
import io.trino.orc.stream.MissingInputStreamSource;
import io.trino.spi.block.Block;
import io.trino.spi.block.Int96ArrayBlock;
import io.trino.spi.block.LongArrayBlock;
import io.trino.spi.block.RunLengthEncodedBlock;
import io.trino.spi.type.DateTimeEncoding;
import io.trino.spi.type.TimeZoneKey;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.TimestampWithTimeZoneType;
import io.trino.spi.type.Timestamps;
import io.trino.spi.type.Type;
import java.io.IOException;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Nullable;
import org.joda.time.DateTimeZone;
import org.openjdk.jol.info.ClassLayout;

public class TimestampColumnReader
implements ColumnReader {
    private static final int[] POWERS_OF_TEN = new int[]{1, 10, 100, 1000, 10000, 100000, 1000000, 10000000, 100000000};
    private static final LocalDateTime ORC_EPOCH = LocalDateTime.of(2015, 1, 1, 0, 0, 0, 0);
    private static final long BASE_INSTANT_IN_SECONDS = ORC_EPOCH.toEpochSecond(ZoneOffset.UTC);
    private static final int INSTANCE_SIZE = ClassLayout.parseClass(TimestampColumnReader.class).instanceSize();
    private final Type type;
    private final OrcColumn column;
    private final TimestampKind timestampKind;
    private long baseTimestampInSeconds;
    private DateTimeZone fileDateTimeZone;
    private int readOffset;
    private int nextBatchSize;
    private InputStreamSource<BooleanInputStream> presentStreamSource = MissingInputStreamSource.missingStreamSource(BooleanInputStream.class);
    @Nullable
    private BooleanInputStream presentStream;
    private InputStreamSource<LongInputStream> secondsStreamSource = MissingInputStreamSource.missingStreamSource(LongInputStream.class);
    @Nullable
    private LongInputStream secondsStream;
    private InputStreamSource<LongInputStream> nanosStreamSource = MissingInputStreamSource.missingStreamSource(LongInputStream.class);
    @Nullable
    private LongInputStream nanosStream;
    private boolean rowGroupOpen;
    private final LocalMemoryContext systemMemoryContext;

    public TimestampColumnReader(Type type, OrcColumn column, LocalMemoryContext systemMemoryContext) throws OrcCorruptionException {
        this.type = Objects.requireNonNull(type, "type is null");
        this.column = Objects.requireNonNull(column, "column is null");
        this.timestampKind = TimestampColumnReader.getTimestampKind(type, column);
        this.systemMemoryContext = Objects.requireNonNull(systemMemoryContext, "systemMemoryContext is null");
    }

    private static TimestampKind getTimestampKind(Type type, OrcColumn column) throws OrcCorruptionException {
        if (type.equals(TimestampType.TIMESTAMP_MILLIS) && column.getColumnType() == OrcType.OrcTypeKind.TIMESTAMP) {
            return TimestampKind.TIMESTAMP_MILLIS;
        }
        if (type.equals(TimestampType.TIMESTAMP_MICROS) && column.getColumnType() == OrcType.OrcTypeKind.TIMESTAMP) {
            return TimestampKind.TIMESTAMP_MICROS;
        }
        if (type.equals(TimestampType.TIMESTAMP_NANOS) && column.getColumnType() == OrcType.OrcTypeKind.TIMESTAMP) {
            return TimestampKind.TIMESTAMP_NANOS;
        }
        if (type.equals(TimestampWithTimeZoneType.TIMESTAMP_TZ_MILLIS) && column.getColumnType() == OrcType.OrcTypeKind.TIMESTAMP_INSTANT) {
            return TimestampKind.INSTANT_MILLIS;
        }
        if (type.equals(TimestampWithTimeZoneType.TIMESTAMP_TZ_MICROS) && column.getColumnType() == OrcType.OrcTypeKind.TIMESTAMP_INSTANT) {
            return TimestampKind.INSTANT_MICROS;
        }
        if (type.equals(TimestampWithTimeZoneType.TIMESTAMP_TZ_NANOS) && column.getColumnType() == OrcType.OrcTypeKind.TIMESTAMP_INSTANT) {
            return TimestampKind.INSTANT_NANOS;
        }
        throw ReaderUtils.invalidStreamType(column, type);
    }

    @Override
    public void prepareNextRead(int batchSize) {
        this.readOffset += this.nextBatchSize;
        this.nextBatchSize = batchSize;
    }

    @Override
    public Block readBlock() throws IOException {
        Block block;
        if (!this.rowGroupOpen) {
            this.openRowGroup();
        }
        if (this.readOffset > 0) {
            if (this.presentStream != null) {
                this.readOffset = this.presentStream.countBitsSet(this.readOffset);
            }
            if (this.readOffset > 0) {
                this.verifyStreamsPresent();
                this.secondsStream.skip(this.readOffset);
                this.nanosStream.skip(this.readOffset);
            }
        }
        if (this.secondsStream == null && this.nanosStream == null) {
            if (this.presentStream == null) {
                throw new OrcCorruptionException(this.column.getOrcDataSourceId(), "Value is null but present stream is missing");
            }
            this.presentStream.skip(this.nextBatchSize);
            block = RunLengthEncodedBlock.create((Type)this.type, null, (int)this.nextBatchSize);
        } else {
            boolean[] isNull;
            int nullCount;
            block = this.presentStream == null ? this.readNonNullBlock() : ((nullCount = this.presentStream.getUnsetBits(this.nextBatchSize, isNull = new boolean[this.nextBatchSize])) == 0 ? this.readNonNullBlock() : (nullCount != this.nextBatchSize ? this.readNullBlock(isNull) : RunLengthEncodedBlock.create((Type)this.type, null, (int)this.nextBatchSize)));
        }
        this.readOffset = 0;
        this.nextBatchSize = 0;
        return block;
    }

    private void verifyStreamsPresent() throws OrcCorruptionException {
        if (this.secondsStream == null) {
            throw new OrcCorruptionException(this.column.getOrcDataSourceId(), "Value is not null but seconds stream is missing");
        }
        if (this.nanosStream == null) {
            throw new OrcCorruptionException(this.column.getOrcDataSourceId(), "Value is not null but nanos stream is missing");
        }
    }

    private Block readNonNullBlock() throws IOException {
        this.verifyStreamsPresent();
        switch (this.timestampKind) {
            case TIMESTAMP_MILLIS: {
                return this.readNonNullTimestampMillis();
            }
            case TIMESTAMP_MICROS: {
                return this.readNonNullTimestampMicros();
            }
            case TIMESTAMP_NANOS: {
                return this.readNonNullTimestampNanos();
            }
            case INSTANT_MILLIS: {
                return this.readNonNullInstantMillis();
            }
            case INSTANT_MICROS: {
                return this.readNonNullInstantMicros();
            }
            case INSTANT_NANOS: {
                return this.readNonNullInstantNanos();
            }
        }
        throw new VerifyException("Unhandled timestmap kind: " + this.timestampKind);
    }

    private Block readNullBlock(boolean[] isNull) throws IOException {
        this.verifyStreamsPresent();
        switch (this.timestampKind) {
            case TIMESTAMP_MILLIS: {
                return this.readNullTimestampMillis(isNull);
            }
            case TIMESTAMP_MICROS: {
                return this.readNullTimestampMicros(isNull);
            }
            case TIMESTAMP_NANOS: {
                return this.readNullTimestampNanos(isNull);
            }
            case INSTANT_MILLIS: {
                return this.readNullInstantMillis(isNull);
            }
            case INSTANT_MICROS: {
                return this.readNullInstantMicros(isNull);
            }
            case INSTANT_NANOS: {
                return this.readNullInstantNanos(isNull);
            }
        }
        throw new VerifyException("Unhandled timestamp kind: " + this.timestampKind);
    }

    private void openRowGroup() throws IOException {
        this.presentStream = this.presentStreamSource.openStream();
        this.secondsStream = this.secondsStreamSource.openStream();
        this.nanosStream = this.nanosStreamSource.openStream();
        this.rowGroupOpen = true;
    }

    @Override
    public void startStripe(ZoneId fileTimeZone, InputStreamSources dictionaryStreamSources, ColumnMetadata<ColumnEncoding> encoding) {
        this.baseTimestampInSeconds = ZonedDateTime.ofLocal(ORC_EPOCH, fileTimeZone, null).toEpochSecond();
        this.fileDateTimeZone = DateTimeZone.forID((String)fileTimeZone.getId());
        this.presentStreamSource = MissingInputStreamSource.missingStreamSource(BooleanInputStream.class);
        this.secondsStreamSource = MissingInputStreamSource.missingStreamSource(LongInputStream.class);
        this.nanosStreamSource = MissingInputStreamSource.missingStreamSource(LongInputStream.class);
        this.readOffset = 0;
        this.nextBatchSize = 0;
        this.presentStream = null;
        this.secondsStream = null;
        this.nanosStream = null;
        this.rowGroupOpen = false;
    }

    @Override
    public void startRowGroup(InputStreamSources dataStreamSources) {
        this.presentStreamSource = dataStreamSources.getInputStreamSource(this.column, Stream.StreamKind.PRESENT, BooleanInputStream.class);
        this.secondsStreamSource = dataStreamSources.getInputStreamSource(this.column, Stream.StreamKind.DATA, LongInputStream.class);
        this.nanosStreamSource = dataStreamSources.getInputStreamSource(this.column, Stream.StreamKind.SECONDARY, LongInputStream.class);
        this.readOffset = 0;
        this.nextBatchSize = 0;
        this.presentStream = null;
        this.secondsStream = null;
        this.nanosStream = null;
        this.rowGroupOpen = false;
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).addValue((Object)this.column).toString();
    }

    @Override
    public void close() {
        this.systemMemoryContext.close();
    }

    @Override
    public long getRetainedSizeInBytes() {
        return INSTANCE_SIZE;
    }

    private boolean isFileUtc() {
        return this.fileDateTimeZone == DateTimeZone.UTC;
    }

    private int decodeNanos(long serialized) throws IOException {
        int zeros = (int)(serialized & 7L);
        int nanos = (int)(serialized >>> 3);
        if (zeros > 0) {
            nanos *= POWERS_OF_TEN[zeros + 1];
        }
        if (nanos < 0 || nanos > 999999999) {
            throw new OrcCorruptionException(this.column.getOrcDataSourceId(), "Nanos field of timestamp is out of range: %s", nanos);
        }
        return nanos;
    }

    private Block readNonNullTimestampMillis() throws IOException {
        long[] millis = new long[this.nextBatchSize];
        for (int i = 0; i < this.nextBatchSize; ++i) {
            millis[i] = this.readTimestampMillis();
        }
        return new LongArrayBlock(this.nextBatchSize, Optional.empty(), millis);
    }

    private Block readNullTimestampMillis(boolean[] isNull) throws IOException {
        long[] millis = new long[isNull.length];
        for (int i = 0; i < isNull.length; ++i) {
            if (isNull[i]) continue;
            millis[i] = this.readTimestampMillis();
        }
        return new LongArrayBlock(isNull.length, Optional.of(isNull), millis);
    }

    private long readTimestampMillis() throws IOException {
        long seconds = this.secondsStream.next();
        long serializedNanos = this.nanosStream.next();
        long millis = (seconds + this.baseTimestampInSeconds) * 1000L;
        long nanos = this.decodeNanos(serializedNanos);
        if (nanos != 0L) {
            if (millis < 0L) {
                millis -= 1000L;
            }
            millis += Timestamps.roundDiv((long)nanos, (long)1000000L);
        }
        if (!this.isFileUtc()) {
            millis = this.fileDateTimeZone.convertUTCToLocal(millis);
        }
        return millis * 1000L;
    }

    private Block readNonNullTimestampMicros() throws IOException {
        long[] micros = new long[this.nextBatchSize];
        for (int i = 0; i < this.nextBatchSize; ++i) {
            micros[i] = this.readTimestampMicros();
        }
        return new LongArrayBlock(this.nextBatchSize, Optional.empty(), micros);
    }

    private Block readNullTimestampMicros(boolean[] isNull) throws IOException {
        long[] micros = new long[this.nextBatchSize];
        for (int i = 0; i < this.nextBatchSize; ++i) {
            if (isNull[i]) continue;
            micros[i] = this.readTimestampMicros();
        }
        return new LongArrayBlock(this.nextBatchSize, Optional.of(isNull), micros);
    }

    private long readTimestampMicros() throws IOException {
        long seconds = this.secondsStream.next();
        long serializedNanos = this.nanosStream.next();
        long micros = (seconds + this.baseTimestampInSeconds) * 1000000L;
        long nanos = this.decodeNanos(serializedNanos);
        if (nanos != 0L) {
            if (micros < 0L) {
                micros -= 1000000L;
            }
            micros += Timestamps.roundDiv((long)nanos, (long)1000L);
        }
        if (!this.isFileUtc()) {
            long millis = Math.floorDiv(micros, 1000);
            int microsFraction = Math.floorMod(micros, 1000);
            millis = this.fileDateTimeZone.convertUTCToLocal(millis);
            micros = millis * 1000L + (long)microsFraction;
        }
        return micros;
    }

    private Block readNonNullTimestampNanos() throws IOException {
        long[] microsValues = new long[this.nextBatchSize];
        int[] picosFractionValues = new int[this.nextBatchSize];
        for (int i = 0; i < this.nextBatchSize; ++i) {
            this.readTimestampNanos(i, microsValues, picosFractionValues);
        }
        return new Int96ArrayBlock(this.nextBatchSize, Optional.empty(), microsValues, picosFractionValues);
    }

    private Block readNullTimestampNanos(boolean[] isNull) throws IOException {
        long[] microsValues = new long[this.nextBatchSize];
        int[] picosFractionValues = new int[this.nextBatchSize];
        for (int i = 0; i < this.nextBatchSize; ++i) {
            if (isNull[i]) continue;
            this.readTimestampNanos(i, microsValues, picosFractionValues);
        }
        return new Int96ArrayBlock(this.nextBatchSize, Optional.of(isNull), microsValues, picosFractionValues);
    }

    private void readTimestampNanos(int i, long[] microsValues, int[] picosFractionValues) throws IOException {
        long seconds = this.secondsStream.next();
        long serializedNanos = this.nanosStream.next();
        long micros = (seconds + this.baseTimestampInSeconds) * 1000000L;
        long nanos = this.decodeNanos(serializedNanos);
        int picosFraction = 0;
        if (nanos != 0L) {
            if (micros < 0L) {
                micros -= 1000000L;
            }
            micros += nanos / 1000L;
            picosFraction = Math.toIntExact((nanos %= 1000L) * 1000L);
        }
        if (!this.isFileUtc()) {
            long millis = Math.floorDiv(micros, 1000);
            int microsFraction = Math.floorMod(micros, 1000);
            millis = this.fileDateTimeZone.convertUTCToLocal(millis);
            micros = millis * 1000L + (long)microsFraction;
        }
        microsValues[i] = micros;
        picosFractionValues[i] = picosFraction;
    }

    private Block readNonNullInstantMillis() throws IOException {
        long[] millis = new long[this.nextBatchSize];
        for (int i = 0; i < this.nextBatchSize; ++i) {
            millis[i] = this.readInstantMillis();
        }
        return new LongArrayBlock(this.nextBatchSize, Optional.empty(), millis);
    }

    private Block readNullInstantMillis(boolean[] isNull) throws IOException {
        long[] millis = new long[this.nextBatchSize];
        for (int i = 0; i < this.nextBatchSize; ++i) {
            if (isNull[i]) continue;
            millis[i] = this.readInstantMillis();
        }
        return new LongArrayBlock(this.nextBatchSize, Optional.of(isNull), millis);
    }

    private long readInstantMillis() throws IOException {
        long seconds = this.secondsStream.next();
        long serializedNanos = this.nanosStream.next();
        long millis = (seconds + BASE_INSTANT_IN_SECONDS) * 1000L;
        long nanos = this.decodeNanos(serializedNanos);
        if (nanos != 0L) {
            if (millis < 0L) {
                millis -= 1000L;
            }
            millis += Timestamps.roundDiv((long)nanos, (long)1000000L);
        }
        return DateTimeEncoding.packDateTimeWithZone((long)millis, (TimeZoneKey)TimeZoneKey.UTC_KEY);
    }

    private Block readNonNullInstantMicros() throws IOException {
        long[] millisValues = new long[this.nextBatchSize];
        int[] picosFractionValues = new int[this.nextBatchSize];
        for (int i = 0; i < this.nextBatchSize; ++i) {
            this.readInstantMicros(i, millisValues, picosFractionValues);
        }
        return new Int96ArrayBlock(this.nextBatchSize, Optional.empty(), millisValues, picosFractionValues);
    }

    private Block readNullInstantMicros(boolean[] isNull) throws IOException {
        long[] millisValues = new long[this.nextBatchSize];
        int[] picosFractionValues = new int[this.nextBatchSize];
        for (int i = 0; i < this.nextBatchSize; ++i) {
            if (isNull[i]) continue;
            this.readInstantMicros(i, millisValues, picosFractionValues);
        }
        return new Int96ArrayBlock(this.nextBatchSize, Optional.of(isNull), millisValues, picosFractionValues);
    }

    private void readInstantMicros(int i, long[] millisValues, int[] picosFractionValues) throws IOException {
        long seconds = this.secondsStream.next();
        long serializedNanos = this.nanosStream.next();
        long millis = (seconds + BASE_INSTANT_IN_SECONDS) * 1000L;
        long nanos = this.decodeNanos(serializedNanos);
        int picosFraction = 0;
        if (nanos != 0L) {
            if (millis < 0L) {
                millis -= 1000L;
            }
            millis += nanos / 1000000L;
            picosFraction = Math.toIntExact(Timestamps.roundDiv((long)(nanos %= 1000000L), (long)1000L)) * 1000000;
        }
        millisValues[i] = DateTimeEncoding.packDateTimeWithZone((long)millis, (TimeZoneKey)TimeZoneKey.UTC_KEY);
        picosFractionValues[i] = picosFraction;
    }

    private Block readNonNullInstantNanos() throws IOException {
        long[] millisValues = new long[this.nextBatchSize];
        int[] picosFractionValues = new int[this.nextBatchSize];
        for (int i = 0; i < this.nextBatchSize; ++i) {
            this.readInstantNanos(i, millisValues, picosFractionValues);
        }
        return new Int96ArrayBlock(this.nextBatchSize, Optional.empty(), millisValues, picosFractionValues);
    }

    private Block readNullInstantNanos(boolean[] isNull) throws IOException {
        long[] millisValues = new long[this.nextBatchSize];
        int[] picosFractionValues = new int[this.nextBatchSize];
        for (int i = 0; i < this.nextBatchSize; ++i) {
            if (isNull[i]) continue;
            this.readInstantNanos(i, millisValues, picosFractionValues);
        }
        return new Int96ArrayBlock(this.nextBatchSize, Optional.of(isNull), millisValues, picosFractionValues);
    }

    private void readInstantNanos(int i, long[] millisValues, int[] picosFractionValues) throws IOException {
        long seconds = this.secondsStream.next();
        long serializedNanos = this.nanosStream.next();
        long millis = (seconds + BASE_INSTANT_IN_SECONDS) * 1000L;
        long nanos = this.decodeNanos(serializedNanos);
        int picosFraction = 0;
        if (nanos != 0L) {
            if (millis < 0L) {
                millis -= 1000L;
            }
            millis += nanos / 1000000L;
            picosFraction = Math.toIntExact((nanos %= 1000000L) * 1000L);
        }
        millisValues[i] = DateTimeEncoding.packDateTimeWithZone((long)millis, (TimeZoneKey)TimeZoneKey.UTC_KEY);
        picosFractionValues[i] = picosFraction;
    }

    private static enum TimestampKind {
        TIMESTAMP_MILLIS,
        TIMESTAMP_MICROS,
        TIMESTAMP_NANOS,
        INSTANT_MILLIS,
        INSTANT_MICROS,
        INSTANT_NANOS;

    }
}

