/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.trino.orc.reader;

import io.trino.memory.context.AggregatedMemoryContext;
import io.trino.orc.OrcBlockFactory;
import io.trino.orc.OrcColumn;
import io.trino.orc.OrcCorruptionException;
import io.trino.orc.OrcReader;
import io.trino.orc.OrcReader.FieldMapperFactory;
import io.trino.spi.type.TimeType;
import io.trino.spi.type.Type;

import static io.trino.orc.metadata.OrcType.OrcTypeKind.LONG;
import static io.trino.orc.reader.ReaderUtils.invalidStreamType;
import static io.trino.spi.type.TimeType.TIME_MICROS;

public final class ColumnReaders
{
    private ColumnReaders() {}

    public static ColumnReader createColumnReader(
            Type type,
            OrcColumn column,
            OrcReader.ProjectedLayout projectedLayout,
            AggregatedMemoryContext systemMemoryContext,
            OrcBlockFactory blockFactory,
            FieldMapperFactory fieldMapperFactory)
            throws OrcCorruptionException
    {
        if (type instanceof TimeType) {
            if (!type.equals(TIME_MICROS) || column.getColumnType() != LONG ||
                    !"TIME".equals(column.getAttributes().get("iceberg.long-type"))) {
                throw invalidStreamType(column, type);
            }
            return new TimeColumnReader(type, column, systemMemoryContext.newLocalMemoryContext(ColumnReaders.class.getSimpleName()));
        }

        switch (column.getColumnType()) {
            case BOOLEAN:
                return new BooleanColumnReader(type, column, systemMemoryContext.newLocalMemoryContext(ColumnReaders.class.getSimpleName()));
            case BYTE:
                return new ByteColumnReader(type, column, systemMemoryContext.newLocalMemoryContext(ColumnReaders.class.getSimpleName()));
            case SHORT:
            case INT:
            case LONG:
            case DATE:
                return new LongColumnReader(type, column, systemMemoryContext.newLocalMemoryContext(ColumnReaders.class.getSimpleName()));
            case FLOAT:
                return new FloatColumnReader(type, column, systemMemoryContext.newLocalMemoryContext(ColumnReaders.class.getSimpleName()));
            case DOUBLE:
                return new DoubleColumnReader(type, column, systemMemoryContext.newLocalMemoryContext(ColumnReaders.class.getSimpleName()));
            case BINARY:
            case STRING:
            case VARCHAR:
            case CHAR:
                return new SliceColumnReader(type, column, systemMemoryContext);
            case TIMESTAMP:
            case TIMESTAMP_INSTANT:
                return new TimestampColumnReader(type, column, systemMemoryContext.newLocalMemoryContext(ColumnReaders.class.getSimpleName()));
            case LIST:
                return new ListColumnReader(type, column, systemMemoryContext, blockFactory, fieldMapperFactory);
            case STRUCT:
                return new StructColumnReader(type, column, projectedLayout, systemMemoryContext, blockFactory, fieldMapperFactory);
            case MAP:
                return new MapColumnReader(type, column, systemMemoryContext, blockFactory, fieldMapperFactory);
            case DECIMAL:
                return new DecimalColumnReader(type, column, systemMemoryContext.newLocalMemoryContext(ColumnReaders.class.getSimpleName()));
            case UNION:
                return new UnionColumnReader(type, column, systemMemoryContext, blockFactory, fieldMapperFactory);
        }
        throw new IllegalArgumentException("Unsupported type: " + column.getColumnType());
    }
}
