/*
 * Decompiled with CFR 0.152.
 */
package io.trino.orc;

import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.primitives.UnsignedBytes;
import io.airlift.slice.SizeOf;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.trino.orc.DictionaryCompressionOptimizer;
import io.trino.orc.OrcCorruptionException;
import io.trino.orc.OrcDataSink;
import io.trino.orc.OrcDataSource;
import io.trino.orc.OrcReader;
import io.trino.orc.OrcWriteValidation;
import io.trino.orc.OrcWriterOptions;
import io.trino.orc.OrcWriterStats;
import io.trino.orc.metadata.ColumnEncoding;
import io.trino.orc.metadata.ColumnMetadata;
import io.trino.orc.metadata.CompressedMetadataWriter;
import io.trino.orc.metadata.CompressionKind;
import io.trino.orc.metadata.Footer;
import io.trino.orc.metadata.Metadata;
import io.trino.orc.metadata.OrcColumnId;
import io.trino.orc.metadata.OrcMetadataWriter;
import io.trino.orc.metadata.OrcType;
import io.trino.orc.metadata.PostScript;
import io.trino.orc.metadata.Stream;
import io.trino.orc.metadata.StripeFooter;
import io.trino.orc.metadata.StripeInformation;
import io.trino.orc.metadata.statistics.BloomFilterBuilder;
import io.trino.orc.metadata.statistics.ColumnStatistics;
import io.trino.orc.metadata.statistics.NoOpBloomFilterBuilder;
import io.trino.orc.metadata.statistics.StripeStatistics;
import io.trino.orc.metadata.statistics.Utf8BloomFilterBuilder;
import io.trino.orc.stream.OrcDataOutput;
import io.trino.orc.stream.StreamDataOutput;
import io.trino.orc.writer.ColumnWriter;
import io.trino.orc.writer.ColumnWriters;
import io.trino.orc.writer.SliceDictionaryColumnWriter;
import io.trino.spi.Page;
import io.trino.spi.type.Type;
import jakarta.annotation.Nullable;
import java.io.Closeable;
import java.io.IOException;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Collectors;

public final class OrcWriter
implements Closeable {
    private static final int INSTANCE_SIZE = SizeOf.instanceSize(OrcWriter.class);
    private static final String TRINO_ORC_WRITER_VERSION_METADATA_KEY = "trino.writer.version";
    private static final String TRINO_ORC_WRITER_VERSION;
    private final OrcWriterStats stats;
    private final OrcDataSink orcDataSink;
    private final List<Type> types;
    private final CompressionKind compression;
    private final int stripeMaxBytes;
    private final int chunkMaxLogicalBytes;
    private final int stripeMaxRowCount;
    private final int rowGroupMaxRowCount;
    private final int maxCompressionBufferSize;
    private final Map<String, String> userMetadata = new HashMap<String, String>();
    private final CompressedMetadataWriter metadataWriter;
    private final List<ClosedStripe> closedStripes = new ArrayList<ClosedStripe>();
    private final ColumnMetadata<OrcType> orcTypes;
    private final List<ColumnWriter> columnWriters;
    private final DictionaryCompressionOptimizer dictionaryCompressionOptimizer;
    private int stripeRowCount;
    private int rowGroupRowCount;
    private long bufferedBytes;
    private long columnWritersRetainedBytes;
    private long closedStripesRetainedBytes;
    private long previouslyRecordedSizeInBytes;
    private boolean closed;
    private long fileRowCount;
    private Optional<ColumnMetadata<ColumnStatistics>> fileStats;
    private List<Long> stripeOffsets;
    private long fileStatsRetainedBytes;
    @Nullable
    private final OrcWriteValidation.OrcWriteValidationBuilder validationBuilder;

    public OrcWriter(OrcDataSink orcDataSink, List<String> columnNames, List<Type> types, ColumnMetadata<OrcType> orcTypes, CompressionKind compression, OrcWriterOptions options, Map<String, String> userMetadata, boolean validate, OrcWriteValidation.OrcWriteValidationMode validationMode, OrcWriterStats stats) {
        this.validationBuilder = validate ? new OrcWriteValidation.OrcWriteValidationBuilder(validationMode, types).setStringStatisticsLimitInBytes(Math.toIntExact(options.getMaxStringStatisticsLimit().toBytes())) : null;
        this.orcDataSink = Objects.requireNonNull(orcDataSink, "orcDataSink is null");
        this.types = ImmutableList.copyOf((Collection)Objects.requireNonNull(types, "types is null"));
        this.compression = Objects.requireNonNull(compression, "compression is null");
        this.recordValidation(validation -> validation.setCompression(compression));
        this.recordValidation(validation -> validation.setTimeZone(ZoneId.of("UTC")));
        Objects.requireNonNull(options, "options is null");
        Preconditions.checkArgument((options.getStripeMaxSize().compareTo(options.getStripeMinSize()) >= 0 ? 1 : 0) != 0, (Object)"stripeMaxSize must be greater than or equal to stripeMinSize");
        int stripeMinBytes = Math.toIntExact(Objects.requireNonNull(options.getStripeMinSize(), "stripeMinSize is null").toBytes());
        this.stripeMaxBytes = Math.toIntExact(Objects.requireNonNull(options.getStripeMaxSize(), "stripeMaxSize is null").toBytes());
        this.chunkMaxLogicalBytes = Math.max(1, this.stripeMaxBytes / 2);
        this.stripeMaxRowCount = options.getStripeMaxRowCount();
        this.rowGroupMaxRowCount = options.getRowGroupMaxRowCount();
        this.recordValidation(validation -> validation.setRowGroupMaxRowCount(this.rowGroupMaxRowCount));
        this.maxCompressionBufferSize = Math.toIntExact(options.getMaxCompressionBufferSize().toBytes());
        this.userMetadata.putAll(Objects.requireNonNull(userMetadata, "userMetadata is null"));
        this.userMetadata.put(TRINO_ORC_WRITER_VERSION_METADATA_KEY, TRINO_ORC_WRITER_VERSION);
        this.metadataWriter = new CompressedMetadataWriter(new OrcMetadataWriter(options.getWriterIdentification()), compression, this.maxCompressionBufferSize);
        this.stats = Objects.requireNonNull(stats, "stats is null");
        Objects.requireNonNull(columnNames, "columnNames is null");
        this.orcTypes = Objects.requireNonNull(orcTypes, "orcTypes is null");
        this.recordValidation(validation -> validation.setColumnNames(columnNames));
        OrcType rootType = orcTypes.get(OrcColumnId.ROOT_COLUMN);
        Preconditions.checkArgument((rootType.getFieldCount() == types.size() ? 1 : 0) != 0);
        ImmutableList.Builder columnWriters = ImmutableList.builder();
        ImmutableSet.Builder sliceColumnWriters = ImmutableSet.builder();
        for (int fieldId = 0; fieldId < types.size(); ++fieldId) {
            OrcColumnId fieldColumnIndex = rootType.getFieldTypeIndex(fieldId);
            Type fieldType = types.get(fieldId);
            ColumnWriter columnWriter = ColumnWriters.createColumnWriter(fieldColumnIndex, orcTypes, fieldType, compression, this.maxCompressionBufferSize, options.getMaxStringStatisticsLimit(), OrcWriter.getBloomFilterBuilder(options, columnNames.get(fieldId)), options.isShouldCompactMinMax());
            columnWriters.add((Object)columnWriter);
            if (columnWriter instanceof SliceDictionaryColumnWriter) {
                sliceColumnWriters.add((Object)((SliceDictionaryColumnWriter)columnWriter));
                continue;
            }
            for (ColumnWriter nestedColumnWriter : columnWriter.getNestedColumnWriters()) {
                if (!(nestedColumnWriter instanceof SliceDictionaryColumnWriter)) continue;
                sliceColumnWriters.add((Object)((SliceDictionaryColumnWriter)nestedColumnWriter));
            }
        }
        this.columnWriters = columnWriters.build();
        this.dictionaryCompressionOptimizer = new DictionaryCompressionOptimizer((Set<? extends DictionaryCompressionOptimizer.DictionaryColumn>)sliceColumnWriters.build(), stripeMinBytes, this.stripeMaxBytes, this.stripeMaxRowCount, Math.toIntExact(Objects.requireNonNull(options.getDictionaryMaxMemory(), "dictionaryMaxMemory is null").toBytes()));
        for (Map.Entry<String, String> entry : this.userMetadata.entrySet()) {
            this.recordValidation(validation -> validation.addMetadataProperty((String)entry.getKey(), Slices.utf8Slice((String)((String)entry.getValue()))));
        }
        this.previouslyRecordedSizeInBytes = this.getRetainedBytes();
        stats.updateSizeInBytes(this.previouslyRecordedSizeInBytes);
    }

    public long getWrittenBytes() {
        return this.orcDataSink.size();
    }

    public long getBufferedBytes() {
        return this.bufferedBytes;
    }

    public int getStripeRowCount() {
        return this.stripeRowCount;
    }

    public long getRetainedBytes() {
        return (long)INSTANCE_SIZE + this.columnWritersRetainedBytes + this.closedStripesRetainedBytes + this.orcDataSink.getRetainedSizeInBytes() + (this.validationBuilder == null ? 0L : this.validationBuilder.getRetainedSize()) + this.fileStatsRetainedBytes;
    }

    public void write(Page page) throws IOException {
        Objects.requireNonNull(page, "page is null");
        if (page.getPositionCount() == 0) {
            return;
        }
        Preconditions.checkArgument((page.getChannelCount() == this.columnWriters.size() ? 1 : 0) != 0);
        if (this.validationBuilder != null) {
            this.validationBuilder.addPage(page);
        }
        int writeOffset = 0;
        while (writeOffset < page.getPositionCount()) {
            Page chunk = page.getRegion(writeOffset, Integer.min(page.getPositionCount() - writeOffset, Integer.min(this.rowGroupMaxRowCount - this.rowGroupRowCount, this.stripeMaxRowCount - this.stripeRowCount)));
            while (chunk.getPositionCount() > 1 && chunk.getLogicalSizeInBytes() > (long)this.chunkMaxLogicalBytes) {
                chunk = page.getRegion(writeOffset, chunk.getPositionCount() / 2);
            }
            writeOffset += chunk.getPositionCount();
            this.writeChunk(chunk);
            this.fileRowCount += (long)chunk.getPositionCount();
        }
        long recordedSizeInBytes = this.getRetainedBytes();
        this.stats.updateSizeInBytes(recordedSizeInBytes - this.previouslyRecordedSizeInBytes);
        this.previouslyRecordedSizeInBytes = recordedSizeInBytes;
    }

    private void writeChunk(Page chunk) throws IOException {
        if (this.rowGroupRowCount == 0) {
            this.columnWriters.forEach(ColumnWriter::beginRowGroup);
        }
        this.bufferedBytes = 0L;
        for (int channel = 0; channel < chunk.getChannelCount(); ++channel) {
            ColumnWriter writer = this.columnWriters.get(channel);
            writer.writeBlock(chunk.getBlock(channel));
            this.bufferedBytes += writer.getBufferedBytes();
        }
        this.rowGroupRowCount += chunk.getPositionCount();
        Preconditions.checkState((this.rowGroupRowCount <= this.rowGroupMaxRowCount ? 1 : 0) != 0);
        this.stripeRowCount += chunk.getPositionCount();
        if (this.rowGroupRowCount == this.rowGroupMaxRowCount) {
            this.finishRowGroup();
        }
        this.dictionaryCompressionOptimizer.optimize(this.bufferedBytes, this.stripeRowCount);
        this.bufferedBytes = Math.toIntExact(this.columnWriters.stream().mapToLong(ColumnWriter::getBufferedBytes).sum());
        if (this.stripeRowCount == this.stripeMaxRowCount) {
            this.flushStripe(OrcWriterStats.FlushReason.MAX_ROWS);
        } else if (this.bufferedBytes > (long)this.stripeMaxBytes) {
            this.flushStripe(OrcWriterStats.FlushReason.MAX_BYTES);
        } else if (this.dictionaryCompressionOptimizer.isFull(this.bufferedBytes)) {
            this.flushStripe(OrcWriterStats.FlushReason.DICTIONARY_FULL);
        }
        this.columnWritersRetainedBytes = this.columnWriters.stream().mapToLong(ColumnWriter::getRetainedBytes).sum();
    }

    private void finishRowGroup() {
        HashMap columnStatistics = new HashMap();
        this.columnWriters.forEach(columnWriter -> columnStatistics.putAll(columnWriter.finishRowGroup()));
        this.recordValidation(validation -> validation.addRowGroupStatistics(columnStatistics));
        this.rowGroupRowCount = 0;
    }

    private void flushStripe(OrcWriterStats.FlushReason flushReason) throws IOException {
        ArrayList<OrcDataOutput> outputData = new ArrayList<OrcDataOutput>();
        long stripeStartOffset = this.orcDataSink.size();
        if (this.closedStripes.isEmpty()) {
            outputData.add(OrcDataOutput.createDataOutput(PostScript.MAGIC));
            stripeStartOffset += (long)PostScript.MAGIC.length();
        }
        outputData.addAll(this.bufferStripeData(stripeStartOffset, flushReason));
        if (flushReason == OrcWriterStats.FlushReason.CLOSED) {
            outputData.addAll(this.bufferFileFooter());
        }
        this.orcDataSink.write(outputData);
        this.columnWriters.forEach(ColumnWriter::reset);
        this.dictionaryCompressionOptimizer.reset();
        this.rowGroupRowCount = 0;
        this.stripeRowCount = 0;
        this.bufferedBytes = Math.toIntExact(this.columnWriters.stream().mapToLong(ColumnWriter::getBufferedBytes).sum());
    }

    private List<OrcDataOutput> bufferStripeData(long stripeStartOffset, OrcWriterStats.FlushReason flushReason) throws IOException {
        if (this.stripeRowCount == 0) {
            Verify.verify((flushReason == OrcWriterStats.FlushReason.CLOSED ? 1 : 0) != 0, (String)"An empty stripe is not allowed", (Object[])new Object[0]);
            this.columnWriters.forEach(ColumnWriter::close);
            return ImmutableList.of();
        }
        if (this.rowGroupRowCount > 0) {
            this.finishRowGroup();
        }
        this.dictionaryCompressionOptimizer.finalOptimize(Math.toIntExact(this.bufferedBytes));
        this.columnWriters.forEach(ColumnWriter::close);
        ArrayList<OrcDataOutput> outputData = new ArrayList<OrcDataOutput>();
        ArrayList<Stream> allStreams = new ArrayList<Stream>(this.columnWriters.size() * 3);
        long indexLength = 0L;
        for (ColumnWriter columnWriter2 : this.columnWriters) {
            for (StreamDataOutput streamDataOutput : columnWriter2.getIndexStreams(this.metadataWriter)) {
                outputData.add(streamDataOutput);
                allStreams.add(streamDataOutput.getStream());
                indexLength += streamDataOutput.size();
            }
            for (StreamDataOutput streamDataOutput : columnWriter2.getBloomFilters(this.metadataWriter)) {
                outputData.add(streamDataOutput);
                allStreams.add(streamDataOutput.getStream());
                indexLength += streamDataOutput.size();
            }
        }
        long dataLength = 0L;
        ArrayList<StreamDataOutput> dataStreams = new ArrayList<StreamDataOutput>(this.columnWriters.size() * 2);
        for (ColumnWriter columnWriter3 : this.columnWriters) {
            List<StreamDataOutput> streams = columnWriter3.getDataStreams();
            dataStreams.addAll(streams);
            dataLength += streams.stream().mapToLong(StreamDataOutput::size).sum();
        }
        Collections.sort(dataStreams);
        for (StreamDataOutput dataStream : dataStreams) {
            outputData.add(dataStream);
            allStreams.add(dataStream.getStream());
        }
        HashMap<OrcColumnId, ColumnEncoding> hashMap = new HashMap<OrcColumnId, ColumnEncoding>();
        this.columnWriters.forEach(columnWriter -> columnEncodings.putAll(columnWriter.getColumnEncodings()));
        HashMap<OrcColumnId, ColumnStatistics> columnStatistics = new HashMap<OrcColumnId, ColumnStatistics>();
        this.columnWriters.forEach(columnWriter -> columnStatistics.putAll(columnWriter.getColumnStripeStatistics()));
        hashMap.put(OrcColumnId.ROOT_COLUMN, new ColumnEncoding(ColumnEncoding.ColumnEncodingKind.DIRECT, 0));
        columnStatistics.put(OrcColumnId.ROOT_COLUMN, new ColumnStatistics(Long.valueOf(this.stripeRowCount), 0L, null, null, null, null, null, null, null, null, null, null));
        StripeFooter stripeFooter = new StripeFooter(allStreams, OrcWriter.toColumnMetadata(hashMap, this.orcTypes.size()), ZoneId.of("UTC"));
        Slice footer = this.metadataWriter.writeStripeFooter(stripeFooter);
        outputData.add(OrcDataOutput.createDataOutput(footer));
        StripeStatistics statistics = new StripeStatistics(OrcWriter.toColumnMetadata(columnStatistics, this.orcTypes.size()));
        this.recordValidation(validation -> validation.addStripeStatistics(stripeStartOffset, statistics));
        StripeInformation stripeInformation = new StripeInformation(this.stripeRowCount, stripeStartOffset, indexLength, dataLength, footer.length());
        ClosedStripe closedStripe = new ClosedStripe(stripeInformation, statistics);
        this.closedStripes.add(closedStripe);
        this.closedStripesRetainedBytes += closedStripe.getRetainedSizeInBytes();
        this.recordValidation(validation -> validation.addStripe(stripeInformation.getNumberOfRows()));
        this.stats.recordStripeWritten(flushReason, stripeInformation.getTotalLength(), stripeInformation.getNumberOfRows(), this.dictionaryCompressionOptimizer.getDictionaryMemoryBytes());
        return outputData;
    }

    @Override
    public void close() throws IOException {
        if (this.closed) {
            return;
        }
        this.closed = true;
        this.stats.updateSizeInBytes(-this.previouslyRecordedSizeInBytes);
        this.previouslyRecordedSizeInBytes = 0L;
        try (OrcDataSink ignored = this.orcDataSink;){
            this.flushStripe(OrcWriterStats.FlushReason.CLOSED);
        }
        this.bufferedBytes = 0L;
    }

    public void updateUserMetadata(Map<String, String> updatedProperties) {
        this.userMetadata.putAll(updatedProperties);
    }

    private List<OrcDataOutput> bufferFileFooter() throws IOException {
        ArrayList<OrcDataOutput> outputData = new ArrayList<OrcDataOutput>();
        Metadata metadata = new Metadata(this.closedStripes.stream().map(ClosedStripe::getStatistics).map(Optional::of).collect(Collectors.toList()));
        Slice metadataSlice = this.metadataWriter.writeMetadata(metadata);
        outputData.add(OrcDataOutput.createDataOutput(metadataSlice));
        this.fileStats = OrcWriter.toFileStats(this.closedStripes.stream().map(ClosedStripe::getStatistics).map(StripeStatistics::getColumnStatistics).collect(Collectors.toList()));
        this.fileStatsRetainedBytes = this.fileStats.map(stats -> stats.stream().mapToLong(ColumnStatistics::getRetainedSizeInBytes).sum()).orElse(0L);
        this.stripeOffsets = (List)this.closedStripes.stream().map(closedStripe -> closedStripe.getStripeInformation().getOffset()).collect(ImmutableList.toImmutableList());
        this.recordValidation(validation -> validation.setFileStatistics(this.fileStats));
        Map<String, Slice> userMetadata = this.userMetadata.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> Slices.utf8Slice((String)((String)entry.getValue()))));
        Footer footer = new Footer(this.fileRowCount, this.rowGroupMaxRowCount == 0 ? OptionalInt.empty() : OptionalInt.of(this.rowGroupMaxRowCount), (List)this.closedStripes.stream().map(ClosedStripe::getStripeInformation).collect(ImmutableList.toImmutableList()), this.orcTypes, this.fileStats, userMetadata, Optional.empty());
        this.closedStripes.clear();
        this.closedStripesRetainedBytes = 0L;
        Slice footerSlice = this.metadataWriter.writeFooter(footer);
        outputData.add(OrcDataOutput.createDataOutput(footerSlice));
        this.recordValidation(validation -> validation.setVersion(this.metadataWriter.getOrcMetadataVersion()));
        Slice postscriptSlice = this.metadataWriter.writePostscript(footerSlice.length(), metadataSlice.length(), this.compression, this.maxCompressionBufferSize);
        outputData.add(OrcDataOutput.createDataOutput(postscriptSlice));
        outputData.add(OrcDataOutput.createDataOutput(Slices.wrappedBuffer((byte[])new byte[]{UnsignedBytes.checkedCast((long)postscriptSlice.length())})));
        return outputData;
    }

    private void recordValidation(Consumer<OrcWriteValidation.OrcWriteValidationBuilder> task) {
        if (this.validationBuilder != null) {
            task.accept(this.validationBuilder);
        }
    }

    public void validate(OrcDataSource input) throws OrcCorruptionException {
        Preconditions.checkState((this.validationBuilder != null ? 1 : 0) != 0, (Object)"validation is not enabled");
        OrcReader.validateFile(this.validationBuilder.build(), input, this.types);
    }

    public long getFileRowCount() {
        Preconditions.checkState((boolean)this.closed, (Object)"File row count is not available until the writing has finished");
        return this.fileRowCount;
    }

    public Optional<ColumnMetadata<ColumnStatistics>> getFileStats() {
        Preconditions.checkState((boolean)this.closed, (Object)"File statistics are not available until the writing has finished");
        return this.fileStats;
    }

    public List<Long> getStripeOffsets() {
        Preconditions.checkState((boolean)this.closed, (Object)"File stripe offsets are not available until the writing has finished");
        return this.stripeOffsets;
    }

    private static Supplier<BloomFilterBuilder> getBloomFilterBuilder(OrcWriterOptions options, String columnName) {
        if (options.isBloomFilterColumn(columnName)) {
            return () -> new Utf8BloomFilterBuilder(options.getRowGroupMaxRowCount(), options.getBloomFilterFpp());
        }
        return NoOpBloomFilterBuilder::new;
    }

    private static <T> ColumnMetadata<T> toColumnMetadata(Map<OrcColumnId, T> data, int expectedSize) {
        Preconditions.checkArgument((data.size() == expectedSize ? 1 : 0) != 0);
        ArrayList<T> list = new ArrayList<T>(expectedSize);
        for (int i = 0; i < expectedSize; ++i) {
            list.add(data.get(new OrcColumnId(i)));
        }
        return new ColumnMetadata(ImmutableList.copyOf(list));
    }

    private static Optional<ColumnMetadata<ColumnStatistics>> toFileStats(List<ColumnMetadata<ColumnStatistics>> stripes) {
        if (stripes.isEmpty()) {
            return Optional.empty();
        }
        int columnCount = stripes.get(0).size();
        Preconditions.checkArgument((boolean)stripes.stream().allMatch(stripe -> columnCount == stripe.size()));
        ImmutableList.Builder fileStats = ImmutableList.builder();
        for (int i = 0; i < columnCount; ++i) {
            OrcColumnId columnId = new OrcColumnId(i);
            fileStats.add((Object)ColumnStatistics.mergeColumnStatistics(stripes.stream().map(stripe -> (ColumnStatistics)stripe.get(columnId)).collect(Collectors.toList())));
        }
        return Optional.of(new ColumnMetadata(fileStats.build()));
    }

    static {
        String version = OrcWriter.class.getPackage().getImplementationVersion();
        TRINO_ORC_WRITER_VERSION = version == null ? "UNKNOWN" : version;
    }

    private static class ClosedStripe {
        private static final int INSTANCE_SIZE = SizeOf.instanceSize(ClosedStripe.class) + SizeOf.instanceSize(StripeInformation.class);
        private final StripeInformation stripeInformation;
        private final StripeStatistics statistics;

        public ClosedStripe(StripeInformation stripeInformation, StripeStatistics statistics) {
            this.stripeInformation = Objects.requireNonNull(stripeInformation, "stripeInformation is null");
            this.statistics = Objects.requireNonNull(statistics, "statistics is null");
        }

        public StripeInformation getStripeInformation() {
            return this.stripeInformation;
        }

        public StripeStatistics getStatistics() {
            return this.statistics;
        }

        public long getRetainedSizeInBytes() {
            return (long)INSTANCE_SIZE + this.statistics.getRetainedSizeInBytes();
        }
    }

    public static enum OrcOperation {
        NONE(-1),
        INSERT(0),
        DELETE(2);

        private final int operationNumber;

        private OrcOperation(int operationNumber) {
            this.operationNumber = operationNumber;
        }

        public int getOperationNumber() {
            return this.operationNumber;
        }
    }
}

