/*
 * Decompiled with CFR 0.152.
 */
package io.trino.parquet.writer;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import io.airlift.slice.DynamicSliceOutput;
import io.airlift.slice.OutputStreamSliceOutput;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceOutput;
import io.airlift.slice.Slices;
import io.airlift.units.DataSize;
import io.trino.parquet.writer.ColumnChunk;
import io.trino.parquet.writer.ColumnWriter;
import io.trino.parquet.writer.MessageTypeConverter;
import io.trino.parquet.writer.ParquetDataOutput;
import io.trino.parquet.writer.ParquetWriterOptions;
import io.trino.parquet.writer.ParquetWriters;
import io.trino.spi.Page;
import io.trino.spi.type.Type;
import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import org.apache.parquet.column.ParquetProperties;
import org.apache.parquet.format.ColumnMetaData;
import org.apache.parquet.format.FileMetaData;
import org.apache.parquet.format.KeyValue;
import org.apache.parquet.format.RowGroup;
import org.apache.parquet.format.Util;
import org.apache.parquet.hadoop.metadata.CompressionCodecName;
import org.apache.parquet.schema.MessageType;
import org.joda.time.DateTimeZone;
import org.openjdk.jol.info.ClassLayout;

public class ParquetWriter
implements Closeable {
    private static final int INSTANCE_SIZE = ClassLayout.parseClass(ParquetWriter.class).instanceSize();
    private static final int CHUNK_MAX_BYTES = Math.toIntExact(DataSize.of((long)128L, (DataSize.Unit)DataSize.Unit.MEGABYTE).toBytes());
    private final OutputStreamSliceOutput outputStream;
    private final ParquetWriterOptions writerOption;
    private final MessageType messageType;
    private final String createdBy;
    private final int chunkMaxLogicalBytes;
    private final Map<List<String>, Type> primitiveTypes;
    private final CompressionCodecName compressionCodecName;
    private final Optional<DateTimeZone> parquetTimeZone;
    private final ImmutableList.Builder<RowGroup> rowGroupBuilder = ImmutableList.builder();
    private List<ColumnWriter> columnWriters;
    private int rows;
    private long bufferedBytes;
    private boolean closed;
    private boolean writeHeader;
    public static final Slice MAGIC = Slices.wrappedBuffer((byte[])"PAR1".getBytes(StandardCharsets.US_ASCII));

    public ParquetWriter(OutputStream outputStream, MessageType messageType, Map<List<String>, Type> primitiveTypes, ParquetWriterOptions writerOption, CompressionCodecName compressionCodecName, String trinoVersion, Optional<DateTimeZone> parquetTimeZone) {
        this.outputStream = new OutputStreamSliceOutput(Objects.requireNonNull(outputStream, "outputstream is null"));
        this.messageType = Objects.requireNonNull(messageType, "messageType is null");
        this.primitiveTypes = Objects.requireNonNull(primitiveTypes, "primitiveTypes is null");
        this.writerOption = Objects.requireNonNull(writerOption, "writerOption is null");
        this.compressionCodecName = Objects.requireNonNull(compressionCodecName, "compressionCodecName is null");
        this.parquetTimeZone = Objects.requireNonNull(parquetTimeZone, "parquetTimeZone is null");
        this.initColumnWriters();
        this.chunkMaxLogicalBytes = Math.max(1, CHUNK_MAX_BYTES / 2);
        this.createdBy = ParquetWriter.formatCreatedBy(Objects.requireNonNull(trinoVersion, "trinoVersion is null"));
    }

    public long getWrittenBytes() {
        return this.outputStream.longSize();
    }

    public long getBufferedBytes() {
        return this.bufferedBytes;
    }

    public long getRetainedBytes() {
        return (long)INSTANCE_SIZE + this.outputStream.getRetainedSize() + this.columnWriters.stream().mapToLong(ColumnWriter::getRetainedBytes).sum();
    }

    public void write(Page page) throws IOException {
        Objects.requireNonNull(page, "page is null");
        Preconditions.checkState((!this.closed ? 1 : 0) != 0, (Object)"writer is closed");
        if (page.getPositionCount() == 0) {
            return;
        }
        Preconditions.checkArgument((page.getChannelCount() == this.columnWriters.size() ? 1 : 0) != 0);
        while (page != null) {
            int chunkRows = Math.min(page.getPositionCount(), this.writerOption.getBatchSize());
            Page chunk = page.getRegion(0, chunkRows);
            while (chunkRows > 1 && chunk.getLogicalSizeInBytes() > (long)this.chunkMaxLogicalBytes) {
                chunk = chunk.getRegion(0, chunkRows /= 2);
            }
            page = chunkRows < page.getPositionCount() ? page.getRegion(chunkRows, page.getPositionCount() - chunkRows) : null;
            this.writeChunk(chunk);
        }
    }

    private void writeChunk(Page page) throws IOException {
        this.bufferedBytes = 0L;
        for (int channel = 0; channel < page.getChannelCount(); ++channel) {
            ColumnWriter writer = this.columnWriters.get(channel);
            writer.writeBlock(new ColumnChunk(page.getBlock(channel)));
            this.bufferedBytes += writer.getBufferedBytes();
        }
        this.rows += page.getPositionCount();
        if (this.bufferedBytes >= this.writerOption.getMaxRowGroupSize()) {
            this.columnWriters.forEach(ColumnWriter::close);
            this.flush();
            this.initColumnWriters();
            this.rows = 0;
            this.bufferedBytes = this.columnWriters.stream().mapToLong(ColumnWriter::getBufferedBytes).sum();
        }
    }

    @Override
    public void close() throws IOException {
        if (this.closed) {
            return;
        }
        this.closed = true;
        try (OutputStreamSliceOutput outputStreamSliceOutput = this.outputStream;){
            this.columnWriters.forEach(ColumnWriter::close);
            this.flush();
            this.writeFooter();
        }
        this.bufferedBytes = 0L;
    }

    private void flush() throws IOException {
        if (!this.writeHeader) {
            ParquetDataOutput.createDataOutput(MAGIC).writeData((SliceOutput)this.outputStream);
            this.writeHeader = true;
        }
        ImmutableList.Builder builder = ImmutableList.builder();
        for (ColumnWriter columnWriter : this.columnWriters) {
            columnWriter.getBuffer().forEach(arg_0 -> ((ImmutableList.Builder)builder).add(arg_0));
        }
        ImmutableList bufferDataList = builder.build();
        long stripeStartOffset = this.outputStream.longSize();
        List metadatas = (List)bufferDataList.stream().map(ColumnWriter.BufferData::getMetaData).collect(ImmutableList.toImmutableList());
        this.updateRowGroups(this.updateColumnMetadataOffset(metadatas, stripeStartOffset));
        bufferDataList.stream().map(ColumnWriter.BufferData::getData).flatMap(Collection::stream).forEach(data -> data.writeData((SliceOutput)this.outputStream));
    }

    private void writeFooter() throws IOException {
        Preconditions.checkState((boolean)this.closed);
        Slice footer = this.getFooter((List<RowGroup>)this.rowGroupBuilder.build(), this.messageType);
        ParquetDataOutput.createDataOutput(footer).writeData((SliceOutput)this.outputStream);
        Slice footerSize = Slices.allocate((int)4);
        footerSize.setInt(0, footer.length());
        ParquetDataOutput.createDataOutput(footerSize).writeData((SliceOutput)this.outputStream);
        ParquetDataOutput.createDataOutput(MAGIC).writeData((SliceOutput)this.outputStream);
    }

    Slice getFooter(List<RowGroup> rowGroups, MessageType messageType) throws IOException {
        FileMetaData fileMetaData = new FileMetaData();
        fileMetaData.setVersion(1);
        fileMetaData.setCreated_by(this.createdBy);
        fileMetaData.setSchema(MessageTypeConverter.toParquetSchema(messageType));
        this.parquetTimeZone.ifPresent(dateTimeZone -> fileMetaData.setKey_value_metadata((List)ImmutableList.of((Object)new KeyValue("writer.time.zone").setValue(dateTimeZone.getID()))));
        long totalRows = rowGroups.stream().mapToLong(RowGroup::getNum_rows).sum();
        fileMetaData.setNum_rows(totalRows);
        fileMetaData.setRow_groups((List)ImmutableList.copyOf(rowGroups));
        DynamicSliceOutput dynamicSliceOutput = new DynamicSliceOutput(40);
        Util.writeFileMetaData((FileMetaData)fileMetaData, (OutputStream)dynamicSliceOutput);
        return dynamicSliceOutput.slice();
    }

    private void updateRowGroups(List<ColumnMetaData> columnMetaData) {
        long totalBytes = columnMetaData.stream().mapToLong(ColumnMetaData::getTotal_compressed_size).sum();
        ImmutableList columnChunks = (ImmutableList)columnMetaData.stream().map(ParquetWriter::toColumnChunk).collect(ImmutableList.toImmutableList());
        this.rowGroupBuilder.add((Object)new RowGroup((List)columnChunks, totalBytes, (long)this.rows));
    }

    private static org.apache.parquet.format.ColumnChunk toColumnChunk(ColumnMetaData metaData) {
        org.apache.parquet.format.ColumnChunk columnChunk = new org.apache.parquet.format.ColumnChunk(0L);
        columnChunk.setMeta_data(metaData);
        return columnChunk;
    }

    private List<ColumnMetaData> updateColumnMetadataOffset(List<ColumnMetaData> columns, long offset) {
        ImmutableList.Builder builder = ImmutableList.builder();
        long currentOffset = offset;
        for (ColumnMetaData column : columns) {
            ColumnMetaData columnMetaData = new ColumnMetaData(column.type, column.encodings, column.path_in_schema, column.codec, column.num_values, column.total_uncompressed_size, column.total_compressed_size, currentOffset);
            columnMetaData.setStatistics(column.getStatistics());
            columnMetaData.setEncoding_stats(column.getEncoding_stats());
            builder.add((Object)columnMetaData);
            currentOffset += column.getTotal_compressed_size();
        }
        return builder.build();
    }

    @VisibleForTesting
    static String formatCreatedBy(String trinoVersion) {
        return "parquet-mr-trino version " + trinoVersion + " (build n/a)";
    }

    private void initColumnWriters() {
        ParquetProperties parquetProperties = ParquetProperties.builder().withWriterVersion(ParquetProperties.WriterVersion.PARQUET_1_0).withPageSize(this.writerOption.getMaxPageSize()).build();
        this.columnWriters = ParquetWriters.getColumnWriters(this.messageType, this.primitiveTypes, parquetProperties, this.compressionCodecName, this.parquetTimeZone);
    }
}

