/*
 * Decompiled with CFR 0.152.
 */
package io.trino.parquet.reader.decoders;

import com.google.common.base.Preconditions;
import io.airlift.slice.Slices;
import io.trino.parquet.ParquetReaderUtils;
import io.trino.parquet.ParquetTypeUtils;
import io.trino.parquet.reader.SimpleSliceInputStream;
import io.trino.parquet.reader.decoders.ShortDecimalFixedWidthByteArrayBatchDecoder;
import io.trino.parquet.reader.decoders.ValueDecoder;
import io.trino.parquet.reader.flat.BitPackingUtils;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.Int128;
import io.trino.spi.type.Type;
import java.util.Objects;
import org.apache.parquet.column.ColumnDescriptor;

public final class PlainValueDecoders {
    private PlainValueDecoders() {
    }

    public static final class UuidPlainValueDecoder
    implements ValueDecoder<long[]> {
        private static final int UUID_SIZE = 16;
        private SimpleSliceInputStream input;

        @Override
        public void init(SimpleSliceInputStream input) {
            this.input = Objects.requireNonNull(input, "input is null");
        }

        @Override
        public void read(long[] values, int offset, int length) {
            int endOffset = (offset + length) * 2;
            for (int currentOutputOffset = offset * 2; currentOutputOffset < endOffset; currentOutputOffset += 2) {
                values[currentOutputOffset] = this.input.readLong();
                values[currentOutputOffset + 1] = this.input.readLong();
            }
        }

        @Override
        public void skip(int n) {
            this.input.skip(n * 16);
        }
    }

    public static final class LongDecimalPlainValueDecoder
    implements ValueDecoder<long[]> {
        private final int typeLength;
        private final byte[] inputBytes;
        private SimpleSliceInputStream input;

        public LongDecimalPlainValueDecoder(int typeLength) {
            Preconditions.checkArgument((typeLength > 0 && typeLength <= 16 ? 1 : 0) != 0, (String)"typeLength %s should be in range (1-16) for a long decimal", (int)typeLength);
            this.typeLength = typeLength;
            this.inputBytes = new byte[typeLength];
        }

        @Override
        public void init(SimpleSliceInputStream input) {
            this.input = Objects.requireNonNull(input, "input is null");
        }

        @Override
        public void read(long[] values, int offset, int length) {
            int endOffset = (offset + length) * 2;
            for (int currentOutputOffset = offset * 2; currentOutputOffset < endOffset; currentOutputOffset += 2) {
                this.input.readBytes(Slices.wrappedBuffer((byte[])this.inputBytes), 0, this.typeLength);
                Int128 value = Int128.fromBigEndian((byte[])this.inputBytes);
                values[currentOutputOffset] = value.getHigh();
                values[currentOutputOffset + 1] = value.getLow();
            }
        }

        @Override
        public void skip(int n) {
            this.input.skip(n * this.typeLength);
        }
    }

    public static final class ShortDecimalFixedLengthByteArrayDecoder
    implements ValueDecoder<long[]> {
        private final int typeLength;
        private final DecimalType decimalType;
        private final ColumnDescriptor descriptor;
        private final ShortDecimalFixedWidthByteArrayBatchDecoder decimalValueDecoder;
        private SimpleSliceInputStream input;

        public ShortDecimalFixedLengthByteArrayDecoder(DecimalType decimalType, ColumnDescriptor descriptor) {
            Preconditions.checkArgument((boolean)decimalType.isShort(), (String)"Decimal type %s is not a short decimal", (Object)decimalType);
            this.decimalType = decimalType;
            this.descriptor = Objects.requireNonNull(descriptor, "descriptor is null");
            this.typeLength = descriptor.getPrimitiveType().getTypeLength();
            Preconditions.checkArgument((this.typeLength > 0 && this.typeLength <= 16 ? 1 : 0) != 0, (String)"Expected column %s to have type length in range (1-16)", (Object)descriptor);
            this.decimalValueDecoder = new ShortDecimalFixedWidthByteArrayBatchDecoder(Math.min(this.typeLength, 8));
        }

        @Override
        public void init(SimpleSliceInputStream input) {
            this.input = Objects.requireNonNull(input, "input is null");
        }

        @Override
        public void read(long[] values, int offset, int length) {
            this.input.ensureBytesAvailable(this.typeLength * length);
            if (this.typeLength <= 8) {
                this.decimalValueDecoder.getShortDecimalValues(this.input, values, offset, length);
                return;
            }
            int extraBytesLength = this.typeLength - 8;
            byte[] inputBytes = this.input.getByteArray();
            int inputBytesOffset = this.input.getByteArrayOffset();
            for (int i = offset; i < offset + length; ++i) {
                ParquetTypeUtils.checkBytesFitInShortDecimal(inputBytes, inputBytesOffset, extraBytesLength, (Type)this.decimalType, this.descriptor);
                values[i] = ParquetTypeUtils.getShortDecimalValue(inputBytes, inputBytesOffset + extraBytesLength, 8);
                inputBytesOffset += this.typeLength;
            }
            this.input.skip(length * this.typeLength);
        }

        @Override
        public void skip(int n) {
            this.input.skip(n * this.typeLength);
        }
    }

    public static final class BooleanPlainValueDecoder
    implements ValueDecoder<byte[]> {
        private SimpleSliceInputStream input;
        private int alreadyReadBits;
        private byte partiallyReadByte;

        @Override
        public void init(SimpleSliceInputStream input) {
            this.input = Objects.requireNonNull(input, "input is null");
            this.alreadyReadBits = 0;
        }

        @Override
        public void read(byte[] values, int offset, int length) {
            if (this.alreadyReadBits != 0) {
                int bitsRemaining = 8 - this.alreadyReadBits;
                int chunkSize = Math.min(bitsRemaining, length);
                BitPackingUtils.unpack(values, offset, this.partiallyReadByte, this.alreadyReadBits, this.alreadyReadBits + chunkSize);
                this.alreadyReadBits = (this.alreadyReadBits + chunkSize) % 8;
                if (length == chunkSize) {
                    return;
                }
                offset += chunkSize;
                length -= chunkSize;
            }
            int bytesToRead = length / 8;
            while (bytesToRead >= 8) {
                long packedLong = this.input.readLong();
                BitPackingUtils.unpack64FromLong(values, offset, packedLong);
                bytesToRead -= 8;
                offset += 64;
            }
            while (bytesToRead >= 1) {
                byte packedByte = this.input.readByte();
                BitPackingUtils.unpack8FromByte(values, offset, packedByte);
                --bytesToRead;
                offset += 8;
            }
            this.alreadyReadBits = length % 8;
            if (this.alreadyReadBits != 0) {
                this.partiallyReadByte = this.input.readByte();
                BitPackingUtils.unpack(values, offset, this.partiallyReadByte, 0, this.alreadyReadBits);
            }
        }

        @Override
        public void skip(int n) {
            if (this.alreadyReadBits != 0) {
                int chunkSize = Math.min(8 - this.alreadyReadBits, n);
                n -= chunkSize;
                this.alreadyReadBits = (this.alreadyReadBits + chunkSize) % 8;
            }
            this.input.skip(n / 8);
            if (n % 8 != 0) {
                this.alreadyReadBits = n % 8;
                this.partiallyReadByte = this.input.readByte();
            }
        }
    }

    public static final class IntToBytePlainValueDecoder
    implements ValueDecoder<byte[]> {
        private SimpleSliceInputStream input;

        @Override
        public void init(SimpleSliceInputStream input) {
            this.input = Objects.requireNonNull(input, "input is null");
        }

        @Override
        public void read(byte[] values, int offset, int length) {
            this.input.ensureBytesAvailable(4 * length);
            int endOffset = offset + length;
            for (int i = offset; i < endOffset; ++i) {
                values[i] = ParquetReaderUtils.toByteExact(this.input.readIntUnsafe());
            }
        }

        @Override
        public void skip(int n) {
            this.input.skip(n * 4);
        }
    }

    public static final class IntToShortPlainValueDecoder
    implements ValueDecoder<short[]> {
        private SimpleSliceInputStream input;

        @Override
        public void init(SimpleSliceInputStream input) {
            this.input = Objects.requireNonNull(input, "input is null");
        }

        @Override
        public void read(short[] values, int offset, int length) {
            this.input.ensureBytesAvailable(4 * length);
            int endOffset = offset + length;
            for (int i = offset; i < endOffset; ++i) {
                values[i] = ParquetReaderUtils.toShortExact(this.input.readIntUnsafe());
            }
        }

        @Override
        public void skip(int n) {
            this.input.skip(n * 4);
        }
    }

    public static final class IntToLongPlainValueDecoder
    implements ValueDecoder<long[]> {
        private SimpleSliceInputStream input;

        @Override
        public void init(SimpleSliceInputStream input) {
            this.input = Objects.requireNonNull(input, "input is null");
        }

        @Override
        public void read(long[] values, int offset, int length) {
            this.input.ensureBytesAvailable(4 * length);
            int endOffset = offset + length;
            for (int i = offset; i < endOffset; ++i) {
                values[i] = this.input.readIntUnsafe();
            }
        }

        @Override
        public void skip(int n) {
            this.input.skip(n * 4);
        }
    }

    public static final class IntPlainValueDecoder
    implements ValueDecoder<int[]> {
        private SimpleSliceInputStream input;

        @Override
        public void init(SimpleSliceInputStream input) {
            this.input = Objects.requireNonNull(input, "input is null");
        }

        @Override
        public void read(int[] values, int offset, int length) {
            this.input.readBytes(Slices.wrappedIntArray((int[])values), offset * 4, length * 4);
        }

        @Override
        public void skip(int n) {
            this.input.skip(n * 4);
        }
    }

    public static final class LongPlainValueDecoder
    implements ValueDecoder<long[]> {
        private SimpleSliceInputStream input;

        @Override
        public void init(SimpleSliceInputStream input) {
            this.input = Objects.requireNonNull(input, "input is null");
        }

        @Override
        public void read(long[] values, int offset, int length) {
            this.input.readBytes(Slices.wrappedLongArray((long[])values), offset * 8, length * 8);
        }

        @Override
        public void skip(int n) {
            this.input.skip(n * 8);
        }
    }
}

