/*
 * Decompiled with CFR 0.152.
 */
package io.trino.parquet.reader.flat;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import io.airlift.slice.Slice;
import io.trino.memory.context.LocalMemoryContext;
import io.trino.parquet.DataPage;
import io.trino.parquet.DataPageV1;
import io.trino.parquet.DataPageV2;
import io.trino.parquet.DictionaryPage;
import io.trino.parquet.ParquetEncoding;
import io.trino.parquet.PrimitiveField;
import io.trino.parquet.reader.ColumnChunk;
import io.trino.parquet.reader.ColumnReader;
import io.trino.parquet.reader.FilteredRowRanges;
import io.trino.parquet.reader.PageReader;
import io.trino.parquet.reader.SimpleSliceInputStream;
import io.trino.parquet.reader.decoders.ValueDecoder;
import io.trino.parquet.reader.decoders.ValueDecoders;
import io.trino.parquet.reader.flat.ColumnAdapter;
import io.trino.parquet.reader.flat.DictionaryDecoder;
import io.trino.parquet.reader.flat.FlatDefinitionLevelDecoder;
import io.trino.parquet.reader.flat.NullsDecoder;
import io.trino.parquet.reader.flat.RowRangesIterator;
import io.trino.parquet.reader.flat.ZeroDefinitionLevelDecoder;
import io.trino.spi.block.Block;
import io.trino.spi.block.DictionaryBlock;
import io.trino.spi.block.RunLengthEncodedBlock;
import io.trino.spi.type.AbstractVariableWidthType;
import io.trino.spi.type.Type;
import java.util.Arrays;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import javax.annotation.Nullable;
import org.apache.parquet.io.ParquetDecodingException;

public class FlatColumnReader<BufferType>
implements ColumnReader {
    private static final int[] EMPTY_DEFINITION_LEVELS = new int[0];
    private static final int[] EMPTY_REPETITION_LEVELS = new int[0];
    private final PrimitiveField field;
    private final ValueDecoder.ValueDecodersProvider<BufferType> decodersProvider;
    private final ColumnAdapter<BufferType> columnAdapter;
    private final LocalMemoryContext memoryContext;
    private PageReader pageReader;
    private RowRangesIterator rowRanges;
    private int readOffset;
    private int remainingPageValueCount;
    @Nullable
    private DictionaryDecoder<BufferType> dictionaryDecoder;
    private boolean produceDictionaryBlock;
    private FlatDefinitionLevelDecoder definitionLevelDecoder;
    private ValueDecoder<BufferType> valueDecoder;
    private int nextBatchSize;

    public FlatColumnReader(PrimitiveField field, ValueDecoder.ValueDecodersProvider<BufferType> decodersProvider, ColumnAdapter<BufferType> columnAdapter, LocalMemoryContext memoryContext) {
        this.field = Objects.requireNonNull(field, "field is null");
        this.decodersProvider = Objects.requireNonNull(decodersProvider, "decoders is null");
        this.columnAdapter = Objects.requireNonNull(columnAdapter, "columnAdapter is null");
        this.memoryContext = Objects.requireNonNull(memoryContext, "memoryContext is null");
    }

    private void seek() {
        int remainingInBatch = this.readOffset;
        while (remainingInBatch > 0) {
            if (this.remainingPageValueCount == 0) {
                if ((remainingInBatch = this.seekToNextPage(remainingInBatch)) == 0) break;
                if (this.remainingPageValueCount == 0) {
                    FlatColumnReader.throwEndOfBatchException(remainingInBatch);
                }
            }
            int chunkSize = Math.min(this.remainingPageValueCount, remainingInBatch);
            int nonNullCount = this.isNonNull() ? chunkSize : this.definitionLevelDecoder.skip(chunkSize);
            this.valueDecoder.skip(nonNullCount);
            remainingInBatch -= this.rowRanges.seekForward(chunkSize);
            this.remainingPageValueCount -= chunkSize;
        }
    }

    @VisibleForTesting
    ColumnChunk readNullable() {
        NullableValuesBuffer<BufferType> valuesBuffer = this.createNullableValuesBuffer(this.nextBatchSize);
        boolean[] isNull = new boolean[this.nextBatchSize];
        int remainingInBatch = this.nextBatchSize;
        int offset = 0;
        while (remainingInBatch > 0) {
            if (this.remainingPageValueCount == 0 && !this.readNextPage()) {
                FlatColumnReader.throwEndOfBatchException(remainingInBatch);
            }
            if (this.skipToRowRangesStart()) continue;
            int chunkSize = this.rowRanges.advanceRange(Math.min(this.remainingPageValueCount, remainingInBatch));
            int nonNullCount = this.definitionLevelDecoder.readNext(isNull, offset, chunkSize);
            valuesBuffer.readNullableValues(this.valueDecoder, isNull, offset, nonNullCount, chunkSize);
            offset += chunkSize;
            remainingInBatch -= chunkSize;
            this.remainingPageValueCount -= chunkSize;
        }
        return valuesBuffer.createNullableBlock(isNull, this.field.getType());
    }

    @VisibleForTesting
    ColumnChunk readNoNull() {
        NonNullValuesBuffer<BufferType> valuesBuffer = this.createNonNullValuesBuffer(this.nextBatchSize);
        int remainingInBatch = this.nextBatchSize;
        int offset = 0;
        while (remainingInBatch > 0) {
            if (this.remainingPageValueCount == 0 && !this.readNextPage()) {
                FlatColumnReader.throwEndOfBatchException(remainingInBatch);
            }
            if (this.skipToRowRangesStart()) continue;
            int chunkSize = this.rowRanges.advanceRange(Math.min(this.remainingPageValueCount, remainingInBatch));
            valuesBuffer.readNonNullValues(this.valueDecoder, offset, chunkSize);
            offset += chunkSize;
            remainingInBatch -= chunkSize;
            this.remainingPageValueCount -= chunkSize;
        }
        return valuesBuffer.createNonNullBlock(this.field.getType());
    }

    private boolean skipToRowRangesStart() {
        int skipCount = Math.toIntExact(this.rowRanges.skipToRangeStart());
        if (skipCount >= this.remainingPageValueCount) {
            this.remainingPageValueCount = 0;
            return true;
        }
        if (skipCount > 0) {
            int nonNullsCount = this.isNonNull() ? skipCount : this.definitionLevelDecoder.skip(skipCount);
            this.valueDecoder.skip(nonNullsCount);
            this.remainingPageValueCount -= skipCount;
        }
        return false;
    }

    private boolean readNextPage() {
        if (!this.pageReader.hasNext()) {
            return false;
        }
        DataPage page = this.readPage();
        this.rowRanges.resetForNewPage(page.getFirstRowIndex());
        return true;
    }

    private int seekToNextPage(int remainingInBatch) {
        while (remainingInBatch > 0 && this.pageReader.hasNext()) {
            DataPage page = this.pageReader.getNextPage();
            this.rowRanges.resetForNewPage(page.getFirstRowIndex());
            if (remainingInBatch < page.getValueCount() || !this.rowRanges.isPageFullyConsumed(page.getValueCount())) {
                this.readPage();
                return remainingInBatch;
            }
            remainingInBatch -= page.getValueCount();
            this.remainingPageValueCount = 0;
            this.pageReader.skipNextPage();
        }
        return remainingInBatch;
    }

    private DataPage readPage() {
        DataPage page = this.pageReader.readPage();
        Objects.requireNonNull(page, "page is null");
        if (page instanceof DataPageV1) {
            this.readFlatPageV1((DataPageV1)page);
        } else if (page instanceof DataPageV2) {
            this.readFlatPageV2((DataPageV2)page);
        }
        int dataPageSizeInBytes = this.pageReader.arePagesCompressed() ? page.getUncompressedSize() : 0;
        long dictionarySizeInBytes = this.dictionaryDecoder == null ? 0L : this.dictionaryDecoder.getRetainedSizeInBytes();
        this.memoryContext.setBytes((long)dataPageSizeInBytes + dictionarySizeInBytes);
        this.remainingPageValueCount = page.getValueCount();
        return page;
    }

    private void readFlatPageV1(DataPageV1 page) {
        Slice buffer = page.getSlice();
        ParquetEncoding definitionEncoding = page.getDefinitionLevelEncoding();
        Preconditions.checkArgument((this.isNonNull() || definitionEncoding == ParquetEncoding.RLE ? 1 : 0) != 0, (Object)("Invalid definition level encoding: " + definitionEncoding));
        int alreadyRead = 0;
        if (definitionEncoding == ParquetEncoding.RLE) {
            if (this.field.getDescriptor().getMaxDefinitionLevel() == 0) {
                this.definitionLevelDecoder = new ZeroDefinitionLevelDecoder();
            } else {
                int bufferSize = buffer.getInt(0);
                this.definitionLevelDecoder = new NullsDecoder(buffer.slice(4, bufferSize));
                alreadyRead = bufferSize + 4;
            }
        }
        this.createValueDecoder(page.getValueEncoding(), buffer.slice(alreadyRead, buffer.length() - alreadyRead));
    }

    private void readFlatPageV2(DataPageV2 page) {
        int maxDefinitionLevel = this.field.getDescriptor().getMaxDefinitionLevel();
        Preconditions.checkArgument((maxDefinitionLevel >= 0 && maxDefinitionLevel <= 1 ? 1 : 0) != 0, (Object)("Invalid max definition level: " + maxDefinitionLevel));
        this.definitionLevelDecoder = new NullsDecoder(page.getDefinitionLevels());
        this.createValueDecoder(page.getDataEncoding(), page.getSlice());
    }

    private void createValueDecoder(ParquetEncoding encoding, Slice data) {
        if (encoding == ParquetEncoding.PLAIN_DICTIONARY || encoding == ParquetEncoding.RLE_DICTIONARY) {
            if (this.dictionaryDecoder == null) {
                throw new ParquetDecodingException(String.format("Dictionary is missing for %s", this.field));
            }
            this.valueDecoder = this.dictionaryDecoder;
        } else {
            this.valueDecoder = this.decodersProvider.create(encoding, this.field);
        }
        this.valueDecoder.init(new SimpleSliceInputStream(data));
    }

    protected boolean isNonNull() {
        return this.field.isRequired() || this.pageReader.hasNoNulls();
    }

    @Override
    public boolean hasPageReader() {
        return this.pageReader != null;
    }

    @Override
    public void setPageReader(PageReader pageReader, Optional<FilteredRowRanges> rowRanges) {
        this.pageReader = Objects.requireNonNull(pageReader, "pageReader");
        DictionaryPage dictionaryPage = pageReader.readDictionaryPage();
        if (dictionaryPage != null) {
            this.dictionaryDecoder = ValueDecoders.getDictionaryDecoder(dictionaryPage, this.columnAdapter, this.decodersProvider.create(ParquetEncoding.PLAIN, this.field), this.isNonNull());
            this.produceDictionaryBlock = this.shouldProduceDictionaryBlock(rowRanges);
        }
        this.rowRanges = RowRangesIterator.createRowRangesIterator(rowRanges);
    }

    @Override
    public void prepareNextRead(int batchSize) {
        this.readOffset += this.nextBatchSize;
        this.nextBatchSize = batchSize;
    }

    @Override
    public ColumnChunk readPrimitive() {
        this.seek();
        ColumnChunk columnChunk = this.isNonNull() ? this.readNoNull() : this.readNullable();
        this.readOffset = 0;
        this.nextBatchSize = 0;
        return columnChunk;
    }

    private static void throwEndOfBatchException(int remainingInBatch) {
        throw new ParquetDecodingException(String.format("Corrupted Parquet file: extra %d values to be consumed when scanning current batch", remainingInBatch));
    }

    private boolean shouldProduceDictionaryBlock(Optional<FilteredRowRanges> filteredRowRanges) {
        if (this.pageReader.hasOnlyDictionaryEncodedPages()) {
            if (!(this.field.getType() instanceof AbstractVariableWidthType)) {
                return false;
            }
            Objects.requireNonNull(this.dictionaryDecoder, "dictionaryDecoder is null");
            return filteredRowRanges.map(rowRanges -> rowRanges.getRowCount() > (long)this.dictionaryDecoder.getDictionarySize()).orElse(true);
        }
        return false;
    }

    private NonNullValuesBuffer<BufferType> createNonNullValuesBuffer(int batchSize) {
        if (this.produceDictionaryBlock) {
            return new DictionaryValuesBuffer<BufferType>(this.dictionaryDecoder, batchSize);
        }
        return new DataValuesBuffer<BufferType>(this.columnAdapter, batchSize);
    }

    private NullableValuesBuffer<BufferType> createNullableValuesBuffer(int batchSize) {
        if (this.produceDictionaryBlock) {
            return new DictionaryValuesBuffer<BufferType>(this.dictionaryDecoder, batchSize);
        }
        return new DataValuesBuffer<BufferType>(this.columnAdapter, batchSize);
    }

    private static interface NullableValuesBuffer<T> {
        public void readNullableValues(ValueDecoder<T> var1, boolean[] var2, int var3, int var4, int var5);

        public ColumnChunk createNullableBlock(boolean[] var1, Type var2);
    }

    private static interface NonNullValuesBuffer<T> {
        public void readNonNullValues(ValueDecoder<T> var1, int var2, int var3);

        public ColumnChunk createNonNullBlock(Type var1);
    }

    private static final class DictionaryValuesBuffer<T>
    implements NonNullValuesBuffer<T>,
    NullableValuesBuffer<T> {
        private final DictionaryDecoder<T> decoder;
        private final int[] ids;
        private final int batchSize;
        private int totalNullsCount;

        private DictionaryValuesBuffer(DictionaryDecoder<T> dictionaryDecoder, int batchSize) {
            this.ids = new int[batchSize];
            this.decoder = dictionaryDecoder;
            this.batchSize = batchSize;
        }

        @Override
        public void readNonNullValues(ValueDecoder<T> valueDecoder, int offset, int chunkSize) {
            this.decoder.readDictionaryIds(this.ids, offset, chunkSize);
        }

        @Override
        public void readNullableValues(ValueDecoder<T> valueDecoder, boolean[] isNull, int offset, int nonNullCount, int valuesCount) {
            if (nonNullCount == 0) {
                Arrays.fill(this.ids, offset, offset + valuesCount, this.decoder.getDictionarySize());
            } else if (nonNullCount == valuesCount) {
                this.decoder.readDictionaryIds(this.ids, offset, valuesCount);
            } else {
                int[] tmpBuffer = new int[nonNullCount];
                this.decoder.readDictionaryIds(tmpBuffer, 0, nonNullCount);
                DictionaryValuesBuffer.unpackDictionaryNullId(tmpBuffer, this.ids, isNull, offset, valuesCount, this.decoder.getDictionarySize());
            }
            this.totalNullsCount += valuesCount - nonNullCount;
        }

        @Override
        public ColumnChunk createNonNullBlock(Type type) {
            Preconditions.checkState((this.totalNullsCount == 0 ? 1 : 0) != 0, (String)"totalNonNullsCount %s should be equal to 0 when creating non-null block", (int)this.totalNullsCount);
            return this.createDictionaryBlock(this.ids, this.decoder.getDictionaryBlock());
        }

        @Override
        public ColumnChunk createNullableBlock(boolean[] isNull, Type type) {
            if (this.totalNullsCount == this.batchSize) {
                return new ColumnChunk(RunLengthEncodedBlock.create((Type)type, null, (int)this.batchSize), EMPTY_DEFINITION_LEVELS, EMPTY_REPETITION_LEVELS);
            }
            return this.createDictionaryBlock(this.ids, this.decoder.getDictionaryBlock());
        }

        private ColumnChunk createDictionaryBlock(int[] dictionaryIds, Block dictionary) {
            int positionsCount = dictionaryIds.length;
            return new ColumnChunk(DictionaryBlock.create((int)positionsCount, (Block)dictionary, (int[])dictionaryIds), EMPTY_DEFINITION_LEVELS, EMPTY_REPETITION_LEVELS, OptionalLong.of(DictionaryValuesBuffer.getMaxDictionaryBlockSize(dictionary, positionsCount)));
        }

        private static void unpackDictionaryNullId(int[] source, int[] destination, boolean[] isNull, int destOffset, int chunkSize, int nullId) {
            int srcOffset = 0;
            for (int i = destOffset; i < destOffset + chunkSize; ++i) {
                destination[i] = isNull[i] ? nullId : source[srcOffset++];
            }
        }

        private static long getMaxDictionaryBlockSize(Block dictionary, long batchSize) {
            double maxDictionaryFractionUsed = Math.min((double)batchSize / (double)dictionary.getPositionCount(), 1.0);
            return (long)((double)(batchSize * 4L) + (double)dictionary.getSizeInBytes() * maxDictionaryFractionUsed);
        }
    }

    private static final class DataValuesBuffer<T>
    implements NonNullValuesBuffer<T>,
    NullableValuesBuffer<T> {
        private final ColumnAdapter<T> columnAdapter;
        private final T values;
        private final int batchSize;
        private int totalNullsCount;

        private DataValuesBuffer(ColumnAdapter<T> columnAdapter, int batchSize) {
            this.values = columnAdapter.createBuffer(batchSize);
            this.columnAdapter = columnAdapter;
            this.batchSize = batchSize;
        }

        @Override
        public void readNonNullValues(ValueDecoder<T> valueDecoder, int offset, int valuesCount) {
            valueDecoder.read(this.values, offset, valuesCount);
        }

        @Override
        public void readNullableValues(ValueDecoder<T> valueDecoder, boolean[] isNull, int offset, int nonNullCount, int valuesCount) {
            if (nonNullCount == 0) {
                T tmpBuffer = this.columnAdapter.createTemporaryBuffer(offset, 0, this.values);
                this.columnAdapter.unpackNullValues(tmpBuffer, this.values, isNull, offset, 0, valuesCount);
            } else if (nonNullCount == valuesCount) {
                valueDecoder.read(this.values, offset, nonNullCount);
            } else {
                T tmpBuffer = this.columnAdapter.createTemporaryBuffer(offset, nonNullCount, this.values);
                valueDecoder.read(tmpBuffer, 0, nonNullCount);
                this.columnAdapter.unpackNullValues(tmpBuffer, this.values, isNull, offset, nonNullCount, valuesCount);
            }
            this.totalNullsCount += valuesCount - nonNullCount;
        }

        @Override
        public ColumnChunk createNonNullBlock(Type type) {
            Preconditions.checkState((this.totalNullsCount == 0 ? 1 : 0) != 0, (String)"totalNonNullsCount %s should be equal to 0 when creating non-null block", (int)this.totalNullsCount);
            return new ColumnChunk(this.columnAdapter.createNonNullBlock(this.values), EMPTY_DEFINITION_LEVELS, EMPTY_REPETITION_LEVELS);
        }

        @Override
        public ColumnChunk createNullableBlock(boolean[] isNull, Type type) {
            if (this.totalNullsCount == this.batchSize) {
                return new ColumnChunk(RunLengthEncodedBlock.create((Type)type, null, (int)this.batchSize), EMPTY_DEFINITION_LEVELS, EMPTY_REPETITION_LEVELS);
            }
            if (this.totalNullsCount == 0) {
                return new ColumnChunk(this.columnAdapter.createNonNullBlock(this.values), EMPTY_DEFINITION_LEVELS, EMPTY_REPETITION_LEVELS);
            }
            return new ColumnChunk(this.columnAdapter.createNullableBlock(isNull, this.values), EMPTY_DEFINITION_LEVELS, EMPTY_REPETITION_LEVELS);
        }
    }
}

