/*
 * Decompiled with CFR 0.152.
 */
package io.trino.parquet.reader;

import com.google.common.base.Preconditions;
import com.google.common.primitives.Booleans;
import com.google.common.primitives.Ints;
import io.airlift.log.Logger;
import io.airlift.slice.SizeOf;
import io.airlift.slice.Slice;
import io.trino.memory.context.LocalMemoryContext;
import io.trino.parquet.DataPage;
import io.trino.parquet.DataPageV1;
import io.trino.parquet.DataPageV2;
import io.trino.parquet.ParquetEncoding;
import io.trino.parquet.ParquetReaderUtils;
import io.trino.parquet.PrimitiveField;
import io.trino.parquet.reader.AbstractColumnReader;
import io.trino.parquet.reader.ColumnChunk;
import io.trino.parquet.reader.SimpleSliceInputStream;
import io.trino.parquet.reader.decoders.RleBitPackingHybridDecoder;
import io.trino.parquet.reader.decoders.ValueDecoder;
import io.trino.parquet.reader.flat.ColumnAdapter;
import io.trino.parquet.reader.flat.DictionaryDecoder;
import io.trino.spi.block.RunLengthEncodedBlock;
import io.trino.spi.type.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import org.apache.parquet.bytes.BytesUtils;

public class NestedColumnReader<BufferType>
extends AbstractColumnReader<BufferType> {
    private static final Logger log = Logger.get(NestedColumnReader.class);
    private final LocalMemoryContext memoryContext;
    private ValueDecoder<int[]> definitionLevelDecoder;
    private ValueDecoder<int[]> repetitionLevelDecoder;
    private ValueDecoder<BufferType> valueDecoder;
    private int[] repetitionBuffer;
    private int readOffset;
    private int nextBatchSize;
    private boolean pageLastRowUnfinished;
    private boolean pageLastRowSkipped;
    private int remainingPageValueCount;
    private int pageValueCount;

    public NestedColumnReader(PrimitiveField field, ValueDecoder.ValueDecodersProvider<BufferType> decodersProvider, ColumnAdapter<BufferType> columnAdapter, LocalMemoryContext memoryContext) {
        super(field, decodersProvider, columnAdapter);
        this.memoryContext = Objects.requireNonNull(memoryContext, "memoryContext is null");
    }

    @Override
    public boolean hasPageReader() {
        return this.pageReader != null;
    }

    @Override
    protected boolean isNonNull() {
        return this.field.isRequired();
    }

    @Override
    public ColumnChunk readPrimitive() {
        this.seek();
        ColumnChunk columnChunk = this.isNonNull() ? this.readNonNull() : this.readNullable();
        this.readOffset = 0;
        this.nextBatchSize = 0;
        return columnChunk;
    }

    @Override
    public void prepareNextRead(int batchSize) {
        this.readOffset += this.nextBatchSize;
        this.nextBatchSize = batchSize;
    }

    private ColumnChunk readNullable() {
        log.debug("readNullable field %s, nextBatchSize %d", new Object[]{this.field, this.nextBatchSize});
        NullableValuesBuffer<BufferType> data = this.createNullableValuesBuffer();
        BooleansBuffer isNull = new BooleansBuffer();
        IntegersBuffer outputRepetitionLevels = new IntegersBuffer();
        IntegersBuffer outputDefinitionLevels = new IntegersBuffer();
        int remainingInBatch = this.nextBatchSize;
        while (remainingInBatch > 0) {
            if (this.remainingPageValueCount == 0) {
                if (!this.readNextPage()) {
                    if (this.pageLastRowUnfinished && remainingInBatch == 1) break;
                    NestedColumnReader.throwEndOfBatchException(remainingInBatch);
                }
                if (this.pageLastRowUnfinished) {
                    int pageIndex = this.readUnfinishedRow();
                    if (this.pageLastRowSkipped) {
                        this.remainingPageValueCount -= pageIndex;
                        int[] definitionLevels = new int[pageIndex];
                        this.definitionLevelDecoder.read(definitionLevels, 0, definitionLevels.length);
                        int existingValueCount = this.countExistingValues(this.field.getDefinitionLevel(), definitionLevels);
                        this.valueDecoder.skip(existingValueCount);
                    } else {
                        if (pageIndex > 0) {
                            int[] definitionLevels = new int[pageIndex];
                            int existingValueCount = this.readDefinitionLevels(outputDefinitionLevels, definitionLevels);
                            this.readNullableValues(data, isNull, outputRepetitionLevels, 0, pageIndex, definitionLevels, existingValueCount);
                        }
                        if (pageIndex == this.pageValueCount) {
                            Preconditions.checkState((boolean)this.pageLastRowUnfinished, (Object)"pageLastRowUnfinished not set when run out of values to read");
                            continue;
                        }
                        --remainingInBatch;
                    }
                }
            }
            if (this.skip(this.field.getDefinitionLevel())) continue;
            ValueCount valueCount = this.getNextPositions(Math.min(this.rowRanges.getRowsLeftInCurrentRange(), remainingInBatch));
            this.rowRanges.advanceRange(valueCount.rows + (this.pageLastRowUnfinished ? 1 : 0));
            int pageValuesIndex = this.pageValueCount - this.remainingPageValueCount;
            int[] definitionLevels = new int[valueCount.values];
            int existingValueCount = this.readDefinitionLevels(outputDefinitionLevels, definitionLevels);
            this.readNullableValues(data, isNull, outputRepetitionLevels, pageValuesIndex, valueCount.values, definitionLevels, existingValueCount);
            remainingInBatch -= valueCount.rows;
        }
        return data.createNullableBlock(isNull, outputDefinitionLevels.getMergedBuffer(), outputRepetitionLevels.getMergedBuffer());
    }

    private ColumnChunk readNonNull() {
        log.debug("readNonNull field %s, nextBatchSize %d", new Object[]{this.field, this.nextBatchSize});
        NonNullValuesBuffer<BufferType> data = this.createNonNullValuesBuffer();
        IntegersBuffer outputRepetitionLevels = new IntegersBuffer();
        IntegersBuffer outputDefinitionLevels = new IntegersBuffer();
        int remainingInBatch = this.nextBatchSize;
        while (remainingInBatch > 0) {
            if (this.remainingPageValueCount == 0) {
                if (!this.readNextPage()) {
                    if (this.pageLastRowUnfinished && remainingInBatch == 1) break;
                    NestedColumnReader.throwEndOfBatchException(remainingInBatch);
                }
                if (this.pageLastRowUnfinished) {
                    int pageIndex = this.readUnfinishedRow();
                    if (this.pageLastRowSkipped) {
                        this.remainingPageValueCount -= pageIndex;
                        int[] definitionLevels = new int[pageIndex];
                        this.definitionLevelDecoder.read(definitionLevels, 0, definitionLevels.length);
                        int existingValueCount = this.countExistingValues(this.field.getDefinitionLevel(), definitionLevels);
                        this.valueDecoder.skip(existingValueCount);
                        if (pageIndex == this.pageValueCount) {
                            continue;
                        }
                    } else {
                        if (pageIndex > 0) {
                            this.readNonNullValues(data, outputRepetitionLevels, outputDefinitionLevels, 0, pageIndex);
                        }
                        if (pageIndex == this.pageValueCount) continue;
                        --remainingInBatch;
                    }
                }
            }
            if (this.skip(this.field.getDefinitionLevel())) continue;
            ValueCount valueCount = this.getNextPositions(Math.min(this.rowRanges.getRowsLeftInCurrentRange(), remainingInBatch));
            this.rowRanges.advanceRange(valueCount.rows + (this.pageLastRowUnfinished ? 1 : 0));
            int pageValuesIndex = this.pageValueCount - this.remainingPageValueCount;
            this.readNonNullValues(data, outputRepetitionLevels, outputDefinitionLevels, pageValuesIndex, valueCount.values);
            remainingInBatch -= valueCount.rows;
        }
        return data.createNonNullBlock(outputDefinitionLevels.getMergedBuffer(), outputRepetitionLevels.getMergedBuffer());
    }

    private void seek() {
        if (this.readOffset > 0) {
            log.debug("seek field %s, readOffset %d, remainingPageValueCount %d, pageLastRowUnfinished %b", new Object[]{this.field, this.readOffset, this.remainingPageValueCount, this.pageLastRowUnfinished});
        }
        int remainingInBatch = this.readOffset;
        while (remainingInBatch > 0) {
            if (this.remainingPageValueCount == 0) {
                if (!this.readNextPage()) break;
                if (this.pageLastRowUnfinished) {
                    int pageIndex = this.readUnfinishedRow();
                    if (pageIndex > 0) {
                        this.seek(pageIndex);
                    }
                    if (this.remainingPageValueCount == 0) {
                        Preconditions.checkState((boolean)this.pageLastRowUnfinished, (Object)"pageLastRowUnfinished not set when run out of values to skip");
                        Preconditions.checkState((boolean)this.pageLastRowSkipped, (Object)"pageLastRowSkipped not set when run out of values to skip");
                        continue;
                    }
                    --remainingInBatch;
                }
            }
            int chunkSize = Math.min(this.remainingPageValueCount, remainingInBatch);
            ValueCount valueCount = this.getNextPositions(Math.toIntExact(chunkSize));
            this.seek(valueCount.values);
            int seek = this.rowRanges.seekForward(valueCount.rows + (this.pageLastRowUnfinished ? 1 : 0));
            remainingInBatch -= seek - (this.pageLastRowUnfinished ? 1 : 0);
        }
    }

    private int readDefinitionLevels(IntegersBuffer outputDefinitionLevels, int[] definitionLevels) {
        this.definitionLevelDecoder.read(definitionLevels, 0, definitionLevels.length);
        outputDefinitionLevels.add(definitionLevels);
        return this.countExistingValues(this.field.getDefinitionLevel() - 1, definitionLevels);
    }

    private void readNullableValues(NullableValuesBuffer<BufferType> data, BooleansBuffer isNull, IntegersBuffer outputRepetitionLevels, int pageValuesIndex, int valueCount, int[] definitionLevels, int existingValueCount) {
        boolean[] isNullChunk = new boolean[existingValueCount];
        isNull.add(isNullChunk);
        int nonNullCount = this.getNulls(definitionLevels, this.field.getDefinitionLevel(), isNullChunk);
        Preconditions.checkState((nonNullCount <= existingValueCount ? 1 : 0) != 0, (String)"nonNullCount %s cannot be greater than existingValueCount %s, field %s", (Object)nonNullCount, (Object)existingValueCount, (Object)this.field);
        outputRepetitionLevels.add(Arrays.copyOfRange(this.repetitionBuffer, pageValuesIndex, pageValuesIndex + valueCount));
        data.readNullableValues(this.valueDecoder, isNullChunk, nonNullCount, existingValueCount);
        this.remainingPageValueCount -= valueCount;
    }

    private void readNonNullValues(NonNullValuesBuffer<BufferType> data, IntegersBuffer outputRepetitionLevels, IntegersBuffer outputDefinitionLevels, int pageValuesIndex, int valueCount) {
        int[] definitionLevels = new int[valueCount];
        this.definitionLevelDecoder.read(definitionLevels, 0, definitionLevels.length);
        int existingValueCount = this.countExistingValues(this.field.getDefinitionLevel(), definitionLevels);
        outputRepetitionLevels.add(Arrays.copyOfRange(this.repetitionBuffer, pageValuesIndex, pageValuesIndex + valueCount));
        outputDefinitionLevels.add(definitionLevels);
        if (existingValueCount > 0) {
            data.readNonNullValues(this.valueDecoder, existingValueCount);
        }
        this.remainingPageValueCount -= valueCount;
    }

    private boolean skip(int minDefinitionLevel) {
        long skipCount = this.rowRanges.skipToRangeStart();
        if (skipCount > 0L) {
            log.debug("skipCount %d, remainingPageValueCount %d", new Object[]{skipCount, this.remainingPageValueCount});
        }
        if (skipCount >= (long)this.remainingPageValueCount) {
            this.remainingPageValueCount = 0;
            this.pageLastRowUnfinished = true;
            this.pageLastRowSkipped = true;
            return true;
        }
        if (skipCount > 0L) {
            ValueCount toSkip = this.getNextPositions(Math.toIntExact(skipCount));
            if (toSkip.values == this.remainingPageValueCount) {
                this.remainingPageValueCount = 0;
                this.pageLastRowSkipped = true;
                return true;
            }
            int[] definitionLevels = new int[toSkip.values];
            this.definitionLevelDecoder.read(definitionLevels, 0, toSkip.values);
            int valuesToSkip = this.countExistingValues(minDefinitionLevel, definitionLevels);
            this.valueDecoder.skip(valuesToSkip);
            this.remainingPageValueCount -= toSkip.values;
        }
        return false;
    }

    private int getNulls(int[] definitionLevels, int maxDefinitionLevel, boolean[] localIsNull) {
        int outputIndex = 0;
        int nonNullCount = 0;
        for (int definitionLevel : definitionLevels) {
            boolean isValueNonNull;
            boolean isValueNull = definitionLevel == maxDefinitionLevel - 1;
            boolean bl = isValueNonNull = definitionLevel == maxDefinitionLevel;
            if (isValueNull) {
                localIsNull[outputIndex] = true;
            }
            outputIndex += ParquetReaderUtils.castToByte(isValueNull | isValueNonNull);
            nonNullCount += ParquetReaderUtils.castToByte(isValueNonNull);
        }
        return nonNullCount;
    }

    private int countExistingValues(int minDefinitionLevel, int[] definitionLevels) {
        int valueCount = 0;
        for (int definitionLevel : definitionLevels) {
            valueCount += ParquetReaderUtils.castToByte(definitionLevel >= minDefinitionLevel);
        }
        return valueCount;
    }

    private int readUnfinishedRow() {
        int pageIndex;
        for (pageIndex = 0; pageIndex < this.remainingPageValueCount && this.repetitionBuffer[pageIndex] != 0; ++pageIndex) {
        }
        return pageIndex;
    }

    private ValueCount getNextPositions(int desiredRowCount) {
        boolean pageReadUptoLastValue;
        int rowCount;
        int valueCount = 0;
        int pageValuesIndex = this.pageValueCount - this.remainingPageValueCount;
        for (rowCount = 0; rowCount < desiredRowCount && valueCount < this.remainingPageValueCount - 1; rowCount += ParquetReaderUtils.castToByte(this.repetitionBuffer[pageValuesIndex + valueCount + 1] == 0), ++valueCount) {
        }
        boolean bl = pageReadUptoLastValue = rowCount != desiredRowCount;
        if (pageReadUptoLastValue) {
            ++valueCount;
            this.pageLastRowUnfinished = true;
            this.pageLastRowSkipped = false;
        } else {
            this.pageLastRowUnfinished = false;
        }
        return new ValueCount(rowCount, valueCount);
    }

    private void seek(int valueCount) {
        int[] definitionLevels = new int[valueCount];
        this.definitionLevelDecoder.read(definitionLevels, 0, definitionLevels.length);
        int maxDefinitionLevel = this.field.getDefinitionLevel();
        int valuesToSkip = 0;
        for (int definitionLevel : definitionLevels) {
            valuesToSkip += ParquetReaderUtils.castToByte(definitionLevel == maxDefinitionLevel);
        }
        this.valueDecoder.skip(valuesToSkip);
        this.remainingPageValueCount -= valueCount;
        if (this.remainingPageValueCount == 0) {
            this.pageLastRowUnfinished = true;
            this.pageLastRowSkipped = true;
        }
    }

    private boolean readNextPage() {
        DataPage page = this.pageReader.readPage();
        if (page == null) {
            return false;
        }
        log.debug("readNextPage field %s, page %s, pageLastRowUnfinished %b", new Object[]{this.field, page, this.pageLastRowUnfinished});
        if (page instanceof DataPageV1) {
            this.readFlatPageV1((DataPageV1)page);
        } else if (page instanceof DataPageV2) {
            this.readFlatPageV2((DataPageV2)page);
        }
        this.pageValueCount = page.getValueCount();
        this.repetitionBuffer = new int[this.pageValueCount];
        this.repetitionLevelDecoder.read(this.repetitionBuffer, 0, this.pageValueCount);
        this.remainingPageValueCount = this.pageValueCount;
        this.rowRanges.resetForNewPage(page.getFirstRowIndex());
        int dataPageSizeInBytes = this.pageReader.arePagesCompressed() ? page.getUncompressedSize() : 0;
        long dictionarySizeInBytes = this.dictionaryDecoder == null ? 0L : this.dictionaryDecoder.getRetainedSizeInBytes();
        long repetitionBufferSizeInBytes = SizeOf.sizeOf((int[])this.repetitionBuffer);
        this.memoryContext.setBytes((long)dataPageSizeInBytes + dictionarySizeInBytes + repetitionBufferSizeInBytes);
        return true;
    }

    private void readFlatPageV1(DataPageV1 page) {
        int bufferSize;
        Slice buffer = page.getSlice();
        ParquetEncoding definitionEncoding = page.getDefinitionLevelEncoding();
        ParquetEncoding repetitionEncoding = page.getRepetitionLevelEncoding();
        int maxDefinitionLevel = this.field.getDefinitionLevel();
        int maxRepetitionLevel = this.field.getRepetitionLevel();
        Preconditions.checkArgument((maxDefinitionLevel == 0 || definitionEncoding == ParquetEncoding.RLE ? 1 : 0) != 0, (Object)("Invalid definition level encoding: " + definitionEncoding));
        Preconditions.checkArgument((maxRepetitionLevel == 0 || repetitionEncoding == ParquetEncoding.RLE ? 1 : 0) != 0, (Object)("Invalid repetition level encoding: " + definitionEncoding));
        if (maxRepetitionLevel > 0) {
            bufferSize = buffer.getInt(0);
            this.repetitionLevelDecoder = new RleBitPackingHybridDecoder(BytesUtils.getWidthFromMaxInt((int)maxRepetitionLevel));
            this.repetitionLevelDecoder.init(new SimpleSliceInputStream(buffer.slice(4, bufferSize)));
            buffer = buffer.slice(bufferSize + 4, buffer.length() - bufferSize - 4);
        } else {
            this.repetitionLevelDecoder = new ValueDecoder.EmptyValueDecoder<int[]>();
        }
        if (maxDefinitionLevel > 0) {
            bufferSize = buffer.getInt(0);
            this.definitionLevelDecoder = new RleBitPackingHybridDecoder(BytesUtils.getWidthFromMaxInt((int)this.field.getDefinitionLevel()));
            this.definitionLevelDecoder.init(new SimpleSliceInputStream(buffer.slice(4, bufferSize)));
            buffer = buffer.slice(bufferSize + 4, buffer.length() - bufferSize - 4);
        } else {
            this.definitionLevelDecoder = new ValueDecoder.EmptyValueDecoder<int[]>();
        }
        this.valueDecoder = this.createValueDecoder(this.decodersProvider, page.getValueEncoding(), buffer);
    }

    private void readFlatPageV2(DataPageV2 page) {
        int maxDefinitionLevel = this.field.getDefinitionLevel();
        int maxRepetitionLevel = this.field.getRepetitionLevel();
        if (maxDefinitionLevel == 0) {
            this.definitionLevelDecoder = new ValueDecoder.EmptyValueDecoder<int[]>();
        } else {
            this.definitionLevelDecoder = new RleBitPackingHybridDecoder(BytesUtils.getWidthFromMaxInt((int)maxDefinitionLevel));
            this.definitionLevelDecoder.init(new SimpleSliceInputStream(page.getDefinitionLevels()));
        }
        if (maxRepetitionLevel == 0) {
            this.repetitionLevelDecoder = new ValueDecoder.EmptyValueDecoder<int[]>();
        } else {
            this.repetitionLevelDecoder = new RleBitPackingHybridDecoder(BytesUtils.getWidthFromMaxInt((int)maxRepetitionLevel));
            this.repetitionLevelDecoder.init(new SimpleSliceInputStream(page.getRepetitionLevels()));
        }
        this.valueDecoder = this.createValueDecoder(this.decodersProvider, page.getDataEncoding(), page.getSlice());
    }

    private NonNullValuesBuffer<BufferType> createNonNullValuesBuffer() {
        if (this.produceDictionaryBlock()) {
            return new DictionaryValuesBuffer(this.field, this.dictionaryDecoder);
        }
        return new DataValuesBuffer(this.field, this.columnAdapter);
    }

    private NullableValuesBuffer<BufferType> createNullableValuesBuffer() {
        if (this.produceDictionaryBlock()) {
            return new DictionaryValuesBuffer(this.field, this.dictionaryDecoder);
        }
        return new DataValuesBuffer(this.field, this.columnAdapter);
    }

    private static interface NullableValuesBuffer<T> {
        public void readNullableValues(ValueDecoder<T> var1, boolean[] var2, int var3, int var4);

        public ColumnChunk createNullableBlock(BooleansBuffer var1, int[] var2, int[] var3);
    }

    private static class BooleansBuffer {
        private final List<boolean[]> buffers = new ArrayList<boolean[]>();

        private BooleansBuffer() {
        }

        private void add(boolean[] buffer) {
            this.buffers.add(buffer);
        }

        private boolean[] getMergedBuffer() {
            if (this.buffers.size() == 1) {
                return this.buffers.get(0);
            }
            return Booleans.concat((boolean[][])((boolean[][])this.buffers.toArray(x$0 -> new boolean[x$0][])));
        }
    }

    private static class IntegersBuffer {
        private final List<int[]> buffers = new ArrayList<int[]>();

        private IntegersBuffer() {
        }

        private void add(int[] buffer) {
            this.buffers.add(buffer);
        }

        private int[] getMergedBuffer() {
            if (this.buffers.size() == 1) {
                return this.buffers.get(0);
            }
            return Ints.concat((int[][])((int[][])this.buffers.toArray(x$0 -> new int[x$0][])));
        }
    }

    record ValueCount(int rows, int values) {
    }

    private static interface NonNullValuesBuffer<T> {
        public void readNonNullValues(ValueDecoder<T> var1, int var2);

        public ColumnChunk createNonNullBlock(int[] var1, int[] var2);
    }

    private static final class DictionaryValuesBuffer<T>
    implements NonNullValuesBuffer<T>,
    NullableValuesBuffer<T> {
        private final PrimitiveField field;
        private final DictionaryDecoder<T> dictionaryDecoder;
        private final IntegersBuffer ids = new IntegersBuffer();
        private int totalExistingValueCount;
        private int totalNonNullsCount;

        private DictionaryValuesBuffer(PrimitiveField field, DictionaryDecoder<T> dictionaryDecoder) {
            this.field = field;
            this.dictionaryDecoder = dictionaryDecoder;
        }

        @Override
        public void readNonNullValues(ValueDecoder<T> valueDecoder, int existingValueCount) {
            int[] positionsBuffer = new int[existingValueCount];
            this.dictionaryDecoder.readDictionaryIds(positionsBuffer, 0, existingValueCount);
            this.ids.add(positionsBuffer);
            this.totalNonNullsCount += existingValueCount;
            this.totalExistingValueCount += existingValueCount;
        }

        @Override
        public void readNullableValues(ValueDecoder<T> valueDecoder, boolean[] isNull, int nonNullCount, int existingValueCount) {
            if (nonNullCount > 0 && nonNullCount == existingValueCount) {
                this.readNonNullValues(valueDecoder, existingValueCount);
                return;
            }
            if (nonNullCount == 0) {
                int[] dummy = new int[existingValueCount];
                Arrays.fill(dummy, this.dictionaryDecoder.getDictionarySize());
                this.ids.add(dummy);
            } else {
                int[] tmpBuffer = new int[nonNullCount];
                this.dictionaryDecoder.readDictionaryIds(tmpBuffer, 0, nonNullCount);
                int[] positionsBuffer = new int[existingValueCount];
                AbstractColumnReader.unpackDictionaryNullId(tmpBuffer, positionsBuffer, isNull, 0, existingValueCount, this.dictionaryDecoder.getDictionarySize());
                this.ids.add(positionsBuffer);
            }
            this.totalNonNullsCount += nonNullCount;
            this.totalExistingValueCount += existingValueCount;
        }

        @Override
        public ColumnChunk createNonNullBlock(int[] definitions, int[] repetitions) {
            Preconditions.checkState((this.totalNonNullsCount == this.totalExistingValueCount ? 1 : 0) != 0, (String)"totalNonNullsCount %s should be equal to totalExistingValueCount %s when creating non-null block", (int)this.totalNonNullsCount, (int)this.totalExistingValueCount);
            log.debug("DictionaryValuesBuffer createNonNullBlock field %s, totalNonNullsCount %d, totalExistingValueCount %d", new Object[]{this.field, this.totalNonNullsCount, this.totalExistingValueCount});
            return AbstractColumnReader.createDictionaryBlock(this.ids.getMergedBuffer(), this.dictionaryDecoder.getDictionaryBlock(), definitions, repetitions);
        }

        @Override
        public ColumnChunk createNullableBlock(BooleansBuffer isNull, int[] definitions, int[] repetitions) {
            log.debug("DictionaryValuesBuffer createNullableBlock field %s, totalNonNullsCount %d, totalExistingValueCount %d", new Object[]{this.field, this.totalNonNullsCount, this.totalExistingValueCount});
            if (this.totalNonNullsCount == 0) {
                return new ColumnChunk(RunLengthEncodedBlock.create((Type)this.field.getType(), null, (int)this.totalExistingValueCount), definitions, repetitions);
            }
            return AbstractColumnReader.createDictionaryBlock(this.ids.getMergedBuffer(), this.dictionaryDecoder.getDictionaryBlock(), definitions, repetitions);
        }
    }

    private static final class DataValuesBuffer<T>
    implements NonNullValuesBuffer<T>,
    NullableValuesBuffer<T> {
        private final PrimitiveField field;
        private final ColumnAdapter<T> columnAdapter;
        private final List<T> valueBuffers = new ArrayList<T>();
        private int totalExistingValueCount;
        private int totalNonNullsCount;

        private DataValuesBuffer(PrimitiveField field, ColumnAdapter<T> columnAdapter) {
            this.field = field;
            this.columnAdapter = columnAdapter;
        }

        @Override
        public void readNonNullValues(ValueDecoder<T> valueDecoder, int existingValueCount) {
            T valueBuffer = this.columnAdapter.createBuffer(existingValueCount);
            valueDecoder.read(valueBuffer, 0, existingValueCount);
            this.valueBuffers.add(valueBuffer);
            this.totalNonNullsCount += existingValueCount;
            this.totalExistingValueCount += existingValueCount;
        }

        @Override
        public void readNullableValues(ValueDecoder<T> valueDecoder, boolean[] isNull, int nonNullCount, int existingValueCount) {
            if (nonNullCount > 0 && nonNullCount == existingValueCount) {
                this.readNonNullValues(valueDecoder, existingValueCount);
                return;
            }
            if (nonNullCount == 0) {
                this.valueBuffers.add(this.columnAdapter.createBuffer(existingValueCount));
            } else {
                T outBuffer = this.columnAdapter.createBuffer(existingValueCount);
                T tmpBuffer = this.columnAdapter.createTemporaryBuffer(0, nonNullCount, outBuffer);
                valueDecoder.read(tmpBuffer, 0, nonNullCount);
                this.columnAdapter.unpackNullValues(tmpBuffer, outBuffer, isNull, 0, nonNullCount, existingValueCount);
                this.valueBuffers.add(outBuffer);
            }
            this.totalNonNullsCount += nonNullCount;
            this.totalExistingValueCount += existingValueCount;
        }

        @Override
        public ColumnChunk createNonNullBlock(int[] definitions, int[] repetitions) {
            Preconditions.checkState((this.totalNonNullsCount == this.totalExistingValueCount ? 1 : 0) != 0, (String)"totalNonNullsCount %s should be equal to totalExistingValueCount %s when creating non-null block", (int)this.totalNonNullsCount, (int)this.totalExistingValueCount);
            log.debug("DataValuesBuffer createNonNullBlock field %s, totalNonNullsCount %d, totalExistingValueCount %d", new Object[]{this.field, this.totalNonNullsCount, this.totalExistingValueCount});
            return new ColumnChunk(this.columnAdapter.createNonNullBlock(this.getMergedValues()), definitions, repetitions);
        }

        @Override
        public ColumnChunk createNullableBlock(BooleansBuffer isNull, int[] definitions, int[] repetitions) {
            log.debug("DataValuesBuffer createNullableBlock field %s, totalNonNullsCount %d, totalExistingValueCount %d", new Object[]{this.field, this.totalNonNullsCount, this.totalExistingValueCount});
            if (this.totalNonNullsCount == 0) {
                return new ColumnChunk(RunLengthEncodedBlock.create((Type)this.field.getType(), null, (int)this.totalExistingValueCount), definitions, repetitions);
            }
            if (this.totalNonNullsCount == this.totalExistingValueCount) {
                return new ColumnChunk(this.columnAdapter.createNonNullBlock(this.getMergedValues()), definitions, repetitions);
            }
            return new ColumnChunk(this.columnAdapter.createNullableBlock(isNull.getMergedBuffer(), this.getMergedValues()), definitions, repetitions);
        }

        private T getMergedValues() {
            if (this.valueBuffers.size() == 1) {
                return this.valueBuffers.get(0);
            }
            return this.columnAdapter.merge(this.valueBuffers);
        }
    }
}

