/*
 * Decompiled with CFR 0.152.
 */
package io.trino.parquet.writer;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import io.airlift.slice.DynamicSliceOutput;
import io.airlift.slice.OutputStreamSliceOutput;
import io.airlift.slice.SizeOf;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceOutput;
import io.airlift.slice.Slices;
import io.trino.memory.context.AggregatedMemoryContext;
import io.trino.parquet.Field;
import io.trino.parquet.ParquetCorruptionException;
import io.trino.parquet.ParquetDataSource;
import io.trino.parquet.ParquetReaderOptions;
import io.trino.parquet.ParquetTypeUtils;
import io.trino.parquet.ParquetWriteValidation;
import io.trino.parquet.reader.MetadataReader;
import io.trino.parquet.reader.ParquetReader;
import io.trino.parquet.writer.ColumnChunk;
import io.trino.parquet.writer.ColumnWriter;
import io.trino.parquet.writer.MessageTypeConverter;
import io.trino.parquet.writer.ParquetDataOutput;
import io.trino.parquet.writer.ParquetWriterOptions;
import io.trino.parquet.writer.ParquetWriters;
import io.trino.spi.Page;
import io.trino.spi.type.Type;
import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import org.apache.parquet.column.ParquetProperties;
import org.apache.parquet.format.ColumnMetaData;
import org.apache.parquet.format.CompressionCodec;
import org.apache.parquet.format.KeyValue;
import org.apache.parquet.format.RowGroup;
import org.apache.parquet.format.Util;
import org.apache.parquet.hadoop.metadata.BlockMetaData;
import org.apache.parquet.hadoop.metadata.FileMetaData;
import org.apache.parquet.hadoop.metadata.ParquetMetadata;
import org.apache.parquet.io.GroupColumnIO;
import org.apache.parquet.io.MessageColumnIO;
import org.apache.parquet.schema.MessageType;
import org.joda.time.DateTimeZone;

public class ParquetWriter
implements Closeable {
    private static final int INSTANCE_SIZE = SizeOf.instanceSize(ParquetWriter.class);
    private final OutputStreamSliceOutput outputStream;
    private final ParquetWriterOptions writerOption;
    private final MessageType messageType;
    private final String createdBy;
    private final int chunkMaxLogicalBytes;
    private final Map<List<String>, Type> primitiveTypes;
    private final CompressionCodec compressionCodec;
    private final Optional<DateTimeZone> parquetTimeZone;
    private final ImmutableList.Builder<RowGroup> rowGroupBuilder = ImmutableList.builder();
    private final Optional<ParquetWriteValidation.ParquetWriteValidationBuilder> validationBuilder;
    private List<ColumnWriter> columnWriters;
    private int rows;
    private long bufferedBytes;
    private boolean closed;
    private boolean writeHeader;
    public static final Slice MAGIC = Slices.wrappedBuffer((byte[])"PAR1".getBytes(StandardCharsets.US_ASCII));

    public ParquetWriter(OutputStream outputStream, MessageType messageType, Map<List<String>, Type> primitiveTypes, ParquetWriterOptions writerOption, CompressionCodec compressionCodec, String trinoVersion, Optional<DateTimeZone> parquetTimeZone, Optional<ParquetWriteValidation.ParquetWriteValidationBuilder> validationBuilder) {
        this.validationBuilder = Objects.requireNonNull(validationBuilder, "validationBuilder is null");
        this.outputStream = new OutputStreamSliceOutput(Objects.requireNonNull(outputStream, "outputstream is null"));
        this.messageType = Objects.requireNonNull(messageType, "messageType is null");
        this.primitiveTypes = Objects.requireNonNull(primitiveTypes, "primitiveTypes is null");
        this.writerOption = Objects.requireNonNull(writerOption, "writerOption is null");
        this.compressionCodec = Objects.requireNonNull(compressionCodec, "compressionCodec is null");
        this.parquetTimeZone = Objects.requireNonNull(parquetTimeZone, "parquetTimeZone is null");
        this.createdBy = ParquetWriter.formatCreatedBy(Objects.requireNonNull(trinoVersion, "trinoVersion is null"));
        this.recordValidation(validation -> validation.setTimeZone(parquetTimeZone.map(DateTimeZone::getID)));
        this.recordValidation(validation -> validation.setColumns(messageType.getColumns()));
        this.recordValidation(validation -> validation.setCreatedBy(this.createdBy));
        this.initColumnWriters();
        this.chunkMaxLogicalBytes = Math.max(1, writerOption.getMaxRowGroupSize() / 2);
    }

    public long getWrittenBytes() {
        return this.outputStream.longSize();
    }

    public long getBufferedBytes() {
        return this.bufferedBytes;
    }

    public long getRetainedBytes() {
        return (long)INSTANCE_SIZE + this.outputStream.getRetainedSize() + this.columnWriters.stream().mapToLong(ColumnWriter::getRetainedBytes).sum() + this.validationBuilder.map(ParquetWriteValidation.ParquetWriteValidationBuilder::getRetainedSize).orElse(0L);
    }

    public void write(Page page) throws IOException {
        Page chunk;
        Objects.requireNonNull(page, "page is null");
        Preconditions.checkState((!this.closed ? 1 : 0) != 0, (Object)"writer is closed");
        if (page.getPositionCount() == 0) {
            return;
        }
        Preconditions.checkArgument((page.getChannelCount() == this.columnWriters.size() ? 1 : 0) != 0);
        Page validationPage = page;
        this.recordValidation(validation -> validation.addPage(validationPage));
        for (int writeOffset = 0; writeOffset < page.getPositionCount(); writeOffset += chunk.getPositionCount()) {
            chunk = page.getRegion(writeOffset, Math.min(page.getPositionCount() - writeOffset, this.writerOption.getBatchSize()));
            while (chunk.getPositionCount() > 1 && chunk.getLogicalSizeInBytes() > (long)this.chunkMaxLogicalBytes) {
                chunk = page.getRegion(writeOffset, chunk.getPositionCount() / 2);
            }
            this.writeChunk(chunk);
        }
    }

    private void writeChunk(Page page) throws IOException {
        this.bufferedBytes = 0L;
        for (int channel = 0; channel < page.getChannelCount(); ++channel) {
            ColumnWriter writer = this.columnWriters.get(channel);
            writer.writeBlock(new ColumnChunk(page.getBlock(channel)));
            this.bufferedBytes += writer.getBufferedBytes();
        }
        this.rows += page.getPositionCount();
        if (this.bufferedBytes >= (long)this.writerOption.getMaxRowGroupSize()) {
            this.columnWriters.forEach(ColumnWriter::close);
            this.flush();
            this.initColumnWriters();
            this.rows = 0;
            this.bufferedBytes = this.columnWriters.stream().mapToLong(ColumnWriter::getBufferedBytes).sum();
        }
    }

    @Override
    public void close() throws IOException {
        if (this.closed) {
            return;
        }
        this.closed = true;
        try (OutputStreamSliceOutput outputStreamSliceOutput = this.outputStream;){
            this.columnWriters.forEach(ColumnWriter::close);
            this.flush();
            this.columnWriters = ImmutableList.of();
            this.writeFooter();
        }
        this.bufferedBytes = 0L;
    }

    public void validate(ParquetDataSource input) throws ParquetCorruptionException {
        Preconditions.checkState((boolean)this.validationBuilder.isPresent(), (Object)"validation is not enabled");
        ParquetWriteValidation writeValidation = this.validationBuilder.get().build();
        try {
            ParquetMetadata parquetMetadata = MetadataReader.readFooter(input, Optional.of(writeValidation));
            try (ParquetReader parquetReader = this.createParquetReader(input, parquetMetadata, writeValidation);){
                Page page = parquetReader.nextPage();
                while (page != null) {
                    page.getLoadedPage();
                    page = parquetReader.nextPage();
                }
            }
        }
        catch (IOException e) {
            if (e instanceof ParquetCorruptionException) {
                throw (ParquetCorruptionException)e;
            }
            throw new ParquetCorruptionException(input.getId(), "Validation failed with exception %s", e);
        }
    }

    private ParquetReader createParquetReader(ParquetDataSource input, ParquetMetadata parquetMetadata, ParquetWriteValidation writeValidation) throws IOException {
        FileMetaData fileMetaData = parquetMetadata.getFileMetaData();
        MessageColumnIO messageColumnIO = ParquetTypeUtils.getColumnIO(fileMetaData.getSchema(), fileMetaData.getSchema());
        ImmutableList.Builder columnFields = ImmutableList.builder();
        for (int i = 0; i < writeValidation.getTypes().size(); ++i) {
            columnFields.add((Object)ParquetTypeUtils.constructField(writeValidation.getTypes().get(i), ParquetTypeUtils.lookupColumnByName((GroupColumnIO)messageColumnIO, writeValidation.getColumnNames().get(i))).orElseThrow());
        }
        long nextStart = 0L;
        ImmutableList.Builder blockStartsBuilder = ImmutableList.builder();
        for (BlockMetaData block : parquetMetadata.getBlocks()) {
            blockStartsBuilder.add((Object)nextStart);
            nextStart += block.getRowCount();
        }
        ImmutableList blockStarts = blockStartsBuilder.build();
        return new ParquetReader(Optional.ofNullable(fileMetaData.getCreatedBy()), (List<Field>)columnFields.build(), parquetMetadata.getBlocks(), (List<Long>)blockStarts, input, this.parquetTimeZone.orElseThrow(), AggregatedMemoryContext.newSimpleAggregatedMemoryContext(), new ParquetReaderOptions(), exception -> {
            Throwables.throwIfUnchecked((Throwable)exception);
            return new RuntimeException((Throwable)exception);
        }, Optional.empty(), Collections.nCopies(blockStarts.size(), Optional.empty()), Optional.of(writeValidation));
    }

    private void recordValidation(Consumer<ParquetWriteValidation.ParquetWriteValidationBuilder> task) {
        this.validationBuilder.ifPresent(task);
    }

    private void flush() throws IOException {
        if (!this.writeHeader) {
            ParquetDataOutput.createDataOutput(MAGIC).writeData((SliceOutput)this.outputStream);
            this.writeHeader = true;
        }
        ImmutableList.Builder builder = ImmutableList.builder();
        for (ColumnWriter columnWriter : this.columnWriters) {
            columnWriter.getBuffer().forEach(arg_0 -> ((ImmutableList.Builder)builder).add(arg_0));
        }
        ImmutableList bufferDataList = builder.build();
        if (this.rows == 0) {
            Verify.verify((boolean)bufferDataList.stream().flatMap(bufferData -> bufferData.getData().stream()).allMatch(dataOutput -> dataOutput.size() == 0), (String)"Buffer should be empty when there are no rows", (Object[])new Object[0]);
            return;
        }
        long stripeStartOffset = this.outputStream.longSize();
        List columns = (List)bufferDataList.stream().map(ColumnWriter.BufferData::getMetaData).collect(ImmutableList.toImmutableList());
        long currentOffset = stripeStartOffset;
        for (ColumnMetaData columnMetaData : columns) {
            columnMetaData.setData_page_offset(currentOffset);
            currentOffset += columnMetaData.getTotal_compressed_size();
        }
        this.updateRowGroups(columns);
        for (ColumnWriter.BufferData bufferData2 : bufferDataList) {
            bufferData2.getData().forEach(data -> data.writeData((SliceOutput)this.outputStream));
        }
    }

    private void writeFooter() throws IOException {
        Preconditions.checkState((boolean)this.closed);
        ImmutableList rowGroups = this.rowGroupBuilder.build();
        Slice footer = this.getFooter((List<RowGroup>)rowGroups, this.messageType);
        this.recordValidation(arg_0 -> ParquetWriter.lambda$writeFooter$8((List)rowGroups, arg_0));
        ParquetDataOutput.createDataOutput(footer).writeData((SliceOutput)this.outputStream);
        Slice footerSize = Slices.allocate((int)4);
        footerSize.setInt(0, footer.length());
        ParquetDataOutput.createDataOutput(footerSize).writeData((SliceOutput)this.outputStream);
        ParquetDataOutput.createDataOutput(MAGIC).writeData((SliceOutput)this.outputStream);
    }

    Slice getFooter(List<RowGroup> rowGroups, MessageType messageType) throws IOException {
        org.apache.parquet.format.FileMetaData fileMetaData = new org.apache.parquet.format.FileMetaData();
        fileMetaData.setVersion(1);
        fileMetaData.setCreated_by(this.createdBy);
        fileMetaData.setSchema(MessageTypeConverter.toParquetSchema(messageType));
        this.parquetTimeZone.ifPresent(dateTimeZone -> fileMetaData.setKey_value_metadata((List)ImmutableList.of((Object)new KeyValue("writer.time.zone").setValue(dateTimeZone.getID()))));
        long totalRows = rowGroups.stream().mapToLong(RowGroup::getNum_rows).sum();
        fileMetaData.setNum_rows(totalRows);
        fileMetaData.setRow_groups((List)ImmutableList.copyOf(rowGroups));
        DynamicSliceOutput dynamicSliceOutput = new DynamicSliceOutput(40);
        Util.writeFileMetaData((org.apache.parquet.format.FileMetaData)fileMetaData, (OutputStream)dynamicSliceOutput);
        return dynamicSliceOutput.slice();
    }

    private void updateRowGroups(List<ColumnMetaData> columnMetaData) {
        long totalCompressedBytes = columnMetaData.stream().mapToLong(ColumnMetaData::getTotal_compressed_size).sum();
        long totalBytes = columnMetaData.stream().mapToLong(ColumnMetaData::getTotal_uncompressed_size).sum();
        ImmutableList columnChunks = (ImmutableList)columnMetaData.stream().map(ParquetWriter::toColumnChunk).collect(ImmutableList.toImmutableList());
        this.rowGroupBuilder.add((Object)new RowGroup((List)columnChunks, totalBytes, (long)this.rows).setTotal_compressed_size(totalCompressedBytes));
    }

    private static org.apache.parquet.format.ColumnChunk toColumnChunk(ColumnMetaData metaData) {
        org.apache.parquet.format.ColumnChunk columnChunk = new org.apache.parquet.format.ColumnChunk(0L);
        columnChunk.setMeta_data(metaData);
        return columnChunk;
    }

    @VisibleForTesting
    static String formatCreatedBy(String trinoVersion) {
        return "parquet-mr-trino version " + trinoVersion + " (build n/a)";
    }

    private void initColumnWriters() {
        ParquetProperties parquetProperties = ParquetProperties.builder().withWriterVersion(ParquetProperties.WriterVersion.PARQUET_1_0).withPageSize(this.writerOption.getMaxPageSize()).build();
        this.columnWriters = ParquetWriters.getColumnWriters(this.messageType, this.primitiveTypes, parquetProperties, this.compressionCodec, this.parquetTimeZone);
    }

    private static /* synthetic */ void lambda$writeFooter$8(List rowGroups, ParquetWriteValidation.ParquetWriteValidationBuilder validation) {
        validation.setRowGroups(rowGroups);
    }
}

