/*
 * Decompiled with CFR 0.152.
 */
package io.trino.parquet.reader;

import com.google.common.base.Preconditions;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.errorprone.annotations.FormatMethod;
import io.airlift.log.Logger;
import io.trino.memory.context.AggregatedMemoryContext;
import io.trino.parquet.ChunkKey;
import io.trino.parquet.Column;
import io.trino.parquet.DiskRange;
import io.trino.parquet.Field;
import io.trino.parquet.GroupField;
import io.trino.parquet.ParquetCorruptionException;
import io.trino.parquet.ParquetDataSource;
import io.trino.parquet.ParquetReaderOptions;
import io.trino.parquet.ParquetValidationUtils;
import io.trino.parquet.ParquetWriteValidation;
import io.trino.parquet.PrimitiveField;
import io.trino.parquet.metadata.BlockMetadata;
import io.trino.parquet.metadata.ColumnChunkMetadata;
import io.trino.parquet.predicate.TupleDomainParquetPredicate;
import io.trino.parquet.reader.ChunkedInputStream;
import io.trino.parquet.reader.ColumnChunk;
import io.trino.parquet.reader.ColumnReader;
import io.trino.parquet.reader.ColumnReaderFactory;
import io.trino.parquet.reader.FilteredOffsetIndex;
import io.trino.parquet.reader.FilteredRowRanges;
import io.trino.parquet.reader.ListColumnReader;
import io.trino.parquet.reader.PageReader;
import io.trino.parquet.reader.ParquetBlockFactory;
import io.trino.parquet.reader.RowGroupInfo;
import io.trino.parquet.reader.StructColumnReader;
import io.trino.plugin.base.metrics.LongCount;
import io.trino.spi.Page;
import io.trino.spi.block.ArrayBlock;
import io.trino.spi.block.Block;
import io.trino.spi.block.DictionaryBlock;
import io.trino.spi.block.MapBlock;
import io.trino.spi.block.RowBlock;
import io.trino.spi.block.RunLengthEncodedBlock;
import io.trino.spi.metrics.Metric;
import io.trino.spi.metrics.Metrics;
import io.trino.spi.type.ArrayType;
import io.trino.spi.type.MapType;
import io.trino.spi.type.RowType;
import io.trino.spi.type.Type;
import jakarta.annotation.Nullable;
import java.io.Closeable;
import java.io.IOException;
import java.lang.invoke.CallSite;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import org.apache.parquet.column.ColumnDescriptor;
import org.apache.parquet.filter2.compat.FilterCompat;
import org.apache.parquet.filter2.predicate.FilterPredicate;
import org.apache.parquet.hadoop.metadata.ColumnPath;
import org.apache.parquet.internal.column.columnindex.OffsetIndex;
import org.apache.parquet.internal.filter2.columnindex.ColumnIndexFilter;
import org.apache.parquet.internal.filter2.columnindex.ColumnIndexStore;
import org.joda.time.DateTimeZone;

public class ParquetReader
implements Closeable {
    private static final Logger log = Logger.get(ParquetReader.class);
    private static final int INITIAL_BATCH_SIZE = 1;
    private static final int BATCH_SIZE_GROWTH_FACTOR = 2;
    public static final String PARQUET_CODEC_METRIC_PREFIX = "ParquetReaderCompressionFormat_";
    public static final String COLUMN_INDEX_ROWS_FILTERED = "ParquetColumnIndexRowsFiltered";
    private final Optional<String> fileCreatedBy;
    private final List<RowGroupInfo> rowGroups;
    private final List<Column> columnFields;
    private final List<PrimitiveField> primitiveFields;
    private final ParquetDataSource dataSource;
    private final ColumnReaderFactory columnReaderFactory;
    private final AggregatedMemoryContext memoryContext;
    private int currentRowGroup = -1;
    private BlockMetadata currentBlockMetadata;
    private long currentGroupRowCount;
    private long firstRowIndexInGroup;
    private long nextRowInGroup;
    private int batchSize;
    private int nextBatchSize = 1;
    private final Map<Integer, ColumnReader> columnReaders;
    private final Map<Integer, Double> maxBytesPerCell;
    private double maxCombinedBytesPerRow;
    private final ParquetReaderOptions options;
    private int maxBatchSize;
    private AggregatedMemoryContext currentRowGroupMemoryContext;
    private final Map<ChunkKey, ChunkedInputStream> chunkReaders;
    private final Optional<ParquetWriteValidation> writeValidation;
    private final Optional<ParquetWriteValidation.WriteChecksumBuilder> writeChecksumBuilder;
    private final Optional<ParquetWriteValidation.StatisticsValidation> rowGroupStatisticsValidation;
    private final FilteredRowRanges[] blockRowRanges;
    private final ParquetBlockFactory blockFactory;
    private final Map<String, Metric<?>> codecMetrics;
    private long columnIndexRowsFiltered = -1L;

    public ParquetReader(Optional<String> fileCreatedBy, List<Column> columnFields, List<RowGroupInfo> rowGroups, ParquetDataSource dataSource, DateTimeZone timeZone, AggregatedMemoryContext memoryContext, ParquetReaderOptions options, Function<Exception, RuntimeException> exceptionTransform, Optional<TupleDomainParquetPredicate> parquetPredicate, Optional<ParquetWriteValidation> writeValidation) throws IOException {
        this.fileCreatedBy = Objects.requireNonNull(fileCreatedBy, "fileCreatedBy is null");
        Objects.requireNonNull(columnFields, "columnFields is null");
        this.columnFields = ImmutableList.copyOf(columnFields);
        this.primitiveFields = ParquetReader.getPrimitiveFields((List)columnFields.stream().map(Column::field).collect(ImmutableList.toImmutableList()));
        this.rowGroups = Objects.requireNonNull(rowGroups, "rowGroups is null");
        this.dataSource = Objects.requireNonNull(dataSource, "dataSource is null");
        this.columnReaderFactory = new ColumnReaderFactory(timeZone);
        this.memoryContext = Objects.requireNonNull(memoryContext, "memoryContext is null");
        this.currentRowGroupMemoryContext = memoryContext.newAggregatedMemoryContext();
        this.options = Objects.requireNonNull(options, "options is null");
        this.maxBatchSize = options.getMaxReadBlockRowCount();
        this.columnReaders = new HashMap<Integer, ColumnReader>();
        this.maxBytesPerCell = new HashMap<Integer, Double>();
        this.writeValidation = Objects.requireNonNull(writeValidation, "writeValidation is null");
        this.validateWrite(validation -> fileCreatedBy.equals(Optional.of(validation.getCreatedBy())), "Expected created by %s, found %s", writeValidation.map(ParquetWriteValidation::getCreatedBy), fileCreatedBy);
        this.validateBlockMetadata(rowGroups);
        this.writeChecksumBuilder = writeValidation.map(validation -> ParquetWriteValidation.WriteChecksumBuilder.createWriteChecksumBuilder(validation.getTypes()));
        this.rowGroupStatisticsValidation = writeValidation.map(validation -> ParquetWriteValidation.StatisticsValidation.createStatisticsValidationBuilder(validation.getTypes()));
        Objects.requireNonNull(parquetPredicate, "parquetPredicate is null");
        Optional<FilterPredicate> filter = Optional.empty();
        if (parquetPredicate.isPresent() && options.isUseColumnIndex()) {
            filter = parquetPredicate.get().toParquetFilter(timeZone);
        }
        this.blockRowRanges = ParquetReader.calculateFilteredRowRanges(rowGroups, filter, this.primitiveFields);
        this.blockFactory = new ParquetBlockFactory(exceptionTransform);
        ArrayListMultimap ranges = ArrayListMultimap.create();
        HashMap<CallSite, LongCount> codecMetrics = new HashMap<CallSite, LongCount>();
        for (int rowGroup = 0; rowGroup < rowGroups.size(); ++rowGroup) {
            BlockMetadata metadata = rowGroups.get(rowGroup).blockMetaData();
            for (PrimitiveField field : this.primitiveFields) {
                int columnId = field.getId();
                ColumnChunkMetadata chunkMetadata = this.getColumnChunkMetaData(metadata, field.getDescriptor());
                ColumnPath columnPath = chunkMetadata.getPath();
                long rowGroupRowCount = metadata.getRowCount();
                long startingPosition = chunkMetadata.getStartingPos();
                long totalLength = chunkMetadata.getTotalSize();
                long totalDataSize = 0L;
                FilteredOffsetIndex filteredOffsetIndex = null;
                if (this.blockRowRanges[rowGroup] != null) {
                    filteredOffsetIndex = this.getFilteredOffsetIndex(this.blockRowRanges[rowGroup], rowGroup, rowGroupRowCount, columnPath);
                }
                if (filteredOffsetIndex == null) {
                    DiskRange range = new DiskRange(startingPosition, totalLength);
                    totalDataSize = range.getLength();
                    ranges.put((Object)new ChunkKey(columnId, rowGroup), (Object)range);
                } else {
                    List<FilteredOffsetIndex.OffsetRange> offsetRanges = filteredOffsetIndex.calculateOffsetRanges(startingPosition);
                    for (FilteredOffsetIndex.OffsetRange offsetRange : offsetRanges) {
                        DiskRange range = new DiskRange(offsetRange.getOffset(), offsetRange.getLength());
                        totalDataSize += range.getLength();
                        ranges.put((Object)new ChunkKey(columnId, rowGroup), (Object)range);
                    }
                    this.columnIndexRowsFiltered = 0L;
                }
                codecMetrics.merge((CallSite)((Object)(PARQUET_CODEC_METRIC_PREFIX + chunkMetadata.getCodec().name())), new LongCount(totalDataSize), LongCount::mergeWith);
            }
        }
        this.codecMetrics = ImmutableMap.copyOf(codecMetrics);
        this.chunkReaders = dataSource.planRead(ranges, memoryContext);
    }

    @Override
    public void close() throws IOException {
        this.columnReaders.clear();
        this.currentRowGroupMemoryContext.close();
        for (ChunkedInputStream chunkedInputStream : this.chunkReaders.values()) {
            chunkedInputStream.close();
        }
        this.dataSource.close();
        if (this.writeChecksumBuilder.isPresent()) {
            ParquetWriteValidation parquetWriteValidation = this.writeValidation.orElseThrow();
            parquetWriteValidation.validateChecksum(this.dataSource.getId(), this.writeChecksumBuilder.get().build());
        }
    }

    public Page nextPage() throws IOException {
        int batchSize = this.nextBatch();
        if (batchSize <= 0) {
            return null;
        }
        this.blockFactory.nextPage();
        Block[] blocks = new Block[this.columnFields.size()];
        for (int channel = 0; channel < this.columnFields.size(); ++channel) {
            Field field = this.columnFields.get(channel).field();
            blocks[channel] = this.blockFactory.createBlock(batchSize, () -> this.readBlock(field));
        }
        Page page = new Page(batchSize, blocks);
        this.validateWritePageChecksum(page);
        return page;
    }

    public long lastBatchStartRow() {
        return this.firstRowIndexInGroup + this.nextRowInGroup - (long)this.batchSize;
    }

    private int nextBatch() throws IOException {
        if (this.nextRowInGroup >= this.currentGroupRowCount && !this.advanceToNextRowGroup()) {
            return -1;
        }
        this.batchSize = Math.min(this.nextBatchSize, this.maxBatchSize);
        this.nextBatchSize = Math.min(this.batchSize * 2, this.options.getMaxReadBlockRowCount());
        this.batchSize = Math.toIntExact(Math.min((long)this.batchSize, this.currentGroupRowCount - this.nextRowInGroup));
        this.nextRowInGroup += (long)this.batchSize;
        this.columnReaders.values().forEach(reader -> reader.prepareNextRead(this.batchSize));
        return this.batchSize;
    }

    private boolean advanceToNextRowGroup() throws IOException {
        this.currentRowGroupMemoryContext.close();
        this.currentRowGroupMemoryContext = this.memoryContext.newAggregatedMemoryContext();
        this.freeCurrentRowGroupBuffers();
        if (this.currentRowGroup >= 0 && this.rowGroupStatisticsValidation.isPresent()) {
            ParquetWriteValidation.StatisticsValidation statisticsValidation = this.rowGroupStatisticsValidation.get();
            this.writeValidation.orElseThrow().validateRowGroupStatistics(this.dataSource.getId(), this.currentBlockMetadata, statisticsValidation.build());
            statisticsValidation.reset();
        }
        ++this.currentRowGroup;
        if (this.currentRowGroup == this.rowGroups.size()) {
            return false;
        }
        RowGroupInfo rowGroupInfo = this.rowGroups.get(this.currentRowGroup);
        this.currentBlockMetadata = rowGroupInfo.blockMetaData();
        this.firstRowIndexInGroup = rowGroupInfo.fileRowOffset();
        this.currentGroupRowCount = this.currentBlockMetadata.getRowCount();
        FilteredRowRanges currentGroupRowRanges = this.blockRowRanges[this.currentRowGroup];
        log.debug("advanceToNextRowGroup dataSource %s, currentRowGroup %d, rowRanges %s, currentBlockMetadata %s", new Object[]{this.dataSource.getId(), this.currentRowGroup, currentGroupRowRanges, this.currentBlockMetadata});
        if (currentGroupRowRanges != null) {
            long rowCount = currentGroupRowRanges.getRowCount();
            this.columnIndexRowsFiltered += this.currentGroupRowCount - rowCount;
            if (rowCount == 0L) {
                return this.advanceToNextRowGroup();
            }
            this.currentGroupRowCount = rowCount;
        }
        this.nextRowInGroup = 0L;
        this.initializeColumnReaders();
        return true;
    }

    private void freeCurrentRowGroupBuffers() {
        if (this.currentRowGroup < 0) {
            return;
        }
        for (int column = 0; column < this.primitiveFields.size(); ++column) {
            ChunkedInputStream chunkedStream = this.chunkReaders.get(new ChunkKey(column, this.currentRowGroup));
            if (chunkedStream == null) continue;
            chunkedStream.close();
        }
    }

    private ColumnChunk readArray(GroupField field) throws IOException {
        List parameters = field.getType().getTypeParameters();
        Preconditions.checkArgument((parameters.size() == 1 ? 1 : 0) != 0, (String)"Arrays must have a single type parameter, found %s", (int)parameters.size());
        Field elementField = field.getChildren().get(0).get();
        ColumnChunk columnChunk = this.readColumnChunk(elementField);
        ListColumnReader.BlockPositions collectionPositions = ListColumnReader.calculateCollectionOffsets(field, columnChunk.getDefinitionLevels(), columnChunk.getRepetitionLevels());
        int positionsCount = collectionPositions.offsets().length - 1;
        ArrayBlock arrayBlock = ArrayBlock.fromElementBlock((int)positionsCount, collectionPositions.isNull(), (int[])collectionPositions.offsets(), (Block)columnChunk.getBlock());
        return new ColumnChunk((Block)arrayBlock, columnChunk.getDefinitionLevels(), columnChunk.getRepetitionLevels());
    }

    private ColumnChunk readMap(GroupField field) throws IOException {
        List parameters = field.getType().getTypeParameters();
        Preconditions.checkArgument((parameters.size() == 2 ? 1 : 0) != 0, (String)"Maps must have two type parameters, found %s", (int)parameters.size());
        Block[] blocks = new Block[parameters.size()];
        ColumnChunk columnChunk = this.readColumnChunk(field.getChildren().get(0).get());
        blocks[0] = columnChunk.getBlock();
        blocks[1] = this.readColumnChunk(field.getChildren().get(1).get()).getBlock();
        ListColumnReader.BlockPositions collectionPositions = ListColumnReader.calculateCollectionOffsets(field, columnChunk.getDefinitionLevels(), columnChunk.getRepetitionLevels());
        MapBlock mapBlock = ((MapType)field.getType()).createBlockFromKeyValue(collectionPositions.isNull(), collectionPositions.offsets(), blocks[0], blocks[1]);
        return new ColumnChunk((Block)mapBlock, columnChunk.getDefinitionLevels(), columnChunk.getRepetitionLevels());
    }

    private ColumnChunk readStruct(GroupField field) throws IOException {
        Block[] blocks = new Block[field.getType().getTypeParameters().size()];
        ColumnChunk columnChunk = null;
        List<Optional<Field>> parameters = field.getChildren();
        for (int i = 0; i < blocks.length; ++i) {
            Optional<Field> parameter = parameters.get(i);
            if (!parameter.isPresent()) continue;
            columnChunk = this.readColumnChunk(parameter.get());
            blocks[i] = columnChunk.getBlock();
        }
        if (columnChunk == null) {
            throw new ParquetCorruptionException(this.dataSource.getId(), "Struct field does not have any children: %s", field);
        }
        StructColumnReader.RowBlockPositions structIsNull = StructColumnReader.calculateStructOffsets(field, columnChunk.getDefinitionLevels(), columnChunk.getRepetitionLevels());
        Optional<boolean[]> isNull = structIsNull.isNull();
        for (int i = 0; i < blocks.length; ++i) {
            if (blocks[i] == null) {
                blocks[i] = RunLengthEncodedBlock.create((Type)((Type)field.getType().getTypeParameters().get(i)), null, (int)structIsNull.positionsCount());
                continue;
            }
            if (!isNull.isPresent()) continue;
            blocks[i] = ParquetReader.toNotNullSupressedBlock(structIsNull.positionsCount(), isNull.get(), blocks[i]);
        }
        RowBlock rowBlock = RowBlock.fromNotNullSuppressedFieldBlocks((int)structIsNull.positionsCount(), structIsNull.isNull(), (Block[])blocks);
        return new ColumnChunk((Block)rowBlock, columnChunk.getDefinitionLevels(), columnChunk.getRepetitionLevels());
    }

    private static Block toNotNullSupressedBlock(int positionCount, boolean[] rowIsNull, Block fieldBlock) {
        int nullIndex = -1;
        if (fieldBlock.mayHaveNull()) {
            for (int position = 0; position < fieldBlock.getPositionCount(); ++position) {
                if (!fieldBlock.isNull(position)) continue;
                nullIndex = position;
                break;
            }
        }
        if (nullIndex == -1) {
            fieldBlock = fieldBlock.getLoadedBlock();
            nullIndex = fieldBlock.getPositionCount();
            fieldBlock = fieldBlock.copyWithAppendedNull();
        }
        int[] dictionaryIds = new int[positionCount];
        int nullSuppressedPosition = 0;
        for (int position = 0; position < positionCount; ++position) {
            dictionaryIds[position] = rowIsNull[position] ? nullIndex : nullSuppressedPosition++;
        }
        return DictionaryBlock.create((int)positionCount, (Block)fieldBlock, (int[])dictionaryIds);
    }

    @Nullable
    private FilteredOffsetIndex getFilteredOffsetIndex(FilteredRowRanges rowRanges, int rowGroup, long rowGroupRowCount, ColumnPath columnPath) {
        Optional<ColumnIndexStore> rowGroupColumnIndexStore = this.rowGroups.get(rowGroup).columnIndexStore();
        if (rowGroupColumnIndexStore.isEmpty()) {
            return null;
        }
        OffsetIndex offsetIndex = Objects.requireNonNull(rowGroupColumnIndexStore.get().getOffsetIndex(columnPath), String.format("Missing OffsetIndex for column %s", columnPath));
        return FilteredOffsetIndex.filterOffsetIndex(offsetIndex, rowRanges.getParquetRowRanges(), rowGroupRowCount);
    }

    private ColumnChunk readPrimitive(PrimitiveField field) throws IOException {
        ColumnChunk columnChunk;
        double bytesPerCell;
        double bytesPerCellDelta;
        ColumnDescriptor columnDescriptor = field.getDescriptor();
        int fieldId = field.getId();
        ColumnReader columnReader = this.columnReaders.get(fieldId);
        if (!columnReader.hasPageReader()) {
            ParquetValidationUtils.validateParquet(this.currentBlockMetadata.getRowCount() > 0L, this.dataSource.getId(), "Row group has 0 rows", new Object[0]);
            ColumnChunkMetadata metadata = this.getColumnChunkMetaData(this.currentBlockMetadata, columnDescriptor);
            FilteredRowRanges rowRanges = this.blockRowRanges[this.currentRowGroup];
            FilteredOffsetIndex offsetIndex = null;
            if (rowRanges != null) {
                offsetIndex = this.getFilteredOffsetIndex(rowRanges, this.currentRowGroup, this.currentBlockMetadata.getRowCount(), metadata.getPath());
            }
            ChunkedInputStream columnChunkInputStream = this.chunkReaders.get(new ChunkKey(fieldId, this.currentRowGroup));
            columnReader.setPageReader(PageReader.createPageReader(this.dataSource.getId(), columnChunkInputStream, metadata, columnDescriptor, offsetIndex, this.fileCreatedBy), Optional.ofNullable(rowRanges));
        }
        if ((bytesPerCellDelta = (bytesPerCell = (double)(columnChunk = columnReader.readPrimitive()).getMaxBlockSize() / (double)this.batchSize) - this.maxBytesPerCell.getOrDefault(fieldId, 0.0)) > 0.0) {
            this.maxCombinedBytesPerRow += bytesPerCellDelta;
            this.maxBatchSize = Math.toIntExact(Math.min((long)this.maxBatchSize, Math.max(1L, (long)((double)this.options.getMaxReadBlockSize().toBytes() / this.maxCombinedBytesPerRow))));
            this.maxBytesPerCell.put(fieldId, bytesPerCell);
        }
        return columnChunk;
    }

    public List<Column> getColumnFields() {
        return this.columnFields;
    }

    public Metrics getMetrics() {
        ImmutableMap.Builder metrics = ImmutableMap.builder().putAll(this.codecMetrics);
        if (this.columnIndexRowsFiltered >= 0L) {
            metrics.put((Object)COLUMN_INDEX_ROWS_FILTERED, (Object)new LongCount(this.columnIndexRowsFiltered));
        }
        return new Metrics((Map)metrics.buildOrThrow());
    }

    private ColumnChunkMetadata getColumnChunkMetaData(BlockMetadata blockMetaData, ColumnDescriptor columnDescriptor) throws IOException {
        for (ColumnChunkMetadata metadata : blockMetaData.getColumns()) {
            if (!metadata.getPath().equals((Object)ColumnPath.get((String[])columnDescriptor.getPath()))) continue;
            return metadata;
        }
        throw new ParquetCorruptionException(this.dataSource.getId(), "Metadata is missing for column: %s", columnDescriptor);
    }

    private void initializeColumnReaders() {
        for (PrimitiveField field : this.primitiveFields) {
            this.columnReaders.put(field.getId(), this.columnReaderFactory.create(field, this.currentRowGroupMemoryContext));
        }
    }

    public static List<PrimitiveField> getPrimitiveFields(List<Field> fields) {
        HashMap primitiveFields = new HashMap();
        fields.forEach(field -> ParquetReader.parseField(field, primitiveFields));
        return ImmutableList.copyOf(primitiveFields.values());
    }

    private static void parseField(Field field, Map<Integer, PrimitiveField> primitiveFields) {
        if (field instanceof PrimitiveField) {
            PrimitiveField primitiveField = (PrimitiveField)field;
            primitiveFields.put(primitiveField.getId(), primitiveField);
        } else if (field instanceof GroupField) {
            GroupField groupField = (GroupField)field;
            groupField.getChildren().stream().flatMap(Optional::stream).forEach(child -> ParquetReader.parseField(child, primitiveFields));
        }
    }

    public Block readBlock(Field field) throws IOException {
        return this.readColumnChunk(field).getBlock();
    }

    private ColumnChunk readColumnChunk(Field field) throws IOException {
        ColumnChunk columnChunk = field.getType() instanceof RowType ? this.readStruct((GroupField)field) : (field.getType() instanceof MapType ? this.readMap((GroupField)field) : (field.getType() instanceof ArrayType ? this.readArray((GroupField)field) : this.readPrimitive((PrimitiveField)field)));
        return columnChunk;
    }

    public ParquetDataSource getDataSource() {
        return this.dataSource;
    }

    public AggregatedMemoryContext getMemoryContext() {
        return this.memoryContext;
    }

    private static FilteredRowRanges[] calculateFilteredRowRanges(List<RowGroupInfo> rowGroups, Optional<FilterPredicate> filter, List<PrimitiveField> primitiveFields) {
        FilteredRowRanges[] blockRowRanges = new FilteredRowRanges[rowGroups.size()];
        if (filter.isEmpty()) {
            return blockRowRanges;
        }
        Set paths = (Set)primitiveFields.stream().map(field -> ColumnPath.get((String[])field.getDescriptor().getPath())).collect(ImmutableSet.toImmutableSet());
        for (int rowGroup = 0; rowGroup < rowGroups.size(); ++rowGroup) {
            RowGroupInfo rowGroupInfo = rowGroups.get(rowGroup);
            Optional<ColumnIndexStore> rowGroupColumnIndexStore = rowGroupInfo.columnIndexStore();
            if (rowGroupColumnIndexStore.isEmpty()) continue;
            BlockMetadata metadata = rowGroupInfo.blockMetaData();
            long rowGroupRowCount = metadata.getRowCount();
            FilteredRowRanges rowRanges = new FilteredRowRanges(ColumnIndexFilter.calculateRowRanges((FilterCompat.Filter)FilterCompat.get((FilterPredicate)filter.get()), (ColumnIndexStore)rowGroupColumnIndexStore.get(), (Set)paths, (long)rowGroupRowCount));
            if (rowRanges.getRowCount() >= rowGroupRowCount) continue;
            blockRowRanges[rowGroup] = rowRanges;
        }
        return blockRowRanges;
    }

    private void validateWritePageChecksum(Page page) {
        if (this.writeChecksumBuilder.isPresent()) {
            page = page.getLoadedPage();
            this.writeChecksumBuilder.get().addPage(page);
            this.rowGroupStatisticsValidation.orElseThrow().addPage(page);
        }
    }

    private void validateBlockMetadata(List<RowGroupInfo> rowGroups) throws ParquetCorruptionException {
        if (this.writeValidation.isPresent()) {
            this.writeValidation.get().validateBlocksMetadata(this.dataSource.getId(), rowGroups);
        }
    }

    @FormatMethod
    private void validateWrite(Predicate<ParquetWriteValidation> test, String messageFormat, Object ... args) throws ParquetCorruptionException {
        if (this.writeValidation.isPresent() && !test.test(this.writeValidation.get())) {
            throw new ParquetCorruptionException(this.dataSource.getId(), "Write validation failed: " + messageFormat, args);
        }
    }
}

