/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.trino.sql.tree;

import com.google.common.collect.ImmutableList;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

import static com.google.common.base.MoreObjects.toStringHelper;
import static java.util.Objects.requireNonNull;

public final class ParameterDeclaration
        extends Node
{
    private final Optional<Identifier> name;
    private final DataType type;

    public ParameterDeclaration(Optional<Identifier> name, DataType type)
    {
        this(Optional.empty(), name, type);
    }

    public ParameterDeclaration(NodeLocation location, Optional<Identifier> name, DataType type)
    {
        this(Optional.of(location), name, type);
    }

    private ParameterDeclaration(Optional<NodeLocation> location, Optional<Identifier> name, DataType type)
    {
        super(location);
        this.name = requireNonNull(name, "name is null");
        this.type = requireNonNull(type, "type is null");
    }

    public Optional<Identifier> getName()
    {
        return name;
    }

    public DataType getType()
    {
        return type;
    }

    @Override
    public List<? extends Node> getChildren()
    {
        return ImmutableList.of();
    }

    @Override
    public <R, C> R accept(AstVisitor<R, C> visitor, C context)
    {
        return visitor.visitParameterDeclaration(this, context);
    }

    @Override
    public boolean equals(Object obj)
    {
        return (obj instanceof ParameterDeclaration other) &&
                Objects.equals(name, other.name) &&
                Objects.equals(type, other.type);
    }

    @Override
    public int hashCode()
    {
        return Objects.hash(name, type);
    }

    @Override
    public String toString()
    {
        return toStringHelper(this)
                .add("name", name)
                .add("type", type)
                .toString();
    }
}
