/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.password.file;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Splitter;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.collect.ImmutableMap;
import io.trino.cache.NonEvictableLoadingCache;
import io.trino.cache.SafeCaches;
import io.trino.plugin.password.Credential;
import io.trino.plugin.password.file.EncryptionUtil;
import io.trino.plugin.password.file.HashedPasswordException;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class PasswordStore {
    private static final Splitter LINE_SPLITTER = Splitter.on((String)":").limit(2);
    private final Map<String, HashedPassword> credentials;
    private final NonEvictableLoadingCache<Credential, Boolean> cache;

    public PasswordStore(File file, int cacheMaxSize) {
        this(PasswordStore.readPasswordFile(file), cacheMaxSize);
    }

    @VisibleForTesting
    public PasswordStore(List<String> lines, int cacheMaxSize) {
        this.credentials = PasswordStore.loadPasswordFile(lines);
        this.cache = SafeCaches.buildNonEvictableCache((CacheBuilder)CacheBuilder.newBuilder().maximumSize((long)cacheMaxSize), (CacheLoader)CacheLoader.from(this::matches));
    }

    public boolean authenticate(String user, String password) {
        return (Boolean)this.cache.getUnchecked((Object)new Credential(user, password));
    }

    private boolean matches(Credential credential) {
        HashedPassword hashed = this.credentials.get(credential.getUser());
        return hashed != null && hashed.matches(credential.getPassword());
    }

    private static Map<String, HashedPassword> loadPasswordFile(List<String> lines) {
        HashMap<String, HashedPassword> users = new HashMap<String, HashedPassword>();
        for (int lineNumber = 1; lineNumber <= lines.size(); ++lineNumber) {
            String line = lines.get(lineNumber - 1).trim();
            if (line.isEmpty()) continue;
            List parts = LINE_SPLITTER.splitToList((CharSequence)line);
            if (parts.size() != 2) {
                throw PasswordStore.invalidFile(lineNumber, "Expected two parts for user and password", null);
            }
            String user = (String)parts.get(0);
            String password = (String)parts.get(1);
            try {
                if (users.put(user, PasswordStore.getHashedPassword(password)) == null) continue;
                throw PasswordStore.invalidFile(lineNumber, "Duplicate user: " + user, null);
            }
            catch (HashedPasswordException e) {
                throw PasswordStore.invalidFile(lineNumber, e.getMessage(), (Throwable)((Object)e));
            }
        }
        return ImmutableMap.copyOf(users);
    }

    private static RuntimeException invalidFile(int lineNumber, String message, Throwable cause) {
        return new TrinoException((ErrorCodeSupplier)StandardErrorCode.CONFIGURATION_INVALID, String.format("Error in password file line %s: %s", lineNumber, message), cause);
    }

    private static List<String> readPasswordFile(File file) {
        try {
            return Files.readAllLines(file.toPath());
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.CONFIGURATION_UNAVAILABLE, "Failed to read password file: " + String.valueOf(file), (Throwable)e);
        }
    }

    private static HashedPassword getHashedPassword(String hashedPassword) {
        switch (EncryptionUtil.getHashingAlgorithm(hashedPassword)) {
            case BCRYPT: {
                return password -> EncryptionUtil.doesBCryptPasswordMatch(password, hashedPassword);
            }
            case PBKDF2: {
                return password -> EncryptionUtil.doesPBKDF2PasswordMatch(password, hashedPassword);
            }
        }
        throw new HashedPasswordException("Hashing algorithm of password cannot be determined");
    }

    public static interface HashedPassword {
        public boolean matches(String var1);
    }
}

