/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.core.data.manager;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.cache.LoadingCache;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Semaphore;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;
import org.apache.commons.configuration.Configuration;
import org.apache.commons.configuration.ConfigurationConverter;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.helix.HelixManager;
import org.apache.helix.store.zk.ZkHelixPropertyStore;
import org.apache.helix.zookeeper.datamodel.ZNRecord;
import org.apache.pinot.common.auth.AuthProviderUtils;
import org.apache.pinot.common.metadata.ZKMetadataProvider;
import org.apache.pinot.common.metadata.segment.SegmentZKMetadata;
import org.apache.pinot.common.metrics.AbstractMetrics;
import org.apache.pinot.common.metrics.ServerGauge;
import org.apache.pinot.common.metrics.ServerMeter;
import org.apache.pinot.common.metrics.ServerMetrics;
import org.apache.pinot.common.restlet.resources.SegmentErrorInfo;
import org.apache.pinot.common.utils.TarGzCompressionUtils;
import org.apache.pinot.common.utils.fetcher.SegmentFetcherFactory;
import org.apache.pinot.core.data.manager.offline.ImmutableSegmentDataManager;
import org.apache.pinot.segment.local.data.manager.SegmentDataManager;
import org.apache.pinot.segment.local.data.manager.TableDataManager;
import org.apache.pinot.segment.local.data.manager.TableDataManagerConfig;
import org.apache.pinot.segment.local.data.manager.TableDataManagerParams;
import org.apache.pinot.segment.local.indexsegment.immutable.ImmutableSegmentLoader;
import org.apache.pinot.segment.local.segment.index.loader.IndexLoadingConfig;
import org.apache.pinot.segment.local.segment.index.loader.LoaderUtils;
import org.apache.pinot.segment.spi.ImmutableSegment;
import org.apache.pinot.segment.spi.SegmentMetadata;
import org.apache.pinot.segment.spi.creator.SegmentGeneratorConfig;
import org.apache.pinot.segment.spi.index.metadata.SegmentMetadataImpl;
import org.apache.pinot.segment.spi.loader.SegmentDirectoryLoader;
import org.apache.pinot.segment.spi.loader.SegmentDirectoryLoaderContext;
import org.apache.pinot.segment.spi.loader.SegmentDirectoryLoaderRegistry;
import org.apache.pinot.segment.spi.store.SegmentDirectory;
import org.apache.pinot.spi.auth.AuthProvider;
import org.apache.pinot.spi.config.table.TableConfig;
import org.apache.pinot.spi.data.Schema;
import org.apache.pinot.spi.env.PinotConfiguration;
import org.apache.pinot.spi.utils.retry.AttemptsExceededException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public abstract class BaseTableDataManager
implements TableDataManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(BaseTableDataManager.class);
    protected final ConcurrentHashMap<String, SegmentDataManager> _segmentDataManagerMap = new ConcurrentHashMap();
    private Semaphore _segmentDownloadSemaphore;
    protected TableDataManagerConfig _tableDataManagerConfig;
    protected String _instanceId;
    protected ZkHelixPropertyStore<ZNRecord> _propertyStore;
    protected ServerMetrics _serverMetrics;
    protected String _tableNameWithType;
    protected String _tableDataDir;
    protected File _indexDir;
    protected File _resourceTmpDir;
    protected Logger _logger;
    protected HelixManager _helixManager;
    protected AuthProvider _authProvider;
    protected long _streamSegmentDownloadUntarRateLimitBytesPerSec;
    protected boolean _isStreamSegmentDownloadUntar;
    protected LoadingCache<Pair<String, String>, SegmentErrorInfo> _errorCache;

    public void init(TableDataManagerConfig tableDataManagerConfig, String instanceId, ZkHelixPropertyStore<ZNRecord> propertyStore, ServerMetrics serverMetrics, HelixManager helixManager, @Nullable LoadingCache<Pair<String, String>, SegmentErrorInfo> errorCache, TableDataManagerParams tableDataManagerParams) {
        int maxParallelSegmentDownloads;
        LOGGER.info("Initializing table data manager for table: {}", (Object)tableDataManagerConfig.getTableName());
        this._tableDataManagerConfig = tableDataManagerConfig;
        this._instanceId = instanceId;
        this._propertyStore = propertyStore;
        this._serverMetrics = serverMetrics;
        this._helixManager = helixManager;
        this._authProvider = AuthProviderUtils.extractAuthProvider((PinotConfiguration)BaseTableDataManager.toPinotConfiguration(this._tableDataManagerConfig.getAuthConfig()), null);
        this._tableNameWithType = tableDataManagerConfig.getTableName();
        this._tableDataDir = tableDataManagerConfig.getDataDir();
        this._indexDir = new File(this._tableDataDir);
        if (!this._indexDir.exists()) {
            Preconditions.checkState((boolean)this._indexDir.mkdirs(), (String)"Unable to create index directory at %s. Please check for available space and write-permissions for this directory.", (Object)this._indexDir);
        }
        this._resourceTmpDir = new File(this._indexDir, "tmp");
        FileUtils.deleteQuietly((File)this._resourceTmpDir);
        if (!this._resourceTmpDir.exists()) {
            Preconditions.checkState((boolean)this._resourceTmpDir.mkdirs(), (String)"Unable to create temp resources directory at %s. Please check for available space and write-permissions for this directory.", (Object)this._resourceTmpDir);
        }
        this._errorCache = errorCache;
        this._streamSegmentDownloadUntarRateLimitBytesPerSec = tableDataManagerParams.getStreamSegmentDownloadUntarRateLimitBytesPerSec();
        this._isStreamSegmentDownloadUntar = tableDataManagerParams.isStreamSegmentDownloadUntar();
        if (this._isStreamSegmentDownloadUntar) {
            LOGGER.info("Using streamed download-untar for segment download! The rate limit interval for streamed download-untar is {} ms", (Object)this._streamSegmentDownloadUntarRateLimitBytesPerSec);
        }
        if ((maxParallelSegmentDownloads = tableDataManagerParams.getMaxParallelSegmentDownloads()) > 0) {
            LOGGER.info("Construct segment download semaphore for Table: {}. Maximum number of parallel segment downloads: {}", (Object)this._tableNameWithType, (Object)maxParallelSegmentDownloads);
            this._segmentDownloadSemaphore = new Semaphore(maxParallelSegmentDownloads, true);
        } else {
            this._segmentDownloadSemaphore = null;
        }
        this._logger = LoggerFactory.getLogger((String)(this._tableNameWithType + "-" + this.getClass().getSimpleName()));
        this.doInit();
        this._logger.info("Initialized table data manager for table: {} with data directory: {}", (Object)this._tableNameWithType, (Object)this._tableDataDir);
    }

    protected abstract void doInit();

    public void start() {
        this._logger.info("Starting table data manager for table: {}", (Object)this._tableNameWithType);
        this.doStart();
        this._logger.info("Started table data manager for table: {}", (Object)this._tableNameWithType);
    }

    protected abstract void doStart();

    public void shutDown() {
        this._logger.info("Shutting down table data manager for table: {}", (Object)this._tableNameWithType);
        this.doShutdown();
        this._logger.info("Shut down table data manager for table: {}", (Object)this._tableNameWithType);
    }

    protected abstract void doShutdown();

    public void addSegment(ImmutableSegment immutableSegment) {
        String segmentName = immutableSegment.getSegmentName();
        this._logger.info("Adding immutable segment: {} to table: {}", (Object)segmentName, (Object)this._tableNameWithType);
        this._serverMetrics.addValueToTableGauge(this._tableNameWithType, (AbstractMetrics.Gauge)ServerGauge.DOCUMENT_COUNT, (long)immutableSegment.getSegmentMetadata().getTotalDocs());
        this._serverMetrics.addValueToTableGauge(this._tableNameWithType, (AbstractMetrics.Gauge)ServerGauge.SEGMENT_COUNT, 1L);
        ImmutableSegmentDataManager newSegmentManager = new ImmutableSegmentDataManager(immutableSegment);
        SegmentDataManager oldSegmentManager = this._segmentDataManagerMap.put(segmentName, newSegmentManager);
        if (oldSegmentManager == null) {
            this._logger.info("Added new immutable segment: {} to table: {}", (Object)segmentName, (Object)this._tableNameWithType);
        } else {
            this._logger.info("Replaced immutable segment: {} of table: {}", (Object)segmentName, (Object)this._tableNameWithType);
            this.releaseSegment(oldSegmentManager);
        }
    }

    public void addSegment(File indexDir, IndexLoadingConfig indexLoadingConfig) throws Exception {
        Schema schema = ZKMetadataProvider.getTableSchema(this._propertyStore, (String)this._tableNameWithType);
        this.addSegment(ImmutableSegmentLoader.load((File)indexDir, (IndexLoadingConfig)indexLoadingConfig, (Schema)schema));
    }

    public void addSegment(String segmentName, TableConfig tableConfig, IndexLoadingConfig indexLoadingConfig) throws Exception {
        throw new UnsupportedOperationException();
    }

    public void removeSegment(String segmentName) {
        this._logger.info("Removing segment: {} from table: {}", (Object)segmentName, (Object)this._tableNameWithType);
        SegmentDataManager segmentDataManager = this._segmentDataManagerMap.remove(segmentName);
        if (segmentDataManager != null) {
            this.releaseSegment(segmentDataManager);
            this._logger.info("Removed segment: {} from table: {}", (Object)segmentName, (Object)this._tableNameWithType);
        } else {
            this._logger.info("Failed to find segment: {} in table: {}", (Object)segmentName, (Object)this._tableNameWithType);
        }
    }

    public List<SegmentDataManager> acquireAllSegments() {
        ArrayList<SegmentDataManager> segmentDataManagers = new ArrayList<SegmentDataManager>();
        for (SegmentDataManager segmentDataManager : this._segmentDataManagerMap.values()) {
            if (!segmentDataManager.increaseReferenceCount()) continue;
            segmentDataManagers.add(segmentDataManager);
        }
        return segmentDataManagers;
    }

    public List<SegmentDataManager> acquireSegments(List<String> segmentNames, List<String> missingSegments) {
        ArrayList<SegmentDataManager> segmentDataManagers = new ArrayList<SegmentDataManager>();
        for (String segmentName : segmentNames) {
            SegmentDataManager segmentDataManager = this._segmentDataManagerMap.get(segmentName);
            if (segmentDataManager != null && segmentDataManager.increaseReferenceCount()) {
                segmentDataManagers.add(segmentDataManager);
                continue;
            }
            missingSegments.add(segmentName);
        }
        return segmentDataManagers;
    }

    @Nullable
    public SegmentDataManager acquireSegment(String segmentName) {
        SegmentDataManager segmentDataManager = this._segmentDataManagerMap.get(segmentName);
        if (segmentDataManager != null && segmentDataManager.increaseReferenceCount()) {
            return segmentDataManager;
        }
        return null;
    }

    public void releaseSegment(SegmentDataManager segmentDataManager) {
        if (segmentDataManager.decreaseReferenceCount()) {
            this.closeSegment(segmentDataManager);
        }
    }

    private void closeSegment(SegmentDataManager segmentDataManager) {
        String segmentName = segmentDataManager.getSegmentName();
        this._logger.info("Closing segment: {} of table: {}", (Object)segmentName, (Object)this._tableNameWithType);
        this._serverMetrics.addValueToTableGauge(this._tableNameWithType, (AbstractMetrics.Gauge)ServerGauge.SEGMENT_COUNT, -1L);
        this._serverMetrics.addMeteredTableValue(this._tableNameWithType, (AbstractMetrics.Meter)ServerMeter.DELETED_SEGMENT_COUNT, 1L);
        this._serverMetrics.addValueToTableGauge(this._tableNameWithType, (AbstractMetrics.Gauge)ServerGauge.DOCUMENT_COUNT, (long)(-segmentDataManager.getSegment().getSegmentMetadata().getTotalDocs()));
        segmentDataManager.destroy();
        this._logger.info("Closed segment: {} of table: {}", (Object)segmentName, (Object)this._tableNameWithType);
    }

    public int getNumSegments() {
        return this._segmentDataManagerMap.size();
    }

    public String getTableName() {
        return this._tableNameWithType;
    }

    public File getTableDataDir() {
        return this._indexDir;
    }

    public void addSegmentError(String segmentName, SegmentErrorInfo segmentErrorInfo) {
        this._errorCache.put((Object)Pair.of((Object)this._tableNameWithType, (Object)segmentName), (Object)segmentErrorInfo);
    }

    public Map<String, SegmentErrorInfo> getSegmentErrors() {
        if (this._errorCache == null) {
            return Collections.emptyMap();
        }
        return this._errorCache.asMap().entrySet().stream().filter(map -> ((String)((Pair)map.getKey()).getLeft()).equals(this._tableNameWithType)).collect(Collectors.toMap(map -> (String)((Pair)map.getKey()).getRight(), Map.Entry::getValue));
    }

    public void reloadSegment(String segmentName, IndexLoadingConfig indexLoadingConfig, SegmentZKMetadata zkMetadata, SegmentMetadata localMetadata, @Nullable Schema schema, boolean forceDownload) throws Exception {
        File indexDir = this.getSegmentDataDir(segmentName);
        try {
            boolean shouldDownload;
            this.createBackup(indexDir);
            boolean bl = shouldDownload = forceDownload || !BaseTableDataManager.hasSameCRC(zkMetadata, localMetadata);
            if (shouldDownload && this.allowDownload(segmentName, zkMetadata)) {
                if (forceDownload) {
                    LOGGER.info("Segment: {} of table: {} is forced to download", (Object)segmentName, (Object)this._tableNameWithType);
                } else {
                    LOGGER.info("Download segment:{} of table: {} as crc changes from: {} to: {}", new Object[]{segmentName, this._tableNameWithType, localMetadata.getCrc(), zkMetadata.getCrc()});
                }
                indexDir = this.downloadSegment(segmentName, zkMetadata);
            } else {
                LOGGER.info("Reload existing segment: {} of table: {}", (Object)segmentName, (Object)this._tableNameWithType);
                try (SegmentDirectory segmentDirectory = this.initSegmentDirectory(segmentName, String.valueOf(zkMetadata.getCrc()), indexLoadingConfig, schema);){
                    segmentDirectory.copyTo(indexDir);
                }
            }
            ImmutableSegment segment = ImmutableSegmentLoader.load((File)indexDir, (IndexLoadingConfig)indexLoadingConfig, (Schema)schema);
            this.addSegment(segment);
            this.removeBackup(indexDir);
        }
        catch (Exception reloadFailureException) {
            try {
                LoaderUtils.reloadFailureRecovery((File)indexDir);
            }
            catch (Exception recoveryFailureException) {
                LOGGER.error("Failed to recover after reload failure", (Throwable)recoveryFailureException);
                reloadFailureException.addSuppressed(recoveryFailureException);
            }
            throw reloadFailureException;
        }
    }

    public void addOrReplaceSegment(String segmentName, IndexLoadingConfig indexLoadingConfig, SegmentZKMetadata zkMetadata, @Nullable SegmentMetadata localMetadata) throws Exception {
        if (localMetadata != null && BaseTableDataManager.hasSameCRC(zkMetadata, localMetadata)) {
            LOGGER.info("Segment: {} of table: {} has crc: {} same as before, already loaded, do nothing", new Object[]{segmentName, this._tableNameWithType, localMetadata.getCrc()});
            return;
        }
        if (localMetadata == null && this.tryLoadExistingSegment(segmentName, indexLoadingConfig, zkMetadata)) {
            return;
        }
        Preconditions.checkState((boolean)this.allowDownload(segmentName, zkMetadata), (String)"Segment: %s of table: %s does not allow download", (Object)segmentName, (Object)this._tableNameWithType);
        if (localMetadata == null) {
            LOGGER.info("Download segment: {} of table: {} as it doesn't exist", (Object)segmentName, (Object)this._tableNameWithType);
        } else {
            LOGGER.info("Download segment: {} of table: {} as crc changes from: {} to: {}", new Object[]{segmentName, this._tableNameWithType, localMetadata.getCrc(), zkMetadata.getCrc()});
        }
        File indexDir = this.downloadSegment(segmentName, zkMetadata);
        this.addSegment(indexDir, indexLoadingConfig);
        LOGGER.info("Downloaded and loaded segment: {} of table: {} with crc: {}", new Object[]{segmentName, this._tableNameWithType, zkMetadata.getCrc()});
    }

    protected boolean allowDownload(String segmentName, SegmentZKMetadata zkMetadata) {
        return true;
    }

    protected File downloadSegment(String segmentName, SegmentZKMetadata zkMetadata) throws Exception {
        return this.downloadSegmentFromDeepStore(segmentName, zkMetadata);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private File downloadSegmentFromDeepStore(String segmentName, SegmentZKMetadata zkMetadata) throws Exception {
        File tempRootDir = this.getTmpSegmentDataDir("tmp-" + segmentName + "-" + UUID.randomUUID());
        FileUtils.forceMkdir((File)tempRootDir);
        if (this._isStreamSegmentDownloadUntar && zkMetadata.getCrypterName() == null) {
            try {
                File untaredSegDir = this.downloadAndStreamUntarWithRateLimit(segmentName, zkMetadata, tempRootDir, this._streamSegmentDownloadUntarRateLimitBytesPerSec);
                File file = this.moveSegment(segmentName, untaredSegDir);
                return file;
            }
            finally {
                FileUtils.deleteQuietly((File)tempRootDir);
            }
        }
        try {
            File tarFile = this.downloadAndDecrypt(segmentName, zkMetadata, tempRootDir);
            File file = this.untarAndMoveSegment(segmentName, tarFile, tempRootDir);
            return file;
        }
        finally {
            FileUtils.deleteQuietly((File)tempRootDir);
        }
    }

    private File moveSegment(String segmentName, File untaredSegDir) throws IOException {
        try {
            File indexDir = this.getSegmentDataDir(segmentName);
            FileUtils.deleteDirectory((File)indexDir);
            FileUtils.moveDirectory((File)untaredSegDir, (File)indexDir);
            return indexDir;
        }
        catch (Exception e) {
            LOGGER.error("Failed to move segment: {} of table: {}", (Object)segmentName, (Object)this._tableNameWithType);
            this._serverMetrics.addMeteredTableValue(this._tableNameWithType, (AbstractMetrics.Meter)ServerMeter.SEGMENT_DIR_MOVEMENT_FAILURES, 1L);
            throw e;
        }
    }

    @VisibleForTesting
    File downloadAndDecrypt(String segmentName, SegmentZKMetadata zkMetadata, File tempRootDir) throws Exception {
        File tarFile = new File(tempRootDir, segmentName + ".tar.gz");
        String uri = zkMetadata.getDownloadUrl();
        try {
            if (this._segmentDownloadSemaphore != null) {
                long startTime = System.currentTimeMillis();
                LOGGER.info("Trying to acquire segment download semaphore for: {}. queue-length: {} ", (Object)segmentName, (Object)this._segmentDownloadSemaphore.getQueueLength());
                this._segmentDownloadSemaphore.acquire();
                LOGGER.info("Acquired segment download semaphore for: {} (lock-time={}ms, queue-length={}).", new Object[]{segmentName, System.currentTimeMillis() - startTime, this._segmentDownloadSemaphore.getQueueLength()});
            }
            SegmentFetcherFactory.fetchAndDecryptSegmentToLocal((String)uri, (File)tarFile, (String)zkMetadata.getCrypterName());
            LOGGER.info("Downloaded tarred segment: {} for table: {} from: {} to: {}, file length: {}", new Object[]{segmentName, this._tableNameWithType, uri, tarFile, tarFile.length()});
            File startTime = tarFile;
            return startTime;
        }
        catch (AttemptsExceededException e) {
            LOGGER.error("Attempts exceeded when downloading segment: {} for table: {} from: {} to: {}", new Object[]{segmentName, this._tableNameWithType, uri, tarFile});
            this._serverMetrics.addMeteredTableValue(this._tableNameWithType, (AbstractMetrics.Meter)ServerMeter.SEGMENT_DOWNLOAD_FAILURES, 1L);
            throw e;
        }
        finally {
            if (this._segmentDownloadSemaphore != null) {
                this._segmentDownloadSemaphore.release();
            }
        }
    }

    private File downloadAndStreamUntarWithRateLimit(String segmentName, SegmentZKMetadata zkMetadata, File tempRootDir, long maxStreamRateInByte) throws Exception {
        if (this._segmentDownloadSemaphore != null) {
            long startTime = System.currentTimeMillis();
            LOGGER.info("Trying to acquire segment download semaphore for: {}. queue-length: {} ", (Object)segmentName, (Object)this._segmentDownloadSemaphore.getQueueLength());
            this._segmentDownloadSemaphore.acquire();
            LOGGER.info("Acquired segment download semaphore for: {} (lock-time={}ms, queue-length={}).", new Object[]{segmentName, System.currentTimeMillis() - startTime, this._segmentDownloadSemaphore.getQueueLength()});
        }
        LOGGER.info("Trying to download segment {} using streamed download-untar with maxStreamRateInByte {}", (Object)segmentName, (Object)maxStreamRateInByte);
        String uri = zkMetadata.getDownloadUrl();
        try {
            File ret = SegmentFetcherFactory.fetchAndStreamUntarToLocal((String)uri, (File)tempRootDir, (long)maxStreamRateInByte);
            LOGGER.info("Download and untarred segment: {} for table: {} from: {}", new Object[]{segmentName, this._tableNameWithType, uri});
            File file = ret;
            return file;
        }
        catch (AttemptsExceededException e) {
            LOGGER.error("Attempts exceeded when stream download-untarring segment: {} for table: {} from: {} to: {}", new Object[]{segmentName, this._tableNameWithType, uri, tempRootDir});
            this._serverMetrics.addMeteredTableValue(this._tableNameWithType, (AbstractMetrics.Meter)ServerMeter.SEGMENT_STREAMED_DOWNLOAD_UNTAR_FAILURES, 1L);
            throw e;
        }
        finally {
            if (this._segmentDownloadSemaphore != null) {
                this._segmentDownloadSemaphore.release();
            }
        }
    }

    @VisibleForTesting
    File untarAndMoveSegment(String segmentName, File tarFile, File tempRootDir) throws IOException {
        File untarDir = new File(tempRootDir, segmentName);
        try {
            File untaredSegDir = (File)TarGzCompressionUtils.untar((File)tarFile, (File)untarDir).get(0);
            LOGGER.info("Uncompressed tar file: {} into target dir: {}", (Object)tarFile, (Object)untarDir);
            File indexDir = this.getSegmentDataDir(segmentName);
            FileUtils.deleteDirectory((File)indexDir);
            FileUtils.moveDirectory((File)untaredSegDir, (File)indexDir);
            LOGGER.info("Successfully downloaded segment: {} of table: {} to index dir: {}", new Object[]{segmentName, this._tableNameWithType, indexDir});
            return indexDir;
        }
        catch (Exception e) {
            LOGGER.error("Failed to untar segment: {} of table: {} from: {} to: {}", new Object[]{segmentName, this._tableNameWithType, tarFile, untarDir});
            this._serverMetrics.addMeteredTableValue(this._tableNameWithType, (AbstractMetrics.Meter)ServerMeter.UNTAR_FAILURES, 1L);
            throw e;
        }
    }

    @VisibleForTesting
    File getSegmentDataDir(String segmentName) {
        return new File(this._indexDir, segmentName);
    }

    @VisibleForTesting
    protected File getTmpSegmentDataDir(String segmentName) {
        return new File(this._resourceTmpDir, segmentName);
    }

    private void createBackup(File indexDir) {
        if (!indexDir.exists()) {
            return;
        }
        File parentDir = indexDir.getParentFile();
        File segmentBackupDir = new File(parentDir, indexDir.getName() + ".segment.bak");
        Preconditions.checkState((boolean)indexDir.renameTo(segmentBackupDir), (String)"Failed to rename index directory: %s to segment backup directory: %s", (Object)indexDir, (Object)segmentBackupDir);
    }

    private void removeBackup(File indexDir) throws IOException {
        File parentDir = indexDir.getParentFile();
        File segmentBackupDir = new File(parentDir, indexDir.getName() + ".segment.bak");
        if (!segmentBackupDir.exists()) {
            return;
        }
        File segmentTempDir = new File(parentDir, indexDir.getName() + ".segment.tmp");
        Preconditions.checkState((boolean)segmentBackupDir.renameTo(segmentTempDir), (String)"Failed to rename segment backup directory: %s to segment temporary directory: %s", (Object)segmentBackupDir, (Object)segmentTempDir);
        FileUtils.deleteDirectory((File)segmentTempDir);
    }

    private boolean tryLoadExistingSegment(String segmentName, IndexLoadingConfig indexLoadingConfig, SegmentZKMetadata zkMetadata) {
        SegmentMetadataImpl segmentMetadata;
        File indexDir = this.getSegmentDataDir(segmentName);
        BaseTableDataManager.recoverReloadFailureQuietly(this._tableNameWithType, segmentName, indexDir);
        Schema schema = ZKMetadataProvider.getTableSchema(this._propertyStore, (String)this._tableNameWithType);
        schema = SegmentGeneratorConfig.updateSchemaWithTimestampIndexes((Schema)schema, (Map)SegmentGeneratorConfig.extractTimestampIndexConfigsFromTableConfig((TableConfig)indexLoadingConfig.getTableConfig()));
        SegmentDirectory segmentDirectory = this.tryInitSegmentDirectory(segmentName, String.valueOf(zkMetadata.getCrc()), indexLoadingConfig, schema);
        SegmentMetadataImpl segmentMetadataImpl = segmentMetadata = segmentDirectory == null ? null : segmentDirectory.getSegmentMetadata();
        if (segmentMetadata == null || !BaseTableDataManager.hasSameCRC(zkMetadata, (SegmentMetadata)segmentMetadata)) {
            if (segmentMetadata == null) {
                LOGGER.info("Segment: {} of table: {} does not exist", (Object)segmentName, (Object)this._tableNameWithType);
            } else if (!BaseTableDataManager.hasSameCRC(zkMetadata, (SegmentMetadata)segmentMetadata)) {
                LOGGER.info("Segment: {} of table: {} has crc change from: {} to: {}", new Object[]{segmentName, this._tableNameWithType, segmentMetadata.getCrc(), zkMetadata.getCrc()});
            }
            BaseTableDataManager.closeSegmentDirectoryQuietly(segmentDirectory);
            return false;
        }
        try {
            if (!ImmutableSegmentLoader.needPreprocess((SegmentDirectory)segmentDirectory, (IndexLoadingConfig)indexLoadingConfig, (Schema)schema)) {
                LOGGER.info("Segment: {} of table: {} is consistent with latest table config and schema", (Object)segmentName, (Object)this._tableNameWithType);
            } else {
                LOGGER.info("Segment: {} of table: {} needs reprocess to reflect latest table config and schema", (Object)segmentName, (Object)this._tableNameWithType);
                segmentDirectory.copyTo(indexDir);
                BaseTableDataManager.closeSegmentDirectoryQuietly(segmentDirectory);
                ImmutableSegmentLoader.preprocess((File)indexDir, (IndexLoadingConfig)indexLoadingConfig, (Schema)schema);
                segmentDirectory = this.initSegmentDirectory(segmentName, String.valueOf(zkMetadata.getCrc()), indexLoadingConfig, schema);
            }
            ImmutableSegment segment = ImmutableSegmentLoader.load((SegmentDirectory)segmentDirectory, (IndexLoadingConfig)indexLoadingConfig, (Schema)schema);
            this.addSegment(segment);
            LOGGER.info("Loaded existing segment: {} of table: {} with crc: {}", new Object[]{segmentName, this._tableNameWithType, zkMetadata.getCrc()});
            return true;
        }
        catch (Exception e) {
            LOGGER.error("Failed to load existing segment: {} of table: {} with crc: {}", new Object[]{segmentName, this._tableNameWithType, zkMetadata.getCrc(), e});
            BaseTableDataManager.closeSegmentDirectoryQuietly(segmentDirectory);
            return false;
        }
    }

    private SegmentDirectory tryInitSegmentDirectory(String segmentName, String segmentCrc, IndexLoadingConfig indexLoadingConfig, Schema schema) {
        try {
            return this.initSegmentDirectory(segmentName, segmentCrc, indexLoadingConfig, schema);
        }
        catch (Exception e) {
            LOGGER.warn("Failed to initialize SegmentDirectory for segment: {} of table: {} with error: {}", new Object[]{segmentName, this._tableNameWithType, e.getMessage()});
            return null;
        }
    }

    private SegmentDirectory initSegmentDirectory(String segmentName, String segmentCrc, IndexLoadingConfig indexLoadingConfig, Schema schema) throws Exception {
        SegmentDirectoryLoaderContext loaderContext = new SegmentDirectoryLoaderContext.Builder().setTableConfig(indexLoadingConfig.getTableConfig()).setSchema(schema).setInstanceId(indexLoadingConfig.getInstanceId()).setSegmentName(segmentName).setSegmentCrc(segmentCrc).setSegmentDirectoryConfigs(indexLoadingConfig.getSegmentDirectoryConfigs()).build();
        SegmentDirectoryLoader segmentDirectoryLoader = SegmentDirectoryLoaderRegistry.getSegmentDirectoryLoader((String)indexLoadingConfig.getSegmentDirectoryLoader());
        File indexDir = this.getSegmentDataDir(segmentName);
        return segmentDirectoryLoader.load(indexDir.toURI(), loaderContext);
    }

    private static boolean hasSameCRC(SegmentZKMetadata zkMetadata, SegmentMetadata localMetadata) {
        return zkMetadata.getCrc() == Long.parseLong(localMetadata.getCrc());
    }

    private static void recoverReloadFailureQuietly(String tableNameWithType, String segmentName, File indexDir) {
        try {
            LoaderUtils.reloadFailureRecovery((File)indexDir);
        }
        catch (Exception e) {
            LOGGER.warn("Failed to recover segment: {} of table: {} due to error: {}", new Object[]{segmentName, tableNameWithType, e.getMessage()});
        }
    }

    private static void closeSegmentDirectoryQuietly(SegmentDirectory segmentDirectory) {
        if (segmentDirectory != null) {
            try {
                segmentDirectory.close();
            }
            catch (Exception e) {
                LOGGER.warn("Failed to close SegmentDirectory due to error: {}", (Object)e.getMessage());
            }
        }
    }

    private static PinotConfiguration toPinotConfiguration(Configuration configuration) {
        if (configuration == null) {
            return new PinotConfiguration();
        }
        return new PinotConfiguration(ConfigurationConverter.getMap((Configuration)configuration));
    }
}

