/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.core.data.table;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.apache.pinot.common.request.context.ExpressionContext;
import org.apache.pinot.common.request.context.FunctionContext;
import org.apache.pinot.common.request.context.OrderByExpressionContext;
import org.apache.pinot.common.utils.DataSchema;
import org.apache.pinot.core.data.table.IntermediateRecord;
import org.apache.pinot.core.data.table.Key;
import org.apache.pinot.core.data.table.Record;
import org.apache.pinot.core.query.aggregation.function.AggregationFunction;
import org.apache.pinot.core.query.aggregation.groupby.GroupByResultHolder;
import org.apache.pinot.core.query.aggregation.groupby.GroupKeyGenerator;
import org.apache.pinot.core.query.postaggregation.PostAggregationFunction;
import org.apache.pinot.core.query.request.context.QueryContext;
import org.apache.pinot.spi.utils.ByteArray;

public class TableResizer {
    private final DataSchema _dataSchema;
    private final int _numGroupByExpressions;
    private final Map<ExpressionContext, Integer> _groupByExpressionIndexMap;
    private final AggregationFunction[] _aggregationFunctions;
    private final Map<FunctionContext, Integer> _aggregationFunctionIndexMap;
    private final int _numOrderByExpressions;
    private final OrderByValueExtractor[] _orderByValueExtractors;
    private final Comparator<IntermediateRecord> _intermediateRecordComparator;

    public TableResizer(DataSchema dataSchema, QueryContext queryContext) {
        this._dataSchema = dataSchema;
        List<ExpressionContext> groupByExpressions = queryContext.getGroupByExpressions();
        assert (groupByExpressions != null);
        this._numGroupByExpressions = groupByExpressions.size();
        this._groupByExpressionIndexMap = new HashMap<ExpressionContext, Integer>();
        for (int i = 0; i < this._numGroupByExpressions; ++i) {
            this._groupByExpressionIndexMap.put(groupByExpressions.get(i), i);
        }
        this._aggregationFunctions = queryContext.getAggregationFunctions();
        assert (this._aggregationFunctions != null);
        this._aggregationFunctionIndexMap = queryContext.getAggregationFunctionIndexMap();
        assert (this._aggregationFunctionIndexMap != null);
        List<OrderByExpressionContext> orderByExpressions = queryContext.getOrderByExpressions();
        assert (orderByExpressions != null);
        this._numOrderByExpressions = orderByExpressions.size();
        this._orderByValueExtractors = new OrderByValueExtractor[this._numOrderByExpressions];
        Comparator[] comparators = new Comparator[this._numOrderByExpressions];
        for (int i = 0; i < this._numOrderByExpressions; ++i) {
            OrderByExpressionContext orderByExpression = orderByExpressions.get(i);
            this._orderByValueExtractors[i] = this.getOrderByValueExtractor(orderByExpression.getExpression());
            comparators[i] = orderByExpression.isAsc() ? Comparator.naturalOrder() : Comparator.reverseOrder();
        }
        boolean nullHandlingEnabled = queryContext.isNullHandlingEnabled();
        this._intermediateRecordComparator = nullHandlingEnabled ? (o1, o2) -> {
            for (int i = 0; i < this._numOrderByExpressions; ++i) {
                Comparable v1 = o1._values[i];
                Comparable v2 = o2._values[i];
                if (v1 == null) {
                    if (v2 == null) continue;
                    return 1;
                }
                if (v2 == null) {
                    return -1;
                }
                int result = comparators[i].compare(v1, v2);
                if (result == 0) continue;
                return result;
            }
            return 0;
        } : (o1, o2) -> {
            for (int i = 0; i < this._numOrderByExpressions; ++i) {
                int result = comparators[i].compare(o1._values[i], o2._values[i]);
                if (result == 0) continue;
                return result;
            }
            return 0;
        };
    }

    private OrderByValueExtractor getOrderByValueExtractor(ExpressionContext expression) {
        if (expression.getType() == ExpressionContext.Type.LITERAL) {
            return new LiteralExtractor(expression.getLiteral());
        }
        Integer groupByExpressionIndex = this._groupByExpressionIndexMap.get(expression);
        if (groupByExpressionIndex != null) {
            return new GroupByExpressionExtractor(groupByExpressionIndex);
        }
        FunctionContext function = expression.getFunction();
        Preconditions.checkState((function != null ? 1 : 0) != 0, (String)"Failed to find ORDER-BY expression: %s in the GROUP-BY clause", (Object)expression);
        if (function.getType() == FunctionContext.Type.AGGREGATION) {
            return new AggregationFunctionExtractor(this._aggregationFunctionIndexMap.get(function));
        }
        return new PostAggregationFunctionExtractor(function);
    }

    private IntermediateRecord getIntermediateRecord(Key key, Record record) {
        Comparable[] orderByValues = new Comparable[this._numOrderByExpressions];
        for (int i = 0; i < this._numOrderByExpressions; ++i) {
            orderByValues[i] = this._orderByValueExtractors[i].extract(record);
        }
        return new IntermediateRecord(key, record, orderByValues);
    }

    public void resizeRecordsMap(Map<Key, Record> recordsMap, int size) {
        int numRecordsToEvict = recordsMap.size() - size;
        if (numRecordsToEvict <= 0) {
            return;
        }
        if (numRecordsToEvict <= size) {
            IntermediateRecord[] recordsToEvict;
            for (IntermediateRecord recordToEvict : recordsToEvict = this.getTopRecordsHeap(recordsMap, numRecordsToEvict, this._intermediateRecordComparator)) {
                recordsMap.remove(recordToEvict._key);
            }
        } else {
            IntermediateRecord[] recordsToRetain = this.getTopRecordsHeap(recordsMap, size, this._intermediateRecordComparator.reversed());
            recordsMap.clear();
            for (IntermediateRecord recordToRetain : recordsToRetain) {
                recordsMap.put(recordToRetain._key, recordToRetain._record);
            }
        }
    }

    private IntermediateRecord[] getTopRecordsHeap(Map<Key, Record> recordsMap, int size, Comparator<IntermediateRecord> comparator) {
        assert (recordsMap.size() > size);
        Iterator<Map.Entry<Key, Record>> mapEntryIterator = recordsMap.entrySet().iterator();
        IntermediateRecord[] heap = new IntermediateRecord[size];
        for (int i = 0; i < size; ++i) {
            Map.Entry<Key, Record> entry = mapEntryIterator.next();
            heap[i] = this.getIntermediateRecord(entry.getKey(), entry.getValue());
        }
        TableResizer.makeHeap(heap, size, comparator);
        while (mapEntryIterator.hasNext()) {
            Map.Entry<Key, Record> entry = mapEntryIterator.next();
            IntermediateRecord intermediateRecord = this.getIntermediateRecord(entry.getKey(), entry.getValue());
            if (comparator.compare(intermediateRecord, heap[0]) <= 0) continue;
            heap[0] = intermediateRecord;
            TableResizer.downHeap(heap, size, 0, comparator);
        }
        return heap;
    }

    private static void makeHeap(IntermediateRecord[] heap, int size, Comparator<IntermediateRecord> c) {
        int i = size >>> 1;
        while (i-- != 0) {
            TableResizer.downHeap(heap, size, i, c);
        }
    }

    private static void downHeap(IntermediateRecord[] heap, int size, int i, Comparator<IntermediateRecord> c) {
        int child;
        IntermediateRecord e = heap[i];
        while ((child = (i << 1) + 1) < size) {
            IntermediateRecord t = heap[child];
            int right = child + 1;
            if (right < size && c.compare(heap[right], t) < 0) {
                child = right;
                t = heap[child];
            }
            if (c.compare(e, t) <= 0) break;
            heap[i] = t;
            i = child;
        }
        heap[i] = e;
    }

    public Collection<Record> getTopRecords(Map<Key, Record> recordsMap, int size, boolean sort) {
        return sort ? this.getSortedTopRecords(recordsMap, size) : this.getUnsortedTopRecords(recordsMap, size);
    }

    @VisibleForTesting
    List<Record> getSortedTopRecords(Map<Key, Record> recordsMap, int size) {
        int numRecords = recordsMap.size();
        if (numRecords == 0) {
            return Collections.emptyList();
        }
        if (numRecords <= size) {
            IntermediateRecord[] intermediateRecords = new IntermediateRecord[numRecords];
            int index = 0;
            for (Map.Entry<Key, Record> entry : recordsMap.entrySet()) {
                intermediateRecords[index++] = this.getIntermediateRecord(entry.getKey(), entry.getValue());
            }
            Arrays.sort(intermediateRecords, this._intermediateRecordComparator);
            Record[] sortedTopRecords = new Record[numRecords];
            for (int i = 0; i < numRecords; ++i) {
                sortedTopRecords[i] = intermediateRecords[i]._record;
            }
            return Arrays.asList(sortedTopRecords);
        }
        Comparator<IntermediateRecord> comparator = this._intermediateRecordComparator.reversed();
        IntermediateRecord[] topRecordsHeap = this.getTopRecordsHeap(recordsMap, size, comparator);
        Record[] sortedTopRecords = new Record[size];
        while (size-- > 0) {
            sortedTopRecords[size] = topRecordsHeap[0]._record;
            topRecordsHeap[0] = topRecordsHeap[size];
            TableResizer.downHeap(topRecordsHeap, size, 0, comparator);
        }
        return Arrays.asList(sortedTopRecords);
    }

    private Collection<Record> getUnsortedTopRecords(Map<Key, Record> recordsMap, int size) {
        int numRecords = recordsMap.size();
        if (numRecords <= size) {
            return recordsMap.values();
        }
        IntermediateRecord[] topRecords = this.getTopRecordsHeap(recordsMap, size, this._intermediateRecordComparator.reversed());
        Record[] unsortedTopRecords = new Record[size];
        int index = 0;
        for (IntermediateRecord topRecord : topRecords) {
            unsortedTopRecords[index++] = topRecord._record;
        }
        return Arrays.asList(unsortedTopRecords);
    }

    public List<IntermediateRecord> trimInSegmentResults(GroupKeyGenerator groupKeyGenerator, GroupByResultHolder[] groupByResultHolders, int size) {
        assert (groupKeyGenerator.getNumKeys() > size);
        Iterator<GroupKeyGenerator.GroupKey> groupKeyIterator = groupKeyGenerator.getGroupKeys();
        Comparator<IntermediateRecord> comparator = this._intermediateRecordComparator.reversed();
        IntermediateRecord[] heap = new IntermediateRecord[size];
        for (int i = 0; i < size; ++i) {
            heap[i] = this.getIntermediateRecord(groupKeyIterator.next(), groupByResultHolders);
        }
        TableResizer.makeHeap(heap, size, comparator);
        while (groupKeyIterator.hasNext()) {
            IntermediateRecord intermediateRecord = this.getIntermediateRecord(groupKeyIterator.next(), groupByResultHolders);
            if (comparator.compare(intermediateRecord, heap[0]) <= 0) continue;
            heap[0] = intermediateRecord;
            TableResizer.downHeap(heap, size, 0, comparator);
        }
        return Arrays.asList(heap);
    }

    private IntermediateRecord getIntermediateRecord(GroupKeyGenerator.GroupKey groupKey, GroupByResultHolder[] groupByResultHolders) {
        int numAggregationFunctions = this._aggregationFunctions.length;
        int numColumns = numAggregationFunctions + this._numGroupByExpressions;
        Object[] keys = groupKey._keys;
        Object[] values = Arrays.copyOf(keys, numColumns);
        int groupId = groupKey._groupId;
        for (int i = 0; i < numAggregationFunctions; ++i) {
            values[this._numGroupByExpressions + i] = this._aggregationFunctions[i].extractGroupByResult(groupByResultHolders[i], groupId);
        }
        return this.getIntermediateRecord(new Key(keys), new Record(values));
    }

    private class PostAggregationFunctionExtractor
    implements OrderByValueExtractor {
        final Object[] _arguments;
        final OrderByValueExtractor[] _argumentExtractors;
        final PostAggregationFunction _postAggregationFunction;

        PostAggregationFunctionExtractor(FunctionContext function) {
            assert (function.getType() == FunctionContext.Type.TRANSFORM);
            List arguments = function.getArguments();
            int numArguments = arguments.size();
            this._arguments = new Object[numArguments];
            this._argumentExtractors = new OrderByValueExtractor[numArguments];
            DataSchema.ColumnDataType[] argumentTypes = new DataSchema.ColumnDataType[numArguments];
            for (int i = 0; i < numArguments; ++i) {
                OrderByValueExtractor argumentExtractor;
                this._argumentExtractors[i] = argumentExtractor = TableResizer.this.getOrderByValueExtractor((ExpressionContext)arguments.get(i));
                argumentTypes[i] = argumentExtractor.getValueType();
            }
            this._postAggregationFunction = new PostAggregationFunction(function.getFunctionName(), argumentTypes);
        }

        @Override
        public DataSchema.ColumnDataType getValueType() {
            return this._postAggregationFunction.getResultType();
        }

        @Override
        public Comparable extract(Record record) {
            int numArguments = this._arguments.length;
            for (int i = 0; i < numArguments; ++i) {
                this._arguments[i] = this._argumentExtractors[i].extract(record);
            }
            Object result = this._postAggregationFunction.invoke(this._arguments);
            if (this._postAggregationFunction.getResultType() == DataSchema.ColumnDataType.BYTES) {
                return new ByteArray((byte[])result);
            }
            return (Comparable)result;
        }
    }

    private class AggregationFunctionExtractor
    implements OrderByValueExtractor {
        final int _index;
        final AggregationFunction _aggregationFunction;

        AggregationFunctionExtractor(int aggregationFunctionIndex) {
            this._index = aggregationFunctionIndex + TableResizer.this._numGroupByExpressions;
            this._aggregationFunction = TableResizer.this._aggregationFunctions[aggregationFunctionIndex];
        }

        @Override
        public DataSchema.ColumnDataType getValueType() {
            return this._aggregationFunction.getFinalResultColumnType();
        }

        @Override
        public Comparable extract(Record record) {
            return this._aggregationFunction.extractFinalResult(record.getValues()[this._index]);
        }
    }

    private class GroupByExpressionExtractor
    implements OrderByValueExtractor {
        final int _index;

        GroupByExpressionExtractor(int groupByExpressionIndex) {
            this._index = groupByExpressionIndex;
        }

        @Override
        public DataSchema.ColumnDataType getValueType() {
            return TableResizer.this._dataSchema.getColumnDataType(this._index);
        }

        @Override
        public Comparable extract(Record record) {
            return (Comparable)record.getValues()[this._index];
        }
    }

    private static class LiteralExtractor
    implements OrderByValueExtractor {
        final String _literal;

        LiteralExtractor(String literal) {
            this._literal = literal;
        }

        @Override
        public DataSchema.ColumnDataType getValueType() {
            return DataSchema.ColumnDataType.STRING;
        }

        public String extract(Record record) {
            return this._literal;
        }
    }

    private static interface OrderByValueExtractor {
        public DataSchema.ColumnDataType getValueType();

        public Comparable extract(Record var1);
    }
}

