/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.core.data.manager;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.helix.HelixManager;
import org.apache.helix.store.zk.ZkHelixPropertyStore;
import org.apache.helix.zookeeper.datamodel.ZNRecord;
import org.apache.pinot.common.auth.AuthProviderUtils;
import org.apache.pinot.common.metadata.ZKMetadataProvider;
import org.apache.pinot.common.metadata.segment.SegmentZKMetadata;
import org.apache.pinot.common.metrics.AbstractMetrics;
import org.apache.pinot.common.metrics.ServerGauge;
import org.apache.pinot.common.metrics.ServerMeter;
import org.apache.pinot.common.metrics.ServerMetrics;
import org.apache.pinot.common.restlet.resources.SegmentErrorInfo;
import org.apache.pinot.common.utils.TarCompressionUtils;
import org.apache.pinot.common.utils.config.TierConfigUtils;
import org.apache.pinot.common.utils.fetcher.SegmentFetcherFactory;
import org.apache.pinot.core.data.manager.offline.ImmutableSegmentDataManager;
import org.apache.pinot.core.util.PeerServerSegmentFinder;
import org.apache.pinot.segment.local.data.manager.SegmentDataManager;
import org.apache.pinot.segment.local.data.manager.StaleSegment;
import org.apache.pinot.segment.local.data.manager.TableDataManager;
import org.apache.pinot.segment.local.indexsegment.immutable.ImmutableSegmentImpl;
import org.apache.pinot.segment.local.indexsegment.immutable.ImmutableSegmentLoader;
import org.apache.pinot.segment.local.segment.index.dictionary.DictionaryIndexType;
import org.apache.pinot.segment.local.segment.index.loader.IndexLoadingConfig;
import org.apache.pinot.segment.local.segment.index.loader.LoaderUtils;
import org.apache.pinot.segment.local.startree.StarTreeBuilderUtils;
import org.apache.pinot.segment.local.startree.v2.builder.StarTreeV2BuilderConfig;
import org.apache.pinot.segment.local.utils.SegmentLocks;
import org.apache.pinot.segment.spi.ColumnMetadata;
import org.apache.pinot.segment.spi.ImmutableSegment;
import org.apache.pinot.segment.spi.IndexSegment;
import org.apache.pinot.segment.spi.SegmentContext;
import org.apache.pinot.segment.spi.SegmentMetadata;
import org.apache.pinot.segment.spi.datasource.DataSource;
import org.apache.pinot.segment.spi.index.FieldIndexConfigs;
import org.apache.pinot.segment.spi.index.FieldIndexConfigsUtil;
import org.apache.pinot.segment.spi.index.IndexType;
import org.apache.pinot.segment.spi.index.StandardIndexes;
import org.apache.pinot.segment.spi.index.metadata.SegmentMetadataImpl;
import org.apache.pinot.segment.spi.index.startree.StarTreeV2;
import org.apache.pinot.segment.spi.index.startree.StarTreeV2Metadata;
import org.apache.pinot.segment.spi.loader.SegmentDirectoryLoader;
import org.apache.pinot.segment.spi.loader.SegmentDirectoryLoaderContext;
import org.apache.pinot.segment.spi.loader.SegmentDirectoryLoaderRegistry;
import org.apache.pinot.segment.spi.partition.PartitionFunction;
import org.apache.pinot.segment.spi.store.SegmentDirectory;
import org.apache.pinot.spi.auth.AuthProvider;
import org.apache.pinot.spi.config.instance.InstanceDataManagerConfig;
import org.apache.pinot.spi.config.table.ColumnPartitionConfig;
import org.apache.pinot.spi.config.table.SegmentPartitionConfig;
import org.apache.pinot.spi.config.table.TableConfig;
import org.apache.pinot.spi.config.table.TableType;
import org.apache.pinot.spi.data.FieldSpec;
import org.apache.pinot.spi.data.Schema;
import org.apache.pinot.spi.env.PinotConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public abstract class BaseTableDataManager
implements TableDataManager {
    protected static final Logger LOGGER = LoggerFactory.getLogger(BaseTableDataManager.class);
    protected final ConcurrentHashMap<String, SegmentDataManager> _segmentDataManagerMap = new ConcurrentHashMap();
    protected final ServerMetrics _serverMetrics = ServerMetrics.get();
    protected InstanceDataManagerConfig _instanceDataManagerConfig;
    protected String _instanceId;
    protected HelixManager _helixManager;
    protected ZkHelixPropertyStore<ZNRecord> _propertyStore;
    protected SegmentLocks _segmentLocks;
    protected TableConfig _tableConfig;
    protected String _tableNameWithType;
    protected String _tableDataDir;
    protected File _indexDir;
    protected File _resourceTmpDir;
    protected Logger _logger;
    protected ExecutorService _segmentPreloadExecutor;
    protected AuthProvider _authProvider;
    protected String _peerDownloadScheme;
    protected long _streamSegmentDownloadUntarRateLimitBytesPerSec;
    protected boolean _isStreamSegmentDownloadUntar;
    private Semaphore _segmentDownloadSemaphore;
    protected Cache<Pair<String, String>, SegmentErrorInfo> _errorCache;
    protected Cache<String, String> _recentlyDeletedSegments;
    protected volatile boolean _shutDown;

    public void init(InstanceDataManagerConfig instanceDataManagerConfig, HelixManager helixManager, SegmentLocks segmentLocks, TableConfig tableConfig, @Nullable ExecutorService segmentPreloadExecutor, @Nullable Cache<Pair<String, String>, SegmentErrorInfo> errorCache) {
        int maxParallelSegmentDownloads;
        LOGGER.info("Initializing table data manager for table: {}", (Object)tableConfig.getTableName());
        this._instanceDataManagerConfig = instanceDataManagerConfig;
        this._instanceId = instanceDataManagerConfig.getInstanceId();
        this._tableConfig = tableConfig;
        this._segmentLocks = segmentLocks;
        this._helixManager = helixManager;
        this._propertyStore = helixManager.getHelixPropertyStore();
        this._segmentPreloadExecutor = segmentPreloadExecutor;
        this._authProvider = AuthProviderUtils.extractAuthProvider((PinotConfiguration)this._instanceDataManagerConfig.getAuthConfig(), null);
        this._tableNameWithType = tableConfig.getTableName();
        this._tableDataDir = this._instanceDataManagerConfig.getInstanceDataDir() + File.separator + this._tableNameWithType;
        this._indexDir = new File(this._tableDataDir);
        if (!this._indexDir.exists()) {
            Preconditions.checkState((boolean)this._indexDir.mkdirs(), (String)"Unable to create index directory at %s. Please check for available space and write-permissions for this directory.", (Object)this._indexDir);
        }
        this._resourceTmpDir = new File(this._indexDir, "tmp");
        FileUtils.deleteQuietly((File)this._resourceTmpDir);
        if (!this._resourceTmpDir.exists()) {
            Preconditions.checkState((boolean)this._resourceTmpDir.mkdirs(), (String)"Unable to create temp resources directory at %s. Please check for available space and write-permissions for this directory.", (Object)this._resourceTmpDir);
        }
        this._errorCache = errorCache;
        this._recentlyDeletedSegments = CacheBuilder.newBuilder().maximumSize((long)instanceDataManagerConfig.getDeletedSegmentsCacheSize()).expireAfterWrite((long)instanceDataManagerConfig.getDeletedSegmentsCacheTtlMinutes(), TimeUnit.MINUTES).build();
        this._peerDownloadScheme = tableConfig.getValidationConfig().getPeerSegmentDownloadScheme();
        if (this._peerDownloadScheme == null) {
            this._peerDownloadScheme = instanceDataManagerConfig.getSegmentPeerDownloadScheme();
        }
        if (this._peerDownloadScheme != null) {
            this._peerDownloadScheme = this._peerDownloadScheme.toLowerCase();
            Preconditions.checkState(("http".equals(this._peerDownloadScheme) || "https".equals(this._peerDownloadScheme) ? 1 : 0) != 0, (String)"Unsupported peer download scheme: %s for table: %s", (Object)this._peerDownloadScheme, (Object)this._tableNameWithType);
        }
        this._streamSegmentDownloadUntarRateLimitBytesPerSec = instanceDataManagerConfig.getStreamSegmentDownloadUntarRateLimit();
        this._isStreamSegmentDownloadUntar = instanceDataManagerConfig.isStreamSegmentDownloadUntar();
        if (this._isStreamSegmentDownloadUntar) {
            LOGGER.info("Using streamed download-untar for segment download! The rate limit interval for streamed download-untar is {} bytes/s", (Object)this._streamSegmentDownloadUntarRateLimitBytesPerSec);
        }
        if ((maxParallelSegmentDownloads = instanceDataManagerConfig.getMaxParallelSegmentDownloads()) > 0) {
            LOGGER.info("Construct segment download semaphore for Table: {}. Maximum number of parallel segment downloads: {}", (Object)this._tableNameWithType, (Object)maxParallelSegmentDownloads);
            this._segmentDownloadSemaphore = new Semaphore(maxParallelSegmentDownloads, true);
        } else {
            this._segmentDownloadSemaphore = null;
        }
        this._logger = LoggerFactory.getLogger((String)(this._tableNameWithType + "-" + this.getClass().getSimpleName()));
        this.doInit();
        this._logger.info("Initialized table data manager with data directory: {}", (Object)this._tableDataDir);
    }

    protected abstract void doInit();

    public String getInstanceId() {
        return this._instanceId;
    }

    public InstanceDataManagerConfig getInstanceDataManagerConfig() {
        return this._instanceDataManagerConfig;
    }

    public synchronized void start() {
        this._logger.info("Starting table data manager");
        this.doStart();
        this._logger.info("Started table data manager");
    }

    protected abstract void doStart();

    public synchronized void shutDown() {
        if (this._shutDown) {
            this._logger.warn("Table data manager is already shut down");
            return;
        }
        this._logger.info("Shutting down table data manager");
        this._shutDown = true;
        this.doShutdown();
        this._logger.info("Shut down table data manager");
    }

    protected abstract void doShutdown();

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void releaseAndRemoveAllSegments() {
        ArrayList<SegmentDataManager> segmentDataManagers;
        ConcurrentHashMap<String, SegmentDataManager> concurrentHashMap = this._segmentDataManagerMap;
        synchronized (concurrentHashMap) {
            segmentDataManagers = new ArrayList<SegmentDataManager>(this._segmentDataManagerMap.values());
            this._segmentDataManagerMap.clear();
        }
        if (!segmentDataManagers.isEmpty()) {
            int numThreads = Math.min(Runtime.getRuntime().availableProcessors(), segmentDataManagers.size());
            ExecutorService stopExecutorService = Executors.newFixedThreadPool(numThreads);
            for (SegmentDataManager segmentDataManager : segmentDataManagers) {
                stopExecutorService.submit(() -> {
                    segmentDataManager.offload();
                    this.releaseSegment(segmentDataManager);
                });
            }
            stopExecutorService.shutdown();
            try {
                if (!stopExecutorService.awaitTermination(10L, TimeUnit.MINUTES)) {
                    stopExecutorService.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                stopExecutorService.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
    }

    public boolean isShutDown() {
        return this._shutDown;
    }

    public Lock getSegmentLock(String segmentName) {
        return this._segmentLocks.getLock(this._tableNameWithType, segmentName);
    }

    public boolean hasSegment(String segmentName) {
        return this._segmentDataManagerMap.containsKey(segmentName);
    }

    public void addSegment(ImmutableSegment immutableSegment) {
        String segmentName = immutableSegment.getSegmentName();
        Preconditions.checkState((!this._shutDown ? 1 : 0) != 0, (String)"Table data manager is already shut down, cannot add segment: %s to table: %s", (Object)segmentName, (Object)this._tableNameWithType);
        this._logger.info("Adding immutable segment: {}", (Object)segmentName);
        this._serverMetrics.addValueToTableGauge(this._tableNameWithType, (AbstractMetrics.Gauge)ServerGauge.DOCUMENT_COUNT, (long)immutableSegment.getSegmentMetadata().getTotalDocs());
        this._serverMetrics.addValueToTableGauge(this._tableNameWithType, (AbstractMetrics.Gauge)ServerGauge.SEGMENT_COUNT, 1L);
        ImmutableSegmentDataManager newSegmentManager = new ImmutableSegmentDataManager(immutableSegment);
        SegmentDataManager oldSegmentManager = this.registerSegment(segmentName, newSegmentManager);
        if (oldSegmentManager == null) {
            this._logger.info("Added new immutable segment: {}", (Object)segmentName);
        } else {
            this._logger.info("Replaced immutable segment: {}", (Object)segmentName);
            oldSegmentManager.offload();
            this.releaseSegment(oldSegmentManager);
        }
    }

    public void addOnlineSegment(String segmentName) throws Exception {
        Preconditions.checkState((!this._shutDown ? 1 : 0) != 0, (String)"Table data manager is already shut down, cannot add ONLINE segment: %s to table: %s", (Object)segmentName, (Object)this._tableNameWithType);
        this._logger.info("Adding ONLINE segment: {}", (Object)segmentName);
        Lock segmentLock = this.getSegmentLock(segmentName);
        segmentLock.lock();
        try {
            this.doAddOnlineSegment(segmentName);
        }
        catch (Exception e) {
            this.addSegmentError(segmentName, new SegmentErrorInfo(System.currentTimeMillis(), "Caught exception while adding ONLINE segment", e));
            throw e;
        }
        finally {
            segmentLock.unlock();
        }
    }

    protected abstract void doAddOnlineSegment(String var1) throws Exception;

    public SegmentZKMetadata fetchZKMetadata(String segmentName) {
        SegmentZKMetadata zkMetadata = ZKMetadataProvider.getSegmentZKMetadata(this._propertyStore, (String)this._tableNameWithType, (String)segmentName);
        Preconditions.checkState((zkMetadata != null ? 1 : 0) != 0, (String)"Failed to find ZK metadata for segment: %s of table: %s", (Object)segmentName, (Object)this._tableNameWithType);
        return zkMetadata;
    }

    public Pair<TableConfig, Schema> fetchTableConfigAndSchema() {
        TableConfig tableConfig = ZKMetadataProvider.getTableConfig(this._propertyStore, (String)this._tableNameWithType);
        Preconditions.checkState((tableConfig != null ? 1 : 0) != 0, (String)"Failed to find table config for table: %s", (Object)this._tableNameWithType);
        Schema schema = ZKMetadataProvider.getTableSchema(this._propertyStore, (TableConfig)tableConfig);
        if (tableConfig.getTableType() == TableType.REALTIME) {
            Preconditions.checkState((schema != null ? 1 : 0) != 0, (String)"Failed to find schema for table: %s", (Object)this._tableNameWithType);
        }
        return Pair.of((Object)tableConfig, (Object)schema);
    }

    public IndexLoadingConfig getIndexLoadingConfig(TableConfig tableConfig, @Nullable Schema schema) {
        IndexLoadingConfig indexLoadingConfig = new IndexLoadingConfig(this._instanceDataManagerConfig, tableConfig, schema);
        indexLoadingConfig.setTableDataDir(this._tableDataDir);
        return indexLoadingConfig;
    }

    public void addNewOnlineSegment(SegmentZKMetadata zkMetadata, IndexLoadingConfig indexLoadingConfig) throws Exception {
        this._logger.info("Adding new ONLINE segment: {}", (Object)zkMetadata.getSegmentName());
        if (!this.tryLoadExistingSegment(zkMetadata, indexLoadingConfig)) {
            this.downloadAndLoadSegment(zkMetadata, indexLoadingConfig);
        }
    }

    protected void replaceSegmentIfCrcMismatch(SegmentDataManager segmentDataManager, SegmentZKMetadata zkMetadata, IndexLoadingConfig indexLoadingConfig) throws Exception {
        String segmentName = segmentDataManager.getSegmentName();
        Preconditions.checkState((boolean)(segmentDataManager instanceof ImmutableSegmentDataManager), (String)"Cannot replace CONSUMING segment: %s in table: %s", (Object)segmentName, (Object)this._tableNameWithType);
        SegmentMetadata localMetadata = segmentDataManager.getSegment().getSegmentMetadata();
        if (BaseTableDataManager.hasSameCRC(zkMetadata, localMetadata)) {
            this._logger.info("Segment: {} has CRC: {} same as before, not replacing it", (Object)segmentName, (Object)localMetadata.getCrc());
            return;
        }
        this._logger.info("Replacing segment: {} because its CRC has changed from: {} to: {}", new Object[]{segmentName, localMetadata.getCrc(), zkMetadata.getCrc()});
        this.downloadAndLoadSegment(zkMetadata, indexLoadingConfig);
        this._logger.info("Replaced segment: {} with new CRC: {}", (Object)segmentName, (Object)zkMetadata.getCrc());
    }

    public void downloadAndLoadSegment(SegmentZKMetadata zkMetadata, IndexLoadingConfig indexLoadingConfig) throws Exception {
        String segmentName = zkMetadata.getSegmentName();
        this._logger.info("Downloading and loading segment: {}", (Object)segmentName);
        File indexDir = this.downloadSegment(zkMetadata);
        this.addSegment(ImmutableSegmentLoader.load((File)indexDir, (IndexLoadingConfig)indexLoadingConfig));
        this._logger.info("Downloaded and loaded segment: {} with CRC: {} on tier: {}", new Object[]{segmentName, zkMetadata.getCrc(), TierConfigUtils.normalizeTierName((String)zkMetadata.getTier())});
    }

    public void replaceSegment(String segmentName) throws Exception {
        Preconditions.checkState((!this._shutDown ? 1 : 0) != 0, (String)"Table data manager is already shut down, cannot replace segment: %s in table: %s", (Object)segmentName, (Object)this._tableNameWithType);
        this._logger.info("Replacing segment: {}", (Object)segmentName);
        Lock segmentLock = this.getSegmentLock(segmentName);
        segmentLock.lock();
        try {
            this.doReplaceSegment(segmentName);
        }
        catch (Exception e) {
            this.addSegmentError(segmentName, new SegmentErrorInfo(System.currentTimeMillis(), "Caught exception while replacing segment", e));
            throw e;
        }
        finally {
            segmentLock.unlock();
        }
    }

    protected void doReplaceSegment(String segmentName) throws Exception {
        SegmentDataManager segmentDataManager = this._segmentDataManagerMap.get(segmentName);
        if (segmentDataManager != null) {
            SegmentZKMetadata zkMetadata = this.fetchZKMetadata(segmentName);
            IndexLoadingConfig indexLoadingConfig = this.fetchIndexLoadingConfig();
            indexLoadingConfig.setSegmentTier(zkMetadata.getTier());
            this.replaceSegmentIfCrcMismatch(segmentDataManager, zkMetadata, indexLoadingConfig);
        } else {
            this._logger.warn("Failed to find segment: {}, skipping replacing it", (Object)segmentName);
        }
    }

    public void offloadSegment(String segmentName) {
        if (this._shutDown) {
            this._logger.info("Table data manager is already shut down, skipping offloading segment: {}", (Object)segmentName);
            return;
        }
        this._logger.info("Offloading segment: {}", (Object)segmentName);
        Lock segmentLock = this.getSegmentLock(segmentName);
        segmentLock.lock();
        try {
            this.doOffloadSegment(segmentName);
        }
        catch (Exception e) {
            this.addSegmentError(segmentName, new SegmentErrorInfo(System.currentTimeMillis(), "Caught exception while offloading segment", e));
            throw e;
        }
        finally {
            segmentLock.unlock();
        }
    }

    public void offloadSegmentUnsafe(String segmentName) {
        if (this._shutDown) {
            this._logger.info("Table data manager is already shut down, skipping offloading segment: {} unsafe", (Object)segmentName);
            return;
        }
        this._logger.info("Offloading segment: {} unsafe", (Object)segmentName);
        try {
            this.doOffloadSegment(segmentName);
        }
        catch (Exception e) {
            this.addSegmentError(segmentName, new SegmentErrorInfo(System.currentTimeMillis(), "Caught exception while offloading segment unsafe", e));
            throw e;
        }
    }

    protected void doOffloadSegment(String segmentName) {
        SegmentDataManager segmentDataManager = this.unregisterSegment(segmentName);
        if (segmentDataManager != null) {
            segmentDataManager.offload();
            this.releaseSegment(segmentDataManager);
            this._logger.info("Offloaded segment: {}", (Object)segmentName);
        } else {
            this._logger.warn("Failed to find segment: {}, skipping offloading it", (Object)segmentName);
        }
    }

    public boolean isSegmentDeletedRecently(String segmentName) {
        return this._recentlyDeletedSegments.getIfPresent((Object)segmentName) != null;
    }

    public List<SegmentDataManager> acquireAllSegments() {
        ArrayList<SegmentDataManager> segmentDataManagers = new ArrayList<SegmentDataManager>();
        for (SegmentDataManager segmentDataManager : this._segmentDataManagerMap.values()) {
            if (!segmentDataManager.increaseReferenceCount()) continue;
            segmentDataManagers.add(segmentDataManager);
        }
        return segmentDataManagers;
    }

    public List<SegmentDataManager> acquireSegments(List<String> segmentNames, List<String> missingSegments) {
        return this.acquireSegments(segmentNames, null, missingSegments);
    }

    public List<SegmentDataManager> acquireSegments(List<String> segmentNames, @Nullable List<String> optionalSegmentNames, List<String> missingSegments) {
        SegmentDataManager segmentDataManager;
        ArrayList<SegmentDataManager> segmentDataManagers = new ArrayList<SegmentDataManager>();
        for (String segmentName : segmentNames) {
            segmentDataManager = this._segmentDataManagerMap.get(segmentName);
            if (segmentDataManager != null && segmentDataManager.increaseReferenceCount()) {
                segmentDataManagers.add(segmentDataManager);
                continue;
            }
            missingSegments.add(segmentName);
        }
        if (optionalSegmentNames != null) {
            for (String segmentName : optionalSegmentNames) {
                segmentDataManager = this._segmentDataManagerMap.get(segmentName);
                if (segmentDataManager == null || !segmentDataManager.increaseReferenceCount()) continue;
                segmentDataManagers.add(segmentDataManager);
            }
        }
        return segmentDataManagers;
    }

    @Nullable
    public SegmentDataManager acquireSegment(String segmentName) {
        SegmentDataManager segmentDataManager = this._segmentDataManagerMap.get(segmentName);
        if (segmentDataManager != null && segmentDataManager.increaseReferenceCount()) {
            return segmentDataManager;
        }
        return null;
    }

    public void releaseSegment(SegmentDataManager segmentDataManager) {
        if (segmentDataManager.decreaseReferenceCount()) {
            this.closeSegment(segmentDataManager);
        }
    }

    private void closeSegment(SegmentDataManager segmentDataManager) {
        String segmentName = segmentDataManager.getSegmentName();
        this._logger.info("Closing segment: {}", (Object)segmentName);
        this._serverMetrics.addValueToTableGauge(this._tableNameWithType, (AbstractMetrics.Gauge)ServerGauge.SEGMENT_COUNT, -1L);
        this._serverMetrics.addMeteredTableValue(this._tableNameWithType, (AbstractMetrics.Meter)ServerMeter.DELETED_SEGMENT_COUNT, 1L);
        this._serverMetrics.addValueToTableGauge(this._tableNameWithType, (AbstractMetrics.Gauge)ServerGauge.DOCUMENT_COUNT, (long)(-segmentDataManager.getSegment().getSegmentMetadata().getTotalDocs()));
        segmentDataManager.destroy();
        this._logger.info("Closed segment: {}", (Object)segmentName);
    }

    public int getNumSegments() {
        return this._segmentDataManagerMap.size();
    }

    public String getTableName() {
        return this._tableNameWithType;
    }

    public File getTableDataDir() {
        return this._indexDir;
    }

    public HelixManager getHelixManager() {
        return this._helixManager;
    }

    public ExecutorService getSegmentPreloadExecutor() {
        return this._segmentPreloadExecutor;
    }

    public void addSegmentError(String segmentName, SegmentErrorInfo segmentErrorInfo) {
        if (this._errorCache != null) {
            this._errorCache.put((Object)Pair.of((Object)this._tableNameWithType, (Object)segmentName), (Object)segmentErrorInfo);
        }
    }

    public Map<String, SegmentErrorInfo> getSegmentErrors() {
        if (this._errorCache != null) {
            HashMap<String, SegmentErrorInfo> segmentErrors = new HashMap<String, SegmentErrorInfo>();
            for (Map.Entry entry : this._errorCache.asMap().entrySet()) {
                Pair tableSegmentPair = (Pair)entry.getKey();
                if (!((String)tableSegmentPair.getLeft()).equals(this._tableNameWithType)) continue;
                segmentErrors.put((String)tableSegmentPair.getRight(), (SegmentErrorInfo)entry.getValue());
            }
            return segmentErrors;
        }
        return Map.of();
    }

    public List<SegmentContext> getSegmentContexts(List<IndexSegment> selectedSegments, Map<String, String> queryOptions) {
        ArrayList<SegmentContext> segmentContexts = new ArrayList<SegmentContext>(selectedSegments.size());
        selectedSegments.forEach(s -> segmentContexts.add(new SegmentContext(s)));
        return segmentContexts;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void reloadSegment(String segmentName, IndexLoadingConfig indexLoadingConfig, SegmentZKMetadata zkMetadata, SegmentMetadata localMetadata, @Nullable Schema schema, boolean forceDownload) throws Exception {
        Preconditions.checkState((!this._shutDown ? 1 : 0) != 0, (String)"Table data manager is already shut down, cannot reload segment: %s of table: %s", (Object)segmentName, (Object)this._tableNameWithType);
        this._logger.info("Reloading segment: {}", (Object)segmentName);
        String segmentTier = this.getSegmentCurrentTier(segmentName);
        indexLoadingConfig.setSegmentTier(segmentTier);
        indexLoadingConfig.setTableDataDir(this._tableDataDir);
        File indexDir = this.getSegmentDataDir(segmentName, segmentTier, indexLoadingConfig.getTableConfig());
        Lock segmentLock = this.getSegmentLock(segmentName);
        segmentLock.lock();
        try {
            boolean shouldDownload;
            boolean bl = shouldDownload = forceDownload || !BaseTableDataManager.hasSameCRC(zkMetadata, localMetadata);
            if (shouldDownload) {
                this.createBackup(indexDir);
                if (forceDownload) {
                    this._logger.info("Force downloading segment: {}", (Object)segmentName);
                } else {
                    this._logger.info("Downloading segment: {} because its CRC has changed from: {} to: {}", new Object[]{segmentName, localMetadata.getCrc(), zkMetadata.getCrc()});
                }
                indexDir = this.downloadSegment(zkMetadata);
            } else {
                this._logger.info("Reloading existing segment: {} on tier: {}", (Object)segmentName, (Object)TierConfigUtils.normalizeTierName((String)segmentTier));
                SegmentDirectory segmentDirectory = this.initSegmentDirectory(segmentName, String.valueOf(zkMetadata.getCrc()), indexLoadingConfig);
                if (this.canReuseExistingDirectoryForReload(zkMetadata, segmentTier, segmentDirectory, indexLoadingConfig, schema)) {
                    this._logger.info("Reloading segment: {} using existing segment directory as no reprocessing needed", (Object)segmentName);
                    ImmutableSegment segment = ImmutableSegmentLoader.load((SegmentDirectory)segmentDirectory, (IndexLoadingConfig)indexLoadingConfig, (Schema)schema);
                    this.addSegment(segment);
                    return;
                }
                this.createBackup(indexDir);
                try {
                    segmentDirectory.copyTo(indexDir);
                }
                finally {
                    segmentDirectory.close();
                }
            }
            indexLoadingConfig.setSegmentTier(zkMetadata.getTier());
            this._logger.info("Loading segment: {} from indexDir: {} to tier: {}", new Object[]{segmentName, indexDir, TierConfigUtils.normalizeTierName((String)zkMetadata.getTier())});
            ImmutableSegment segment = ImmutableSegmentLoader.load((File)indexDir, (IndexLoadingConfig)indexLoadingConfig, (Schema)schema);
            this.addSegment(segment);
            this.removeBackup(indexDir);
        }
        catch (Exception reloadFailureException) {
            try {
                LoaderUtils.reloadFailureRecovery((File)indexDir);
            }
            catch (Exception recoveryFailureException) {
                this._logger.error("Failed to recover segment: {} after reload failure", (Object)segmentName, (Object)recoveryFailureException);
                reloadFailureException.addSuppressed(recoveryFailureException);
            }
            this.addSegmentError(segmentName, new SegmentErrorInfo(System.currentTimeMillis(), "Caught exception while reloading segment", reloadFailureException));
            throw reloadFailureException;
        }
        finally {
            segmentLock.unlock();
        }
        this._logger.info("Reloaded segment: {}", (Object)segmentName);
    }

    private boolean canReuseExistingDirectoryForReload(SegmentZKMetadata segmentZKMetadata, String currentSegmentTier, SegmentDirectory segmentDirectory, IndexLoadingConfig indexLoadingConfig, Schema schema) throws Exception {
        SegmentDirectoryLoader segmentDirectoryLoader = SegmentDirectoryLoaderRegistry.getSegmentDirectoryLoader((String)indexLoadingConfig.getSegmentDirectoryLoader());
        return !segmentDirectoryLoader.needsTierMigration(segmentZKMetadata.getTier(), currentSegmentTier) && !ImmutableSegmentLoader.needPreprocess((SegmentDirectory)segmentDirectory, (IndexLoadingConfig)indexLoadingConfig, (Schema)schema);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    protected SegmentDataManager registerSegment(String segmentName, SegmentDataManager segmentDataManager) {
        SegmentDataManager oldSegmentDataManager;
        ConcurrentHashMap<String, SegmentDataManager> concurrentHashMap = this._segmentDataManagerMap;
        synchronized (concurrentHashMap) {
            oldSegmentDataManager = this._segmentDataManagerMap.put(segmentName, segmentDataManager);
        }
        this._recentlyDeletedSegments.invalidate((Object)segmentName);
        return oldSegmentDataManager;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    protected SegmentDataManager unregisterSegment(String segmentName) {
        this._recentlyDeletedSegments.put((Object)segmentName, (Object)segmentName);
        ConcurrentHashMap<String, SegmentDataManager> concurrentHashMap = this._segmentDataManagerMap;
        synchronized (concurrentHashMap) {
            return this._segmentDataManagerMap.remove(segmentName);
        }
    }

    protected File downloadSegment(SegmentZKMetadata zkMetadata) throws Exception {
        String segmentName = zkMetadata.getSegmentName();
        String downloadUrl = zkMetadata.getDownloadUrl();
        Preconditions.checkState((downloadUrl != null ? 1 : 0) != 0, (String)"Failed to find download URL in ZK metadata for segment: %s of table: %s", (Object)segmentName, (Object)this._tableNameWithType);
        try {
            if (!"".equals(downloadUrl)) {
                try {
                    return this.downloadSegmentFromDeepStore(zkMetadata);
                }
                catch (Exception e) {
                    if (this._peerDownloadScheme != null) {
                        this._logger.warn("Caught exception while downloading segment: {} from: {}, trying to download from peers", new Object[]{segmentName, downloadUrl, e});
                        return this.downloadSegmentFromPeers(zkMetadata);
                    }
                    throw e;
                }
            }
            return this.downloadSegmentFromPeers(zkMetadata);
        }
        catch (Exception e) {
            this._serverMetrics.addMeteredTableValue(this._tableNameWithType, (AbstractMetrics.Meter)ServerMeter.SEGMENT_DOWNLOAD_FAILURES, 1L);
            throw e;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private File downloadSegmentFromDeepStore(SegmentZKMetadata zkMetadata) throws Exception {
        String segmentName = zkMetadata.getSegmentName();
        String downloadUrl = zkMetadata.getDownloadUrl();
        this._logger.info("Downloading segment: {} from: {}", (Object)segmentName, (Object)downloadUrl);
        File tempRootDir = this.getTmpSegmentDataDir("tmp-" + segmentName + "-" + UUID.randomUUID());
        if (this._segmentDownloadSemaphore != null) {
            long startTime = System.currentTimeMillis();
            this._logger.info("Acquiring segment download semaphore for segment: {}, queue-length: {} ", (Object)segmentName, (Object)this._segmentDownloadSemaphore.getQueueLength());
            this._segmentDownloadSemaphore.acquire();
            this._logger.info("Acquired segment download semaphore for segment: {} (lock-time={}ms, queue-length={}).", new Object[]{segmentName, System.currentTimeMillis() - startTime, this._segmentDownloadSemaphore.getQueueLength()});
        }
        try {
            File untarredSegmentDir;
            if (this._isStreamSegmentDownloadUntar && zkMetadata.getCrypterName() == null) {
                this._logger.info("Downloading segment: {} using streamed download-untar with maxStreamRateInByte: {}", (Object)segmentName, (Object)this._streamSegmentDownloadUntarRateLimitBytesPerSec);
                AtomicInteger failedAttempts = new AtomicInteger(0);
                try {
                    untarredSegmentDir = SegmentFetcherFactory.fetchAndStreamUntarToLocal((String)downloadUrl, (File)tempRootDir, (long)this._streamSegmentDownloadUntarRateLimitBytesPerSec, (AtomicInteger)failedAttempts);
                    this._logger.info("Downloaded and untarred segment: {} from: {}, failed attempts: {}", new Object[]{segmentName, downloadUrl, failedAttempts.get()});
                }
                finally {
                    this._serverMetrics.addMeteredTableValue(this._tableNameWithType, (AbstractMetrics.Meter)ServerMeter.SEGMENT_STREAMED_DOWNLOAD_UNTAR_FAILURES, (long)failedAttempts.get());
                }
            } else {
                File segmentTarFile = new File(tempRootDir, segmentName + ".tar.compressed");
                SegmentFetcherFactory.fetchAndDecryptSegmentToLocal((String)downloadUrl, (File)segmentTarFile, (String)zkMetadata.getCrypterName());
                this._logger.info("Downloaded tarred segment: {} from: {} to: {}, file length: {}", new Object[]{segmentName, downloadUrl, segmentTarFile, segmentTarFile.length()});
                untarredSegmentDir = this.untarSegment(segmentName, segmentTarFile, tempRootDir);
            }
            File indexDir = this.moveSegment(segmentName, untarredSegmentDir);
            this._logger.info("Downloaded segment: {} from: {} to: {}", new Object[]{segmentName, downloadUrl, indexDir});
            File file = indexDir;
            return file;
        }
        catch (Exception e) {
            this._serverMetrics.addMeteredTableValue(this._tableNameWithType, (AbstractMetrics.Meter)ServerMeter.SEGMENT_DOWNLOAD_FROM_REMOTE_FAILURES, 1L);
            throw e;
        }
        finally {
            if (this._segmentDownloadSemaphore != null) {
                this._segmentDownloadSemaphore.release();
            }
            FileUtils.deleteQuietly((File)tempRootDir);
        }
    }

    private File downloadSegmentFromPeers(SegmentZKMetadata zkMetadata) throws Exception {
        String segmentName = zkMetadata.getSegmentName();
        Preconditions.checkState((this._peerDownloadScheme != null ? 1 : 0) != 0, (String)"Peer download is not enabled for table: %s", (Object)this._tableNameWithType);
        this._logger.info("Downloading segment: {} from peers", (Object)segmentName);
        File tempRootDir = this.getTmpSegmentDataDir("tmp-" + segmentName + "-" + UUID.randomUUID());
        File segmentTarFile = new File(tempRootDir, segmentName + ".tar.compressed");
        try {
            SegmentFetcherFactory.fetchAndDecryptSegmentToLocal((String)segmentName, (String)this._peerDownloadScheme, () -> {
                List<URI> peerServerURIs = PeerServerSegmentFinder.getPeerServerURIs(this._helixManager, this._tableNameWithType, segmentName, this._peerDownloadScheme);
                Collections.shuffle(peerServerURIs);
                return peerServerURIs;
            }, (File)segmentTarFile, (String)zkMetadata.getCrypterName());
            this._logger.info("Downloaded tarred segment: {} from peers to: {}, file length: {}", new Object[]{segmentName, segmentTarFile, segmentTarFile.length()});
            File indexDir = this.untarAndMoveSegment(segmentName, segmentTarFile, tempRootDir);
            this._logger.info("Downloaded segment: {} from peers to: {}", (Object)segmentName, (Object)indexDir);
            File file = indexDir;
            return file;
        }
        catch (Exception e) {
            this._serverMetrics.addMeteredTableValue(this._tableNameWithType, (AbstractMetrics.Meter)ServerMeter.SEGMENT_DOWNLOAD_FROM_PEERS_FAILURES, 1L);
            throw e;
        }
        finally {
            FileUtils.deleteQuietly((File)tempRootDir);
        }
    }

    private File untarSegment(String segmentName, File segmentTarFile, File tempRootDir) throws IOException {
        File untarDir = new File(tempRootDir, segmentName);
        this._logger.info("Untarring segment: {} from: {} to: {}", new Object[]{segmentName, segmentTarFile, untarDir});
        try {
            File untarredSegmentDir = (File)TarCompressionUtils.untar((File)segmentTarFile, (File)untarDir).get(0);
            this._logger.info("Untarred segment: {} into: {}", (Object)segmentName, (Object)untarredSegmentDir);
            return untarredSegmentDir;
        }
        catch (Exception e) {
            this._serverMetrics.addMeteredTableValue(this._tableNameWithType, (AbstractMetrics.Meter)ServerMeter.UNTAR_FAILURES, 1L);
            throw e;
        }
    }

    private File moveSegment(String segmentName, File untarredSegmentDir) throws IOException {
        File indexDir = this.getSegmentDataDir(segmentName);
        try {
            FileUtils.deleteDirectory((File)indexDir);
            FileUtils.moveDirectory((File)untarredSegmentDir, (File)indexDir);
            return indexDir;
        }
        catch (Exception e) {
            this._serverMetrics.addMeteredTableValue(this._tableNameWithType, (AbstractMetrics.Meter)ServerMeter.SEGMENT_DIR_MOVEMENT_FAILURES, 1L);
            throw e;
        }
    }

    @VisibleForTesting
    File untarAndMoveSegment(String segmentName, File segmentTarFile, File tempRootDir) throws IOException {
        return this.moveSegment(segmentName, this.untarSegment(segmentName, segmentTarFile, tempRootDir));
    }

    @VisibleForTesting
    File getSegmentDataDir(String segmentName) {
        return new File(this._indexDir, segmentName);
    }

    public File getSegmentDataDir(String segmentName, @Nullable String segmentTier, TableConfig tableConfig) {
        if (segmentTier == null) {
            return this.getSegmentDataDir(segmentName);
        }
        String tierDataDir = TierConfigUtils.getDataDirForTier((TableConfig)tableConfig, (String)segmentTier, (Map)this._instanceDataManagerConfig.getTierConfigs());
        if (StringUtils.isEmpty((CharSequence)tierDataDir)) {
            return this.getSegmentDataDir(segmentName);
        }
        File tierTableDataDir = new File(tierDataDir, this._tableNameWithType);
        return new File(tierTableDataDir, segmentName);
    }

    @Nullable
    private String getSegmentCurrentTier(String segmentName) {
        SegmentDataManager segment = this._segmentDataManagerMap.get(segmentName);
        if (segment != null && segment.getSegment() instanceof ImmutableSegment) {
            return ((ImmutableSegment)segment.getSegment()).getTier();
        }
        return null;
    }

    @VisibleForTesting
    protected File getTmpSegmentDataDir(String segmentName) throws IOException {
        File tmpDir = new File(this._resourceTmpDir, segmentName);
        if (tmpDir.exists()) {
            FileUtils.deleteQuietly((File)tmpDir);
        }
        FileUtils.forceMkdir((File)tmpDir);
        return tmpDir;
    }

    private void createBackup(File indexDir) {
        if (!indexDir.exists()) {
            return;
        }
        File parentDir = indexDir.getParentFile();
        File segmentBackupDir = new File(parentDir, indexDir.getName() + ".segment.bak");
        Preconditions.checkState((boolean)indexDir.renameTo(segmentBackupDir), (String)"Failed to rename index directory: %s to segment backup directory: %s", (Object)indexDir, (Object)segmentBackupDir);
    }

    private void removeBackup(File indexDir) throws IOException {
        File parentDir = indexDir.getParentFile();
        File segmentBackupDir = new File(parentDir, indexDir.getName() + ".segment.bak");
        if (!segmentBackupDir.exists()) {
            return;
        }
        File segmentTempDir = new File(parentDir, indexDir.getName() + ".segment.tmp");
        Preconditions.checkState((boolean)segmentBackupDir.renameTo(segmentTempDir), (String)"Failed to rename segment backup directory: %s to segment temporary directory: %s", (Object)segmentBackupDir, (Object)segmentTempDir);
        FileUtils.deleteDirectory((File)segmentTempDir);
    }

    public boolean tryLoadExistingSegment(SegmentZKMetadata zkMetadata, IndexLoadingConfig indexLoadingConfig) {
        SegmentMetadataImpl segmentMetadata;
        String segmentName = zkMetadata.getSegmentName();
        Preconditions.checkState((!this._shutDown ? 1 : 0) != 0, (String)"Table data manager is already shut down, cannot load existing segment: %s of table: %s", (Object)segmentName, (Object)this._tableNameWithType);
        String segmentTier = zkMetadata.getTier();
        File indexDir = this.getSegmentDataDir(segmentName, segmentTier, indexLoadingConfig.getTableConfig());
        BaseTableDataManager.recoverReloadFailureQuietly(this._tableNameWithType, segmentName, indexDir);
        SegmentDirectory segmentDirectory = this.tryInitSegmentDirectory(segmentName, String.valueOf(zkMetadata.getCrc()), indexLoadingConfig);
        SegmentMetadataImpl segmentMetadataImpl = segmentMetadata = segmentDirectory == null ? null : segmentDirectory.getSegmentMetadata();
        if (segmentMetadata == null || !BaseTableDataManager.hasSameCRC(zkMetadata, (SegmentMetadata)segmentMetadata)) {
            if (segmentMetadata == null) {
                this._logger.info("Segment: {} does not exist", (Object)segmentName);
            } else if (!BaseTableDataManager.hasSameCRC(zkMetadata, (SegmentMetadata)segmentMetadata)) {
                this._logger.info("Segment: {} has CRC changed from: {} to: {}", new Object[]{segmentName, segmentMetadata.getCrc(), zkMetadata.getCrc()});
            }
            BaseTableDataManager.closeSegmentDirectoryQuietly(segmentDirectory);
            return false;
        }
        try {
            Schema schema = indexLoadingConfig.getSchema();
            if (!ImmutableSegmentLoader.needPreprocess((SegmentDirectory)segmentDirectory, (IndexLoadingConfig)indexLoadingConfig, (Schema)schema)) {
                this._logger.info("Segment: {} is consistent with latest table config and schema", (Object)segmentName);
            } else {
                this._logger.info("Segment: {} needs reprocess to reflect latest table config and schema", (Object)segmentName);
                segmentDirectory.copyTo(indexDir);
                BaseTableDataManager.closeSegmentDirectoryQuietly(segmentDirectory);
                ImmutableSegmentLoader.preprocess((File)indexDir, (IndexLoadingConfig)indexLoadingConfig, (Schema)schema);
                segmentDirectory = this.initSegmentDirectory(segmentName, String.valueOf(zkMetadata.getCrc()), indexLoadingConfig);
            }
            ImmutableSegment segment = ImmutableSegmentLoader.load((SegmentDirectory)segmentDirectory, (IndexLoadingConfig)indexLoadingConfig, (Schema)schema);
            this.addSegment(segment);
            this._logger.info("Loaded existing segment: {} with CRC: {} on tier: {}", new Object[]{segmentName, zkMetadata.getCrc(), TierConfigUtils.normalizeTierName((String)segmentTier)});
            return true;
        }
        catch (Exception e) {
            this._logger.error("Failed to load existing segment: {} with CRC: {} on tier: {}", new Object[]{segmentName, zkMetadata.getCrc(), TierConfigUtils.normalizeTierName((String)segmentTier), e});
            BaseTableDataManager.closeSegmentDirectoryQuietly(segmentDirectory);
            return false;
        }
    }

    @Nullable
    private SegmentDirectory tryInitSegmentDirectory(String segmentName, String segmentCrc, IndexLoadingConfig indexLoadingConfig) {
        try {
            return this.initSegmentDirectory(segmentName, segmentCrc, indexLoadingConfig);
        }
        catch (Exception e) {
            this._logger.warn("Failed to initialize SegmentDirectory for segment: {} with error: {}", (Object)segmentName, (Object)e.getMessage());
            return null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean needReloadSegments() throws Exception {
        IndexLoadingConfig indexLoadingConfig = this.fetchIndexLoadingConfig();
        List<SegmentDataManager> segmentDataManagers = this.acquireAllSegments();
        boolean needReload = false;
        try {
            for (SegmentDataManager segmentDataManager : segmentDataManagers) {
                ImmutableSegmentImpl immutableSegment;
                IndexSegment segment = segmentDataManager.getSegment();
                if (!(segment instanceof ImmutableSegmentImpl) || !(immutableSegment = (ImmutableSegmentImpl)segment).isReloadNeeded(indexLoadingConfig)) continue;
                needReload = true;
                break;
            }
        }
        finally {
            for (SegmentDataManager segmentDataManager : segmentDataManagers) {
                this.releaseSegment(segmentDataManager);
            }
        }
        return needReload;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<StaleSegment> getStaleSegments(TableConfig tableConfig, Schema schema) {
        ArrayList<StaleSegment> staleSegments = new ArrayList<StaleSegment>();
        List<SegmentDataManager> segmentDataManagers = this.acquireAllSegments();
        long startTime = System.currentTimeMillis();
        try {
            for (SegmentDataManager segmentDataManager : segmentDataManagers) {
                StaleSegment response = this.isSegmentStale(tableConfig, schema, segmentDataManager);
                if (!response.isStale()) continue;
                staleSegments.add(response);
            }
        }
        finally {
            for (SegmentDataManager segmentDataManager : segmentDataManagers) {
                this.releaseSegment(segmentDataManager);
            }
            LOGGER.info("Time Taken to get stale segments: {} ms", (Object)(System.currentTimeMillis() - startTime));
        }
        return staleSegments;
    }

    protected StaleSegment isSegmentStale(TableConfig tableConfig, Schema schema, SegmentDataManager segmentDataManager) {
        List builderConfigFromTableConfigs;
        NavigableSet columnsInSegment;
        String tableNameWithType = tableConfig.getTableName();
        Map indexConfigsMap = FieldIndexConfigsUtil.createIndexConfigsByColName((TableConfig)tableConfig, (Schema)schema);
        String segmentName = segmentDataManager.getSegmentName();
        IndexSegment segment = segmentDataManager.getSegment();
        SegmentMetadata segmentMetadata = segment.getSegmentMetadata();
        Set segmentPhysicalColumns = segment.getPhysicalColumnNames();
        String timeColumn = tableConfig.getValidationConfig().getTimeColumnName();
        if (!(timeColumn == null || segmentMetadata.getTimeColumn() != null && segmentMetadata.getTimeColumn().equals(timeColumn))) {
            LOGGER.debug("tableNameWithType: {}, segmentName: {}, change: time column", (Object)tableNameWithType, (Object)segmentName);
            return new StaleSegment(segmentName, true, "time column");
        }
        List sortedColumns = tableConfig.getIndexingConfig().getSortedColumn();
        String sortedColumn = CollectionUtils.isNotEmpty((Collection)sortedColumns) ? (String)sortedColumns.get(0) : null;
        String partitionColumn = null;
        ColumnPartitionConfig partitionConfig = null;
        SegmentPartitionConfig segmentPartitionConfig = tableConfig.getIndexingConfig().getSegmentPartitionConfig();
        if (segmentPartitionConfig != null && segmentPartitionConfig.getColumnPartitionMap().size() == 1) {
            Map.Entry entry = segmentPartitionConfig.getColumnPartitionMap().entrySet().iterator().next();
            partitionColumn = (String)entry.getKey();
            partitionConfig = (ColumnPartitionConfig)entry.getValue();
        }
        if (!(columnsInSegment = segmentMetadata.getAllColumns()).containsAll(schema.getPhysicalColumnNames())) {
            LOGGER.debug("tableNameWithType: {}, segmentName: {}, change: column added", (Object)tableNameWithType, (Object)segmentName);
            return new StaleSegment(segmentName, true, "column added");
        }
        Set noDictionaryColumns = FieldIndexConfigsUtil.columnsWithIndexDisabled((IndexType)StandardIndexes.dictionary(), (Map)indexConfigsMap);
        Set bloomFilters = FieldIndexConfigsUtil.columnsWithIndexEnabled((IndexType)StandardIndexes.bloomFilter(), (Map)indexConfigsMap);
        Set jsonIndex = FieldIndexConfigsUtil.columnsWithIndexEnabled((IndexType)StandardIndexes.json(), (Map)indexConfigsMap);
        Set invertedIndex = FieldIndexConfigsUtil.columnsWithIndexEnabled((IndexType)StandardIndexes.inverted(), (Map)indexConfigsMap);
        Set nullValueVectorIndex = FieldIndexConfigsUtil.columnsWithIndexEnabled((IndexType)StandardIndexes.nullValueVector(), (Map)indexConfigsMap);
        Set rangeIndex = FieldIndexConfigsUtil.columnsWithIndexEnabled((IndexType)StandardIndexes.range(), (Map)indexConfigsMap);
        Set h3Indexes = FieldIndexConfigsUtil.columnsWithIndexEnabled((IndexType)StandardIndexes.h3(), (Map)indexConfigsMap);
        Set fstIndexes = FieldIndexConfigsUtil.columnsWithIndexEnabled((IndexType)StandardIndexes.fst(), (Map)indexConfigsMap);
        Set textIndexes = FieldIndexConfigsUtil.columnsWithIndexEnabled((IndexType)StandardIndexes.text(), (Map)indexConfigsMap);
        List starTreeIndexConfigsFromTableConfig = tableConfig.getIndexingConfig().getStarTreeIndexConfigs();
        List starTreeIndexMetadata = segment.getStarTrees();
        ArrayList<StarTreeV2BuilderConfig> builderConfigFromSegmentMetadata = new ArrayList<StarTreeV2BuilderConfig>();
        if (starTreeIndexMetadata != null) {
            for (StarTreeV2 starTreeV2 : starTreeIndexMetadata) {
                builderConfigFromSegmentMetadata.add(StarTreeV2BuilderConfig.fromMetadata((StarTreeV2Metadata)starTreeV2.getMetadata()));
            }
        }
        if (!StarTreeBuilderUtils.areStarTreeBuilderConfigListsEqual((List)(builderConfigFromTableConfigs = StarTreeBuilderUtils.generateBuilderConfigs((List)starTreeIndexConfigsFromTableConfig, (boolean)tableConfig.getIndexingConfig().isEnableDefaultStarTree(), (SegmentMetadata)segmentMetadata)), builderConfigFromSegmentMetadata)) {
            return new StaleSegment(segmentName, true, "startree index");
        }
        for (String columnName : segmentPhysicalColumns) {
            ColumnMetadata columnMetadata = segmentMetadata.getColumnMetadataFor(columnName);
            FieldSpec fieldSpecInSchema = schema.getFieldSpecFor(columnName);
            DataSource source = segment.getDataSource(columnName);
            Preconditions.checkNotNull((Object)columnMetadata);
            Preconditions.checkNotNull((Object)source);
            if (fieldSpecInSchema == null) {
                LOGGER.debug("tableNameWithType: {}, columnName: {}, segmentName: {}, change: column deleted", new Object[]{tableNameWithType, columnName, segmentName});
                return new StaleSegment(segmentName, true, "column deleted: " + columnName);
            }
            if (columnMetadata.getFieldType().compareTo((Enum)fieldSpecInSchema.getFieldType()) != 0) {
                LOGGER.debug("tableNameWithType: {}, columnName: {}, segmentName: {}, change: field type", new Object[]{tableNameWithType, columnName, segmentName});
                return new StaleSegment(segmentName, true, "field type changed: " + columnName);
            }
            if (!columnMetadata.getDataType().equals((Object)fieldSpecInSchema.getDataType())) {
                LOGGER.debug("tableNameWithType: {}, columnName: {}, segmentName: {}, change: data type", new Object[]{tableNameWithType, columnName, segmentName});
                return new StaleSegment(segmentName, true, "data type changed: " + columnName);
            }
            if (columnMetadata.isSingleValue() != fieldSpecInSchema.isSingleValueField()) {
                LOGGER.debug("tableNameWithType: {}, columnName: {}, segmentName: {}, change: single / multi value", new Object[]{tableNameWithType, columnName, segmentName});
                return new StaleSegment(segmentName, true, "single / multi value changed: " + columnName);
            }
            boolean colHasDictionary = columnMetadata.hasDictionary();
            if (colHasDictionary == noDictionaryColumns.contains(columnName)) {
                boolean incompatible;
                boolean bl = incompatible = colHasDictionary || DictionaryIndexType.ignoreDictionaryOverride((boolean)tableConfig.getIndexingConfig().isOptimizeDictionary(), (boolean)tableConfig.getIndexingConfig().isOptimizeDictionaryForMetrics(), (double)tableConfig.getIndexingConfig().getNoDictionarySizeRatioThreshold(), (Double)tableConfig.getIndexingConfig().getNoDictionaryCardinalityRatioThreshold(), (FieldSpec)fieldSpecInSchema, (FieldIndexConfigs)((FieldIndexConfigs)indexConfigsMap.get(columnName)), (int)columnMetadata.getCardinality(), (int)columnMetadata.getTotalNumberOfEntries());
                if (incompatible) {
                    LOGGER.debug("tableNameWithType: {}, columnName: {}, segmentName: {}, change: dictionary encoding,", new Object[]{tableNameWithType, columnName, segmentName});
                    return new StaleSegment(segmentName, true, "dictionary encoding changed: " + columnName);
                }
                LOGGER.debug("tableNameWithType: {}, segmentName: {}, no change as dictionary overrides applied to col: {}", new Object[]{tableNameWithType, segmentName, columnName});
            }
            if (columnName.equals(sortedColumn) && !columnMetadata.isSorted()) {
                LOGGER.debug("tableNameWithType: {}, columnName: {}, segmentName: {}, change: sort column", new Object[]{tableNameWithType, columnName, segmentName});
                return new StaleSegment(segmentName, true, "sort column changed: " + columnName);
            }
            if (Objects.isNull(source.getBloomFilter()) == bloomFilters.contains(columnName)) {
                LOGGER.debug("tableNameWithType: {}, columnName: {}, segmentName: {}, change: bloom filter changed", new Object[]{tableNameWithType, columnName, segmentName});
                return new StaleSegment(segmentName, true, "bloom filter changed: " + columnName);
            }
            if (Objects.isNull(source.getJsonIndex()) == jsonIndex.contains(columnName)) {
                LOGGER.debug("tableNameWithType: {}, columnName: {}, segmentName: {}, change: json index changed", new Object[]{tableNameWithType, columnName, segmentName});
                return new StaleSegment(segmentName, true, "json index changed: " + columnName);
            }
            if (Objects.isNull(source.getTextIndex()) == textIndexes.contains(columnName)) {
                LOGGER.debug("tableNameWithType: {}, columnName: {}, segmentName: {}, change: text index changed", new Object[]{tableNameWithType, columnName, segmentName});
                return new StaleSegment(segmentName, true, "text index changed: " + columnName);
            }
            if (Objects.isNull(source.getFSTIndex()) == fstIndexes.contains(columnName)) {
                LOGGER.debug("tableNameWithType: {}, columnName: {}, segmentName: {}, change: fst index changed", new Object[]{tableNameWithType, columnName, segmentName});
                return new StaleSegment(segmentName, true, "fst index changed: " + columnName);
            }
            if (Objects.isNull(source.getH3Index()) == h3Indexes.contains(columnName)) {
                LOGGER.debug("tableNameWithType: {}, columnName: {}, segmentName: {}, change: h3 index changed", new Object[]{tableNameWithType, columnName, segmentName});
                return new StaleSegment(segmentName, true, "hst index changed: " + columnName);
            }
            if (columnMetadata.isSorted()) {
                if (Objects.isNull(source.getInvertedIndex()) && invertedIndex.contains(columnName)) {
                    LOGGER.debug("tableNameWithType: {}, columnName: {}, segmentName: {}, change: inverted index added to sorted column", new Object[]{tableNameWithType, columnName, segmentName});
                    return new StaleSegment(segmentName, true, "invert index added to sort column: " + columnName);
                }
            } else if (Objects.isNull(source.getInvertedIndex()) == invertedIndex.contains(columnName)) {
                LOGGER.debug("tableNameWithType: {}, columnName: {}, segmentName: {}, change: inverted index changed on unsorted column", new Object[]{tableNameWithType, columnName, segmentName});
                return new StaleSegment(segmentName, true, "inverted index changed on unsorted column: " + columnName);
            }
            if (!Objects.isNull(source.getNullValueVector()) && !nullValueVectorIndex.contains(columnName)) {
                LOGGER.debug("tableNameWithType: {}, columnName: {}, segmentName: {}, change: null value vector index removed from column and cannot be added back to this segment.", new Object[]{tableNameWithType, columnName, segmentName});
                return new StaleSegment(segmentName, true, "null value vector index removed from column: " + columnName);
            }
            if (Objects.isNull(source.getRangeIndex()) == rangeIndex.contains(columnName)) {
                LOGGER.debug("tableNameWithType: {}, columnName: {}, segmentName: {}, change: range index changed", new Object[]{tableNameWithType, columnName, segmentName});
                return new StaleSegment(segmentName, true, "range index changed: " + columnName);
            }
            if (!columnName.equals(partitionColumn)) continue;
            PartitionFunction partitionFunction = columnMetadata.getPartitionFunction();
            if (partitionFunction == null) {
                LOGGER.debug("tableNameWithType: {}, columnName: {}, segmentName: {}, change: partition function", new Object[]{tableNameWithType, columnName, segmentName});
                return new StaleSegment(segmentName, true, "partition function added: " + columnName);
            }
            if (!partitionFunction.getName().equalsIgnoreCase(partitionConfig.getFunctionName())) {
                LOGGER.debug("tableNameWithType: {}, columnName: {}, segmentName: {}, change: partition function name", new Object[]{tableNameWithType, columnName, segmentName});
                return new StaleSegment(segmentName, true, "partition function name changed: " + columnName);
            }
            if (partitionFunction.getNumPartitions() != partitionConfig.getNumPartitions()) {
                LOGGER.debug("tableNameWithType: {}, columnName: {},, segmentName: {}, change: num partitions", new Object[]{tableNameWithType, columnName, segmentName});
                return new StaleSegment(segmentName, true, "num partitions changed: " + columnName);
            }
            Set partitions = columnMetadata.getPartitions();
            if (partitions != null && partitions.size() == 1) continue;
            LOGGER.debug("tableNameWithType: {}, columnName: {}, segmentName: {}, change: partitions", new Object[]{tableNameWithType, columnName, segmentName});
            return new StaleSegment(segmentName, true, "partitions changed: " + columnName);
        }
        return new StaleSegment(segmentName, false, null);
    }

    private SegmentDirectory initSegmentDirectory(String segmentName, String segmentCrc, IndexLoadingConfig indexLoadingConfig) throws Exception {
        SegmentDirectoryLoaderContext loaderContext = new SegmentDirectoryLoaderContext.Builder().setTableConfig(indexLoadingConfig.getTableConfig()).setSchema(indexLoadingConfig.getSchema()).setInstanceId(indexLoadingConfig.getInstanceId()).setTableDataDir(indexLoadingConfig.getTableDataDir()).setSegmentName(segmentName).setSegmentCrc(segmentCrc).setSegmentTier(indexLoadingConfig.getSegmentTier()).setInstanceTierConfigs(indexLoadingConfig.getInstanceTierConfigs()).setSegmentDirectoryConfigs(indexLoadingConfig.getSegmentDirectoryConfigs()).build();
        SegmentDirectoryLoader segmentDirectoryLoader = SegmentDirectoryLoaderRegistry.getSegmentDirectoryLoader((String)indexLoadingConfig.getSegmentDirectoryLoader());
        File indexDir = this.getSegmentDataDir(segmentName, indexLoadingConfig.getSegmentTier(), indexLoadingConfig.getTableConfig());
        return segmentDirectoryLoader.load(indexDir.toURI(), loaderContext);
    }

    private static boolean hasSameCRC(SegmentZKMetadata zkMetadata, SegmentMetadata localMetadata) {
        return zkMetadata.getCrc() == Long.parseLong(localMetadata.getCrc());
    }

    private static void recoverReloadFailureQuietly(String tableNameWithType, String segmentName, File indexDir) {
        try {
            LoaderUtils.reloadFailureRecovery((File)indexDir);
        }
        catch (Exception e) {
            LOGGER.warn("Failed to recover segment: {} of table: {} due to error: {}", new Object[]{segmentName, tableNameWithType, e.getMessage()});
        }
    }

    private static void closeSegmentDirectoryQuietly(SegmentDirectory segmentDirectory) {
        if (segmentDirectory != null) {
            try {
                segmentDirectory.close();
            }
            catch (Exception e) {
                LOGGER.warn("Failed to close SegmentDirectory due to error: {}", (Object)e.getMessage());
            }
        }
    }
}

