/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.core.data.manager.realtime;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import java.time.Clock;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import javax.annotation.Nullable;
import org.apache.pinot.common.metrics.AbstractMetrics;
import org.apache.pinot.common.metrics.ServerGauge;
import org.apache.pinot.common.metrics.ServerMetrics;
import org.apache.pinot.common.utils.LLCSegmentName;
import org.apache.pinot.core.data.manager.realtime.RealtimeTableDataManager;
import org.apache.pinot.spi.stream.LongMsgOffset;
import org.apache.pinot.spi.stream.StreamPartitionMsgOffset;
import org.apache.pinot.spi.utils.builder.TableNameBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IngestionDelayTracker {
    private static final Logger LOGGER = LoggerFactory.getLogger(IngestionDelayTracker.class);
    private static final int SCHEDULED_EXECUTOR_THREAD_TICK_INTERVAL_MS = 300000;
    private static final int PARTITION_TIMEOUT_MS = 600000;
    private static final int INITIAL_SCHEDULED_EXECUTOR_THREAD_DELAY_MS = 100;
    private static final int IGNORED_SEGMENT_CACHE_TIME_MINUTES = 10;
    private final Map<Integer, IngestionInfo> _ingestionInfoMap = new ConcurrentHashMap<Integer, IngestionInfo>();
    private final Map<Integer, Long> _partitionsMarkedForVerification = new ConcurrentHashMap<Integer, Long>();
    private final Cache<String, Boolean> _segmentsToIgnore = CacheBuilder.newBuilder().expireAfterAccess(10L, TimeUnit.MINUTES).build();
    private final ScheduledExecutorService _scheduledExecutor = Executors.newScheduledThreadPool(2);
    private final ServerMetrics _serverMetrics;
    private final String _tableNameWithType;
    private final String _metricName;
    private final RealtimeTableDataManager _realTimeTableDataManager;
    private final Supplier<Boolean> _isServerReadyToServeQueries;
    private Clock _clock;

    @VisibleForTesting
    public IngestionDelayTracker(ServerMetrics serverMetrics, final String tableNameWithType, RealtimeTableDataManager realtimeTableDataManager, int scheduledExecutorThreadTickIntervalMs, Supplier<Boolean> isServerReadyToServeQueries) throws RuntimeException {
        this._serverMetrics = serverMetrics;
        this._tableNameWithType = tableNameWithType;
        this._metricName = tableNameWithType;
        this._realTimeTableDataManager = realtimeTableDataManager;
        this._clock = Clock.systemUTC();
        this._isServerReadyToServeQueries = isServerReadyToServeQueries;
        if (scheduledExecutorThreadTickIntervalMs <= 0) {
            throw new RuntimeException("Illegal timer timeout argument, expected > 0, got=" + scheduledExecutorThreadTickIntervalMs + " for table=" + this._tableNameWithType);
        }
        ThreadFactory threadFactory = new ThreadFactory(){
            private final ThreadFactory _defaultFactory = Executors.defaultThreadFactory();

            @Override
            public Thread newThread(Runnable r) {
                Thread thread = this._defaultFactory.newThread(r);
                thread.setName("IngestionDelayTimerThread-" + TableNameBuilder.extractRawTableName((String)tableNameWithType));
                return thread;
            }
        };
        ((ScheduledThreadPoolExecutor)this._scheduledExecutor).setThreadFactory(threadFactory);
        this._scheduledExecutor.scheduleWithFixedDelay(this::timeoutInactivePartitions, 100L, scheduledExecutorThreadTickIntervalMs, TimeUnit.MILLISECONDS);
    }

    public IngestionDelayTracker(ServerMetrics serverMetrics, String tableNameWithType, RealtimeTableDataManager tableDataManager, Supplier<Boolean> isServerReadyToServeQueries) {
        this(serverMetrics, tableNameWithType, tableDataManager, 300000, isServerReadyToServeQueries);
    }

    private long getIngestionDelayMs(long ingestionTimeMs) {
        if (ingestionTimeMs < 0L) {
            return 0L;
        }
        long agedIngestionDelayMs = this._clock.millis() - ingestionTimeMs;
        agedIngestionDelayMs = Math.max(agedIngestionDelayMs, 0L);
        return agedIngestionDelayMs;
    }

    private void removePartitionId(int partitionId) {
        this._ingestionInfoMap.compute(partitionId, (k, v) -> {
            if (v != null) {
                this._serverMetrics.removePartitionGauge(this._metricName, partitionId, (AbstractMetrics.Gauge)ServerGauge.REALTIME_INGESTION_DELAY_MS);
                this._serverMetrics.removePartitionGauge(this._metricName, partitionId, (AbstractMetrics.Gauge)ServerGauge.END_TO_END_REALTIME_INGESTION_DELAY_MS);
                this._serverMetrics.removePartitionGauge(this._metricName, partitionId, (AbstractMetrics.Gauge)ServerGauge.REALTIME_INGESTION_OFFSET_LAG);
                this._serverMetrics.removePartitionGauge(this._metricName, partitionId, (AbstractMetrics.Gauge)ServerGauge.REALTIME_INGESTION_UPSTREAM_OFFSET);
                this._serverMetrics.removePartitionGauge(this._metricName, partitionId, (AbstractMetrics.Gauge)ServerGauge.REALTIME_INGESTION_CONSUMING_OFFSET);
            }
            return null;
        });
        this._partitionsMarkedForVerification.remove(partitionId);
    }

    private List<Integer> getPartitionsToBeVerified() {
        ArrayList<Integer> partitionsToVerify = new ArrayList<Integer>();
        for (Map.Entry<Integer, Long> entry : this._partitionsMarkedForVerification.entrySet()) {
            long timeMarked = this._clock.millis() - entry.getValue();
            if (timeMarked <= 600000L) continue;
            partitionsToVerify.add(entry.getKey());
        }
        return partitionsToVerify;
    }

    @VisibleForTesting
    void setClock(Clock clock) {
        this._clock = clock;
    }

    public void updateIngestionMetrics(String segmentName, int partitionId, long ingestionTimeMs, long firstStreamIngestionTimeMs, @Nullable StreamPartitionMsgOffset currentOffset, @Nullable StreamPartitionMsgOffset latestOffset) {
        if (!this._isServerReadyToServeQueries.get().booleanValue() || this._realTimeTableDataManager.isShutDown()) {
            return;
        }
        if (ingestionTimeMs < 0L && firstStreamIngestionTimeMs < 0L && (currentOffset == null || latestOffset == null)) {
            return;
        }
        this._ingestionInfoMap.compute(partitionId, (k, v) -> {
            if (this._segmentsToIgnore.getIfPresent((Object)segmentName) != null) {
                return v;
            }
            if (v == null) {
                if (ingestionTimeMs > 0L) {
                    this._serverMetrics.setOrUpdatePartitionGauge(this._metricName, partitionId, (AbstractMetrics.Gauge)ServerGauge.REALTIME_INGESTION_DELAY_MS, () -> this.getPartitionIngestionDelayMs(partitionId));
                }
                if (firstStreamIngestionTimeMs > 0L) {
                    this._serverMetrics.setOrUpdatePartitionGauge(this._metricName, partitionId, (AbstractMetrics.Gauge)ServerGauge.END_TO_END_REALTIME_INGESTION_DELAY_MS, () -> this.getPartitionEndToEndIngestionDelayMs(partitionId));
                }
                if (currentOffset != null && latestOffset != null) {
                    this._serverMetrics.setOrUpdatePartitionGauge(this._metricName, partitionId, (AbstractMetrics.Gauge)ServerGauge.REALTIME_INGESTION_OFFSET_LAG, () -> this.getPartitionIngestionOffsetLag(partitionId));
                }
                if (currentOffset != null) {
                    this._serverMetrics.setOrUpdatePartitionGauge(this._metricName, partitionId, (AbstractMetrics.Gauge)ServerGauge.REALTIME_INGESTION_CONSUMING_OFFSET, () -> this.getPartitionIngestionConsumingOffset(partitionId));
                }
                if (latestOffset != null) {
                    this._serverMetrics.setOrUpdatePartitionGauge(this._metricName, partitionId, (AbstractMetrics.Gauge)ServerGauge.REALTIME_INGESTION_UPSTREAM_OFFSET, () -> this.getPartitionIngestionUpstreamOffset(partitionId));
                }
            }
            return new IngestionInfo(ingestionTimeMs, firstStreamIngestionTimeMs, currentOffset, latestOffset);
        });
        this._partitionsMarkedForVerification.remove(partitionId);
    }

    public void stopTrackingPartitionIngestionDelay(int partitionId) {
        this.removePartitionId(partitionId);
    }

    public void stopTrackingPartitionIngestionDelay(String segmentName) {
        this._segmentsToIgnore.put((Object)segmentName, (Object)true);
        this.removePartitionId(new LLCSegmentName(segmentName).getPartitionGroupId());
    }

    public void timeoutInactivePartitions() {
        Set<Integer> partitionsHostedByThisServer;
        if (!this._isServerReadyToServeQueries.get().booleanValue()) {
            return;
        }
        List<Integer> partitionsToVerify = this.getPartitionsToBeVerified();
        if (partitionsToVerify.isEmpty()) {
            return;
        }
        try {
            partitionsHostedByThisServer = this._realTimeTableDataManager.getHostedPartitionsGroupIds();
        }
        catch (Exception e) {
            LOGGER.error("Failed to get partitions hosted by this server, table={}, exception={}:{}", new Object[]{this._tableNameWithType, e.getClass(), e.getMessage()});
            return;
        }
        for (int partitionId : partitionsToVerify) {
            if (partitionsHostedByThisServer.contains(partitionId)) continue;
            this.removePartitionId(partitionId);
        }
    }

    public void markPartitionForVerification(String segmentName) {
        if (!this._isServerReadyToServeQueries.get().booleanValue() || this._segmentsToIgnore.getIfPresent((Object)segmentName) != null) {
            return;
        }
        this._partitionsMarkedForVerification.put(new LLCSegmentName(segmentName).getPartitionGroupId(), this._clock.millis());
    }

    public long getPartitionIngestionTimeMs(int partitionId) {
        IngestionInfo ingestionInfo = this._ingestionInfoMap.get(partitionId);
        return ingestionInfo != null ? ingestionInfo._ingestionTimeMs : Long.MIN_VALUE;
    }

    public long getPartitionIngestionDelayMs(int partitionId) {
        IngestionInfo ingestionInfo = this._ingestionInfoMap.get(partitionId);
        return ingestionInfo != null ? this.getIngestionDelayMs(ingestionInfo._ingestionTimeMs) : 0L;
    }

    public long getPartitionEndToEndIngestionDelayMs(int partitionId) {
        IngestionInfo ingestionInfo = this._ingestionInfoMap.get(partitionId);
        return ingestionInfo != null ? this.getIngestionDelayMs(ingestionInfo._firstStreamIngestionTimeMs) : 0L;
    }

    public long getPartitionIngestionOffsetLag(int partitionId) {
        IngestionInfo ingestionInfo = this._ingestionInfoMap.get(partitionId);
        if (ingestionInfo == null) {
            return 0L;
        }
        StreamPartitionMsgOffset currentOffset = ingestionInfo._currentOffset;
        StreamPartitionMsgOffset latestOffset = ingestionInfo._latestOffset;
        if (currentOffset == null || latestOffset == null) {
            return 0L;
        }
        if (!(currentOffset instanceof LongMsgOffset) || !(latestOffset instanceof LongMsgOffset)) {
            return 0L;
        }
        return ((LongMsgOffset)latestOffset).getOffset() - ((LongMsgOffset)currentOffset).getOffset();
    }

    public long getPartitionIngestionConsumingOffset(int partitionId) {
        IngestionInfo ingestionInfo = this._ingestionInfoMap.get(partitionId);
        if (ingestionInfo == null) {
            return 0L;
        }
        StreamPartitionMsgOffset currentOffset = ingestionInfo._currentOffset;
        if (currentOffset == null) {
            return 0L;
        }
        if (!(currentOffset instanceof LongMsgOffset)) {
            return 0L;
        }
        return ((LongMsgOffset)currentOffset).getOffset();
    }

    public long getPartitionIngestionUpstreamOffset(int partitionId) {
        IngestionInfo ingestionInfo = this._ingestionInfoMap.get(partitionId);
        if (ingestionInfo == null) {
            return 0L;
        }
        StreamPartitionMsgOffset latestOffset = ingestionInfo._latestOffset;
        if (latestOffset == null) {
            return 0L;
        }
        if (!(latestOffset instanceof LongMsgOffset)) {
            return 0L;
        }
        return ((LongMsgOffset)latestOffset).getOffset();
    }

    public void shutdown() {
        this._scheduledExecutor.shutdown();
        if (!this._isServerReadyToServeQueries.get().booleanValue()) {
            return;
        }
        for (Integer partitionId : this._ingestionInfoMap.keySet()) {
            this.removePartitionId(partitionId);
        }
    }

    private static class IngestionInfo {
        final long _ingestionTimeMs;
        final long _firstStreamIngestionTimeMs;
        final StreamPartitionMsgOffset _currentOffset;
        final StreamPartitionMsgOffset _latestOffset;

        IngestionInfo(long ingestionTimeMs, long firstStreamIngestionTimeMs, @Nullable StreamPartitionMsgOffset currentOffset, @Nullable StreamPartitionMsgOffset latestOffset) {
            this._ingestionTimeMs = ingestionTimeMs;
            this._firstStreamIngestionTimeMs = firstStreamIngestionTimeMs;
            this._currentOffset = currentOffset;
            this._latestOffset = latestOffset;
        }
    }
}

