/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.core.query.scheduler;

import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import com.google.common.util.concurrent.MoreExecutors;
import java.util.List;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.LongAccumulator;
import org.apache.pinot.common.exception.QueryException;
import org.apache.pinot.common.metrics.AbstractMetrics;
import org.apache.pinot.common.metrics.ServerMeter;
import org.apache.pinot.common.metrics.ServerMetrics;
import org.apache.pinot.common.metrics.ServerQueryPhase;
import org.apache.pinot.common.metrics.ServerTimer;
import org.apache.pinot.common.utils.config.QueryOptionsUtils;
import org.apache.pinot.core.query.executor.QueryExecutor;
import org.apache.pinot.core.query.request.ServerQueryRequest;
import org.apache.pinot.core.query.scheduler.OutOfCapacityException;
import org.apache.pinot.core.query.scheduler.QueryScheduler;
import org.apache.pinot.core.query.scheduler.SchedulerQueryContext;
import org.apache.pinot.core.query.scheduler.SecondaryWorkloadQueue;
import org.apache.pinot.core.query.scheduler.resources.BinaryWorkloadResourceManager;
import org.apache.pinot.core.query.scheduler.resources.QueryExecutorService;
import org.apache.pinot.spi.env.PinotConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BinaryWorkloadScheduler
extends QueryScheduler {
    private static final Logger LOGGER = LoggerFactory.getLogger(BinaryWorkloadScheduler.class);
    public static final String MAX_SECONDARY_QUERIES = "binarywlm.maxSecondaryRunnerThreads";
    public static final int DEFAULT_MAX_SECONDARY_QUERIES = 5;
    private final int _numSecondaryRunners;
    private final Semaphore _secondaryRunnerSemaphore;
    private final SecondaryWorkloadQueue _secondaryQueryQ;
    Thread _scheduler;

    public BinaryWorkloadScheduler(PinotConfiguration config, QueryExecutor queryExecutor, ServerMetrics metrics, LongAccumulator latestQueryTime) {
        super(config, queryExecutor, new BinaryWorkloadResourceManager(config), metrics, latestQueryTime);
        this._secondaryQueryQ = new SecondaryWorkloadQueue(config, this._resourceManager);
        this._numSecondaryRunners = config.getProperty(MAX_SECONDARY_QUERIES, 5);
        LOGGER.info("numSecondaryRunners={}", (Object)this._numSecondaryRunners);
        this._secondaryRunnerSemaphore = new Semaphore(this._numSecondaryRunners);
    }

    @Override
    public String name() {
        return "BinaryWorkloadScheduler";
    }

    @Override
    public ListenableFuture<byte[]> submit(ServerQueryRequest queryRequest) {
        if (!this._isRunning) {
            return this.immediateErrorResponse(queryRequest, QueryException.SERVER_SCHEDULER_DOWN_ERROR);
        }
        queryRequest.getTimerContext().startNewPhaseTimer(ServerQueryPhase.SCHEDULER_WAIT);
        if (!QueryOptionsUtils.isSecondaryWorkload(queryRequest.getQueryContext().getQueryOptions())) {
            QueryExecutorService queryExecutorService = this._resourceManager.getExecutorService(queryRequest, null);
            ListenableFutureTask<byte[]> queryTask = this.createQueryFutureTask(queryRequest, queryExecutorService);
            this._resourceManager.getQueryRunners().submit(queryTask);
            return queryTask;
        }
        SchedulerQueryContext schedQueryContext = new SchedulerQueryContext(queryRequest);
        try {
            this._serverMetrics.addMeteredTableValue(queryRequest.getTableNameWithType(), (AbstractMetrics.Meter)ServerMeter.NUM_SECONDARY_QUERIES, 1L);
            this._secondaryQueryQ.put(schedQueryContext);
        }
        catch (OutOfCapacityException e) {
            LOGGER.error("Out of capacity for query {} table {}, message: {}", new Object[]{queryRequest.getRequestId(), queryRequest.getTableNameWithType(), e.getMessage()});
            return this.immediateErrorResponse(queryRequest, QueryException.SERVER_OUT_OF_CAPACITY_ERROR);
        }
        catch (Exception e) {
            LOGGER.error("Internal error for query {} table {}, message {}", new Object[]{queryRequest.getRequestId(), queryRequest.getTableNameWithType(), e.getMessage()});
            return this.immediateErrorResponse(queryRequest, QueryException.SERVER_SCHEDULER_DOWN_ERROR);
        }
        return schedQueryContext.getResultFuture();
    }

    @Override
    public void start() {
        super.start();
        this._scheduler = this.getScheduler();
        this._scheduler.setName("scheduler");
        this._scheduler.setPriority(10);
        this._scheduler.setDaemon(true);
        this._scheduler.start();
    }

    private Thread getScheduler() {
        return new Thread(new Runnable(){

            @Override
            public void run() {
                while (BinaryWorkloadScheduler.this._isRunning) {
                    try {
                        BinaryWorkloadScheduler.this._secondaryRunnerSemaphore.acquire();
                    }
                    catch (InterruptedException e) {
                        if (!BinaryWorkloadScheduler.this._isRunning) {
                            LOGGER.info("Shutting down scheduler");
                            break;
                        }
                        LOGGER.error("Interrupt while acquiring semaphore. Exiting.", (Throwable)e);
                        break;
                    }
                    try {
                        final SchedulerQueryContext request = BinaryWorkloadScheduler.this._secondaryQueryQ.take();
                        if (request == null) continue;
                        ServerQueryRequest queryRequest = request.getQueryRequest();
                        final QueryExecutorService executor = BinaryWorkloadScheduler.this._resourceManager.getExecutorService(queryRequest, request.getSchedulerGroup());
                        ListenableFutureTask<byte[]> queryFutureTask = BinaryWorkloadScheduler.this.createQueryFutureTask(queryRequest, executor);
                        queryFutureTask.addListener(new Runnable(){

                            @Override
                            public void run() {
                                executor.releaseWorkers();
                                request.getSchedulerGroup().endQuery();
                                BinaryWorkloadScheduler.this._secondaryRunnerSemaphore.release();
                                BinaryWorkloadScheduler.this.checkStopResourceManager();
                            }
                        }, MoreExecutors.directExecutor());
                        BinaryWorkloadScheduler.this.updateSecondaryWorkloadMetrics(queryRequest);
                        request.setResultFuture((ListenableFuture<byte[]>)queryFutureTask);
                        request.getSchedulerGroup().startQuery();
                        BinaryWorkloadScheduler.this._resourceManager.getQueryRunners().submit(queryFutureTask);
                    }
                    catch (Throwable t) {
                        LOGGER.error("Error in scheduler thread. This is indicative of a bug. Please report this. Server will continue with errors", t);
                    }
                }
                if (BinaryWorkloadScheduler.this._isRunning) {
                    throw new RuntimeException("FATAL: Scheduler thread is quitting.....something went horribly wrong.....!!!");
                }
                BinaryWorkloadScheduler.this.failAllPendingQueries();
            }
        });
    }

    private void updateSecondaryWorkloadMetrics(ServerQueryRequest queryRequest) {
        long timeInQMs = System.currentTimeMillis() - queryRequest.getTimerContext().getQueryArrivalTimeMs();
        this._serverMetrics.addTimedTableValue(queryRequest.getTableNameWithType(), (AbstractMetrics.Timer)ServerTimer.SECONDARY_Q_WAIT_TIME_MS, timeInQMs, TimeUnit.MILLISECONDS);
        this._serverMetrics.addMeteredTableValue(queryRequest.getTableNameWithType(), (AbstractMetrics.Meter)ServerMeter.NUM_SECONDARY_QUERIES_SCHEDULED, 1L);
    }

    @Override
    public void stop() {
        super.stop();
        if (this._scheduler != null) {
            this._scheduler.interrupt();
        }
    }

    private void checkStopResourceManager() {
        if (!this._isRunning && this._secondaryRunnerSemaphore.availablePermits() == this._numSecondaryRunners) {
            this._resourceManager.stop();
        }
    }

    private synchronized void failAllPendingQueries() {
        List<SchedulerQueryContext> pending = this._secondaryQueryQ.drain();
        for (SchedulerQueryContext queryContext : pending) {
            queryContext.setResultFuture(this.immediateErrorResponse(queryContext.getQueryRequest(), QueryException.SERVER_SCHEDULER_DOWN_ERROR));
        }
    }
}

