/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.pinot;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.trino.plugin.base.util.JsonTypeUtil;
import io.trino.plugin.pinot.PinotColumnHandle;
import io.trino.plugin.pinot.PinotErrorCode;
import io.trino.plugin.pinot.PinotException;
import io.trino.plugin.pinot.client.PinotDataFetcher;
import io.trino.plugin.pinot.client.PinotDataTableWithSize;
import io.trino.plugin.pinot.conversion.PinotTimestamps;
import io.trino.plugin.pinot.decoders.VarbinaryDecoder;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.Page;
import io.trino.spi.PageBuilder;
import io.trino.spi.TrinoException;
import io.trino.spi.block.Block;
import io.trino.spi.block.BlockBuilder;
import io.trino.spi.connector.ConnectorPageSource;
import io.trino.spi.connector.SourcePage;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.RealType;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.Type;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.pinot.common.datatable.DataTable;
import org.apache.pinot.common.utils.DataSchema;
import org.roaringbitmap.RoaringBitmap;

public class PinotSegmentPageSource
implements ConnectorPageSource {
    private final List<PinotColumnHandle> columnHandles;
    private final List<Type> columnTypes;
    private final long targetSegmentPageSizeBytes;
    private final PinotDataFetcher pinotDataFetcher;
    private long completedBytes;
    private long estimatedMemoryUsageInBytes;
    private PinotDataTableWithSize currentDataTable;
    private boolean closed;

    public PinotSegmentPageSource(long targetSegmentPageSizeBytes, List<PinotColumnHandle> columnHandles, PinotDataFetcher pinotDataFetcher) {
        this.columnHandles = Objects.requireNonNull(columnHandles, "columnHandles is null");
        this.columnTypes = columnHandles.stream().map(PinotColumnHandle::getDataType).collect(Collectors.toList());
        this.targetSegmentPageSizeBytes = targetSegmentPageSizeBytes;
        this.pinotDataFetcher = Objects.requireNonNull(pinotDataFetcher, "pinotDataFetcher is null");
    }

    public long getCompletedBytes() {
        return this.completedBytes;
    }

    public long getReadTimeNanos() {
        return this.pinotDataFetcher.getReadTimeNanos();
    }

    public long getMemoryUsage() {
        return this.estimatedMemoryUsageInBytes;
    }

    public boolean isFinished() {
        return this.closed || this.pinotDataFetcher.isDataFetched() && this.pinotDataFetcher.endOfData();
    }

    public SourcePage getNextSourcePage() {
        if (this.isFinished()) {
            this.close();
            return null;
        }
        if (!this.pinotDataFetcher.isDataFetched()) {
            this.pinotDataFetcher.fetchData();
            this.estimatedMemoryUsageInBytes = this.pinotDataFetcher.getMemoryUsageBytes();
        }
        if (this.pinotDataFetcher.endOfData()) {
            this.close();
            return null;
        }
        long pageSizeBytes = 0L;
        PageBuilder pageBuilder = new PageBuilder(this.columnTypes);
        while (!this.pinotDataFetcher.endOfData() && pageSizeBytes < this.targetSegmentPageSizeBytes) {
            if (this.currentDataTable != null) {
                this.estimatedMemoryUsageInBytes -= this.currentDataTable.estimatedSizeInBytes();
            }
            this.currentDataTable = this.pinotDataFetcher.getNextDataTable();
            this.estimatedMemoryUsageInBytes += this.currentDataTable.estimatedSizeInBytes();
            pageSizeBytes += this.currentDataTable.estimatedSizeInBytes();
            pageBuilder.declarePositions(this.currentDataTable.dataTable().getNumberOfRows());
            Map<Integer, RoaringBitmap> nullRowIds = PinotSegmentPageSource.buildColumnIdToNullRowId(this.currentDataTable.dataTable(), this.columnHandles);
            for (int rowIndex = 0; rowIndex < this.currentDataTable.dataTable().getNumberOfRows(); ++rowIndex) {
                for (int columnHandleIdx = 0; columnHandleIdx < this.columnHandles.size(); ++columnHandleIdx) {
                    BlockBuilder blockBuilder = pageBuilder.getBlockBuilder(columnHandleIdx);
                    Type columnType = this.columnTypes.get(columnHandleIdx);
                    if (nullRowIds.containsKey(columnHandleIdx) && nullRowIds.get(columnHandleIdx).contains(rowIndex)) {
                        blockBuilder.appendNull();
                        continue;
                    }
                    this.writeBlock(blockBuilder, columnType, rowIndex, columnHandleIdx);
                }
            }
        }
        return SourcePage.create((Page)pageBuilder.build());
    }

    private static Map<Integer, RoaringBitmap> buildColumnIdToNullRowId(DataTable dataTable, List<PinotColumnHandle> columnHandles) {
        ImmutableMap.Builder nullRowIds = ImmutableMap.builder();
        for (int i = 0; i < columnHandles.size(); ++i) {
            RoaringBitmap nullRowId = dataTable.getNullRowIds(i);
            if (nullRowId == null) continue;
            nullRowIds.put((Object)i, (Object)nullRowId);
        }
        return nullRowIds.buildOrThrow();
    }

    public void close() {
        if (this.closed) {
            return;
        }
        this.closed = true;
    }

    private void writeBlock(BlockBuilder blockBuilder, Type columnType, int rowIdx, int columnIdx) {
        Class javaType = columnType.getJavaType();
        DataSchema.ColumnDataType pinotColumnType = this.currentDataTable.dataTable().getDataSchema().getColumnDataType(columnIdx);
        if (javaType.equals(Boolean.TYPE)) {
            this.writeBooleanBlock(blockBuilder, columnType, rowIdx, columnIdx);
        } else if (javaType.equals(Long.TYPE)) {
            if (columnType instanceof TimestampType) {
                this.writeShortTimestampBlock(blockBuilder, columnType, rowIdx, columnIdx);
            } else {
                this.writeLongBlock(blockBuilder, columnType, rowIdx, columnIdx);
            }
        } else if (javaType.equals(Double.TYPE)) {
            this.writeDoubleBlock(blockBuilder, columnType, rowIdx, columnIdx);
        } else if (javaType.equals(Slice.class)) {
            this.writeSliceBlock(blockBuilder, columnType, rowIdx, columnIdx);
        } else if (javaType.equals(Block.class)) {
            this.writeArrayBlock(blockBuilder, columnType, rowIdx, columnIdx);
        } else {
            throw new TrinoException((ErrorCodeSupplier)PinotErrorCode.PINOT_UNSUPPORTED_COLUMN_TYPE, String.format("Failed to write column %s. pinotColumnType %s, javaType %s", this.columnHandles.get(columnIdx).getColumnName(), pinotColumnType, javaType));
        }
    }

    private void writeBooleanBlock(BlockBuilder blockBuilder, Type columnType, int rowIndex, int columnIndex) {
        columnType.writeBoolean(blockBuilder, this.getBoolean(rowIndex, columnIndex));
        ++this.completedBytes;
    }

    private void writeLongBlock(BlockBuilder blockBuilder, Type columnType, int rowIndex, int columnIndex) {
        columnType.writeLong(blockBuilder, this.getLong(rowIndex, columnIndex));
        this.completedBytes += 8L;
    }

    private void writeDoubleBlock(BlockBuilder blockBuilder, Type columnType, int rowIndex, int columnIndex) {
        columnType.writeDouble(blockBuilder, this.getDouble(rowIndex, columnIndex));
        this.completedBytes += 8L;
    }

    private void writeSliceBlock(BlockBuilder blockBuilder, Type columnType, int rowIndex, int columnIndex) {
        Slice slice = this.getSlice(rowIndex, columnIndex);
        columnType.writeSlice(blockBuilder, slice, 0, slice.length());
        this.completedBytes += (long)slice.getBytes().length;
    }

    private void writeArrayBlock(BlockBuilder blockBuilder, Type columnType, int rowIndex, int columnIndex) {
        Block block = this.getArrayBlock(rowIndex, columnIndex);
        columnType.writeObject(blockBuilder, (Object)block);
        this.completedBytes += block.getSizeInBytes();
    }

    private void writeShortTimestampBlock(BlockBuilder blockBuilder, Type columnType, int rowIndex, int columnIndex) {
        columnType.writeLong(blockBuilder, PinotTimestamps.toMicros(this.getLong(rowIndex, columnIndex)));
        this.completedBytes += 8L;
    }

    private Type getType(int columnIndex) {
        Preconditions.checkArgument((columnIndex < this.columnHandles.size() ? 1 : 0) != 0, (Object)"Invalid field index");
        return this.columnHandles.get(columnIndex).getDataType();
    }

    private boolean getBoolean(int rowIdx, int columnIndex) {
        return this.currentDataTable.dataTable().getInt(rowIdx, columnIndex) != 0;
    }

    private long getLong(int rowIndex, int columnIndex) {
        DataSchema.ColumnDataType dataType = this.currentDataTable.dataTable().getDataSchema().getColumnDataType(columnIndex);
        return switch (dataType) {
            case DataSchema.ColumnDataType.DOUBLE -> (long)this.currentDataTable.dataTable().getDouble(rowIndex, columnIndex);
            case DataSchema.ColumnDataType.INT -> this.currentDataTable.dataTable().getInt(rowIndex, columnIndex);
            case DataSchema.ColumnDataType.FLOAT -> Float.floatToIntBits(this.currentDataTable.dataTable().getFloat(rowIndex, columnIndex));
            case DataSchema.ColumnDataType.LONG, DataSchema.ColumnDataType.TIMESTAMP -> this.currentDataTable.dataTable().getLong(rowIndex, columnIndex);
            default -> throw new PinotException(PinotErrorCode.PINOT_DECODE_ERROR, Optional.empty(), String.format("Unexpected pinot type: '%s'", dataType));
        };
    }

    private double getDouble(int rowIndex, int columnIndex) {
        DataSchema.ColumnDataType dataType = this.currentDataTable.dataTable().getDataSchema().getColumnDataType(columnIndex);
        if (dataType.equals((Object)DataSchema.ColumnDataType.FLOAT)) {
            return this.currentDataTable.dataTable().getFloat(rowIndex, columnIndex);
        }
        return this.currentDataTable.dataTable().getDouble(rowIndex, columnIndex);
    }

    private Block getArrayBlock(int rowIndex, int columnIndex) {
        BlockBuilder blockBuilder;
        Type trinoType = this.getType(columnIndex);
        Type elementType = (Type)trinoType.getTypeParameters().get(0);
        DataSchema.ColumnDataType columnType = this.currentDataTable.dataTable().getDataSchema().getColumnDataType(columnIndex);
        switch (columnType) {
            case INT_ARRAY: {
                int[] intArray = this.currentDataTable.dataTable().getIntArray(rowIndex, columnIndex);
                blockBuilder = elementType.createBlockBuilder(null, intArray.length);
                for (int element : intArray) {
                    IntegerType.INTEGER.writeInt(blockBuilder, element);
                }
                break;
            }
            case LONG_ARRAY: {
                long[] longArray = this.currentDataTable.dataTable().getLongArray(rowIndex, columnIndex);
                blockBuilder = elementType.createBlockBuilder(null, longArray.length);
                for (long element : longArray) {
                    BigintType.BIGINT.writeLong(blockBuilder, element);
                }
                break;
            }
            case FLOAT_ARRAY: {
                float[] floatArray = this.currentDataTable.dataTable().getFloatArray(rowIndex, columnIndex);
                blockBuilder = elementType.createBlockBuilder(null, floatArray.length);
                for (float element : floatArray) {
                    RealType.REAL.writeFloat(blockBuilder, element);
                }
                break;
            }
            case DOUBLE_ARRAY: {
                double[] doubleArray = this.currentDataTable.dataTable().getDoubleArray(rowIndex, columnIndex);
                blockBuilder = elementType.createBlockBuilder(null, doubleArray.length);
                for (double element : doubleArray) {
                    elementType.writeDouble(blockBuilder, element);
                }
                break;
            }
            case STRING_ARRAY: {
                String[] stringArray = this.currentDataTable.dataTable().getStringArray(rowIndex, columnIndex);
                blockBuilder = elementType.createBlockBuilder(null, stringArray.length);
                for (String element : stringArray) {
                    Slice slice = this.getUtf8Slice(element);
                    elementType.writeSlice(blockBuilder, slice, 0, slice.length());
                }
                break;
            }
            default: {
                throw new UnsupportedOperationException(String.format("Unexpected pinot type '%s'", columnType));
            }
        }
        return blockBuilder.build();
    }

    private Slice getSlice(int rowIndex, int columnIndex) {
        Type trinoType = this.getType(columnIndex);
        DataTable dataTable = this.currentDataTable.dataTable();
        if (trinoType instanceof VarcharType) {
            String field = dataTable.getString(rowIndex, columnIndex);
            return this.getUtf8Slice(field);
        }
        if (trinoType instanceof VarbinaryType) {
            if (dataTable.getVersion() >= 4) {
                try {
                    return Slices.wrappedBuffer((byte[])dataTable.getBytes(rowIndex, columnIndex).getBytes());
                }
                catch (NullPointerException e) {
                    return Slices.wrappedBuffer((byte[])new byte[0]);
                }
            }
            return Slices.wrappedBuffer((byte[])VarbinaryDecoder.toBytes(dataTable.getString(rowIndex, columnIndex)));
        }
        if (trinoType.getTypeSignature().getBase().equalsIgnoreCase("json")) {
            String field = dataTable.getString(rowIndex, columnIndex);
            return JsonTypeUtil.jsonParse((Slice)this.getUtf8Slice(field));
        }
        return Slices.EMPTY_SLICE;
    }

    private Slice getUtf8Slice(String value) {
        if (Strings.isNullOrEmpty((String)value)) {
            return Slices.EMPTY_SLICE;
        }
        return Slices.utf8Slice((String)value);
    }
}

