/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.segment.spi.memory;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import org.apache.pinot.segment.spi.memory.CleanerUtil;
import org.apache.pinot.segment.spi.memory.CompoundDataBuffer;
import org.apache.pinot.segment.spi.memory.DataBuffer;
import org.apache.pinot.segment.spi.memory.PinotDataBuffer;
import org.apache.pinot.segment.spi.memory.PinotOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PagedPinotOutputStream
extends PinotOutputStream {
    private final PageAllocator _allocator;
    private final int _pageSize;
    private final ArrayList<ByteBuffer> _pages;
    private ByteBuffer _currentPage;
    private long _currentPageStartOffset;
    private int _offsetInPage;
    private long _written = 0L;
    private static final Logger LOGGER = LoggerFactory.getLogger(PagedPinotOutputStream.class);

    public PagedPinotOutputStream(PageAllocator allocator) {
        this._pageSize = allocator.pageSize();
        this._allocator = allocator;
        this._pages = new ArrayList(8);
        this._currentPage = this._allocator.allocate().order(ByteOrder.BIG_ENDIAN);
        this._currentPageStartOffset = 0L;
        this._pages.add(this._currentPage);
    }

    public static PagedPinotOutputStream createHeap() {
        return new PagedPinotOutputStream(HeapPageAllocator.createSmall());
    }

    private void nextPage() {
        this.moveCurrentOffset(this.remainingInPage());
    }

    private int remainingInPage() {
        return this._pageSize - this._offsetInPage;
    }

    public ByteBuffer[] getPages() {
        boolean lastPageIsEmpty;
        int numPages = this._pages.size();
        long lastPageStart = (long)(numPages - 1) * (long)this._pageSize;
        assert (lastPageStart >= 0L) : "lastPageStart=" + lastPageStart;
        assert (lastPageStart <= this._written) : "lastPageStart=" + lastPageStart + ", _written=" + this._written;
        boolean bl = lastPageIsEmpty = this._written == lastPageStart;
        if (lastPageIsEmpty) {
            --numPages;
        }
        if (numPages == 0) {
            return new ByteBuffer[0];
        }
        ByteBuffer[] result = new ByteBuffer[numPages];
        for (int i = 0; i < numPages; ++i) {
            ByteBuffer byteBuffer = this._pages.get(i);
            ByteBuffer page = byteBuffer.asReadOnlyBuffer();
            page.clear();
            result[i] = page;
        }
        if (!lastPageIsEmpty) {
            result[numPages - 1].limit((int)(this._written - lastPageStart));
        }
        return result;
    }

    @Override
    public long getCurrentOffset() {
        return this._currentPageStartOffset + (long)this._offsetInPage;
    }

    @Override
    public void seek(long newPos) {
        if (newPos < 0L) {
            throw new IllegalArgumentException("New position cannot be negative");
        }
        if (newPos == 0L) {
            this._currentPage = this._pages.get(0);
            this._offsetInPage = 0;
        } else {
            int pageIdx = (int)(newPos / (long)this._pageSize);
            if (pageIdx >= this._pages.size()) {
                this._pages.ensureCapacity(pageIdx + 1);
                while (this._pages.size() <= pageIdx) {
                    this._pages.add(this._allocator.allocate().order(ByteOrder.BIG_ENDIAN));
                }
            }
            int offsetInPage = (int)(newPos % (long)this._pageSize);
            this._currentPage = this._pages.get(pageIdx);
            this._currentPageStartOffset = (long)pageIdx * (long)this._pageSize;
            this._offsetInPage = offsetInPage;
        }
        this._written = Math.max(this._written, newPos);
    }

    @Override
    public void write(int b) throws IOException {
        if (this.remainingInPage() == 0) {
            this.nextPage();
        }
        this._currentPage.put(this._offsetInPage++, (byte)b);
        this._written = Math.max(this._written, (long)this._offsetInPage + this._currentPageStartOffset);
    }

    @Override
    public void writeInt(int v) throws IOException {
        if (this.remainingInPage() >= 4) {
            this._currentPage.putInt(this._offsetInPage, v);
            this._offsetInPage += 4;
            this._written = Math.max(this._written, (long)this._offsetInPage + this._currentPageStartOffset);
        } else {
            super.writeInt(v);
        }
    }

    @Override
    public void writeLong(long v) throws IOException {
        if (this.remainingInPage() >= 8) {
            this._currentPage.putLong(this._offsetInPage, v);
            this._offsetInPage += 8;
            this._written = Math.max(this._written, (long)this._offsetInPage + this._currentPageStartOffset);
        } else {
            super.writeLong(v);
        }
    }

    @Override
    public void writeShort(int v) throws IOException {
        if (this.remainingInPage() >= 2) {
            this._currentPage.putShort(this._offsetInPage, (short)v);
            this._offsetInPage += 2;
            this._written = Math.max(this._written, (long)this._offsetInPage + this._currentPageStartOffset);
        } else {
            super.writeShort(v);
        }
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        if (this.remainingInPage() >= len) {
            this._currentPage.position(this._offsetInPage);
            this._currentPage.put(b, off, len);
            this._offsetInPage += len;
            this._currentPage.position(0);
        } else {
            int written = 0;
            while (written < len) {
                int remainingInPage = this.remainingInPage();
                if (remainingInPage == 0) {
                    this.nextPage();
                    continue;
                }
                int toWrite = Math.min(len - written, remainingInPage);
                this._currentPage.position(this._offsetInPage);
                this._currentPage.put(b, off + written, toWrite);
                this._currentPage.position(0);
                written += toWrite;
                this._offsetInPage += toWrite;
            }
        }
        this._written = Math.max(this._written, (long)this._offsetInPage + this._currentPageStartOffset);
    }

    @Override
    public void write(DataBuffer input, long offset, long length) throws IOException {
        if ((long)this.remainingInPage() >= length) {
            int intLength = (int)length;
            input.copyTo(offset, this._currentPage, this._offsetInPage, intLength);
            this._offsetInPage += intLength;
        } else {
            long written = 0L;
            while (written < length) {
                int remainingInPage = this.remainingInPage();
                if (remainingInPage == 0) {
                    this.nextPage();
                    continue;
                }
                int toWrite = (int)Math.min(length - written, (long)remainingInPage);
                input.copyTo(offset + written, this._currentPage, this._offsetInPage, toWrite);
                written += (long)toWrite;
                this._offsetInPage += toWrite;
            }
        }
        this._written = Math.max(this._written, (long)this._offsetInPage + this._currentPageStartOffset);
    }

    public DataBuffer asBuffer(ByteOrder order, boolean owner) {
        if (this._written == 0L) {
            return PinotDataBuffer.empty();
        }
        ByteBuffer[] pages = this.getPages();
        for (int i = 0; i < pages.length; ++i) {
            ByteBuffer page = pages[i];
            if (page.remaining() == this._pageSize || i == pages.length - 1 && page.hasRemaining()) continue;
            throw new IllegalArgumentException("Unexpected remaining bytes in page " + i + ": " + page.remaining());
        }
        return new CompoundDataBuffer(pages, order, owner);
    }

    public int getPageSize() {
        return this._pageSize;
    }

    @Override
    public void close() throws IOException {
        IOException ex = null;
        for (ByteBuffer page : this._pages) {
            try {
                this._allocator.release(page);
            }
            catch (IOException e) {
                if (ex == null) {
                    ex = e;
                    continue;
                }
                ex.addSuppressed(e);
            }
        }
        if (ex != null) {
            throw ex;
        }
    }

    public static class DirectPageAllocator
    extends PageAllocator {
        private final int _pageSize;
        private final boolean _release;

        public DirectPageAllocator(int pageSize) {
            this(pageSize, false);
        }

        public DirectPageAllocator(int pageSize, boolean release) {
            Preconditions.checkArgument((pageSize > 0 ? 1 : 0) != 0, (Object)"Page size must be positive");
            this._pageSize = pageSize;
            this._release = release;
        }

        public static DirectPageAllocator createSmall(boolean release) {
            return new DirectPageAllocator(16384, release);
        }

        public static DirectPageAllocator createLarge(boolean release) {
            return new DirectPageAllocator(0x100000, release);
        }

        @Override
        public int pageSize() {
            return this._pageSize;
        }

        @Override
        public ByteBuffer allocate() {
            return ByteBuffer.allocateDirect(this._pageSize);
        }

        @Override
        public void release(ByteBuffer buffer) throws IOException {
            if (this._release && CleanerUtil.UNMAP_SUPPORTED) {
                CleanerUtil.getCleaner().freeBuffer(buffer);
            }
        }
    }

    public static class HeapPageAllocator
    extends PageAllocator {
        private final int _pageSize;

        public static HeapPageAllocator createSmall() {
            return new HeapPageAllocator(16384);
        }

        public static HeapPageAllocator createLarge() {
            return new HeapPageAllocator(0x100000);
        }

        public HeapPageAllocator(int pageSize) {
            Preconditions.checkArgument((pageSize > 0 ? 1 : 0) != 0, (Object)"Page size must be positive");
            this._pageSize = pageSize;
        }

        @Override
        public int pageSize() {
            return this._pageSize;
        }

        @Override
        public ByteBuffer allocate() {
            return ByteBuffer.allocate(this._pageSize);
        }

        @Override
        public void release(ByteBuffer buffer) {
        }
    }

    public static abstract class PageAllocator {
        public static final int MIN_RECOMMENDED_PAGE_SIZE = 16384;
        public static final int MAX_RECOMMENDED_PAGE_SIZE = 0x100000;

        public abstract int pageSize();

        public abstract ByteBuffer allocate();

        public abstract void release(ByteBuffer var1) throws IOException;
    }
}

