/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.segment.local.utils;

import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import org.apache.pinot.segment.local.utils.GeometryType;
import org.apache.pinot.segment.local.utils.GeometryUtils;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.LinearRing;
import org.locationtech.jts.geom.MultiPoint;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.geom.TopologyException;

public final class GeometrySerializer {
    private static final int TYPE_SIZE = 1;
    private static final int COORDINATE_SIZE = 16;

    public static byte[] serialize(Geometry geometry) {
        return GeometrySerializer.writeGeometry(geometry);
    }

    public static Geometry deserialize(byte[] bytes) {
        return GeometrySerializer.readGeometry(bytes);
    }

    private static byte[] writeGeometry(Geometry geometry) {
        byte[] bytes = new byte[GeometrySerializer.getByteSize(geometry)];
        ByteBuffer byteBuffer = ByteBuffer.wrap(bytes);
        GeometrySerializer.writeGeometryByteBuffer(byteBuffer, geometry);
        return bytes;
    }

    private static Geometry readGeometry(byte[] bytes) {
        return GeometrySerializer.readGeometry(ByteBuffer.wrap(bytes));
    }

    private static Geometry readGeometry(ByteBuffer byteBuffer) {
        byte typeByte = byteBuffer.get();
        GeometryType type = GeometrySerializer.readGeometryType(typeByte);
        GeometryFactory factory = GeometrySerializer.getGeometryFactory(typeByte);
        Geometry geometry = GeometrySerializer.readGeometry(byteBuffer, type, factory);
        return geometry;
    }

    private static Geometry readGeometry(ByteBuffer byteBuffer, GeometryType type, GeometryFactory factory) {
        switch (type) {
            case POINT: {
                return GeometrySerializer.readPoint(byteBuffer, factory);
            }
            case MULTI_POINT: {
                return GeometrySerializer.readMultiPoint(byteBuffer, factory);
            }
            case LINE_STRING: {
                return GeometrySerializer.readPolyline(byteBuffer, false, factory);
            }
            case MULTI_LINE_STRING: {
                return GeometrySerializer.readPolyline(byteBuffer, true, factory);
            }
            case POLYGON: {
                return GeometrySerializer.readPolygon(byteBuffer, false, factory);
            }
            case MULTI_POLYGON: {
                return GeometrySerializer.readPolygon(byteBuffer, true, factory);
            }
            case GEOMETRY_COLLECTION: {
                return GeometrySerializer.readGeometryCollection(byteBuffer, factory);
            }
        }
        throw new UnsupportedOperationException("Unexpected type: " + type);
    }

    private static Point readPoint(ByteBuffer byteBuffer, GeometryFactory factory) {
        Coordinate coordinates = GeometrySerializer.readCoordinate(byteBuffer);
        if (Double.isNaN(coordinates.x) || Double.isNaN(coordinates.y)) {
            return factory.createPoint();
        }
        return factory.createPoint(coordinates);
    }

    private static Coordinate readCoordinate(ByteBuffer byteBuffer) {
        return new Coordinate(byteBuffer.getDouble(), byteBuffer.getDouble());
    }

    private static Coordinate[] readCoordinates(ByteBuffer byteBuffer, int count) {
        Preconditions.checkArgument((count > 0 ? 1 : 0) != 0, (Object)"Count shall be positive");
        Coordinate[] coordinates = new Coordinate[count];
        for (int i = 0; i < count; ++i) {
            coordinates[i] = GeometrySerializer.readCoordinate(byteBuffer);
        }
        return coordinates;
    }

    private static Geometry readMultiPoint(ByteBuffer byteBuffer, GeometryFactory factory) {
        int pointCount = byteBuffer.getInt();
        Point[] points = new Point[pointCount];
        for (int i = 0; i < pointCount; ++i) {
            points[i] = GeometrySerializer.readPoint(byteBuffer, factory);
        }
        return factory.createMultiPoint(points);
    }

    private static GeometryType readGeometryType(byte typeByte) {
        return GeometryType.fromID(typeByte & 0x7F);
    }

    private static Geometry readPolyline(ByteBuffer byteBuffer, boolean multitype, GeometryFactory factory) {
        int partCount = byteBuffer.getInt();
        if (partCount == 0) {
            if (multitype) {
                return factory.createMultiLineString();
            }
            return factory.createLineString();
        }
        int pointCount = byteBuffer.getInt();
        int[] startIndexes = new int[partCount];
        for (int i = 0; i < partCount; ++i) {
            startIndexes[i] = byteBuffer.getInt();
        }
        int[] partLengths = new int[partCount];
        if (partCount > 1) {
            partLengths[0] = startIndexes[1];
            for (int i = 1; i < partCount - 1; ++i) {
                partLengths[i] = startIndexes[i + 1] - startIndexes[i];
            }
        }
        partLengths[partCount - 1] = pointCount - startIndexes[partCount - 1];
        LineString[] lineStrings = new LineString[partCount];
        for (int i = 0; i < partCount; ++i) {
            lineStrings[i] = factory.createLineString(GeometrySerializer.readCoordinates(byteBuffer, partLengths[i]));
        }
        if (multitype) {
            return factory.createMultiLineString(lineStrings);
        }
        Preconditions.checkArgument((lineStrings.length == 1 ? 1 : 0) != 0, (Object)"The remaining line string must have only one node");
        return lineStrings[0];
    }

    private static Geometry readPolygon(ByteBuffer byteBuffer, boolean multitype, GeometryFactory factory) {
        int partCount = byteBuffer.getInt();
        if (partCount == 0) {
            if (multitype) {
                return factory.createMultiPolygon();
            }
            return factory.createPolygon();
        }
        int pointCount = byteBuffer.getInt();
        int[] startIndexes = new int[partCount];
        for (int i = 0; i < partCount; ++i) {
            startIndexes[i] = byteBuffer.getInt();
        }
        int[] partLengths = new int[partCount];
        if (partCount > 1) {
            partLengths[0] = startIndexes[1];
            for (int i = 1; i < partCount - 1; ++i) {
                partLengths[i] = startIndexes[i + 1] - startIndexes[i];
            }
        }
        partLengths[partCount - 1] = pointCount - startIndexes[partCount - 1];
        LinearRing shell = null;
        ArrayList<LinearRing> holes = new ArrayList<LinearRing>();
        ArrayList<Polygon> polygons = new ArrayList<Polygon>();
        try {
            for (int i = 0; i < partCount; ++i) {
                Coordinate[] coordinates = GeometrySerializer.readCoordinates(byteBuffer, partLengths[i]);
                if (GeometrySerializer.isClockwise(coordinates)) {
                    if (shell != null) {
                        polygons.add(factory.createPolygon(shell, holes.toArray(new LinearRing[0])));
                        holes.clear();
                    }
                    shell = factory.createLinearRing(coordinates);
                    continue;
                }
                holes.add(factory.createLinearRing(coordinates));
            }
            polygons.add(factory.createPolygon(shell, holes.toArray(new LinearRing[0])));
        }
        catch (IllegalArgumentException e) {
            throw new TopologyException("Error constructing Polygon: " + e.getMessage());
        }
        if (multitype) {
            return factory.createMultiPolygon(polygons.toArray(new Polygon[0]));
        }
        return (Geometry)Iterables.getOnlyElement(polygons);
    }

    private static Geometry readGeometryCollection(ByteBuffer byteBuffer, GeometryFactory factory) {
        ArrayList<Geometry> geometries = new ArrayList<Geometry>();
        while (byteBuffer.hasRemaining()) {
            byte typeByte = byteBuffer.get();
            GeometryType type = GeometrySerializer.readGeometryType(typeByte);
            GeometryFactory geometryFactory = GeometrySerializer.getGeometryFactory(typeByte);
            geometries.add(GeometrySerializer.readGeometry(byteBuffer, type, geometryFactory));
        }
        return factory.createGeometryCollection(geometries.toArray(new Geometry[0]));
    }

    private static boolean isClockwise(Coordinate[] coordinates) {
        return GeometrySerializer.isClockwise(coordinates, 0, coordinates.length);
    }

    private static boolean isClockwise(Coordinate[] coordinates, int start, int end) {
        double area = 0.0;
        for (int i = start + 1; i < end; ++i) {
            area += (coordinates[i].x - coordinates[i - 1].x) * (coordinates[i].y + coordinates[i - 1].y);
        }
        return (area += (coordinates[start].x - coordinates[end - 1].x) * (coordinates[start].y + coordinates[end - 1].y)) > 0.0;
    }

    private static GeometryFactory getGeometryFactory(byte typeByte) {
        return typeByte < 0 ? GeometryUtils.GEOGRAPHY_FACTORY : GeometryUtils.GEOMETRY_FACTORY;
    }

    private static void writeGeometryByteBuffer(ByteBuffer byteBuffer, Geometry geometry) {
        switch (geometry.getGeometryType()) {
            case "Point": {
                GeometrySerializer.writePoint(byteBuffer, (Point)geometry);
                break;
            }
            case "MultiPoint": {
                GeometrySerializer.writeMultiPoint(byteBuffer, (MultiPoint)geometry);
                break;
            }
            case "LineString": 
            case "LinearRing": {
                GeometrySerializer.writePolyline(byteBuffer, geometry, false);
                break;
            }
            case "MultiLineString": {
                GeometrySerializer.writePolyline(byteBuffer, geometry, true);
                break;
            }
            case "Polygon": {
                GeometrySerializer.writePolygon(byteBuffer, geometry, false);
                break;
            }
            case "MultiPolygon": {
                GeometrySerializer.writePolygon(byteBuffer, geometry, true);
                break;
            }
            case "GeometryCollection": {
                GeometrySerializer.writeGeometryCollection(byteBuffer, geometry);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unsupported geometry type : " + geometry.getGeometryType());
            }
        }
    }

    private static int getByteSize(Geometry geometry) {
        int size = 1;
        switch (geometry.getGeometryType()) {
            case "Point": {
                size += 16;
                break;
            }
            case "MultiPoint": {
                size += 4 + geometry.getNumPoints() * 16;
                break;
            }
            case "LineString": 
            case "LinearRing": {
                size += GeometrySerializer.getPolylineByteSize(geometry, false);
                break;
            }
            case "MultiLineString": {
                size += GeometrySerializer.getPolylineByteSize(geometry, true);
                break;
            }
            case "Polygon": {
                size += GeometrySerializer.getPolygonByteSize(geometry, false);
                break;
            }
            case "MultiPolygon": {
                size += GeometrySerializer.getPolygonByteSize(geometry, true);
                break;
            }
            case "GeometryCollection": {
                size += GeometrySerializer.getGeometryCollectionByteSize(geometry);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unsupported geometry type : " + geometry.getGeometryType());
            }
        }
        return size;
    }

    private static void writeType(ByteBuffer byteBuffer, GeometryType serializationType, int srid) {
        byte type = Integer.valueOf(serializationType.id()).byteValue();
        if (srid == 4326) {
            type = (byte)(type | 0xFFFFFF80);
        }
        byteBuffer.put(type);
    }

    private static void writePoint(ByteBuffer byteBuffer, Point point) {
        GeometrySerializer.writeType(byteBuffer, GeometryType.POINT, point.getSRID());
        if (point.isEmpty()) {
            byteBuffer.putDouble(Double.NaN);
            byteBuffer.putDouble(Double.NaN);
        } else {
            GeometrySerializer.writeCoordinate(byteBuffer, point.getCoordinate());
        }
    }

    private static void writeCoordinate(ByteBuffer byteBuffer, Coordinate coordinate) {
        byteBuffer.putDouble(coordinate.getX());
        byteBuffer.putDouble(coordinate.getY());
    }

    private static void writeMultiPoint(ByteBuffer byteBuffer, MultiPoint geometry) {
        GeometrySerializer.writeType(byteBuffer, GeometryType.MULTI_POINT, geometry.getSRID());
        byteBuffer.putInt(geometry.getNumPoints());
        for (Coordinate coordinate : geometry.getCoordinates()) {
            GeometrySerializer.writeCoordinate(byteBuffer, coordinate);
        }
    }

    private static int getPolylineByteSize(Geometry geometry, boolean multitype) {
        int numPoints = geometry.getNumPoints();
        int numParts = multitype ? geometry.getNumGeometries() : (numPoints > 0 ? 1 : 0);
        return 8 + numParts * 4 + numPoints * 16;
    }

    private static void writePolyline(ByteBuffer byteBuffer, Geometry geometry, boolean multitype) {
        int numParts;
        int numPoints = geometry.getNumPoints();
        if (multitype) {
            numParts = geometry.getNumGeometries();
            GeometrySerializer.writeType(byteBuffer, GeometryType.MULTI_LINE_STRING, geometry.getSRID());
        } else {
            numParts = numPoints > 0 ? 1 : 0;
            GeometrySerializer.writeType(byteBuffer, GeometryType.LINE_STRING, geometry.getSRID());
        }
        byteBuffer.putInt(numParts);
        byteBuffer.putInt(numPoints);
        int partIndex = 0;
        for (int i = 0; i < numParts; ++i) {
            byteBuffer.putInt(partIndex);
            partIndex += geometry.getGeometryN(i).getNumPoints();
        }
        GeometrySerializer.writeCoordinates(byteBuffer, geometry.getCoordinates());
    }

    private static void writeCoordinates(ByteBuffer byteBuffer, Coordinate[] coordinates) {
        for (Coordinate coordinate : coordinates) {
            GeometrySerializer.writeCoordinate(byteBuffer, coordinate);
        }
    }

    private static int getPolygonByteSize(Geometry geometry, boolean multitype) {
        int numGeometries = geometry.getNumGeometries();
        int numParts = 0;
        int numPoints = geometry.getNumPoints();
        for (int i = 0; i < numGeometries; ++i) {
            Polygon polygon = (Polygon)geometry.getGeometryN(i);
            if (polygon.getNumPoints() <= 0) continue;
            numParts += polygon.getNumInteriorRing() + 1;
        }
        int size = 8;
        if (numParts == 0) {
            return size;
        }
        return size + numParts * 4 + numPoints * 16;
    }

    private static void writePolygon(ByteBuffer byteBuffer, Geometry geometry, boolean multitype) {
        int numGeometries = geometry.getNumGeometries();
        int numParts = 0;
        int numPoints = geometry.getNumPoints();
        for (int i = 0; i < numGeometries; ++i) {
            Polygon polygon = (Polygon)geometry.getGeometryN(i);
            if (polygon.getNumPoints() <= 0) continue;
            numParts += polygon.getNumInteriorRing() + 1;
        }
        if (multitype) {
            GeometrySerializer.writeType(byteBuffer, GeometryType.MULTI_POLYGON, geometry.getSRID());
        } else {
            GeometrySerializer.writeType(byteBuffer, GeometryType.POLYGON, geometry.getSRID());
        }
        byteBuffer.putInt(numParts);
        byteBuffer.putInt(numPoints);
        if (numParts == 0) {
            return;
        }
        int[] partIndexes = new int[numParts];
        boolean[] shellPart = new boolean[numParts];
        int currentPart = 0;
        int currentPoint = 0;
        for (int i = 0; i < numGeometries; ++i) {
            Polygon polygon = (Polygon)geometry.getGeometryN(i);
            partIndexes[currentPart] = currentPoint;
            shellPart[currentPart] = true;
            ++currentPart;
            currentPoint += polygon.getExteriorRing().getNumPoints();
            int holesCount = polygon.getNumInteriorRing();
            for (int holeIndex = 0; holeIndex < holesCount; ++holeIndex) {
                partIndexes[currentPart] = currentPoint;
                shellPart[currentPart] = false;
                ++currentPart;
                currentPoint += polygon.getInteriorRingN(holeIndex).getNumPoints();
            }
        }
        for (int partIndex : partIndexes) {
            byteBuffer.putInt(partIndex);
        }
        Coordinate[] coordinates = geometry.getCoordinates();
        GeometrySerializer.canonicalizePolygonCoordinates(coordinates, partIndexes, shellPart);
        GeometrySerializer.writeCoordinates(byteBuffer, coordinates);
    }

    private static void canonicalizePolygonCoordinates(Coordinate[] coordinates, int[] partIndexes, boolean[] shellPart) {
        for (int part = 0; part < partIndexes.length - 1; ++part) {
            GeometrySerializer.canonicalizePolygonCoordinates(coordinates, partIndexes[part], partIndexes[part + 1], shellPart[part]);
        }
        if (partIndexes.length > 0) {
            GeometrySerializer.canonicalizePolygonCoordinates(coordinates, partIndexes[partIndexes.length - 1], coordinates.length, shellPart[partIndexes.length - 1]);
        }
    }

    private static void canonicalizePolygonCoordinates(Coordinate[] coordinates, int start, int end, boolean isShell) {
        boolean isClockwise = GeometrySerializer.isClockwise(coordinates, start, end);
        if (isShell && !isClockwise || !isShell && isClockwise) {
            GeometrySerializer.reverse(coordinates, start, end);
        }
    }

    private static void reverse(Coordinate[] coordinates, int start, int end) {
        Preconditions.checkArgument((start <= end ? 1 : 0) != 0, (Object)"start must be less or equal than end");
        for (int i = start; i < start + (end - start) / 2; ++i) {
            Coordinate buffer = coordinates[i];
            coordinates[i] = coordinates[start + end - i - 1];
            coordinates[start + end - i - 1] = buffer;
        }
    }

    private static int getGeometryCollectionByteSize(Geometry collection) {
        int size = 0;
        for (int geometryIndex = 0; geometryIndex < collection.getNumGeometries(); ++geometryIndex) {
            Geometry geometry = collection.getGeometryN(geometryIndex);
            size += GeometrySerializer.getByteSize(geometry);
        }
        return size;
    }

    private static void writeGeometryCollection(ByteBuffer byteBuffer, Geometry collection) {
        GeometrySerializer.writeType(byteBuffer, GeometryType.GEOMETRY_COLLECTION, collection.getSRID());
        for (int geometryIndex = 0; geometryIndex < collection.getNumGeometries(); ++geometryIndex) {
            Geometry geometry = collection.getGeometryN(geometryIndex);
            GeometrySerializer.writeGeometryByteBuffer(byteBuffer, geometry);
        }
    }

    private GeometrySerializer() {
    }
}

