/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.raptor.legacy.storage;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import io.airlift.log.Logger;
import io.trino.plugin.raptor.legacy.RaptorErrorCode;
import io.trino.plugin.raptor.legacy.storage.StorageManagerConfig;
import io.trino.plugin.raptor.legacy.storage.StorageService;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.TrinoException;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.nio.file.Files;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.regex.Pattern;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.inject.Inject;

public class FileStorageService
implements StorageService {
    private static final Logger log = Logger.get(FileStorageService.class);
    private static final Pattern HEX_DIRECTORY = Pattern.compile("[0-9a-f]{2}");
    private static final String FILE_EXTENSION = ".orc";
    private final File baseStorageDir;
    private final File baseStagingDir;
    private final File baseQuarantineDir;

    @Inject
    public FileStorageService(StorageManagerConfig config) {
        this(config.getDataDirectory());
    }

    public FileStorageService(File dataDirectory) {
        File baseDataDir = Objects.requireNonNull(dataDirectory, "dataDirectory is null");
        this.baseStorageDir = new File(baseDataDir, "storage");
        this.baseStagingDir = new File(baseDataDir, "staging");
        this.baseQuarantineDir = new File(baseDataDir, "quarantine");
    }

    @Override
    @PostConstruct
    public void start() {
        this.deleteStagingFilesAsync();
        this.createParents(new File(this.baseStagingDir, "."));
        this.createParents(new File(this.baseStorageDir, "."));
        this.createParents(new File(this.baseQuarantineDir, "."));
    }

    @Override
    public long getAvailableBytes() {
        return this.baseStorageDir.getUsableSpace();
    }

    @PreDestroy
    public void stop() throws IOException {
        FileStorageService.deleteDirectory(this.baseStagingDir);
    }

    @Override
    public File getStorageFile(UUID shardUuid) {
        return FileStorageService.getFileSystemPath(this.baseStorageDir, shardUuid);
    }

    @Override
    public File getStagingFile(UUID shardUuid) {
        String name = FileStorageService.getFileSystemPath(new File("/"), shardUuid).getName();
        return new File(this.baseStagingDir, name);
    }

    @Override
    public File getQuarantineFile(UUID shardUuid) {
        String name = FileStorageService.getFileSystemPath(new File("/"), shardUuid).getName();
        return new File(this.baseQuarantineDir, name);
    }

    @Override
    public Set<UUID> getStorageShards() {
        ImmutableSet.Builder shards = ImmutableSet.builder();
        for (File level1 : FileStorageService.listFiles(this.baseStorageDir, FileStorageService::isHexDirectory)) {
            for (File level2 : FileStorageService.listFiles(level1, FileStorageService::isHexDirectory)) {
                for (File file : FileStorageService.listFiles(level2, path -> true)) {
                    if (!file.isFile()) continue;
                    FileStorageService.uuidFromFileName(file.getName()).ifPresent(arg_0 -> ((ImmutableSet.Builder)shards).add(arg_0));
                }
            }
        }
        return shards.build();
    }

    @Override
    public void createParents(File file) {
        File dir = file.getParentFile();
        if (!dir.mkdirs() && !dir.isDirectory()) {
            throw new TrinoException((ErrorCodeSupplier)RaptorErrorCode.RAPTOR_ERROR, "Failed creating directories: " + dir);
        }
    }

    public static File getFileSystemPath(File base, UUID shardUuid) {
        String uuid = shardUuid.toString().toLowerCase(Locale.ENGLISH);
        return base.toPath().resolve(uuid.substring(0, 2)).resolve(uuid.substring(2, 4)).resolve(uuid + FILE_EXTENSION).toFile();
    }

    private void deleteStagingFilesAsync() {
        List<File> files = FileStorageService.listFiles(this.baseStagingDir, null);
        if (!files.isEmpty()) {
            new Thread(() -> {
                for (File file : files) {
                    try {
                        Files.deleteIfExists(file.toPath());
                    }
                    catch (IOException e) {
                        log.warn((Throwable)e, "Failed to delete file: %s", new Object[]{file.getAbsolutePath()});
                    }
                }
            }, "background-staging-delete").start();
        }
    }

    private static void deleteDirectory(File dir) throws IOException {
        if (!dir.exists()) {
            return;
        }
        File[] files = dir.listFiles();
        if (files == null) {
            throw new IOException("Failed to list directory: " + dir);
        }
        for (File file : files) {
            Files.deleteIfExists(file.toPath());
        }
        Files.deleteIfExists(dir.toPath());
    }

    private static List<File> listFiles(File dir, FileFilter filter) {
        Object[] files = dir.listFiles(filter);
        if (files == null) {
            return ImmutableList.of();
        }
        return ImmutableList.copyOf((Object[])files);
    }

    private static boolean isHexDirectory(File file) {
        return file.isDirectory() && HEX_DIRECTORY.matcher(file.getName()).matches();
    }

    private static Optional<UUID> uuidFromFileName(String name) {
        if (name.endsWith(FILE_EXTENSION)) {
            name = name.substring(0, name.length() - FILE_EXTENSION.length());
            return FileStorageService.uuidFromString(name);
        }
        return Optional.empty();
    }

    private static Optional<UUID> uuidFromString(String value) {
        try {
            return Optional.of(UUID.fromString(value));
        }
        catch (IllegalArgumentException e) {
            return Optional.empty();
        }
    }
}

