/*
 * Decompiled with CFR 0.152.
 */
package io.trino.parquet.writer;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import io.airlift.slice.DynamicSliceOutput;
import io.airlift.slice.OutputStreamSliceOutput;
import io.airlift.slice.SizeOf;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceOutput;
import io.airlift.slice.Slices;
import io.trino.memory.context.AggregatedMemoryContext;
import io.trino.parquet.Column;
import io.trino.parquet.ParquetCorruptionException;
import io.trino.parquet.ParquetDataSource;
import io.trino.parquet.ParquetReaderOptions;
import io.trino.parquet.ParquetTypeUtils;
import io.trino.parquet.ParquetWriteValidation;
import io.trino.parquet.metadata.BlockMetadata;
import io.trino.parquet.metadata.FileMetadata;
import io.trino.parquet.metadata.ParquetMetadata;
import io.trino.parquet.metadata.PrunedBlockMetadata;
import io.trino.parquet.reader.MetadataReader;
import io.trino.parquet.reader.ParquetReader;
import io.trino.parquet.reader.RowGroupInfo;
import io.trino.parquet.writer.ColumnChunk;
import io.trino.parquet.writer.ColumnWriter;
import io.trino.parquet.writer.MessageTypeConverter;
import io.trino.parquet.writer.ParquetDataOutput;
import io.trino.parquet.writer.ParquetWriterOptions;
import io.trino.parquet.writer.ParquetWriters;
import io.trino.spi.Page;
import io.trino.spi.connector.SourcePage;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.RealType;
import io.trino.spi.type.Type;
import io.trino.spi.type.UuidType;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import jakarta.annotation.Nullable;
import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.function.Consumer;
import org.apache.parquet.column.ColumnDescriptor;
import org.apache.parquet.column.values.bloomfilter.BloomFilter;
import org.apache.parquet.format.BloomFilterAlgorithm;
import org.apache.parquet.format.BloomFilterCompression;
import org.apache.parquet.format.BloomFilterHash;
import org.apache.parquet.format.BloomFilterHeader;
import org.apache.parquet.format.ColumnMetaData;
import org.apache.parquet.format.CompressionCodec;
import org.apache.parquet.format.FileMetaData;
import org.apache.parquet.format.KeyValue;
import org.apache.parquet.format.RowGroup;
import org.apache.parquet.format.SplitBlockAlgorithm;
import org.apache.parquet.format.Uncompressed;
import org.apache.parquet.format.Util;
import org.apache.parquet.format.XxHash;
import org.apache.parquet.io.GroupColumnIO;
import org.apache.parquet.io.MessageColumnIO;
import org.apache.parquet.schema.MessageType;
import org.joda.time.DateTimeZone;

public class ParquetWriter
implements Closeable {
    private static final int INSTANCE_SIZE = SizeOf.instanceSize(ParquetWriter.class);
    public static final List<Type> SUPPORTED_BLOOM_FILTER_TYPES = ImmutableList.of((Object)BigintType.BIGINT, (Object)DoubleType.DOUBLE, (Object)IntegerType.INTEGER, (Object)RealType.REAL, (Object)UuidType.UUID, (Object)VarbinaryType.VARBINARY, (Object)VarcharType.VARCHAR);
    private final OutputStreamSliceOutput outputStream;
    private final ParquetWriterOptions writerOption;
    private final MessageType messageType;
    private final int chunkMaxBytes;
    private final Map<List<String>, Type> primitiveTypes;
    private final CompressionCodec compressionCodec;
    private final Optional<DateTimeZone> parquetTimeZone;
    private final FileFooter fileFooter;
    private final ImmutableList.Builder<List<Optional<BloomFilter>>> bloomFilterGroups = ImmutableList.builder();
    private final Optional<ParquetWriteValidation.ParquetWriteValidationBuilder> validationBuilder;
    private List<ColumnWriter> columnWriters;
    private int rows;
    private long bufferedBytes;
    private boolean closed;
    private boolean writeHeader;
    @Nullable
    private FileMetaData fileMetaData;
    public static final Slice MAGIC = Slices.wrappedBuffer((byte[])"PAR1".getBytes(StandardCharsets.US_ASCII));

    public ParquetWriter(OutputStream outputStream, MessageType messageType, Map<List<String>, Type> primitiveTypes, ParquetWriterOptions writerOption, CompressionCodec compressionCodec, String trinoVersion, Optional<DateTimeZone> parquetTimeZone, Optional<ParquetWriteValidation.ParquetWriteValidationBuilder> validationBuilder) {
        this.validationBuilder = Objects.requireNonNull(validationBuilder, "validationBuilder is null");
        this.outputStream = new OutputStreamSliceOutput(Objects.requireNonNull(outputStream, "outputStream is null"));
        this.messageType = Objects.requireNonNull(messageType, "messageType is null");
        this.primitiveTypes = Objects.requireNonNull(primitiveTypes, "primitiveTypes is null");
        this.writerOption = Objects.requireNonNull(writerOption, "writerOption is null");
        this.compressionCodec = Objects.requireNonNull(compressionCodec, "compressionCodec is null");
        this.parquetTimeZone = Objects.requireNonNull(parquetTimeZone, "parquetTimeZone is null");
        String createdBy = ParquetWriter.formatCreatedBy(Objects.requireNonNull(trinoVersion, "trinoVersion is null"));
        this.fileFooter = new FileFooter(messageType, createdBy, parquetTimeZone);
        this.recordValidation(validation -> validation.setTimeZone(parquetTimeZone.map(DateTimeZone::getID)));
        this.recordValidation(validation -> validation.setColumns(messageType.getColumns()));
        this.recordValidation(validation -> validation.setCreatedBy(createdBy));
        this.initColumnWriters();
        this.chunkMaxBytes = Math.max(1, writerOption.getMaxRowGroupSize() / 2);
    }

    public long getWrittenBytes() {
        return this.outputStream.longSize();
    }

    public long getBufferedBytes() {
        return this.bufferedBytes;
    }

    public long getRetainedBytes() {
        return (long)INSTANCE_SIZE + this.outputStream.getRetainedSize() + this.columnWriters.stream().mapToLong(ColumnWriter::getRetainedBytes).sum() + this.validationBuilder.map(ParquetWriteValidation.ParquetWriteValidationBuilder::getRetainedSize).orElse(0L);
    }

    public void write(Page page) throws IOException {
        Page chunk;
        Objects.requireNonNull(page, "page is null");
        Preconditions.checkState((!this.closed ? 1 : 0) != 0, (Object)"writer is closed");
        if (page.getPositionCount() == 0) {
            return;
        }
        Preconditions.checkArgument((page.getChannelCount() == this.columnWriters.size() ? 1 : 0) != 0);
        this.recordValidation(validation -> validation.addPage(page));
        for (int writeOffset = 0; writeOffset < page.getPositionCount(); writeOffset += chunk.getPositionCount()) {
            chunk = page.getRegion(writeOffset, Math.min(page.getPositionCount() - writeOffset, this.writerOption.getBatchSize()));
            while (chunk.getPositionCount() > 1 && chunk.getSizeInBytes() > (long)this.chunkMaxBytes) {
                chunk = page.getRegion(writeOffset, chunk.getPositionCount() / 2);
            }
            this.writeChunk(chunk);
        }
    }

    private void writeChunk(Page page) throws IOException {
        this.bufferedBytes = 0L;
        for (int channel = 0; channel < page.getChannelCount(); ++channel) {
            ColumnWriter writer = this.columnWriters.get(channel);
            writer.writeBlock(new ColumnChunk(page.getBlock(channel)));
            this.bufferedBytes += writer.getBufferedBytes();
        }
        this.rows += page.getPositionCount();
        if (this.bufferedBytes >= (long)this.writerOption.getMaxRowGroupSize()) {
            this.columnWriters.forEach(ColumnWriter::close);
            this.flush();
            this.initColumnWriters();
            this.rows = 0;
            this.bufferedBytes = this.columnWriters.stream().mapToLong(ColumnWriter::getBufferedBytes).sum();
        }
    }

    @Override
    public void close() throws IOException {
        if (this.closed) {
            return;
        }
        this.closed = true;
        try (OutputStreamSliceOutput outputStreamSliceOutput = this.outputStream;){
            this.columnWriters.forEach(ColumnWriter::close);
            this.flush();
            this.columnWriters = ImmutableList.of();
            this.fileMetaData = this.fileFooter.createFileMetadata();
            this.writeBloomFilters(this.fileMetaData.getRow_groups(), (List<List<Optional<BloomFilter>>>)this.bloomFilterGroups.build());
            this.writeFooter();
        }
        this.bufferedBytes = 0L;
    }

    public void validate(ParquetDataSource input) throws ParquetCorruptionException {
        Preconditions.checkState((boolean)this.validationBuilder.isPresent(), (Object)"validation is not enabled");
        ParquetWriteValidation writeValidation = this.validationBuilder.get().build();
        try {
            ParquetMetadata parquetMetadata = MetadataReader.readFooter(input, Optional.of(writeValidation));
            try (ParquetReader parquetReader = this.createParquetReader(input, parquetMetadata, writeValidation);){
                SourcePage page = parquetReader.nextPage();
                while (page != null) {
                    page.getPage();
                    page = parquetReader.nextPage();
                }
            }
        }
        catch (IOException e) {
            if (e instanceof ParquetCorruptionException) {
                ParquetCorruptionException pce = (ParquetCorruptionException)e;
                throw pce;
            }
            throw new ParquetCorruptionException(input.getId(), "Validation failed with exception %s", e);
        }
    }

    public FileMetaData getFileMetaData() {
        Preconditions.checkState((boolean)this.closed, (Object)"fileMetaData is available only after writer is closed");
        return Objects.requireNonNull(this.fileMetaData, "fileMetaData is null");
    }

    private ParquetReader createParquetReader(ParquetDataSource input, ParquetMetadata parquetMetadata, ParquetWriteValidation writeValidation) throws IOException {
        FileMetadata fileMetaData = parquetMetadata.getFileMetaData();
        MessageColumnIO messageColumnIO = ParquetTypeUtils.getColumnIO(fileMetaData.getSchema(), fileMetaData.getSchema());
        ImmutableList.Builder columnFields = ImmutableList.builder();
        for (int i = 0; i < writeValidation.getTypes().size(); ++i) {
            columnFields.add((Object)new Column(messageColumnIO.getName(), ParquetTypeUtils.constructField(writeValidation.getTypes().get(i), ParquetTypeUtils.lookupColumnByName((GroupColumnIO)messageColumnIO, writeValidation.getColumnNames().get(i))).orElseThrow()));
        }
        Map<List<String>, ColumnDescriptor> descriptorsByPath = ParquetTypeUtils.getDescriptors(fileMetaData.getSchema(), fileMetaData.getSchema());
        long nextStart = 0L;
        ImmutableList.Builder rowGroupInfoBuilder = ImmutableList.builder();
        for (BlockMetadata block : parquetMetadata.getBlocks()) {
            rowGroupInfoBuilder.add((Object)new RowGroupInfo(PrunedBlockMetadata.createPrunedColumnsMetadata(block, input.getId(), descriptorsByPath), nextStart, Optional.empty()));
            nextStart += block.rowCount();
        }
        return new ParquetReader(Optional.ofNullable(fileMetaData.getCreatedBy()), (List<Column>)columnFields.build(), false, (List<RowGroupInfo>)rowGroupInfoBuilder.build(), input, this.parquetTimeZone.orElseThrow(), AggregatedMemoryContext.newSimpleAggregatedMemoryContext(), ParquetReaderOptions.defaultOptions(), exception -> {
            Throwables.throwIfUnchecked((Throwable)exception);
            return new RuntimeException((Throwable)exception);
        }, Optional.empty(), Optional.of(writeValidation));
    }

    private void recordValidation(Consumer<ParquetWriteValidation.ParquetWriteValidationBuilder> task) {
        this.validationBuilder.ifPresent(task);
    }

    private void flush() throws IOException {
        if (!this.writeHeader) {
            ParquetDataOutput.createDataOutput(MAGIC).writeData((SliceOutput)this.outputStream);
            this.writeHeader = true;
        }
        ImmutableList.Builder builder = ImmutableList.builder();
        for (ColumnWriter columnWriter : this.columnWriters) {
            columnWriter.getBuffer().forEach(arg_0 -> ((ImmutableList.Builder)builder).add(arg_0));
        }
        ImmutableList bufferDataList = builder.build();
        if (this.rows == 0) {
            Verify.verify((boolean)bufferDataList.stream().flatMap(bufferData -> bufferData.getData().stream()).allMatch(dataOutput -> dataOutput.size() == 0), (String)"Buffer should be empty when there are no rows", (Object[])new Object[0]);
            return;
        }
        long currentOffset = this.outputStream.longSize();
        ImmutableList.Builder columnMetaDataBuilder = ImmutableList.builder();
        for (ColumnWriter.BufferData bufferData2 : bufferDataList) {
            ColumnMetaData columnMetaData = bufferData2.getMetaData();
            OptionalInt dictionaryPageSize = bufferData2.getDictionaryPageSize();
            if (dictionaryPageSize.isPresent()) {
                columnMetaData.setDictionary_page_offset(currentOffset);
            }
            columnMetaData.setData_page_offset(currentOffset + (long)dictionaryPageSize.orElse(0));
            columnMetaDataBuilder.add((Object)columnMetaData);
            currentOffset += columnMetaData.getTotal_compressed_size();
        }
        this.updateRowGroups((List<ColumnMetaData>)columnMetaDataBuilder.build(), this.outputStream.longSize());
        for (ColumnWriter.BufferData bufferData2 : bufferDataList) {
            bufferData2.getData().forEach(data -> data.writeData((SliceOutput)this.outputStream));
        }
        this.bloomFilterGroups.add((Object)((List)bufferDataList.stream().map(ColumnWriter.BufferData::getBloomFilter).collect(ImmutableList.toImmutableList())));
    }

    private void writeFooter() throws IOException {
        Preconditions.checkState((boolean)this.closed);
        Slice footer = ParquetWriter.serializeFooter(this.fileMetaData);
        this.recordValidation(validation -> validation.setRowGroups(this.fileMetaData.getRow_groups()));
        ParquetDataOutput.createDataOutput(footer).writeData((SliceOutput)this.outputStream);
        Slice footerSize = Slices.allocate((int)4);
        footerSize.setInt(0, footer.length());
        ParquetDataOutput.createDataOutput(footerSize).writeData((SliceOutput)this.outputStream);
        ParquetDataOutput.createDataOutput(MAGIC).writeData((SliceOutput)this.outputStream);
    }

    private void writeBloomFilters(List<RowGroup> rowGroups, List<List<Optional<BloomFilter>>> rowGroupBloomFilters) {
        Preconditions.checkArgument((rowGroups.size() == rowGroupBloomFilters.size() ? 1 : 0) != 0, (String)"Row groups size %s should match row group Bloom filter size %s", (int)rowGroups.size(), (int)rowGroupBloomFilters.size());
        for (int group = 0; group < rowGroups.size(); ++group) {
            List columns = rowGroups.get(group).getColumns();
            List<Optional<BloomFilter>> bloomFilters = rowGroupBloomFilters.get(group);
            for (int i = 0; i < columns.size(); ++i) {
                if (bloomFilters.get(i).isEmpty()) continue;
                BloomFilter bloomFilter = bloomFilters.get(i).orElseThrow();
                long bloomFilterOffset = this.outputStream.longSize();
                try {
                    Util.writeBloomFilterHeader((BloomFilterHeader)new BloomFilterHeader(bloomFilter.getBitsetSize(), BloomFilterAlgorithm.BLOCK((SplitBlockAlgorithm)new SplitBlockAlgorithm()), BloomFilterHash.XXHASH((XxHash)new XxHash()), BloomFilterCompression.UNCOMPRESSED((Uncompressed)new Uncompressed())), (OutputStream)this.outputStream, null, null);
                    bloomFilter.writeTo((OutputStream)this.outputStream);
                    ((org.apache.parquet.format.ColumnChunk)columns.get(i)).getMeta_data().setBloom_filter_offset(bloomFilterOffset);
                    continue;
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            }
        }
    }

    private void updateRowGroups(List<ColumnMetaData> columnMetaData, long fileOffset) {
        long totalCompressedBytes = columnMetaData.stream().mapToLong(ColumnMetaData::getTotal_compressed_size).sum();
        long totalBytes = columnMetaData.stream().mapToLong(ColumnMetaData::getTotal_uncompressed_size).sum();
        List columnChunks = (List)columnMetaData.stream().map(ParquetWriter::toColumnChunk).collect(ImmutableList.toImmutableList());
        this.fileFooter.addRowGroup(new RowGroup(columnChunks, totalBytes, (long)this.rows).setTotal_compressed_size(totalCompressedBytes).setFile_offset(fileOffset));
    }

    private static Slice serializeFooter(FileMetaData fileMetaData) throws IOException {
        DynamicSliceOutput dynamicSliceOutput = new DynamicSliceOutput(40);
        Util.writeFileMetaData((FileMetaData)fileMetaData, (OutputStream)dynamicSliceOutput);
        return dynamicSliceOutput.slice();
    }

    private static org.apache.parquet.format.ColumnChunk toColumnChunk(ColumnMetaData metaData) {
        org.apache.parquet.format.ColumnChunk columnChunk = new org.apache.parquet.format.ColumnChunk(0L);
        columnChunk.setMeta_data(metaData);
        return columnChunk;
    }

    @VisibleForTesting
    static String formatCreatedBy(String trinoVersion) {
        return "parquet-mr-trino version " + trinoVersion + " (build n/a)";
    }

    private void initColumnWriters() {
        this.columnWriters = ParquetWriters.getColumnWriters(this.messageType, this.primitiveTypes, this.compressionCodec, this.writerOption, this.parquetTimeZone);
    }

    private static class FileFooter {
        private final MessageType messageType;
        private final String createdBy;
        private final Optional<DateTimeZone> parquetTimeZone;
        @Nullable
        private ImmutableList.Builder<RowGroup> rowGroupBuilder = ImmutableList.builder();

        private FileFooter(MessageType messageType, String createdBy, Optional<DateTimeZone> parquetTimeZone) {
            this.messageType = messageType;
            this.createdBy = createdBy;
            this.parquetTimeZone = parquetTimeZone;
        }

        public void addRowGroup(RowGroup rowGroup) {
            Preconditions.checkState((this.rowGroupBuilder != null ? 1 : 0) != 0, (Object)"rowGroupBuilder is null");
            this.rowGroupBuilder.add((Object)rowGroup);
        }

        public FileMetaData createFileMetadata() {
            Preconditions.checkState((this.rowGroupBuilder != null ? 1 : 0) != 0, (Object)"rowGroupBuilder is null");
            ImmutableList rowGroups = this.rowGroupBuilder.build();
            this.rowGroupBuilder = null;
            long totalRows = rowGroups.stream().mapToLong(RowGroup::getNum_rows).sum();
            FileMetaData fileMetaData = new FileMetaData(1, MessageTypeConverter.toParquetSchema(this.messageType), totalRows, (List)ImmutableList.copyOf((Collection)rowGroups));
            fileMetaData.setCreated_by(this.createdBy);
            this.parquetTimeZone.ifPresent(dateTimeZone -> fileMetaData.setKey_value_metadata((List)ImmutableList.of((Object)new KeyValue("writer.time.zone").setValue(dateTimeZone.getID()))));
            return fileMetaData;
        }
    }
}

