/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.resourcegroups.db;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.inject.Inject;
import io.airlift.bootstrap.LifeCycleManager;
import io.airlift.concurrent.Threads;
import io.airlift.log.Logger;
import io.airlift.stats.CounterStat;
import io.airlift.units.Duration;
import io.trino.plugin.resourcegroups.AbstractResourceConfigurationManager;
import io.trino.plugin.resourcegroups.ManagerSpec;
import io.trino.plugin.resourcegroups.ResourceGroupIdTemplate;
import io.trino.plugin.resourcegroups.ResourceGroupSelector;
import io.trino.plugin.resourcegroups.ResourceGroupSpec;
import io.trino.plugin.resourcegroups.SelectorSpec;
import io.trino.plugin.resourcegroups.db.DbResourceGroupConfig;
import io.trino.plugin.resourcegroups.db.DbSourceExactMatchSelector;
import io.trino.plugin.resourcegroups.db.ForEnvironment;
import io.trino.plugin.resourcegroups.db.ResourceGroupGlobalProperties;
import io.trino.plugin.resourcegroups.db.ResourceGroupSpecBuilder;
import io.trino.plugin.resourcegroups.db.ResourceGroupsDao;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.memory.ClusterMemoryPoolManager;
import io.trino.spi.resourcegroups.ResourceGroup;
import io.trino.spi.resourcegroups.ResourceGroupId;
import io.trino.spi.resourcegroups.SelectionContext;
import io.trino.spi.resourcegroups.SelectionCriteria;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import java.util.AbstractMap;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import org.weakref.jmx.Managed;
import org.weakref.jmx.Nested;

public class DbResourceGroupConfigurationManager
extends AbstractResourceConfigurationManager {
    private static final Logger log = Logger.get(DbResourceGroupConfigurationManager.class);
    private final Optional<LifeCycleManager> lifeCycleManager;
    private final ResourceGroupsDao dao;
    private final Map<ResourceGroupId, ResourceGroupSpec> specsUsedToConfigureGroups = new ConcurrentHashMap<ResourceGroupId, ResourceGroupSpec>();
    private final ResourceGroupToTemplateMap configuredGroups = new ResourceGroupToTemplateMap();
    private final AtomicReference<List<ResourceGroupSpec>> rootGroups = new AtomicReference<ImmutableList>(ImmutableList.of());
    private final AtomicReference<List<ResourceGroupSelector>> selectors = new AtomicReference();
    private final AtomicReference<Optional<Duration>> cpuQuotaPeriod = new AtomicReference(Optional.empty());
    private final ScheduledExecutorService configExecutor = Executors.newSingleThreadScheduledExecutor(Threads.daemonThreadsNamed((String)"DbResourceGroupConfigurationManager"));
    private final AtomicBoolean started = new AtomicBoolean();
    private final AtomicLong lastRefresh = new AtomicLong();
    private final String environment;
    private final Duration maxRefreshInterval;
    private final Duration refreshInterval;
    private final boolean exactMatchSelectorEnabled;
    private final CounterStat refreshFailures = new CounterStat();

    @Inject
    public DbResourceGroupConfigurationManager(LifeCycleManager lifeCycleManager, ClusterMemoryPoolManager memoryPoolManager, DbResourceGroupConfig config, ResourceGroupsDao dao, @ForEnvironment String environment) {
        this(Optional.of(lifeCycleManager), memoryPoolManager, config, dao, environment);
    }

    @VisibleForTesting
    DbResourceGroupConfigurationManager(ClusterMemoryPoolManager memoryPoolManager, DbResourceGroupConfig config, ResourceGroupsDao dao, String environment) {
        this(Optional.empty(), memoryPoolManager, config, dao, environment);
    }

    private DbResourceGroupConfigurationManager(Optional<LifeCycleManager> lifeCycleManager, ClusterMemoryPoolManager memoryPoolManager, DbResourceGroupConfig config, ResourceGroupsDao dao, String environment) {
        super(memoryPoolManager);
        this.lifeCycleManager = Objects.requireNonNull(lifeCycleManager, "lifeCycleManager is null");
        Objects.requireNonNull(dao, "daoProvider is null");
        this.environment = Objects.requireNonNull(environment, "environment is null");
        this.maxRefreshInterval = config.getMaxRefreshInterval();
        this.refreshInterval = config.getRefreshInterval();
        this.exactMatchSelectorEnabled = config.getExactMatchSelectorEnabled();
        this.dao = dao;
        this.load();
    }

    @Override
    protected Optional<Duration> getCpuQuotaPeriod() {
        return this.cpuQuotaPeriod.get();
    }

    @Override
    protected List<ResourceGroupSpec> getRootGroups() {
        if (this.lastRefresh.get() == 0L) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.CONFIGURATION_UNAVAILABLE, "Root groups cannot be fetched from database");
        }
        if (this.selectors.get().isEmpty()) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.CONFIGURATION_INVALID, "No root groups are configured");
        }
        return this.rootGroups.get();
    }

    @Override
    protected void configureGroup(ResourceGroup group, ResourceGroupSpec groupSpec) {
        super.configureGroup(group, groupSpec);
        this.specsUsedToConfigureGroups.put(group.getId(), groupSpec);
    }

    @PreDestroy
    public void destroy() {
        this.configExecutor.shutdownNow();
    }

    @PostConstruct
    public void start() {
        if (this.started.compareAndSet(false, true)) {
            this.configExecutor.scheduleWithFixedDelay(this::load, 1000L, this.refreshInterval.toMillis(), TimeUnit.MILLISECONDS);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void configure(ResourceGroup group, SelectionContext<ResourceGroupIdTemplate> criteria) {
        ResourceGroupSpec groupSpec = this.getMatchingSpec(group, criteria);
        this.configuredGroups.put((ResourceGroupIdTemplate)criteria.getContext(), group);
        ResourceGroup resourceGroup = this.getRootGroup(group.getId());
        synchronized (resourceGroup) {
            this.configureGroup(group, groupSpec);
        }
    }

    public Optional<SelectionContext<ResourceGroupIdTemplate>> match(SelectionCriteria criteria) {
        if (this.lastRefresh.get() == 0L) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.CONFIGURATION_UNAVAILABLE, "Selectors cannot be fetched from database");
        }
        if (this.selectors.get().isEmpty()) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.CONFIGURATION_INVALID, "No selectors are configured");
        }
        return this.selectors.get().stream().map(s -> s.match(criteria)).filter(Optional::isPresent).map(Optional::get).findFirst();
    }

    @VisibleForTesting
    public List<ResourceGroupSelector> getSelectors() {
        if (this.lastRefresh.get() == 0L) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.CONFIGURATION_UNAVAILABLE, "Selectors cannot be fetched from database");
        }
        if (this.selectors.get().isEmpty()) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.CONFIGURATION_INVALID, "No selectors are configured");
        }
        return this.selectors.get();
    }

    private synchronized Optional<Duration> getCpuQuotaPeriodFromDb() {
        List<ResourceGroupGlobalProperties> globalProperties = this.dao.getResourceGroupGlobalProperties();
        Preconditions.checkState((globalProperties.size() <= 1 ? 1 : 0) != 0, (Object)"There is more than one cpu_quota_period");
        return !globalProperties.isEmpty() ? globalProperties.get(0).getCpuQuotaPeriod() : Optional.empty();
    }

    @VisibleForTesting
    public synchronized void load() {
        try {
            Map.Entry<ManagerSpec, Map<ResourceGroupIdTemplate, ResourceGroupSpec>> specsFromDb = this.buildSpecsFromDb();
            ManagerSpec managerSpec = specsFromDb.getKey();
            Map<ResourceGroupIdTemplate, ResourceGroupSpec> newResourceGroupSpecs = specsFromDb.getValue();
            Map<ResourceGroupIdTemplate, Set<ResourceGroup>> templateToGroup = this.configuredGroups.getAllTemplateToGroupsMappings();
            Map<ResourceGroup, ResourceGroupSpec> changedGroups = this.findChangedGroups(templateToGroup, newResourceGroupSpecs);
            Set<ResourceGroup> deletedGroups = this.findDeletedGroups(templateToGroup, newResourceGroupSpecs);
            this.cpuQuotaPeriod.set(managerSpec.getCpuQuotaPeriod());
            this.rootGroups.set(managerSpec.getRootGroups());
            List<ResourceGroupSelector> selectors = this.buildSelectors(managerSpec);
            if (this.exactMatchSelectorEnabled) {
                ImmutableList.Builder builder = ImmutableList.builder();
                builder.add((Object)new DbSourceExactMatchSelector(this.environment, this.dao));
                builder.addAll(selectors);
                this.selectors.set((List<ResourceGroupSelector>)builder.build());
            } else {
                this.selectors.set(selectors);
            }
            this.configureChangedGroups(changedGroups);
            this.disableDeletedGroups(deletedGroups);
            if (this.lastRefresh.get() > 0L) {
                for (ResourceGroup resourceGroup : deletedGroups) {
                    log.info("Resource group deleted '%s'", new Object[]{resourceGroup.getId()});
                }
                for (Map.Entry entry : changedGroups.entrySet()) {
                    log.info("Resource group '%s' changed to %s", new Object[]{((ResourceGroup)entry.getKey()).getId(), entry.getValue()});
                }
            } else {
                log.info("Loaded %s selectors and %s resource groups from database", new Object[]{this.selectors.get().size(), this.specsUsedToConfigureGroups.size()});
            }
            this.lastRefresh.set(System.nanoTime());
        }
        catch (Throwable e) {
            if (Duration.succinctNanos((long)(System.nanoTime() - this.lastRefresh.get())).compareTo(this.maxRefreshInterval) > 0) {
                this.lastRefresh.set(0L);
            }
            this.refreshFailures.update(1L);
            log.error(e, "Error loading configuration from db");
        }
    }

    private Map<ResourceGroup, ResourceGroupSpec> findChangedGroups(Map<ResourceGroupIdTemplate, Set<ResourceGroup>> templateToGroups, Map<ResourceGroupIdTemplate, ResourceGroupSpec> newResourceGroupSpecs) {
        ImmutableMap.Builder changedGroups = ImmutableMap.builder();
        for (Map.Entry<ResourceGroupIdTemplate, Set<ResourceGroup>> entry : templateToGroups.entrySet()) {
            ResourceGroupSpec newSpec = newResourceGroupSpecs.get(entry.getKey());
            if (newSpec == null) continue;
            Set changedGroupsForCurrentTemplate = (Set)entry.getValue().stream().filter(resourceGroupId -> {
                ResourceGroupSpec previousSpec = this.specsUsedToConfigureGroups.get(resourceGroupId.getId());
                return previousSpec == null || !previousSpec.sameConfig(newSpec);
            }).collect(ImmutableSet.toImmutableSet());
            for (ResourceGroup group : changedGroupsForCurrentTemplate) {
                changedGroups.put((Object)group, (Object)newSpec);
            }
        }
        return changedGroups.buildOrThrow();
    }

    private Set<ResourceGroup> findDeletedGroups(Map<ResourceGroupIdTemplate, Set<ResourceGroup>> templateToGroups, Map<ResourceGroupIdTemplate, ResourceGroupSpec> newResourceGroupSpecs) {
        return (Set)templateToGroups.entrySet().stream().filter(entry -> !newResourceGroupSpecs.containsKey(entry.getKey())).flatMap(entry -> ((Set)entry.getValue()).stream()).filter(resourceGroup -> !resourceGroup.isDisabled()).collect(ImmutableSet.toImmutableSet());
    }

    private synchronized void populateFromDbHelper(Map<Long, ResourceGroupSpecBuilder> recordMap, Set<Long> rootGroupIds, Map<Long, ResourceGroupIdTemplate> resourceGroupIdTemplateMap, Map<Long, Set<Long>> subGroupIdsToBuild) {
        List<ResourceGroupSpecBuilder> records = this.dao.getResourceGroups(this.environment);
        for (ResourceGroupSpecBuilder record : records) {
            recordMap.put(record.getId(), record);
            if (record.getParentId().isEmpty()) {
                rootGroupIds.add(record.getId());
                resourceGroupIdTemplateMap.put(record.getId(), new ResourceGroupIdTemplate(record.getNameTemplate().toString()));
                continue;
            }
            subGroupIdsToBuild.computeIfAbsent(record.getParentId().get(), k -> new HashSet()).add(record.getId());
        }
    }

    private synchronized Map.Entry<ManagerSpec, Map<ResourceGroupIdTemplate, ResourceGroupSpec>> buildSpecsFromDb() {
        HashMap<ResourceGroupIdTemplate, ResourceGroupSpec> resourceGroupSpecs = new HashMap<ResourceGroupIdTemplate, ResourceGroupSpec>();
        HashSet<Long> rootGroupIds = new HashSet<Long>();
        HashMap<Long, ResourceGroupSpec> resourceGroupSpecMap = new HashMap<Long, ResourceGroupSpec>();
        HashMap<Long, ResourceGroupIdTemplate> resourceGroupIdTemplateMap = new HashMap<Long, ResourceGroupIdTemplate>();
        HashMap<Long, ResourceGroupSpecBuilder> recordMap = new HashMap<Long, ResourceGroupSpecBuilder>();
        HashMap<Long, Set<Long>> subGroupIdsToBuild = new HashMap<Long, Set<Long>>();
        this.populateFromDbHelper(recordMap, rootGroupIds, resourceGroupIdTemplateMap, subGroupIdsToBuild);
        LinkedList<Long> queue = new LinkedList<Long>(rootGroupIds);
        while (!queue.isEmpty()) {
            Long id = queue.pollFirst();
            resourceGroupIdTemplateMap.computeIfAbsent(id, k -> {
                ResourceGroupSpecBuilder builder = (ResourceGroupSpecBuilder)recordMap.get(id);
                return ResourceGroupIdTemplate.forSubGroupNamed((ResourceGroupIdTemplate)resourceGroupIdTemplateMap.get(builder.getParentId().get()), builder.getNameTemplate().toString());
            });
            Set<Long> childrenToBuild = subGroupIdsToBuild.getOrDefault(id, (Set<Long>)ImmutableSet.of());
            if (childrenToBuild.isEmpty()) {
                ResourceGroupSpecBuilder builder = (ResourceGroupSpecBuilder)recordMap.get(id);
                ResourceGroupSpec resourceGroupSpec = builder.build();
                resourceGroupSpecMap.put(id, resourceGroupSpec);
                resourceGroupSpecs.put((ResourceGroupIdTemplate)resourceGroupIdTemplateMap.get(id), resourceGroupSpec);
                builder.getParentId().ifPresent(parentId -> {
                    ((ResourceGroupSpecBuilder)recordMap.get(parentId)).addSubGroup(resourceGroupSpec);
                    ((Set)subGroupIdsToBuild.get(parentId)).remove(id);
                });
                continue;
            }
            queue.addFirst(id);
            queue.addAll(0, childrenToBuild);
        }
        List<ResourceGroupSpec> rootGroups = rootGroupIds.stream().map(resourceGroupSpecMap::get).collect(Collectors.toList());
        List<SelectorSpec> selectors = this.dao.getSelectors(this.environment).stream().map(selectorRecord -> new SelectorSpec(selectorRecord.getUserRegex(), selectorRecord.getUserGroupRegex(), selectorRecord.getSourceRegex(), selectorRecord.getQueryType(), selectorRecord.getClientTags(), selectorRecord.getSelectorResourceEstimate(), (ResourceGroupIdTemplate)resourceGroupIdTemplateMap.get(selectorRecord.getResourceGroupId()))).collect(Collectors.toList());
        ManagerSpec managerSpec = new ManagerSpec(rootGroups, selectors, this.getCpuQuotaPeriodFromDb());
        this.validateRootGroups(managerSpec);
        return new AbstractMap.SimpleImmutableEntry<ManagerSpec, Map<ResourceGroupIdTemplate, ResourceGroupSpec>>(managerSpec, resourceGroupSpecs);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void configureChangedGroups(Map<ResourceGroup, ResourceGroupSpec> changedGroups) {
        for (Map.Entry<ResourceGroup, ResourceGroupSpec> entry : changedGroups.entrySet()) {
            ResourceGroup group = entry.getKey();
            ResourceGroupSpec groupSpec = entry.getValue();
            ResourceGroup resourceGroup = this.getRootGroup(group.getId());
            synchronized (resourceGroup) {
                this.configureGroup(group, groupSpec);
            }
        }
    }

    private synchronized void disableDeletedGroups(Set<ResourceGroup> deletedGroups) {
        for (ResourceGroup group : deletedGroups) {
            group.setDisabled(true);
        }
    }

    private ResourceGroup getRootGroup(ResourceGroupId groupId) {
        Optional parent = groupId.getParent();
        while (parent.isPresent()) {
            groupId = (ResourceGroupId)parent.get();
            parent = groupId.getParent();
        }
        return this.configuredGroups.get(groupId);
    }

    @Managed
    @Nested
    public CounterStat getRefreshFailures() {
        return this.refreshFailures;
    }

    public void shutdown() {
        this.lifeCycleManager.ifPresent(LifeCycleManager::stop);
    }

    private static class ResourceGroupToTemplateMap {
        private final Map<ResourceGroupId, ResourceGroup> groups = new ConcurrentHashMap<ResourceGroupId, ResourceGroup>();
        private final Map<ResourceGroupId, ResourceGroupIdTemplate> groupIdToTemplate = new HashMap<ResourceGroupId, ResourceGroupIdTemplate>();
        private final Map<ResourceGroupIdTemplate, Set<ResourceGroup>> templateToGroups = new HashMap<ResourceGroupIdTemplate, Set<ResourceGroup>>();

        private ResourceGroupToTemplateMap() {
        }

        synchronized void put(ResourceGroupIdTemplate newTemplate, ResourceGroup group) {
            ResourceGroup previousGroup = this.groups.putIfAbsent(group.getId(), group);
            Preconditions.checkState((previousGroup == null || previousGroup == group ? 1 : 0) != 0, (Object)"Unexpected resource group instance");
            ResourceGroupIdTemplate previousTemplate = this.groupIdToTemplate.put(group.getId(), newTemplate);
            if (previousTemplate != null) {
                this.templateToGroups.get(previousTemplate).remove(group);
            }
            this.templateToGroups.computeIfAbsent(newTemplate, resourceGroupIdTemplate -> new HashSet()).add(group);
        }

        ResourceGroup get(ResourceGroupId groupId) {
            return this.groups.get(groupId);
        }

        synchronized Map<ResourceGroupIdTemplate, Set<ResourceGroup>> getAllTemplateToGroupsMappings() {
            return (Map)this.templateToGroups.entrySet().stream().collect(ImmutableMap.toImmutableMap(Map.Entry::getKey, entry -> ImmutableSet.copyOf((Collection)((Collection)entry.getValue()))));
        }
    }
}

