/*
 * Decompiled with CFR 0.152.
 */
package io.trino.filesystem.memory;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.cache.Cache;
import com.google.common.collect.ImmutableList;
import com.google.inject.Inject;
import io.airlift.slice.SizeOf;
import io.airlift.slice.Slice;
import io.airlift.units.DataSize;
import io.airlift.units.Duration;
import io.trino.cache.EvictableCacheBuilder;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoInput;
import io.trino.filesystem.TrinoInputFile;
import io.trino.filesystem.TrinoInputStream;
import io.trino.filesystem.cache.TrinoFileSystemCache;
import io.trino.filesystem.memory.MemoryFileSystemCacheConfig;
import io.trino.filesystem.memory.MemoryInput;
import io.trino.filesystem.memory.MemoryInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.NoSuchFileException;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.weakref.jmx.Managed;

public final class MemoryFileSystemCache
implements TrinoFileSystemCache {
    private final Cache<String, Optional<Slice>> cache;
    private final int maxContentLengthBytes;
    private final AtomicLong largeFileSkippedCount = new AtomicLong();

    @Inject
    public MemoryFileSystemCache(MemoryFileSystemCacheConfig config) {
        this(config.getCacheTtl(), config.getMaxSize(), config.getMaxContentLength());
    }

    private MemoryFileSystemCache(Duration expireAfterWrite, DataSize maxSize, DataSize maxContentLength) {
        Preconditions.checkArgument((maxContentLength.compareTo(DataSize.of((long)1L, (DataSize.Unit)DataSize.Unit.GIGABYTE)) <= 0 ? 1 : 0) != 0, (Object)"maxContentLength must be less than or equal to 1GB");
        this.cache = EvictableCacheBuilder.newBuilder().maximumWeight(maxSize.toBytes()).weigher((key, value) -> Math.toIntExact(SizeOf.estimatedSizeOf((String)key) + SizeOf.sizeOf((Optional)value, Slice::getRetainedSize))).expireAfterWrite(expireAfterWrite.toMillis(), TimeUnit.MILLISECONDS).shareNothingWhenDisabled().recordStats().build();
        this.maxContentLengthBytes = Math.toIntExact(maxContentLength.toBytes());
    }

    @Override
    public TrinoInput cacheInput(TrinoInputFile delegate, String key) throws IOException {
        Optional<Slice> cachedEntry = this.getOrLoadFromCache(key, delegate);
        if (cachedEntry.isEmpty()) {
            this.largeFileSkippedCount.incrementAndGet();
            return delegate.newInput();
        }
        return new MemoryInput(delegate.location(), cachedEntry.get());
    }

    @Override
    public TrinoInputStream cacheStream(TrinoInputFile delegate, String key) throws IOException {
        Optional<Slice> cachedEntry = this.getOrLoadFromCache(key, delegate);
        if (cachedEntry.isEmpty()) {
            this.largeFileSkippedCount.incrementAndGet();
            return delegate.newStream();
        }
        return new MemoryInputStream(delegate.location(), cachedEntry.get());
    }

    @Override
    public long cacheLength(TrinoInputFile delegate, String key) throws IOException {
        Optional<Slice> cachedEntry = this.getOrLoadFromCache(key, delegate);
        if (cachedEntry.isEmpty()) {
            this.largeFileSkippedCount.incrementAndGet();
            return delegate.length();
        }
        return cachedEntry.get().length();
    }

    @Override
    public void expire(Location location) throws IOException {
        List expired = (List)this.cache.asMap().keySet().stream().filter(key -> key.startsWith(location.path())).collect(ImmutableList.toImmutableList());
        this.cache.invalidateAll((Iterable)expired);
    }

    @Override
    public void expire(Collection<Location> locations) throws IOException {
        List expired = (List)this.cache.asMap().keySet().stream().filter(key -> locations.stream().map(Location::path).anyMatch(key::startsWith)).collect(ImmutableList.toImmutableList());
        this.cache.invalidateAll((Iterable)expired);
    }

    @Managed
    public void flushCache() {
        this.cache.invalidateAll();
    }

    @Managed
    public long getHitCount() {
        return this.cache.stats().hitCount();
    }

    @Managed
    public long getRequestCount() {
        return this.cache.stats().requestCount();
    }

    @Managed
    public long getLargeFileSkippedCount() {
        return this.largeFileSkippedCount.get();
    }

    @VisibleForTesting
    boolean isCached(String key) {
        Optional cachedEntry = (Optional)this.cache.getIfPresent((Object)key);
        return cachedEntry != null && cachedEntry.isPresent();
    }

    private Optional<Slice> getOrLoadFromCache(String key, TrinoInputFile delegate) throws IOException {
        try {
            return (Optional)this.cache.get((Object)key, () -> this.load(delegate));
        }
        catch (ExecutionException e) {
            throw MemoryFileSystemCache.handleException(delegate.location(), e.getCause());
        }
    }

    private Optional<Slice> load(TrinoInputFile delegate) throws IOException {
        long fileSize = delegate.length();
        if (fileSize > (long)this.maxContentLengthBytes) {
            return Optional.empty();
        }
        try (TrinoInput trinoInput = delegate.newInput();){
            Optional<Slice> optional = Optional.of(trinoInput.readTail(Math.toIntExact(fileSize)));
            return optional;
        }
    }

    private static IOException handleException(Location location, Throwable cause) throws IOException {
        if (cause instanceof FileNotFoundException || cause instanceof NoSuchFileException) {
            throw MemoryFileSystemCache.withCause(new FileNotFoundException(location.toString()), cause);
        }
        if (cause instanceof FileAlreadyExistsException) {
            throw MemoryFileSystemCache.withCause(new FileAlreadyExistsException(location.toString()), cause);
        }
        throw new IOException(cause.getMessage() + ": " + String.valueOf(location), cause);
    }

    private static <T extends Throwable> T withCause(T throwable, Throwable cause) {
        throwable.initCause(cause);
        return throwable;
    }
}

