/*
 * Decompiled with CFR 0.152.
 */
package io.trino.filesystem.azure;

import com.azure.core.http.HttpClient;
import com.azure.core.http.rest.PagedIterable;
import com.azure.core.util.ClientOptions;
import com.azure.core.util.TracingOptions;
import com.azure.storage.blob.BlobClient;
import com.azure.storage.blob.BlobContainerClient;
import com.azure.storage.blob.BlobContainerClientBuilder;
import com.azure.storage.blob.models.BlobItem;
import com.azure.storage.blob.models.ListBlobsOptions;
import com.azure.storage.blob.models.UserDelegationKey;
import com.azure.storage.blob.sas.BlobSasPermission;
import com.azure.storage.blob.sas.BlobServiceSasSignatureValues;
import com.azure.storage.blob.specialized.BlockBlobClient;
import com.azure.storage.common.sas.SasProtocol;
import com.azure.storage.file.datalake.DataLakeDirectoryClient;
import com.azure.storage.file.datalake.DataLakeFileClient;
import com.azure.storage.file.datalake.DataLakeFileSystemClient;
import com.azure.storage.file.datalake.DataLakeServiceClient;
import com.azure.storage.file.datalake.DataLakeServiceClientBuilder;
import com.azure.storage.file.datalake.models.DataLakeRequestConditions;
import com.azure.storage.file.datalake.models.DataLakeStorageException;
import com.azure.storage.file.datalake.models.ListPathsOptions;
import com.azure.storage.file.datalake.models.PathItem;
import com.azure.storage.file.datalake.options.DataLakePathDeleteOptions;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import io.airlift.units.DataSize;
import io.airlift.units.Duration;
import io.trino.filesystem.FileIterator;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoFileSystemException;
import io.trino.filesystem.TrinoInputFile;
import io.trino.filesystem.TrinoOutputFile;
import io.trino.filesystem.UriLocation;
import io.trino.filesystem.azure.AzureAuth;
import io.trino.filesystem.azure.AzureAuthAccessKey;
import io.trino.filesystem.azure.AzureAuthOauth;
import io.trino.filesystem.azure.AzureBlobFileIterator;
import io.trino.filesystem.azure.AzureDataLakeFileIterator;
import io.trino.filesystem.azure.AzureInputFile;
import io.trino.filesystem.azure.AzureLocation;
import io.trino.filesystem.azure.AzureOutputFile;
import io.trino.filesystem.azure.AzureUtils;
import io.trino.filesystem.encryption.EncryptionKey;
import java.io.IOException;
import java.lang.runtime.SwitchBootstraps;
import java.net.URI;
import java.time.Instant;
import java.time.OffsetDateTime;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.UUID;
import java.util.function.Predicate;

public class AzureFileSystem
implements TrinoFileSystem {
    private final HttpClient httpClient;
    private final TracingOptions tracingOptions;
    private final AzureAuth azureAuth;
    private final String endpoint;
    private final int readBlockSizeBytes;
    private final long writeBlockSizeBytes;
    private final int maxWriteConcurrency;
    private final long maxSingleUploadSizeBytes;

    public AzureFileSystem(HttpClient httpClient, TracingOptions tracingOptions, AzureAuth azureAuth, String endpoint, DataSize readBlockSize, DataSize writeBlockSize, int maxWriteConcurrency, DataSize maxSingleUploadSize) {
        this.httpClient = Objects.requireNonNull(httpClient, "httpClient is null");
        this.tracingOptions = Objects.requireNonNull(tracingOptions, "tracingOptions is null");
        this.azureAuth = Objects.requireNonNull(azureAuth, "azureAuth is null");
        this.endpoint = Objects.requireNonNull(endpoint, "endpoint is null");
        this.readBlockSizeBytes = Math.toIntExact(readBlockSize.toBytes());
        this.writeBlockSizeBytes = writeBlockSize.toBytes();
        Preconditions.checkArgument((maxWriteConcurrency >= 0 ? 1 : 0) != 0, (Object)"maxWriteConcurrency is negative");
        this.maxWriteConcurrency = maxWriteConcurrency;
        this.maxSingleUploadSizeBytes = maxSingleUploadSize.toBytes();
    }

    public TrinoInputFile newInputFile(Location location) {
        AzureLocation azureLocation = new AzureLocation(location);
        BlobClient client = this.createBlobClient(azureLocation, Optional.empty());
        return new AzureInputFile(azureLocation, OptionalLong.empty(), Optional.empty(), client, this.readBlockSizeBytes);
    }

    public TrinoInputFile newEncryptedInputFile(Location location, EncryptionKey key) {
        AzureLocation azureLocation = new AzureLocation(location);
        BlobClient client = this.createBlobClient(azureLocation, Optional.of(key));
        return new AzureInputFile(azureLocation, OptionalLong.empty(), Optional.empty(), client, this.readBlockSizeBytes);
    }

    public TrinoInputFile newInputFile(Location location, long length) {
        AzureLocation azureLocation = new AzureLocation(location);
        BlobClient client = this.createBlobClient(azureLocation, Optional.empty());
        return new AzureInputFile(azureLocation, OptionalLong.of(length), Optional.empty(), client, this.readBlockSizeBytes);
    }

    public TrinoInputFile newEncryptedInputFile(Location location, long length, EncryptionKey key) {
        AzureLocation azureLocation = new AzureLocation(location);
        BlobClient client = this.createBlobClient(azureLocation, Optional.of(key));
        return new AzureInputFile(azureLocation, OptionalLong.of(length), Optional.empty(), client, this.readBlockSizeBytes);
    }

    public TrinoInputFile newInputFile(Location location, long length, Instant lastModified) {
        AzureLocation azureLocation = new AzureLocation(location);
        BlobClient client = this.createBlobClient(azureLocation, Optional.empty());
        return new AzureInputFile(azureLocation, OptionalLong.of(length), Optional.of(lastModified), client, this.readBlockSizeBytes);
    }

    public TrinoInputFile newEncryptedInputFile(Location location, long length, Instant lastModified, EncryptionKey key) {
        AzureLocation azureLocation = new AzureLocation(location);
        BlobClient client = this.createBlobClient(azureLocation, Optional.of(key));
        return new AzureInputFile(azureLocation, OptionalLong.of(length), Optional.of(lastModified), client, this.readBlockSizeBytes);
    }

    public TrinoOutputFile newOutputFile(Location location) {
        AzureLocation azureLocation = new AzureLocation(location);
        BlobClient client = this.createBlobClient(azureLocation, Optional.empty());
        return new AzureOutputFile(azureLocation, client, this.writeBlockSizeBytes, this.maxWriteConcurrency, this.maxSingleUploadSizeBytes);
    }

    public TrinoOutputFile newEncryptedOutputFile(Location location, EncryptionKey key) {
        AzureLocation azureLocation = new AzureLocation(location);
        BlobClient client = this.createBlobClient(azureLocation, Optional.of(key));
        return new AzureOutputFile(azureLocation, client, this.writeBlockSizeBytes, this.maxWriteConcurrency, this.maxSingleUploadSizeBytes);
    }

    public void deleteFile(Location location) throws IOException {
        location.verifyValidFileLocation();
        AzureLocation azureLocation = new AzureLocation(location);
        BlobClient client = this.createBlobClient(azureLocation, Optional.empty());
        try {
            client.delete();
        }
        catch (RuntimeException e) {
            if (AzureUtils.isFileNotFoundException(e)) {
                return;
            }
            throw AzureUtils.handleAzureException(e, "deleting file", azureLocation);
        }
    }

    public void deleteDirectory(Location location) throws IOException {
        AzureLocation azureLocation = new AzureLocation(location);
        try {
            if (this.isHierarchicalNamespaceEnabled(azureLocation)) {
                this.deleteGen2Directory(azureLocation);
            } else {
                this.deleteBlobDirectory(azureLocation);
            }
        }
        catch (RuntimeException e) {
            throw AzureUtils.handleAzureException(e, "deleting directory", azureLocation);
        }
    }

    private void deleteGen2Directory(AzureLocation location) throws IOException {
        DataLakeFileSystemClient fileSystemClient = this.createFileSystemClient(location, Optional.empty());
        DataLakePathDeleteOptions deleteRecursiveOptions = new DataLakePathDeleteOptions().setIsRecursive(true);
        if (location.path().isEmpty()) {
            for (PathItem pathItem : fileSystemClient.listPaths()) {
                if (pathItem.isDirectory()) {
                    fileSystemClient.deleteDirectoryIfExistsWithResponse(pathItem.getName(), deleteRecursiveOptions, null, null);
                    continue;
                }
                fileSystemClient.deleteFileIfExists(pathItem.getName());
            }
        } else {
            DataLakeDirectoryClient directoryClient = AzureFileSystem.createDirectoryClient(fileSystemClient, location.path());
            if (directoryClient.exists().booleanValue()) {
                if (!directoryClient.getProperties().isDirectory().booleanValue()) {
                    throw new TrinoFileSystemException("Location is not a directory: " + String.valueOf(location));
                }
                directoryClient.deleteIfExistsWithResponse(deleteRecursiveOptions, null, null);
            }
        }
    }

    private void deleteBlobDirectory(AzureLocation location) {
        BlobContainerClient blobContainerClient = this.createBlobContainerClient(location, Optional.empty());
        PagedIterable blobItems = blobContainerClient.listBlobs(new ListBlobsOptions().setPrefix(location.directoryPath()), null);
        for (BlobItem item : blobItems) {
            blobContainerClient.getBlobClient(item.getName()).deleteIfExists();
        }
    }

    public void renameFile(Location source, Location target) throws IOException {
        source.verifyValidFileLocation();
        target.verifyValidFileLocation();
        AzureLocation sourceLocation = new AzureLocation(source);
        AzureLocation targetLocation = new AzureLocation(target);
        if (!sourceLocation.account().equals(targetLocation.account())) {
            throw new TrinoFileSystemException("Cannot rename across storage accounts");
        }
        if (!Objects.equals(sourceLocation.container(), targetLocation.container())) {
            throw new TrinoFileSystemException("Cannot rename across storage account containers");
        }
        this.renameGen2File(sourceLocation, targetLocation);
    }

    private void renameGen2File(AzureLocation source, AzureLocation target) throws IOException {
        try {
            DataLakeFileSystemClient fileSystemClient = this.createFileSystemClient(source, Optional.empty());
            DataLakeFileClient dataLakeFileClient = AzureFileSystem.createFileClient(fileSystemClient, source.path());
            if (dataLakeFileClient.getProperties().isDirectory().booleanValue()) {
                throw new TrinoFileSystemException("Rename file from %s to %s, source is a directory".formatted(source, target));
            }
            AzureFileSystem.createDirectoryIfNotExists(fileSystemClient, target.location().parentDirectory().path());
            dataLakeFileClient.renameWithResponse(null, target.path(), null, new DataLakeRequestConditions().setIfNoneMatch("*"), null, null);
        }
        catch (RuntimeException e) {
            throw new IOException("Rename file from %s to %s failed".formatted(source, target), e);
        }
    }

    public FileIterator listFiles(Location location) throws IOException {
        AzureLocation azureLocation = new AzureLocation(location);
        try {
            return this.isHierarchicalNamespaceEnabled(azureLocation) ? this.listGen2Files(azureLocation) : this.listBlobFiles(azureLocation);
        }
        catch (RuntimeException e) {
            throw AzureUtils.handleAzureException(e, "listing files", azureLocation);
        }
    }

    private FileIterator listGen2Files(AzureLocation location) throws IOException {
        PagedIterable pathItems;
        DataLakeFileSystemClient fileSystemClient = this.createFileSystemClient(location, Optional.empty());
        if (location.path().isEmpty()) {
            pathItems = fileSystemClient.listPaths(new ListPathsOptions().setRecursive(true), null);
        } else {
            DataLakeDirectoryClient directoryClient = AzureFileSystem.createDirectoryClient(fileSystemClient, location.path());
            if (!directoryClient.exists().booleanValue()) {
                return FileIterator.empty();
            }
            if (!directoryClient.getProperties().isDirectory().booleanValue()) {
                throw new TrinoFileSystemException("Location is not a directory: " + String.valueOf(location));
            }
            pathItems = directoryClient.listPaths(true, false, null, null);
        }
        return new AzureDataLakeFileIterator(location, pathItems.stream().filter(Predicate.not(PathItem::isDirectory)).iterator());
    }

    private FileIterator listBlobFiles(AzureLocation location) {
        PagedIterable blobItems = this.createBlobContainerClient(location, Optional.empty()).listBlobs(new ListBlobsOptions().setPrefix(location.directoryPath()), null);
        return new AzureBlobFileIterator(location, blobItems.iterator());
    }

    public Optional<Boolean> directoryExists(Location location) throws IOException {
        AzureLocation azureLocation = new AzureLocation(location);
        if (location.path().isEmpty()) {
            return Optional.of(true);
        }
        if (!this.isHierarchicalNamespaceEnabled(azureLocation)) {
            if (this.listFiles(location).hasNext()) {
                return Optional.of(true);
            }
            return Optional.empty();
        }
        try {
            DataLakeFileSystemClient fileSystemClient = this.createFileSystemClient(azureLocation, Optional.empty());
            DataLakeFileClient fileClient = AzureFileSystem.createFileClient(fileSystemClient, azureLocation.path());
            return Optional.of(fileClient.getProperties().isDirectory());
        }
        catch (DataLakeStorageException e) {
            if (e.getStatusCode() == 404) {
                return Optional.of(false);
            }
            throw AzureUtils.handleAzureException((RuntimeException)((Object)e), "checking directory existence", azureLocation);
        }
        catch (RuntimeException e) {
            throw AzureUtils.handleAzureException(e, "checking directory existence", azureLocation);
        }
    }

    public void createDirectory(Location location) throws IOException {
        AzureLocation azureLocation = new AzureLocation(location);
        if (!this.isHierarchicalNamespaceEnabled(azureLocation)) {
            return;
        }
        try {
            DataLakeFileSystemClient fileSystemClient = this.createFileSystemClient(azureLocation, Optional.empty());
            DataLakeDirectoryClient directoryClient = AzureFileSystem.createDirectoryIfNotExists(fileSystemClient, azureLocation.path());
            if (!directoryClient.getProperties().isDirectory().booleanValue()) {
                throw new TrinoFileSystemException("Location is not a directory: " + String.valueOf(azureLocation));
            }
        }
        catch (RuntimeException e) {
            throw AzureUtils.handleAzureException(e, "creating directory", azureLocation);
        }
    }

    public void renameDirectory(Location source, Location target) throws IOException {
        AzureLocation sourceLocation = new AzureLocation(source);
        AzureLocation targetLocation = new AzureLocation(target);
        if (!sourceLocation.account().equals(targetLocation.account())) {
            throw new TrinoFileSystemException("Cannot rename across storage accounts");
        }
        if (!Objects.equals(sourceLocation.container(), targetLocation.container())) {
            throw new TrinoFileSystemException("Cannot rename across storage account containers");
        }
        if (!this.isHierarchicalNamespaceEnabled(sourceLocation)) {
            throw new TrinoFileSystemException("Azure non-hierarchical does not support directory renames");
        }
        if (sourceLocation.path().isEmpty() || targetLocation.path().isEmpty()) {
            throw new TrinoFileSystemException("Cannot rename %s to %s".formatted(source, target));
        }
        try {
            DataLakeFileSystemClient fileSystemClient = this.createFileSystemClient(sourceLocation, Optional.empty());
            DataLakeDirectoryClient directoryClient = AzureFileSystem.createDirectoryClient(fileSystemClient, sourceLocation.path());
            if (!directoryClient.exists().booleanValue()) {
                throw new TrinoFileSystemException("Source directory does not exist: " + String.valueOf(source));
            }
            if (!directoryClient.getProperties().isDirectory().booleanValue()) {
                throw new TrinoFileSystemException("Source is not a directory: " + String.valueOf(source));
            }
            directoryClient.rename(null, targetLocation.path());
        }
        catch (RuntimeException e) {
            throw new IOException("Rename directory from %s to %s failed".formatted(source, target), e);
        }
    }

    public Set<Location> listDirectories(Location location) throws IOException {
        AzureLocation azureLocation = new AzureLocation(location);
        try {
            return this.isHierarchicalNamespaceEnabled(azureLocation) ? this.listGen2Directories(azureLocation) : this.listBlobDirectories(azureLocation);
        }
        catch (RuntimeException e) {
            throw AzureUtils.handleAzureException(e, "listing files", azureLocation);
        }
    }

    public Optional<Location> createTemporaryDirectory(Location targetPath, String temporaryPrefix, String relativePrefix) throws IOException {
        Location temporary;
        AzureLocation azureLocation = new AzureLocation(targetPath);
        if (!this.isHierarchicalNamespaceEnabled(azureLocation)) {
            return Optional.empty();
        }
        if (temporaryPrefix.startsWith("/")) {
            String prefix = temporaryPrefix;
            while (prefix.startsWith("/")) {
                prefix = prefix.substring(1);
            }
            temporary = azureLocation.baseLocation().appendPath(prefix);
        } else {
            temporary = targetPath.appendPath(temporaryPrefix);
        }
        temporary = temporary.appendPath(UUID.randomUUID().toString());
        this.createDirectory(temporary);
        return Optional.of(temporary);
    }

    public Optional<UriLocation> preSignedUri(Location location, Duration ttl) throws IOException {
        AzureAuth azureAuth = this.azureAuth;
        Objects.requireNonNull(azureAuth);
        AzureAuth azureAuth2 = azureAuth;
        int n = 0;
        return switch (SwitchBootstraps.typeSwitch("typeSwitch", new Object[]{AzureAuthOauth.class, AzureAuthAccessKey.class}, (AzureAuth)azureAuth2, n)) {
            case 0 -> this.oauth2PresignedUri(location, ttl, Optional.empty());
            case 1 -> this.accessKeyPresignedUri(location, ttl, Optional.empty());
            default -> throw new UnsupportedOperationException("Unsupported azure auth: " + String.valueOf(this.azureAuth));
        };
    }

    public Optional<UriLocation> encryptedPreSignedUri(Location location, Duration ttl, EncryptionKey key) throws IOException {
        AzureAuth azureAuth = this.azureAuth;
        Objects.requireNonNull(azureAuth);
        AzureAuth azureAuth2 = azureAuth;
        int n = 0;
        return switch (SwitchBootstraps.typeSwitch("typeSwitch", new Object[]{AzureAuthOauth.class, AzureAuthAccessKey.class}, (AzureAuth)azureAuth2, n)) {
            case 0 -> this.oauth2PresignedUri(location, ttl, Optional.of(key));
            case 1 -> this.accessKeyPresignedUri(location, ttl, Optional.of(key));
            default -> throw new UnsupportedOperationException("Unsupported azure auth: " + String.valueOf(this.azureAuth));
        };
    }

    private Optional<UriLocation> oauth2PresignedUri(Location location, Duration ttl, Optional<EncryptionKey> key) throws IOException {
        AzureLocation azureLocation = new AzureLocation(location);
        BlobContainerClient client = this.createBlobContainerClient(azureLocation, Optional.empty());
        OffsetDateTime startTime = OffsetDateTime.now();
        OffsetDateTime expiryTime = startTime.plus(ttl.toMillis(), ChronoUnit.MILLIS);
        UserDelegationKey userDelegationKey = client.getServiceClient().getUserDelegationKey(startTime, expiryTime);
        BlobSasPermission blobSasPermission = new BlobSasPermission().setReadPermission(true);
        BlobServiceSasSignatureValues sasValues = new BlobServiceSasSignatureValues(expiryTime, blobSasPermission).setStartTime(startTime).setProtocol(SasProtocol.HTTPS_ONLY);
        BlobClient blobClient = this.createBlobClient(azureLocation, key);
        String sasToken = blobClient.generateUserDelegationSas(sasValues, userDelegationKey);
        try {
            return Optional.of(new UriLocation(URI.create(blobClient.getBlobUrl() + "?" + sasToken), AzureFileSystem.preSignedHeaders(key)));
        }
        catch (Exception e) {
            throw new IOException("Failed to generate pre-signed URI", e);
        }
    }

    private Optional<UriLocation> accessKeyPresignedUri(Location location, Duration ttl, Optional<EncryptionKey> key) throws IOException {
        AzureLocation azureLocation = new AzureLocation(location);
        BlobClient client = this.createBlobClient(azureLocation, key);
        BlobSasPermission blobSasPermission = new BlobSasPermission().setReadPermission(true);
        OffsetDateTime startTime = OffsetDateTime.now();
        OffsetDateTime expiryTime = startTime.plus(ttl.toMillis(), ChronoUnit.MILLIS);
        BlobServiceSasSignatureValues values = new BlobServiceSasSignatureValues(expiryTime, blobSasPermission).setStartTime(startTime).setExpiryTime(expiryTime).setPermissions(blobSasPermission);
        this.createBlobContainerClient(azureLocation, key).generateSas(values);
        try {
            return Optional.of(new UriLocation(URI.create(client.getBlobUrl() + "?" + client.generateSas(values)), AzureFileSystem.preSignedHeaders(key)));
        }
        catch (Exception e) {
            throw new IOException("Failed to generate pre-signed URI", e);
        }
    }

    private static Map<String, List<String>> preSignedHeaders(Optional<EncryptionKey> key) {
        if (key.isEmpty()) {
            return ImmutableMap.of();
        }
        EncryptionKey encryption = key.get();
        ImmutableMap.Builder headers = ImmutableMap.builderWithExpectedSize((int)3);
        headers.put((Object)"x-ms-encryption-algorithm", List.of(encryption.algorithm()));
        headers.put((Object)"x-ms-encryption-key", List.of(AzureUtils.encodedKey(encryption)));
        headers.put((Object)"x-ms-encryption-key-sha256", List.of(AzureUtils.keySha256Checksum(encryption)));
        return headers.buildOrThrow();
    }

    private Set<Location> listGen2Directories(AzureLocation location) throws IOException {
        PagedIterable pathItems;
        DataLakeFileSystemClient fileSystemClient = this.createFileSystemClient(location, Optional.empty());
        if (location.path().isEmpty()) {
            pathItems = fileSystemClient.listPaths();
        } else {
            DataLakeDirectoryClient directoryClient = AzureFileSystem.createDirectoryClient(fileSystemClient, location.path());
            if (!directoryClient.exists().booleanValue()) {
                return ImmutableSet.of();
            }
            if (!directoryClient.getProperties().isDirectory().booleanValue()) {
                throw new TrinoFileSystemException("Location is not a directory: " + String.valueOf(location));
            }
            pathItems = directoryClient.listPaths(false, false, null, null);
        }
        Location baseLocation = location.baseLocation();
        return (Set)pathItems.stream().filter(PathItem::isDirectory).map(item -> baseLocation.appendPath(item.getName() + "/")).collect(ImmutableSet.toImmutableSet());
    }

    private Set<Location> listBlobDirectories(AzureLocation location) {
        Location baseLocation = location.baseLocation();
        return (Set)this.createBlobContainerClient(location, Optional.empty()).listBlobsByHierarchy(location.directoryPath()).stream().filter(BlobItem::isPrefix).map(item -> baseLocation.appendPath(item.getName())).collect(ImmutableSet.toImmutableSet());
    }

    private boolean isHierarchicalNamespaceEnabled(AzureLocation location) throws IOException {
        try {
            BlockBlobClient blockBlobClient = this.createBlobContainerClient(location, Optional.empty()).getBlobClient("/").getBlockBlobClient();
            return blockBlobClient.exists();
        }
        catch (RuntimeException e) {
            throw new IOException("Checking whether hierarchical namespace is enabled for the location %s failed".formatted(location), e);
        }
    }

    private String validatedEndpoint(AzureLocation location) {
        if (!location.endpoint().equals(this.endpoint)) {
            throw new IllegalArgumentException("Location does not match configured Azure endpoint: " + String.valueOf(location));
        }
        return location.endpoint();
    }

    private BlobClient createBlobClient(AzureLocation location, Optional<EncryptionKey> key) {
        return this.createBlobContainerClient(location, key).getBlobClient(location.path());
    }

    private BlobContainerClient createBlobContainerClient(AzureLocation location, Optional<EncryptionKey> key) {
        Objects.requireNonNull(location, "location is null");
        BlobContainerClientBuilder builder = new BlobContainerClientBuilder().httpClient(this.httpClient).clientOptions(new ClientOptions().setTracingOptions(this.tracingOptions)).endpoint("https://%s.blob.%s".formatted(location.account(), this.validatedEndpoint(location)));
        key.ifPresent(encryption -> builder.customerProvidedKey(AzureUtils.blobCustomerProvidedKey(encryption)));
        this.azureAuth.setAuth(location.account(), builder);
        location.container().ifPresent(arg_0 -> ((BlobContainerClientBuilder)builder).containerName(arg_0));
        return builder.buildClient();
    }

    private DataLakeFileSystemClient createFileSystemClient(AzureLocation location, Optional<EncryptionKey> key) {
        Objects.requireNonNull(location, "location is null");
        DataLakeServiceClientBuilder builder = new DataLakeServiceClientBuilder().httpClient(this.httpClient).clientOptions(new ClientOptions().setTracingOptions(this.tracingOptions)).endpoint("https://%s.dfs.%s".formatted(location.account(), this.validatedEndpoint(location)));
        key.ifPresent(encryption -> builder.customerProvidedKey(AzureUtils.lakeCustomerProvidedKey(encryption)));
        this.azureAuth.setAuth(location.account(), builder);
        DataLakeServiceClient client = builder.buildClient();
        DataLakeFileSystemClient fileSystemClient = client.getFileSystemClient(location.container().orElseThrow());
        if (!fileSystemClient.exists()) {
            throw new IllegalArgumentException();
        }
        return fileSystemClient;
    }

    private static DataLakeDirectoryClient createDirectoryClient(DataLakeFileSystemClient fileSystemClient, String directoryName) {
        return fileSystemClient.getDirectoryClient(directoryName);
    }

    private static DataLakeFileClient createFileClient(DataLakeFileSystemClient fileSystemClient, String fileName) {
        return fileSystemClient.getFileClient(fileName);
    }

    private static DataLakeDirectoryClient createDirectoryIfNotExists(DataLakeFileSystemClient fileSystemClient, String name) {
        return fileSystemClient.createDirectoryIfNotExists(name);
    }
}

