/*
 * Decompiled with CFR 0.152.
 */
package io.trino.filesystem.azure;

import com.azure.storage.blob.BlobClient;
import com.azure.storage.blob.models.BlobRange;
import com.azure.storage.blob.options.BlobInputStreamOptions;
import com.azure.storage.blob.specialized.BlobInputStream;
import com.google.common.base.Preconditions;
import io.trino.filesystem.TrinoInputStream;
import io.trino.filesystem.azure.AzureLocation;
import io.trino.filesystem.azure.AzureUtils;
import java.io.EOFException;
import java.io.IOException;
import java.util.Objects;

class AzureInputStream
extends TrinoInputStream {
    private final AzureLocation location;
    private final BlobClient blobClient;
    private final int readBlockSizeBytes;
    private final long fileSize;
    private BlobInputStream stream;
    private long currentPosition;
    private long nextPosition;
    private boolean closed;

    public AzureInputStream(AzureLocation location, BlobClient blobClient, int readBlockSizeBytes) throws IOException {
        this.location = Objects.requireNonNull(location, "location is null");
        this.blobClient = Objects.requireNonNull(blobClient, "blobClient is null");
        Preconditions.checkArgument((readBlockSizeBytes >= 0 ? 1 : 0) != 0, (Object)"readBlockSizeBytes is negative");
        this.readBlockSizeBytes = readBlockSizeBytes;
        this.openStream(0L);
        this.fileSize = this.stream.getProperties().getBlobSize();
    }

    public int available() throws IOException {
        this.ensureOpen();
        this.repositionStream();
        return this.stream.available();
    }

    public long getPosition() {
        return this.nextPosition;
    }

    public void seek(long newPosition) throws IOException {
        this.ensureOpen();
        if (newPosition < 0L) {
            throw new IOException("Negative seek offset");
        }
        if (newPosition > this.fileSize) {
            throw new IOException("Cannot seek to %s. File size is %s: %s".formatted(newPosition, this.fileSize, this.location));
        }
        this.nextPosition = newPosition;
    }

    public int read() throws IOException {
        this.ensureOpen();
        this.repositionStream();
        try {
            int value = this.stream.read();
            if (value >= 0) {
                ++this.currentPosition;
                ++this.nextPosition;
            }
            return value;
        }
        catch (RuntimeException e) {
            throw AzureUtils.handleAzureException(e, "reading file", this.location);
        }
    }

    public int read(byte[] buffer, int offset, int length) throws IOException {
        Objects.checkFromIndexSize(offset, length, buffer.length);
        this.ensureOpen();
        this.repositionStream();
        try {
            int readSize = this.stream.read(buffer, offset, length);
            if (readSize > 0) {
                this.currentPosition += (long)readSize;
                this.nextPosition += (long)readSize;
            }
            return readSize;
        }
        catch (RuntimeException e) {
            throw AzureUtils.handleAzureException(e, "reading file", this.location);
        }
    }

    public long skip(long n) throws IOException {
        this.ensureOpen();
        long skipSize = Math.clamp(n, 0L, this.fileSize - this.nextPosition);
        this.nextPosition += skipSize;
        return skipSize;
    }

    public void skipNBytes(long n) throws IOException {
        this.ensureOpen();
        if (n <= 0L) {
            return;
        }
        long position = this.nextPosition + n;
        if (position < 0L || position > this.fileSize) {
            throw new EOFException("Unable to skip %s bytes (position=%s, fileSize=%s): %s".formatted(n, this.nextPosition, this.fileSize, this.location));
        }
        this.nextPosition = position;
    }

    private void ensureOpen() throws IOException {
        if (this.closed) {
            throw new IOException("Output stream closed: " + String.valueOf(this.location));
        }
    }

    public void close() throws IOException {
        if (!this.closed) {
            this.closed = true;
            try {
                this.stream.close();
            }
            catch (RuntimeException e) {
                throw AzureUtils.handleAzureException(e, "closing file", this.location);
            }
        }
    }

    private void openStream(long offset) throws IOException {
        try {
            BlobInputStreamOptions options = new BlobInputStreamOptions().setRange(new BlobRange(offset)).setBlockSize(Integer.valueOf(this.readBlockSizeBytes));
            this.stream = this.blobClient.openInputStream(options);
            this.currentPosition = offset;
        }
        catch (RuntimeException e) {
            throw AzureUtils.handleAzureException(e, "reading file", this.location);
        }
    }

    private void repositionStream() throws IOException {
        if (this.nextPosition == this.currentPosition) {
            return;
        }
        if (this.nextPosition > this.currentPosition) {
            long bytesToSkip = this.nextPosition - this.currentPosition;
            this.stream.skipNBytes(bytesToSkip);
        } else {
            this.stream.close();
            this.openStream(this.nextPosition);
        }
        this.currentPosition = this.nextPosition;
    }
}

