/*
 * Decompiled with CFR 0.152.
 */
package io.trino.operator;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import io.airlift.slice.SizeOf;
import io.trino.operator.AppendOnlyVariableWidthData;
import io.trino.operator.FlatHashStrategy;
import io.trino.operator.GroupByHashMode;
import io.trino.operator.UpdateMemory;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.block.Block;
import io.trino.spi.block.BlockBuilder;
import io.trino.spi.type.BigintType;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.VarHandle;
import java.nio.ByteOrder;
import java.util.Arrays;
import java.util.Objects;

public final class FlatHash {
    private static final int INSTANCE_SIZE = SizeOf.instanceSize(FlatHash.class);
    private static final double DEFAULT_LOAD_FACTOR = 0.9375;
    private static final int RECORDS_PER_GROUP_SHIFT = 10;
    private static final int RECORDS_PER_GROUP = 1024;
    private static final int RECORDS_PER_GROUP_MASK = 1023;
    private static final int VECTOR_LENGTH = 8;
    private static final VarHandle LONG_HANDLE = MethodHandles.byteArrayViewVarHandle(long[].class, ByteOrder.LITTLE_ENDIAN);
    private final FlatHashStrategy flatHashStrategy;
    private final AppendOnlyVariableWidthData variableWidthData;
    private final UpdateMemory checkMemoryReservation;
    private final boolean hasPrecomputedHash;
    private final boolean cacheHashValue;
    private final int fixedRecordSize;
    private final int variableWidthOffset;
    private final int fixedValueOffset;
    private byte[] control;
    private int[] groupIdsByHash;
    private byte[][] fixedSizeRecords;
    private long fixedRecordGroupsRetainedSize;
    private long temporaryRehashRetainedSize;
    private int capacity;
    private int mask;
    private int nextGroupId;
    private int maxFill;

    private static int computeCapacity(int maxSize, double loadFactor) {
        int capacity = (int)((double)maxSize / loadFactor);
        return Math.max(Math.toIntExact(1L << 64 - Long.numberOfLeadingZeros(capacity - 1)), 16);
    }

    private static int calculateMaxFill(int capacity) {
        return Math.toIntExact((long)capacity * 15L / 16L);
    }

    public FlatHash(FlatHashStrategy flatHashStrategy, GroupByHashMode hashMode, int expectedSize, UpdateMemory checkMemoryReservation) {
        this.flatHashStrategy = Objects.requireNonNull(flatHashStrategy, "flatHashStrategy is null");
        this.checkMemoryReservation = Objects.requireNonNull(checkMemoryReservation, "checkMemoryReservation is null");
        boolean hasVariableData = flatHashStrategy.isAnyVariableWidth();
        this.variableWidthData = hasVariableData ? new AppendOnlyVariableWidthData() : null;
        Objects.requireNonNull(hashMode, "hashMode is null");
        this.hasPrecomputedHash = hashMode.isHashPrecomputed();
        this.cacheHashValue = hashMode.isHashCached();
        this.variableWidthOffset = this.cacheHashValue ? 8 : 0;
        this.fixedValueOffset = this.variableWidthOffset + (hasVariableData ? 8 : 0);
        this.fixedRecordSize = this.fixedValueOffset + flatHashStrategy.getTotalFlatFixedLength();
        this.capacity = Math.max(8, FlatHash.computeCapacity(expectedSize, 0.9375));
        this.mask = this.capacity - 1;
        this.maxFill = FlatHash.calculateMaxFill(this.capacity);
        int groupsRequired = FlatHash.recordGroupsRequiredForCapacity(this.capacity);
        this.control = new byte[this.capacity + 8];
        this.groupIdsByHash = new int[this.capacity];
        Arrays.fill(this.groupIdsByHash, -1);
        this.fixedSizeRecords = new byte[groupsRequired][];
    }

    public FlatHash(FlatHash other) {
        this.flatHashStrategy = other.flatHashStrategy;
        this.checkMemoryReservation = other.checkMemoryReservation;
        this.variableWidthData = other.variableWidthData == null ? null : new AppendOnlyVariableWidthData(other.variableWidthData);
        this.hasPrecomputedHash = other.hasPrecomputedHash;
        this.cacheHashValue = other.cacheHashValue;
        this.fixedRecordSize = other.fixedRecordSize;
        this.variableWidthOffset = other.variableWidthOffset;
        this.fixedValueOffset = other.fixedValueOffset;
        this.fixedRecordGroupsRetainedSize = other.fixedRecordGroupsRetainedSize;
        this.capacity = other.capacity;
        this.mask = other.mask;
        this.nextGroupId = other.nextGroupId;
        this.maxFill = other.maxFill;
        this.control = Arrays.copyOf(other.control, other.control.length);
        this.groupIdsByHash = Arrays.copyOf(other.groupIdsByHash, other.groupIdsByHash.length);
        this.fixedSizeRecords = (byte[][])Arrays.stream(other.fixedSizeRecords).map(fixedSizeRecords -> fixedSizeRecords == null ? null : Arrays.copyOf(fixedSizeRecords, ((byte[])fixedSizeRecords).length)).toArray(x$0 -> new byte[x$0][]);
    }

    public long getEstimatedSize() {
        return FlatHash.sumExact(INSTANCE_SIZE, this.fixedRecordGroupsRetainedSize, this.temporaryRehashRetainedSize, SizeOf.sizeOf((byte[])this.control), SizeOf.sizeOf((int[])this.groupIdsByHash), SizeOf.sizeOf((Object[])this.fixedSizeRecords), this.variableWidthData == null ? 0L : this.variableWidthData.getRetainedSizeBytes());
    }

    public int size() {
        return this.nextGroupId;
    }

    public int getCapacity() {
        return this.capacity;
    }

    public long hashPosition(int groupId) {
        if (groupId < 0) {
            throw new IllegalArgumentException("groupId is negative");
        }
        byte[] fixedSizeRecords = this.getFixedSizeRecords(groupId);
        int fixedRecordOffset = this.getFixedRecordOffset(groupId);
        if (this.cacheHashValue) {
            return LONG_HANDLE.get(fixedSizeRecords, fixedRecordOffset);
        }
        byte[] variableWidthChunk = null;
        int variableChunkOffset = 0;
        if (this.variableWidthData != null) {
            variableWidthChunk = this.variableWidthData.getChunk(fixedSizeRecords, fixedRecordOffset + this.variableWidthOffset);
            variableChunkOffset = AppendOnlyVariableWidthData.getChunkOffset(fixedSizeRecords, fixedRecordOffset + this.variableWidthOffset);
        }
        try {
            return this.flatHashStrategy.hash(fixedSizeRecords, fixedRecordOffset + this.fixedValueOffset, variableWidthChunk, variableChunkOffset);
        }
        catch (Throwable throwable) {
            Throwables.throwIfUnchecked((Throwable)throwable);
            throw new RuntimeException(throwable);
        }
    }

    public void appendTo(int groupId, BlockBuilder[] blockBuilders) {
        Preconditions.checkArgument((groupId < this.nextGroupId ? 1 : 0) != 0, (Object)"groupId out of range");
        byte[] fixedSizeRecords = this.getFixedSizeRecords(groupId);
        int recordOffset = this.getFixedRecordOffset(groupId);
        byte[] variableWidthChunk = null;
        int variableChunkOffset = 0;
        if (this.variableWidthData != null) {
            variableWidthChunk = this.variableWidthData.getChunk(fixedSizeRecords, recordOffset + this.variableWidthOffset);
            variableChunkOffset = AppendOnlyVariableWidthData.getChunkOffset(fixedSizeRecords, recordOffset + this.variableWidthOffset);
        }
        this.flatHashStrategy.readFlat(fixedSizeRecords, recordOffset + this.fixedValueOffset, variableWidthChunk, variableChunkOffset, blockBuilders);
        if (this.hasPrecomputedHash) {
            BigintType.BIGINT.writeLong(blockBuilders[blockBuilders.length - 1], LONG_HANDLE.get(fixedSizeRecords, recordOffset));
        }
    }

    public void computeHashes(Block[] blocks, long[] hashes, int offset, int length) {
        if (this.hasPrecomputedHash) {
            Block hashBlock = blocks[blocks.length - 1];
            for (int i = 0; i < length; ++i) {
                hashes[i] = BigintType.BIGINT.getLong(hashBlock, offset + i);
            }
        } else {
            this.flatHashStrategy.hashBlocksBatched(blocks, hashes, offset, length);
        }
    }

    public int putIfAbsent(Block[] blocks, int position) {
        long hash = this.hasPrecomputedHash ? BigintType.BIGINT.getLong(blocks[blocks.length - 1], position) : this.flatHashStrategy.hash(blocks, position);
        return this.putIfAbsent(blocks, position, hash);
    }

    public int putIfAbsent(Block[] blocks, int position, long hash) {
        int index = this.getIndex(blocks, position, hash);
        if (index >= 0) {
            int groupId = this.groupIdsByHash[index];
            if (groupId < 0) {
                throw new IllegalStateException("groupId out of range");
            }
            return groupId;
        }
        index = -index - 1;
        int groupId = this.addNewGroup(index, blocks, position, hash);
        if (this.nextGroupId >= this.maxFill) {
            this.rehash(0);
        }
        return groupId;
    }

    private int getIndex(Block[] blocks, int position, long hash) {
        byte hashPrefix = (byte)(hash & 0x7FL | 0x80L);
        int bucket = this.bucket((int)(hash >> 7));
        int step = 1;
        long repeated = FlatHash.repeat(hashPrefix);
        long controlVector;
        int matchIndex;
        while ((matchIndex = this.matchInVector(blocks, position, hash, bucket, repeated, controlVector = LONG_HANDLE.get(this.control, bucket))) < 0) {
            int emptyIndex = this.findEmptyInVector(controlVector, bucket);
            if (emptyIndex >= 0) {
                return -emptyIndex - 1;
            }
            bucket = this.bucket(bucket + step);
            step += 8;
        }
        return matchIndex;
    }

    private int matchInVector(Block[] blocks, int position, long hash, int vectorStartBucket, long repeated, long controlVector) {
        for (long controlMatches = FlatHash.match(controlVector, repeated); controlMatches != 0L; controlMatches &= controlMatches - 1L) {
            int index = this.bucket(vectorStartBucket + (Long.numberOfTrailingZeros(controlMatches) >>> 3));
            int groupId = this.groupIdsByHash[index];
            if (!this.valueIdentical(groupId, blocks, position, hash)) continue;
            return index;
        }
        return -1;
    }

    private int findEmptyInVector(long vector, int vectorStartBucket) {
        long controlMatches = FlatHash.match(vector, 0L);
        if (controlMatches == 0L) {
            return -1;
        }
        int slot = Long.numberOfTrailingZeros(controlMatches) >>> 3;
        return this.bucket(vectorStartBucket + slot);
    }

    private int addNewGroup(int index, Block[] blocks, int position, long hash) {
        int groupId;
        this.setControl(index, (byte)(hash & 0x7FL | 0x80L));
        this.groupIdsByHash[index] = groupId = this.nextGroupId++;
        int recordGroupIndex = FlatHash.recordGroupIndexForGroupId(groupId);
        int fixedRecordOffset = this.getFixedRecordOffset(groupId);
        byte[] fixedSizeRecords = this.fixedSizeRecords[recordGroupIndex];
        if (fixedRecordOffset == 0) {
            if (fixedSizeRecords != null) {
                throw new IllegalStateException("fixedSizeRecords already exists");
            }
            fixedSizeRecords = new byte[Math.multiplyExact(1024, this.fixedRecordSize)];
            this.fixedSizeRecords[recordGroupIndex] = fixedSizeRecords;
            this.fixedRecordGroupsRetainedSize = Math.addExact(this.fixedRecordGroupsRetainedSize, SizeOf.sizeOf((byte[])fixedSizeRecords));
        }
        if (this.cacheHashValue) {
            LONG_HANDLE.set(fixedSizeRecords, fixedRecordOffset, hash);
        }
        byte[] variableWidthChunk = null;
        int variableWidthChunkOffset = 0;
        if (this.variableWidthData != null) {
            int variableWidthSize = this.flatHashStrategy.getTotalVariableWidth(blocks, position);
            variableWidthChunk = this.variableWidthData.allocate(fixedSizeRecords, fixedRecordOffset + this.variableWidthOffset, variableWidthSize);
            variableWidthChunkOffset = AppendOnlyVariableWidthData.getChunkOffset(fixedSizeRecords, fixedRecordOffset + this.variableWidthOffset);
        }
        this.flatHashStrategy.writeFlat(blocks, position, fixedSizeRecords, fixedRecordOffset + this.fixedValueOffset, variableWidthChunk, variableWidthChunkOffset);
        return groupId;
    }

    private void setControl(int index, byte hashPrefix) {
        this.control[index] = hashPrefix;
        if (index < 8) {
            this.control[index + this.capacity] = hashPrefix;
        }
    }

    public boolean ensureAvailableCapacity(int batchSize) {
        long requiredMaxFill = this.nextGroupId + batchSize;
        if (requiredMaxFill >= (long)this.maxFill) {
            long minimumRequiredCapacity = (requiredMaxFill + 1L) * 16L / 15L;
            return this.tryRehash(Math.toIntExact(minimumRequiredCapacity));
        }
        return true;
    }

    private boolean tryRehash(int minimumRequiredCapacity) {
        int newCapacity = this.computeNewCapacity(minimumRequiredCapacity);
        this.temporaryRehashRetainedSize = Math.multiplyExact((long)newCapacity, 5);
        if (!this.checkMemoryReservation.update()) {
            return false;
        }
        this.rehash(minimumRequiredCapacity);
        return true;
    }

    private void rehash(int minimumRequiredCapacity) {
        this.capacity = this.computeNewCapacity(minimumRequiredCapacity);
        this.maxFill = FlatHash.calculateMaxFill(this.capacity);
        this.mask = this.capacity - 1;
        this.fixedSizeRecords = (byte[][])Arrays.copyOf(this.fixedSizeRecords, FlatHash.recordGroupsRequiredForCapacity(this.capacity));
        this.control = new byte[this.capacity + 8];
        this.groupIdsByHash = new int[this.capacity];
        Arrays.fill(this.groupIdsByHash, -1);
        int groupId = 0;
        while (groupId < this.nextGroupId) {
            long hash = this.hashPosition(groupId);
            byte hashPrefix = (byte)(hash & 0x7FL | 0x80L);
            int bucket = this.bucket((int)(hash >> 7));
            int step = 1;
            while (true) {
                long controlVector;
                int emptyIndex;
                if ((emptyIndex = this.findEmptyInVector(controlVector = LONG_HANDLE.get(this.control, bucket), bucket)) >= 0) {
                    this.setControl(emptyIndex, hashPrefix);
                    if (this.groupIdsByHash[emptyIndex] != -1) {
                        throw new IllegalStateException("groupId mapping already exists at index");
                    }
                    break;
                }
                bucket = this.bucket(bucket + step);
                step += 8;
            }
            this.groupIdsByHash[emptyIndex] = groupId++;
        }
        this.temporaryRehashRetainedSize = 0L;
        this.checkMemoryReservation.update();
    }

    private int bucket(int hash) {
        return hash & this.mask;
    }

    private static int recordGroupIndexForGroupId(int groupId) {
        return groupId >> 10;
    }

    private byte[] getFixedSizeRecords(int groupId) {
        return this.fixedSizeRecords[FlatHash.recordGroupIndexForGroupId(groupId)];
    }

    private int getFixedRecordOffset(int groupId) {
        return (groupId & 0x3FF) * this.fixedRecordSize;
    }

    private boolean valueIdentical(int groupId, Block[] rightBlocks, int rightPosition, long rightHash) {
        Preconditions.checkArgument((groupId >= 0 ? 1 : 0) != 0, (Object)"groupId is negative");
        byte[] fixedSizeRecords = this.getFixedSizeRecords(groupId);
        int fixedRecordsOffset = this.getFixedRecordOffset(groupId);
        if (this.cacheHashValue && rightHash != LONG_HANDLE.get(fixedSizeRecords, fixedRecordsOffset)) {
            return false;
        }
        byte[] variableWidthChunk = null;
        int variableWidthChunkOffset = 0;
        if (this.variableWidthData != null) {
            variableWidthChunk = this.variableWidthData.getChunk(fixedSizeRecords, fixedRecordsOffset + this.variableWidthOffset);
            variableWidthChunkOffset = AppendOnlyVariableWidthData.getChunkOffset(fixedSizeRecords, fixedRecordsOffset + this.variableWidthOffset);
        }
        return this.flatHashStrategy.valueIdentical(fixedSizeRecords, fixedRecordsOffset + this.fixedValueOffset, variableWidthChunk, variableWidthChunkOffset, rightBlocks, rightPosition);
    }

    private int computeNewCapacity(int minimumRequiredCapacity) {
        Preconditions.checkArgument((minimumRequiredCapacity >= 0 ? 1 : 0) != 0, (Object)"minimumRequiredCapacity must be positive");
        long newCapacityLong = (long)this.capacity * 2L;
        while (newCapacityLong < (long)minimumRequiredCapacity) {
            newCapacityLong = Math.multiplyExact(newCapacityLong, 2);
        }
        if (newCapacityLong > Integer.MAX_VALUE) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INSUFFICIENT_RESOURCES, "Size of hash table cannot exceed 1 billion entries");
        }
        return Math.toIntExact(newCapacityLong);
    }

    private static long repeat(byte value) {
        return (long)(value & 0xFF) * 0x101010101010101L;
    }

    private static long match(long vector, long repeatedValue) {
        long comparison = vector ^ repeatedValue;
        return comparison - 0x101010101010101L & (comparison ^ 0xFFFFFFFFFFFFFFFFL) & 0x8080808080808080L;
    }

    private static int recordGroupsRequiredForCapacity(int capacity) {
        Preconditions.checkArgument((capacity > 0 ? 1 : 0) != 0, (Object)"capacity must be positive");
        return Math.max(1, capacity + 1 >> 10);
    }

    public static long sumExact(long ... values) {
        long result = 0L;
        for (long value : values) {
            result = Math.addExact(result, value);
        }
        return result;
    }

    public FlatHash copy() {
        return new FlatHash(this);
    }
}

