/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.accumulo.index;

import com.google.common.base.MoreObjects;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.MultimapBuilder;
import com.google.common.collect.Streams;
import io.airlift.concurrent.BoundedExecutor;
import io.airlift.concurrent.Threads;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import io.trino.collect.cache.NonEvictableLoadingCache;
import io.trino.collect.cache.SafeCaches;
import io.trino.plugin.accumulo.AccumuloErrorCode;
import io.trino.plugin.accumulo.conf.AccumuloConfig;
import io.trino.plugin.accumulo.index.Indexer;
import io.trino.plugin.accumulo.model.AccumuloColumnConstraint;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.PreDestroy;
import javax.inject.Inject;
import org.apache.accumulo.core.client.BatchScanner;
import org.apache.accumulo.core.client.Connector;
import org.apache.accumulo.core.data.Key;
import org.apache.accumulo.core.data.PartialKey;
import org.apache.accumulo.core.data.Range;
import org.apache.accumulo.core.data.Value;
import org.apache.accumulo.core.security.Authorizations;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.hadoop.io.Text;

public class ColumnCardinalityCache {
    private static final Logger LOG = Logger.get(ColumnCardinalityCache.class);
    private final Connector connector;
    private final ExecutorService coreExecutor;
    private final BoundedExecutor executorService;
    private final NonEvictableLoadingCache<CacheKey, Long> cache;

    @Inject
    public ColumnCardinalityCache(Connector connector, AccumuloConfig config) {
        this.connector = Objects.requireNonNull(connector, "connector is null");
        int size = Objects.requireNonNull(config, "config is null").getCardinalityCacheSize();
        Duration expireDuration = config.getCardinalityCacheExpiration();
        this.coreExecutor = Executors.newCachedThreadPool(Threads.daemonThreadsNamed((String)"cardinality-lookup-%s"));
        this.executorService = new BoundedExecutor((Executor)this.coreExecutor, 4 * Runtime.getRuntime().availableProcessors());
        LOG.debug("Created new cache size %d expiry %s", new Object[]{size, expireDuration});
        this.cache = SafeCaches.buildNonEvictableCache((CacheBuilder)CacheBuilder.newBuilder().maximumSize((long)size).expireAfterWrite(expireDuration.toMillis(), TimeUnit.MILLISECONDS), (CacheLoader)new CardinalityCacheLoader());
    }

    @PreDestroy
    public void shutdown() {
        this.coreExecutor.shutdownNow();
    }

    public Multimap<Long, AccumuloColumnConstraint> getCardinalities(String schema, String table, Authorizations auths, Multimap<AccumuloColumnConstraint, Range> idxConstraintRangePairs, long earlyReturnThreshold, Duration pollingDuration) {
        ExecutorCompletionService executor = new ExecutorCompletionService((Executor)this.executorService);
        idxConstraintRangePairs.asMap().forEach((key, value) -> executor.submit(() -> {
            long cardinality = this.getColumnCardinality(schema, table, auths, key.getFamily(), key.getQualifier(), (Collection<Range>)value);
            LOG.debug("Cardinality for column %s is %s", new Object[]{key.getName(), cardinality});
            return Pair.of((Object)cardinality, (Object)key);
        }));
        ListMultimap cardinalityToConstraints = MultimapBuilder.treeKeys().arrayListValues().build();
        try {
            boolean earlyReturn = false;
            int numTasks = idxConstraintRangePairs.asMap().entrySet().size();
            do {
                Thread.sleep(pollingDuration.toMillis());
                for (int i = 0; i < numTasks; ++i) {
                    Future futureCardinality = executor.poll();
                    if (futureCardinality == null || !futureCardinality.isDone()) continue;
                    Pair columnCardinality = (Pair)futureCardinality.get();
                    cardinalityToConstraints.put((Object)((Long)columnCardinality.getLeft()), (Object)((AccumuloColumnConstraint)columnCardinality.getRight()));
                }
                Optional smallestCardinality = cardinalityToConstraints.entries().stream().findFirst();
                if (!smallestCardinality.isPresent() || (Long)((Map.Entry)smallestCardinality.get()).getKey() > earlyReturnThreshold) continue;
                LOG.info("Cardinality %s, is below threshold. Returning early while other tasks finish", new Object[]{smallestCardinality});
                earlyReturn = true;
            } while (!earlyReturn && cardinalityToConstraints.entries().size() < numTasks);
        }
        catch (InterruptedException | ExecutionException e) {
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            throw new TrinoException((ErrorCodeSupplier)AccumuloErrorCode.UNEXPECTED_ACCUMULO_ERROR, "Exception when getting cardinality", (Throwable)e);
        }
        return ImmutableMultimap.copyOf((Multimap)cardinalityToConstraints);
    }

    public long getColumnCardinality(String schema, String table, Authorizations auths, String family, String qualifier, Collection<Range> colValues) throws ExecutionException {
        LOG.debug("Getting cardinality for %s:%s", new Object[]{family, qualifier});
        Collection exactRanges = colValues.stream().filter(ColumnCardinalityCache::isExact).map(range -> new CacheKey(schema, table, family, qualifier, (Range)range, auths)).collect(Collectors.toList());
        LOG.debug("Column values contain %s exact ranges of %s", new Object[]{exactRanges.size(), colValues.size()});
        long sum = this.cache.getAll((Iterable)exactRanges).values().stream().mapToLong(Long::longValue).sum();
        if (exactRanges.size() != colValues.size()) {
            for (Range range2 : colValues) {
                if (ColumnCardinalityCache.isExact(range2)) continue;
                sum += ((Long)this.cache.get((Object)new CacheKey(schema, table, family, qualifier, range2, auths))).longValue();
            }
        }
        return sum;
    }

    private static boolean isExact(Range range) {
        return !range.isInfiniteStartKey() && !range.isInfiniteStopKey() && range.getStartKey().followingKey(PartialKey.ROW).equals((Object)range.getEndKey());
    }

    private class CardinalityCacheLoader
    extends CacheLoader<CacheKey, Long> {
        private CardinalityCacheLoader() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Long load(CacheKey key) throws Exception {
            LOG.debug("Loading a non-exact range from Accumulo: %s", new Object[]{key});
            String metricsTable = Indexer.getMetricsTableName(key.getSchema(), key.getTable());
            Text columnFamily = new Text(Indexer.getIndexColumnFamily(key.getFamily().getBytes(StandardCharsets.UTF_8), key.getQualifier().getBytes(StandardCharsets.UTF_8)).array());
            BatchScanner scanner = ColumnCardinalityCache.this.connector.createBatchScanner(metricsTable, key.auths, 10);
            scanner.setRanges((Collection)ColumnCardinalityCache.this.connector.tableOperations().splitRangeByTablets(metricsTable, key.range, Integer.MAX_VALUE));
            scanner.fetchColumn(columnFamily, Indexer.CARDINALITY_CQ_AS_TEXT);
            try {
                Long l = Streams.stream((Iterable)scanner).map(Map.Entry::getValue).map(Value::toString).mapToLong(Long::parseLong).sum();
                return l;
            }
            finally {
                scanner.close();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Map<CacheKey, Long> loadAll(Iterable<? extends CacheKey> keys) throws Exception {
            int size = Iterables.size(keys);
            if (size == 0) {
                return ImmutableMap.of();
            }
            LOG.debug("Loading %s exact ranges from Accumulo", new Object[]{size});
            CacheKey anyKey = (CacheKey)Streams.stream(keys).findAny().get();
            if (Streams.stream(keys).anyMatch(k -> !k.getSchema().equals(anyKey.getSchema()) || !k.getTable().equals(anyKey.getTable()) || !k.getFamily().equals(anyKey.getFamily()) || !k.getQualifier().equals(anyKey.getQualifier()))) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.FUNCTION_IMPLEMENTATION_ERROR, "loadAll called with a non-homogeneous collection of cache keys");
            }
            Map rangeToKey = Streams.stream(keys).collect(Collectors.toMap(CacheKey::getRange, Function.identity()));
            LOG.debug("rangeToKey size is %s", new Object[]{rangeToKey.size()});
            String metricsTable = Indexer.getMetricsTableName(anyKey.getSchema(), anyKey.getTable());
            Text columnFamily = new Text(Indexer.getIndexColumnFamily(anyKey.getFamily().getBytes(StandardCharsets.UTF_8), anyKey.getQualifier().getBytes(StandardCharsets.UTF_8)).array());
            try (BatchScanner scanner = ColumnCardinalityCache.this.connector.createBatchScanner(metricsTable, anyKey.getAuths(), 10);){
                scanner.setRanges((Collection)Streams.stream(keys).map(CacheKey::getRange).collect(Collectors.toList()));
                scanner.fetchColumn(columnFamily, Indexer.CARDINALITY_CQ_AS_TEXT);
                HashMap<CacheKey, Long> rangeValues = new HashMap<CacheKey, Long>();
                Streams.stream(keys).forEach(key -> rangeValues.put((CacheKey)key, 0L));
                for (Map.Entry entry : scanner) {
                    rangeValues.put((CacheKey)rangeToKey.get(Range.exact((Text)((Key)entry.getKey()).getRow())), Long.parseLong(((Value)entry.getValue()).toString()));
                }
                HashMap<CacheKey, Long> hashMap = rangeValues;
                return hashMap;
            }
        }
    }

    private static class CacheKey {
        private final String schema;
        private final String table;
        private final String family;
        private final String qualifier;
        private final Range range;
        private final Authorizations auths;

        public CacheKey(String schema, String table, String family, String qualifier, Range range, Authorizations auths) {
            this.schema = Objects.requireNonNull(schema, "schema is null");
            this.table = Objects.requireNonNull(table, "table is null");
            this.family = Objects.requireNonNull(family, "family is null");
            this.qualifier = Objects.requireNonNull(qualifier, "qualifier is null");
            this.range = Objects.requireNonNull(range, "range is null");
            this.auths = Objects.requireNonNull(auths, "auths is null");
        }

        public String getSchema() {
            return this.schema;
        }

        public String getTable() {
            return this.table;
        }

        public String getFamily() {
            return this.family;
        }

        public String getQualifier() {
            return this.qualifier;
        }

        public Range getRange() {
            return this.range;
        }

        public Authorizations getAuths() {
            return this.auths;
        }

        public int hashCode() {
            return Objects.hash(this.schema, this.table, this.family, this.qualifier, this.range);
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null || this.getClass() != obj.getClass()) {
                return false;
            }
            CacheKey other = (CacheKey)obj;
            return Objects.equals(this.schema, other.schema) && Objects.equals(this.table, other.table) && Objects.equals(this.family, other.family) && Objects.equals(this.qualifier, other.qualifier) && Objects.equals(this.range, other.range);
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("schema", (Object)this.schema).add("table", (Object)this.table).add("family", (Object)this.family).add("qualifier", (Object)this.qualifier).add("range", (Object)this.range).toString();
        }
    }
}

