/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.pinot;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import io.airlift.log.Logger;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.trino.plugin.pinot.PinotColumnHandle;
import io.trino.plugin.pinot.PinotErrorCode;
import io.trino.plugin.pinot.PinotException;
import io.trino.plugin.pinot.PinotSessionProperties;
import io.trino.plugin.pinot.PinotSplit;
import io.trino.plugin.pinot.client.PinotQueryClient;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.Page;
import io.trino.spi.PageBuilder;
import io.trino.spi.TrinoException;
import io.trino.spi.block.Block;
import io.trino.spi.block.BlockBuilder;
import io.trino.spi.connector.ConnectorPageSource;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.type.Type;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.pinot.common.utils.DataSchema;
import org.apache.pinot.common.utils.DataTable;
import org.apache.pinot.core.transport.ServerInstance;

public class PinotSegmentPageSource
implements ConnectorPageSource {
    private static final Logger LOG = Logger.get(PinotSegmentPageSource.class);
    private final List<PinotColumnHandle> columnHandles;
    private final PinotSplit split;
    private final PinotQueryClient pinotQueryClient;
    private final ConnectorSession session;
    private final String query;
    private final int limitForSegmentQueries;
    private final AtomicLong currentRowCount = new AtomicLong();
    private final int estimatedNonNumericColumnSize;
    private List<Type> columnTypes;
    private LinkedList<PinotDataTableWithSize> dataTableList = new LinkedList();
    private long completedBytes;
    private long readTimeNanos;
    private long estimatedMemoryUsageInBytes;
    private PinotDataTableWithSize currentDataTable;
    private boolean closed;
    private boolean isPinotDataFetched;

    public PinotSegmentPageSource(ConnectorSession session, int estimatedNonNumericColumnSize, int limitForSegmentQueries, PinotQueryClient pinotQueryClient, PinotSplit split, List<PinotColumnHandle> columnHandles, String query) {
        this.limitForSegmentQueries = limitForSegmentQueries;
        this.estimatedNonNumericColumnSize = estimatedNonNumericColumnSize;
        this.split = Objects.requireNonNull(split, "split is null");
        this.pinotQueryClient = Objects.requireNonNull(pinotQueryClient, "pinotQueryClient is null");
        this.columnHandles = Objects.requireNonNull(columnHandles, "columnHandles is null");
        this.session = Objects.requireNonNull(session, "session is null");
        this.query = Objects.requireNonNull(query, "query is null");
    }

    private static void checkExceptions(DataTable dataTable, PinotSplit split, String query) {
        Map metadata = dataTable.getMetadata();
        ArrayList exceptions = new ArrayList();
        metadata.forEach((k, v) -> {
            if (k.startsWith("Exception")) {
                exceptions.add(v);
            }
        });
        if (!exceptions.isEmpty()) {
            throw new PinotException(PinotErrorCode.PINOT_EXCEPTION, Optional.of(query), String.format("Encountered %d pinot exceptions for split %s: %s", exceptions.size(), split, exceptions));
        }
    }

    public long getCompletedBytes() {
        return this.completedBytes;
    }

    public long getReadTimeNanos() {
        return this.readTimeNanos;
    }

    public long getMemoryUsage() {
        return this.estimatedMemoryUsageInBytes;
    }

    public boolean isFinished() {
        return this.closed || this.isPinotDataFetched && this.dataTableList.isEmpty();
    }

    public Page getNextPage() {
        if (this.isFinished()) {
            this.close();
            return null;
        }
        if (!this.isPinotDataFetched) {
            this.fetchPinotData();
        }
        if (this.currentDataTable != null) {
            this.estimatedMemoryUsageInBytes -= (long)this.currentDataTable.getEstimatedSizeInBytes();
        }
        if (this.dataTableList.size() == 0) {
            this.close();
            return null;
        }
        this.currentDataTable = this.dataTableList.pop();
        PageBuilder pageBuilder = new PageBuilder(this.columnTypes);
        pageBuilder.declarePositions(this.currentDataTable.getDataTable().getNumberOfRows());
        for (int columnHandleIdx = 0; columnHandleIdx < this.columnHandles.size(); ++columnHandleIdx) {
            BlockBuilder blockBuilder = pageBuilder.getBlockBuilder(columnHandleIdx);
            Type columnType = this.columnTypes.get(columnHandleIdx);
            this.writeBlock(blockBuilder, columnType, columnHandleIdx);
        }
        return pageBuilder.build();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void fetchPinotData() {
        long startTimeNanos = System.nanoTime();
        try {
            Map<ServerInstance, DataTable> dataTableMap = this.queryPinot();
            dataTableMap.values().stream().filter(table -> table != null && table.getNumberOfRows() > 0).forEach(dataTable -> {
                PinotSegmentPageSource.checkExceptions(dataTable, this.split, this.query);
                this.checkTooManyRows((DataTable)dataTable);
                int estimatedTableSizeInBytes = IntStream.rangeClosed(0, dataTable.getDataSchema().size() - 1).map(i -> this.getEstimatedColumnSizeInBytes(dataTable.getDataSchema().getColumnDataType(i)) * dataTable.getNumberOfRows()).reduce(0, Integer::sum);
                this.dataTableList.add(new PinotDataTableWithSize((DataTable)dataTable, estimatedTableSizeInBytes));
                this.estimatedMemoryUsageInBytes += (long)estimatedTableSizeInBytes;
            });
            this.columnTypes = this.columnHandles.stream().map(columnHandle -> columnHandle.getDataType()).collect(Collectors.toList());
            this.isPinotDataFetched = true;
        }
        finally {
            this.readTimeNanos += System.nanoTime() - startTimeNanos;
        }
    }

    private void checkTooManyRows(DataTable dataTable) {
        if (this.currentRowCount.addAndGet(dataTable.getNumberOfRows()) > (long)this.limitForSegmentQueries) {
            throw new PinotException(PinotErrorCode.PINOT_EXCEPTION, Optional.of(this.query), String.format("Segment query returned '%s' rows per split, maximum allowed is '%s' rows.", this.currentRowCount.get(), this.limitForSegmentQueries));
        }
    }

    private Map<ServerInstance, DataTable> queryPinot() {
        String host = this.split.getSegmentHost().orElseThrow(() -> new PinotException(PinotErrorCode.PINOT_INVALID_PQL_GENERATED, Optional.empty(), "Expected the segment split to contain the host"));
        LOG.info("Query '%s' on host '%s' for segment splits: %s", new Object[]{this.query, this.split.getSegmentHost(), this.split.getSegments()});
        return ImmutableMap.copyOf(this.pinotQueryClient.queryPinotServerForDataTable(this.query, host, this.split.getSegments(), PinotSessionProperties.getConnectionTimeout(this.session).toMillis(), PinotSessionProperties.getPinotRetryCount(this.session)));
    }

    public void close() {
        if (this.closed) {
            return;
        }
        this.closed = true;
    }

    private void writeBlock(BlockBuilder blockBuilder, Type columnType, int columnIdx) {
        Class javaType = columnType.getJavaType();
        DataSchema.ColumnDataType pinotColumnType = this.currentDataTable.getDataTable().getDataSchema().getColumnDataType(columnIdx);
        if (javaType.equals(Boolean.TYPE)) {
            this.writeBooleanBlock(blockBuilder, columnType, columnIdx);
        } else if (javaType.equals(Long.TYPE)) {
            this.writeLongBlock(blockBuilder, columnType, columnIdx);
        } else if (javaType.equals(Double.TYPE)) {
            this.writeDoubleBlock(blockBuilder, columnType, columnIdx);
        } else if (javaType.equals(Slice.class)) {
            this.writeSliceBlock(blockBuilder, columnType, columnIdx);
        } else if (javaType.equals(Block.class)) {
            this.writeArrayBlock(blockBuilder, columnType, columnIdx);
        } else {
            throw new TrinoException((ErrorCodeSupplier)PinotErrorCode.PINOT_UNSUPPORTED_COLUMN_TYPE, String.format("Failed to write column %s. pinotColumnType %s, javaType %s", this.columnHandles.get(columnIdx).getColumnName(), pinotColumnType, javaType));
        }
    }

    private void writeBooleanBlock(BlockBuilder blockBuilder, Type columnType, int columnIndex) {
        for (int i = 0; i < this.currentDataTable.getDataTable().getNumberOfRows(); ++i) {
            columnType.writeBoolean(blockBuilder, this.getBoolean(i, columnIndex));
            ++this.completedBytes;
        }
    }

    private void writeLongBlock(BlockBuilder blockBuilder, Type columnType, int columnIndex) {
        for (int i = 0; i < this.currentDataTable.getDataTable().getNumberOfRows(); ++i) {
            columnType.writeLong(blockBuilder, this.getLong(i, columnIndex));
            this.completedBytes += 8L;
        }
    }

    private void writeDoubleBlock(BlockBuilder blockBuilder, Type columnType, int columnIndex) {
        for (int i = 0; i < this.currentDataTable.getDataTable().getNumberOfRows(); ++i) {
            columnType.writeDouble(blockBuilder, this.getDouble(i, columnIndex));
            this.completedBytes += 8L;
        }
    }

    private void writeSliceBlock(BlockBuilder blockBuilder, Type columnType, int columnIndex) {
        for (int i = 0; i < this.currentDataTable.getDataTable().getNumberOfRows(); ++i) {
            Slice slice = this.getSlice(i, columnIndex);
            columnType.writeSlice(blockBuilder, slice, 0, slice.length());
            this.completedBytes += (long)slice.getBytes().length;
        }
    }

    private void writeArrayBlock(BlockBuilder blockBuilder, Type columnType, int columnIndex) {
        for (int i = 0; i < this.currentDataTable.getDataTable().getNumberOfRows(); ++i) {
            Block block = this.getArrayBlock(i, columnIndex);
            columnType.writeObject(blockBuilder, (Object)block);
            this.completedBytes += block.getSizeInBytes();
        }
    }

    Type getType(int columnIndex) {
        Preconditions.checkArgument((columnIndex < this.columnHandles.size() ? 1 : 0) != 0, (Object)"Invalid field index");
        return this.columnHandles.get(columnIndex).getDataType();
    }

    boolean getBoolean(int rowIdx, int columnIndex) {
        return this.currentDataTable.getDataTable().getInt(rowIdx, columnIndex) != 0;
    }

    long getLong(int rowIndex, int columnIndex) {
        DataSchema.ColumnDataType dataType = this.currentDataTable.getDataTable().getDataSchema().getColumnDataType(columnIndex);
        switch (dataType) {
            case DOUBLE: {
                return (long)this.currentDataTable.getDataTable().getDouble(rowIndex, columnIndex);
            }
            case INT: {
                return this.currentDataTable.getDataTable().getInt(rowIndex, columnIndex);
            }
            case FLOAT: {
                return Float.floatToIntBits(this.currentDataTable.getDataTable().getFloat(rowIndex, columnIndex));
            }
            case LONG: {
                return this.currentDataTable.getDataTable().getLong(rowIndex, columnIndex);
            }
        }
        throw new PinotException(PinotErrorCode.PINOT_DECODE_ERROR, Optional.empty(), String.format("Unexpected pinot type: '%s'", dataType));
    }

    double getDouble(int rowIndex, int columnIndex) {
        DataSchema.ColumnDataType dataType = this.currentDataTable.getDataTable().getDataSchema().getColumnDataType(columnIndex);
        if (dataType.equals((Object)DataSchema.ColumnDataType.FLOAT)) {
            return this.currentDataTable.getDataTable().getFloat(rowIndex, columnIndex);
        }
        return this.currentDataTable.getDataTable().getDouble(rowIndex, columnIndex);
    }

    Block getArrayBlock(int rowIndex, int columnIndex) {
        BlockBuilder blockBuilder;
        Type trinoType = this.getType(columnIndex);
        Type elementType = (Type)trinoType.getTypeParameters().get(0);
        DataSchema.ColumnDataType columnType = this.currentDataTable.getDataTable().getDataSchema().getColumnDataType(columnIndex);
        switch (columnType) {
            case INT_ARRAY: {
                int[] intArray = this.currentDataTable.getDataTable().getIntArray(rowIndex, columnIndex);
                blockBuilder = elementType.createBlockBuilder(null, intArray.length);
                for (int element : intArray) {
                    blockBuilder.writeInt(element);
                }
                break;
            }
            case LONG_ARRAY: {
                long[] longArray = this.currentDataTable.getDataTable().getLongArray(rowIndex, columnIndex);
                blockBuilder = elementType.createBlockBuilder(null, longArray.length);
                for (long element : longArray) {
                    blockBuilder.writeLong(element);
                }
                break;
            }
            case FLOAT_ARRAY: {
                float[] floatArray = this.currentDataTable.getDataTable().getFloatArray(rowIndex, columnIndex);
                blockBuilder = elementType.createBlockBuilder(null, floatArray.length);
                for (float element : floatArray) {
                    blockBuilder.writeInt(Float.floatToIntBits(element));
                }
                break;
            }
            case DOUBLE_ARRAY: {
                double[] doubleArray = this.currentDataTable.getDataTable().getDoubleArray(rowIndex, columnIndex);
                blockBuilder = elementType.createBlockBuilder(null, doubleArray.length);
                for (double element : doubleArray) {
                    elementType.writeDouble(blockBuilder, element);
                }
                break;
            }
            case STRING_ARRAY: {
                String[] stringArray = this.currentDataTable.getDataTable().getStringArray(rowIndex, columnIndex);
                blockBuilder = elementType.createBlockBuilder(null, stringArray.length);
                for (String element : stringArray) {
                    Slice slice = this.getUtf8Slice(element);
                    elementType.writeSlice(blockBuilder, slice, 0, slice.length());
                }
                break;
            }
            default: {
                throw new UnsupportedOperationException(String.format("Unexpected pinot type '%s'", columnType));
            }
        }
        return blockBuilder.build();
    }

    Slice getSlice(int rowIndex, int columnIndex) {
        Type trinoType = this.getType(columnIndex);
        if (trinoType instanceof VarcharType) {
            String field = this.currentDataTable.getDataTable().getString(rowIndex, columnIndex);
            return this.getUtf8Slice(field);
        }
        if (trinoType instanceof VarbinaryType) {
            return Slices.wrappedBuffer((byte[])PinotSegmentPageSource.toBytes(this.currentDataTable.getDataTable().getString(rowIndex, columnIndex)));
        }
        return Slices.EMPTY_SLICE;
    }

    static byte[] toBytes(String stringValue) {
        try {
            return Hex.decodeHex((char[])stringValue.toCharArray());
        }
        catch (DecoderException e) {
            throw new IllegalArgumentException("Value: " + stringValue + " is not Hex encoded", e);
        }
    }

    Slice getUtf8Slice(String value) {
        if (Strings.isNullOrEmpty((String)value)) {
            return Slices.EMPTY_SLICE;
        }
        return Slices.utf8Slice((String)value);
    }

    private int getEstimatedColumnSizeInBytes(DataSchema.ColumnDataType dataType) {
        if (dataType.isNumber()) {
            switch (dataType) {
                case LONG: {
                    return 8;
                }
                case FLOAT: {
                    return 4;
                }
                case DOUBLE: {
                    return 8;
                }
            }
            return 4;
        }
        return this.estimatedNonNumericColumnSize;
    }

    private static class PinotDataTableWithSize {
        DataTable dataTable;
        int estimatedSizeInBytes;

        PinotDataTableWithSize(DataTable dataTable, int estimatedSizeInBytes) {
            this.dataTable = dataTable;
            this.estimatedSizeInBytes = estimatedSizeInBytes;
        }

        DataTable getDataTable() {
            return this.dataTable;
        }

        int getEstimatedSizeInBytes() {
            return this.estimatedSizeInBytes;
        }
    }
}

