/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.iceberg;

import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import io.airlift.json.JsonCodec;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.trino.plugin.hive.HdfsEnvironment;
import io.trino.plugin.hive.util.ConfigurationUtils;
import io.trino.plugin.iceberg.CommitTaskData;
import io.trino.plugin.iceberg.IcebergColumnHandle;
import io.trino.plugin.iceberg.IcebergErrorCode;
import io.trino.plugin.iceberg.IcebergFileFormat;
import io.trino.plugin.iceberg.IcebergFileWriter;
import io.trino.plugin.iceberg.IcebergFileWriterFactory;
import io.trino.plugin.iceberg.IcebergSessionProperties;
import io.trino.plugin.iceberg.MetricsWrapper;
import io.trino.plugin.iceberg.PartitionData;
import io.trino.plugin.iceberg.PartitionTransforms;
import io.trino.plugin.iceberg.util.Timestamps;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.Page;
import io.trino.spi.PageIndexer;
import io.trino.spi.PageIndexerFactory;
import io.trino.spi.TrinoException;
import io.trino.spi.block.Block;
import io.trino.spi.connector.ConnectorPageSink;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.BooleanType;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.Decimals;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.RealType;
import io.trino.spi.type.SmallintType;
import io.trino.spi.type.TimeType;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.TimestampWithTimeZoneType;
import io.trino.spi.type.TinyintType;
import io.trino.spi.type.Type;
import io.trino.spi.type.UuidType;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.mapred.JobConf;
import org.apache.iceberg.MetricsConfig;
import org.apache.iceberg.PartitionField;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.Schema;
import org.apache.iceberg.StructLike;
import org.apache.iceberg.io.LocationProvider;
import org.apache.iceberg.transforms.Transform;

public class IcebergPageSink
implements ConnectorPageSink {
    private static final int MAX_PAGE_POSITIONS = 4096;
    private final int maxOpenWriters;
    private final Schema outputSchema;
    private final PartitionSpec partitionSpec;
    private final LocationProvider locationProvider;
    private final IcebergFileWriterFactory fileWriterFactory;
    private final HdfsEnvironment hdfsEnvironment;
    private final HdfsEnvironment.HdfsContext hdfsContext;
    private final JobConf jobConf;
    private final JsonCodec<CommitTaskData> jsonCodec;
    private final ConnectorSession session;
    private final IcebergFileFormat fileFormat;
    private final MetricsConfig metricsConfig;
    private final PagePartitioner pagePartitioner;
    private final long targetMaxFileSize;
    private final List<WriteContext> writers = new ArrayList<WriteContext>();
    private final List<WriteContext> closedWriters = new ArrayList<WriteContext>();
    private final Collection<Slice> commitTasks = new ArrayList<Slice>();
    private long writtenBytes;
    private long memoryUsage;
    private long validationCpuNanos;

    public IcebergPageSink(Schema outputSchema, PartitionSpec partitionSpec, LocationProvider locationProvider, IcebergFileWriterFactory fileWriterFactory, PageIndexerFactory pageIndexerFactory, HdfsEnvironment hdfsEnvironment, HdfsEnvironment.HdfsContext hdfsContext, List<IcebergColumnHandle> inputColumns, JsonCodec<CommitTaskData> jsonCodec, ConnectorSession session, IcebergFileFormat fileFormat, Map<String, String> storageProperties, int maxOpenWriters) {
        Objects.requireNonNull(inputColumns, "inputColumns is null");
        this.outputSchema = Objects.requireNonNull(outputSchema, "outputSchema is null");
        this.partitionSpec = Objects.requireNonNull(partitionSpec, "partitionSpec is null");
        this.locationProvider = Objects.requireNonNull(locationProvider, "locationProvider is null");
        this.fileWriterFactory = Objects.requireNonNull(fileWriterFactory, "fileWriterFactory is null");
        this.hdfsEnvironment = Objects.requireNonNull(hdfsEnvironment, "hdfsEnvironment is null");
        this.hdfsContext = Objects.requireNonNull(hdfsContext, "hdfsContext is null");
        this.jobConf = ConfigurationUtils.toJobConf((Configuration)hdfsEnvironment.getConfiguration(hdfsContext, new Path(locationProvider.newDataLocation("data-file"))));
        this.jsonCodec = Objects.requireNonNull(jsonCodec, "jsonCodec is null");
        this.session = Objects.requireNonNull(session, "session is null");
        this.fileFormat = Objects.requireNonNull(fileFormat, "fileFormat is null");
        this.metricsConfig = MetricsConfig.fromProperties(Objects.requireNonNull(storageProperties, "storageProperties is null"));
        this.maxOpenWriters = maxOpenWriters;
        this.pagePartitioner = new PagePartitioner(pageIndexerFactory, IcebergPageSink.toPartitionColumns(inputColumns, partitionSpec));
        this.targetMaxFileSize = IcebergSessionProperties.getTargetMaxFileSize(session);
    }

    public long getCompletedBytes() {
        return this.writtenBytes;
    }

    public long getMemoryUsage() {
        return this.memoryUsage;
    }

    public long getValidationCpuNanos() {
        return this.validationCpuNanos;
    }

    public CompletableFuture<?> appendPage(Page page) {
        this.hdfsEnvironment.doAs(this.session.getIdentity(), () -> this.doAppend(page));
        return NOT_BLOCKED;
    }

    public CompletableFuture<Collection<Slice>> finish() {
        for (WriteContext context : this.writers) {
            this.closeWriter(context);
        }
        this.writtenBytes = this.closedWriters.stream().mapToLong(writer -> writer.getWriter().getWrittenBytes()).sum();
        this.validationCpuNanos = this.closedWriters.stream().mapToLong(writer -> writer.getWriter().getValidationCpuNanos()).sum();
        return CompletableFuture.completedFuture(this.commitTasks);
    }

    public void abort() {
        Throwable error = null;
        for (WriteContext context : Iterables.concat(this.writers, this.closedWriters)) {
            try {
                if (context == null) continue;
                context.getWriter().rollback();
            }
            catch (Throwable t) {
                if (error == null) {
                    error = new RuntimeException("Exception during rollback");
                }
                error.addSuppressed(t);
            }
        }
        if (error != null) {
            throw error;
        }
    }

    private void doAppend(Page page) {
        while (page.getPositionCount() > 4096) {
            Page chunk = page.getRegion(0, 4096);
            page = page.getRegion(4096, page.getPositionCount() - 4096);
            this.writePage(chunk);
        }
        this.writePage(page);
    }

    private void writePage(Page page) {
        int index;
        int[] writerIndexes = this.getWriterIndexes(page);
        int[] sizes = new int[this.writers.size()];
        int[] nArray = writerIndexes;
        int n = nArray.length;
        for (int i = 0; i < n; ++i) {
            int n2 = index = nArray[i];
            sizes[n2] = sizes[n2] + 1;
        }
        int[][] writerPositions = new int[this.writers.size()][];
        int[] counts = new int[this.writers.size()];
        int position = 0;
        while (position < page.getPositionCount()) {
            index = writerIndexes[position];
            int count = counts[index];
            if (count == 0) {
                writerPositions[index] = new int[sizes[index]];
            }
            writerPositions[index][count] = position++;
            int n3 = index;
            counts[n3] = counts[n3] + 1;
        }
        for (int index2 = 0; index2 < writerPositions.length; ++index2) {
            int[] positions = writerPositions[index2];
            if (positions == null) continue;
            Page pageForWriter = page;
            if (positions.length != page.getPositionCount()) {
                Verify.verify((positions.length == counts[index2] ? 1 : 0) != 0);
                pageForWriter = pageForWriter.getPositions(positions, 0, positions.length);
            }
            IcebergFileWriter writer = this.writers.get(index2).getWriter();
            long currentWritten = writer.getWrittenBytes();
            long currentMemory = writer.getMemoryUsage();
            writer.appendRows(pageForWriter);
            this.writtenBytes += writer.getWrittenBytes() - currentWritten;
            this.memoryUsage += writer.getMemoryUsage() - currentMemory;
        }
    }

    private int[] getWriterIndexes(Page page) {
        int[] writerIndexes = this.pagePartitioner.partitionPage(page);
        if (this.pagePartitioner.getMaxIndex() >= this.maxOpenWriters) {
            throw new TrinoException((ErrorCodeSupplier)IcebergErrorCode.ICEBERG_TOO_MANY_OPEN_PARTITIONS, String.format("Exceeded limit of %s open writers for partitions", this.maxOpenWriters));
        }
        while (this.writers.size() <= this.pagePartitioner.getMaxIndex()) {
            this.writers.add(null);
        }
        for (int position = 0; position < page.getPositionCount(); ++position) {
            int writerIndex = writerIndexes[position];
            WriteContext writer = this.writers.get(writerIndex);
            if (writer != null) {
                if (writer.getWrittenBytes() <= this.targetMaxFileSize) continue;
                this.closeWriter(writer);
            }
            Optional<PartitionData> partitionData = IcebergPageSink.getPartitionData(this.pagePartitioner.getColumns(), page, position);
            writer = this.createWriter(partitionData);
            this.writers.set(writerIndex, writer);
        }
        Verify.verify((this.writers.size() == this.pagePartitioner.getMaxIndex() + 1 ? 1 : 0) != 0);
        Verify.verify((!this.writers.contains(null) ? 1 : 0) != 0);
        return writerIndexes;
    }

    private void closeWriter(WriteContext writeContext) {
        long currentWritten = writeContext.getWriter().getWrittenBytes();
        long currentMemory = writeContext.getWriter().getMemoryUsage();
        writeContext.getWriter().commit();
        this.writtenBytes += writeContext.getWriter().getWrittenBytes() - currentWritten;
        this.memoryUsage += writeContext.getWriter().getMemoryUsage() - currentMemory;
        CommitTaskData task = new CommitTaskData(writeContext.getPath().toString(), writeContext.getWriter().getWrittenBytes(), new MetricsWrapper(writeContext.getWriter().getMetrics()), writeContext.getPartitionData().map(PartitionData::toJson));
        this.commitTasks.add(Slices.wrappedBuffer((byte[])this.jsonCodec.toJsonBytes((Object)task)));
        this.closedWriters.add(writeContext);
    }

    private WriteContext createWriter(Optional<PartitionData> partitionData) {
        String fileName = this.fileFormat.toIceberg().addExtension(this.session.getQueryId() + "-" + UUID.randomUUID());
        Path outputPath = partitionData.map(partition -> new Path(this.locationProvider.newDataLocation(this.partitionSpec, (StructLike)partition, fileName))).orElse(new Path(this.locationProvider.newDataLocation(fileName)));
        IcebergFileWriter writer = this.fileWriterFactory.createFileWriter(outputPath, this.outputSchema, this.jobConf, this.session, this.hdfsContext, this.fileFormat, this.metricsConfig);
        return new WriteContext(writer, outputPath, partitionData);
    }

    private static Optional<PartitionData> getPartitionData(List<PartitionColumn> columns, Page page, int position) {
        if (columns.isEmpty()) {
            return Optional.empty();
        }
        Object[] values = new Object[columns.size()];
        for (int i = 0; i < columns.size(); ++i) {
            PartitionColumn column = columns.get(i);
            Block block = page.getBlock(column.getSourceChannel());
            Type type = column.getSourceType();
            Object value = IcebergPageSink.getIcebergValue(block, position, type);
            values[i] = IcebergPageSink.applyTransform(column.getField().transform(), value);
        }
        return Optional.of(new PartitionData(values));
    }

    private static Object applyTransform(Transform<?, ?> transform, Object value) {
        return transform.apply(value);
    }

    public static Object getIcebergValue(Block block, int position, Type type) {
        if (block.isNull(position)) {
            return null;
        }
        if (type instanceof BigintType) {
            return type.getLong(block, position);
        }
        if (type instanceof IntegerType || type instanceof SmallintType || type instanceof TinyintType || type instanceof DateType) {
            return Math.toIntExact(type.getLong(block, position));
        }
        if (type instanceof BooleanType) {
            return type.getBoolean(block, position);
        }
        if (type instanceof DecimalType) {
            return Decimals.readBigDecimal((DecimalType)((DecimalType)type), (Block)block, (int)position);
        }
        if (type instanceof RealType) {
            return Float.valueOf(Float.intBitsToFloat(Math.toIntExact(type.getLong(block, position))));
        }
        if (type instanceof DoubleType) {
            return type.getDouble(block, position);
        }
        if (type.equals(TimeType.TIME_MICROS)) {
            return type.getLong(block, position) / 1000000L;
        }
        if (type.equals(TimestampType.TIMESTAMP_MICROS)) {
            return type.getLong(block, position);
        }
        if (type.equals(TimestampWithTimeZoneType.TIMESTAMP_TZ_MICROS)) {
            return Timestamps.timestampTzToMicros(Timestamps.getTimestampTz(block, position));
        }
        if (type instanceof VarbinaryType) {
            return type.getSlice(block, position).getBytes();
        }
        if (type instanceof VarcharType) {
            return type.getSlice(block, position).toStringUtf8();
        }
        if (type instanceof UuidType) {
            return UuidType.trinoUuidToJavaUuid((Slice)type.getSlice(block, position));
        }
        throw new UnsupportedOperationException("Type not supported as partition column: " + type.getDisplayName());
    }

    private static List<PartitionColumn> toPartitionColumns(List<IcebergColumnHandle> handles, PartitionSpec partitionSpec) {
        HashMap<Integer, Integer> idChannels = new HashMap<Integer, Integer>();
        for (int i = 0; i < handles.size(); ++i) {
            idChannels.put(handles.get(i).getId(), i);
        }
        return (List)partitionSpec.fields().stream().map(field -> {
            Integer channel = (Integer)idChannels.get(field.sourceId());
            Preconditions.checkArgument((channel != null ? 1 : 0) != 0, (String)"partition field not found: %s", (Object)field);
            Type inputType = ((IcebergColumnHandle)handles.get(channel)).getType();
            PartitionTransforms.ColumnTransform transform = PartitionTransforms.getColumnTransform(field, inputType);
            return new PartitionColumn((PartitionField)field, channel, inputType, transform.getType(), transform.getBlockTransform());
        }).collect(ImmutableList.toImmutableList());
    }

    private static class PartitionColumn {
        private final PartitionField field;
        private final int sourceChannel;
        private final Type sourceType;
        private final Type resultType;
        private final Function<Block, Block> blockTransform;

        public PartitionColumn(PartitionField field, int sourceChannel, Type sourceType, Type resultType, Function<Block, Block> blockTransform) {
            this.field = Objects.requireNonNull(field, "field is null");
            this.sourceChannel = sourceChannel;
            this.sourceType = Objects.requireNonNull(sourceType, "sourceType is null");
            this.resultType = Objects.requireNonNull(resultType, "resultType is null");
            this.blockTransform = Objects.requireNonNull(blockTransform, "blockTransform is null");
        }

        public PartitionField getField() {
            return this.field;
        }

        public int getSourceChannel() {
            return this.sourceChannel;
        }

        public Type getSourceType() {
            return this.sourceType;
        }

        public Type getResultType() {
            return this.resultType;
        }

        public Function<Block, Block> getBlockTransform() {
            return this.blockTransform;
        }
    }

    private static class PagePartitioner {
        private final PageIndexer pageIndexer;
        private final List<PartitionColumn> columns;

        public PagePartitioner(PageIndexerFactory pageIndexerFactory, List<PartitionColumn> columns) {
            this.pageIndexer = pageIndexerFactory.createPageIndexer((List)columns.stream().map(PartitionColumn::getResultType).collect(ImmutableList.toImmutableList()));
            this.columns = ImmutableList.copyOf(columns);
        }

        public int[] partitionPage(Page page) {
            Block[] blocks = new Block[this.columns.size()];
            for (int i = 0; i < this.columns.size(); ++i) {
                PartitionColumn column = this.columns.get(i);
                Block block = page.getBlock(column.getSourceChannel());
                blocks[i] = column.getBlockTransform().apply(block);
            }
            Page transformed = new Page(page.getPositionCount(), blocks);
            return this.pageIndexer.indexPage(transformed);
        }

        public int getMaxIndex() {
            return this.pageIndexer.getMaxIndex();
        }

        public List<PartitionColumn> getColumns() {
            return this.columns;
        }
    }

    private static class WriteContext {
        private final IcebergFileWriter writer;
        private final Path path;
        private final Optional<PartitionData> partitionData;

        public WriteContext(IcebergFileWriter writer, Path path, Optional<PartitionData> partitionData) {
            this.writer = Objects.requireNonNull(writer, "writer is null");
            this.path = Objects.requireNonNull(path, "path is null");
            this.partitionData = Objects.requireNonNull(partitionData, "partitionData is null");
        }

        public IcebergFileWriter getWriter() {
            return this.writer;
        }

        public Path getPath() {
            return this.path;
        }

        public Optional<PartitionData> getPartitionData() {
            return this.partitionData;
        }

        public long getWrittenBytes() {
            return this.writer.getWrittenBytes();
        }
    }
}

