/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.iceberg;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.base.Suppliers;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterators;
import com.google.common.collect.Sets;
import com.google.common.collect.Streams;
import io.airlift.units.DataSize;
import io.airlift.units.Duration;
import io.trino.plugin.iceberg.ExpressionConverter;
import io.trino.plugin.iceberg.IcebergColumnHandle;
import io.trino.plugin.iceberg.IcebergFileFormat;
import io.trino.plugin.iceberg.IcebergSplit;
import io.trino.plugin.iceberg.IcebergTableHandle;
import io.trino.plugin.iceberg.IcebergTypes;
import io.trino.plugin.iceberg.IcebergUtil;
import io.trino.plugin.iceberg.TypeConverter;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.HostAddress;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ColumnHandle;
import io.trino.spi.connector.ConnectorPartitionHandle;
import io.trino.spi.connector.ConnectorSplitSource;
import io.trino.spi.connector.Constraint;
import io.trino.spi.connector.DynamicFilter;
import io.trino.spi.predicate.Domain;
import io.trino.spi.predicate.NullableValue;
import io.trino.spi.predicate.Range;
import io.trino.spi.predicate.TupleDomain;
import io.trino.spi.predicate.ValueSet;
import io.trino.spi.type.TypeManager;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.function.Supplier;
import javax.annotation.Nullable;
import org.apache.iceberg.CombinedScanTask;
import org.apache.iceberg.DataFile;
import org.apache.iceberg.FileScanTask;
import org.apache.iceberg.Schema;
import org.apache.iceberg.TableScan;
import org.apache.iceberg.expressions.Expression;
import org.apache.iceberg.io.CloseableIterable;
import org.apache.iceberg.types.Conversions;
import org.apache.iceberg.types.Type;

public class IcebergSplitSource
implements ConnectorSplitSource {
    private static final ConnectorSplitSource.ConnectorSplitBatch EMPTY_BATCH = new ConnectorSplitSource.ConnectorSplitBatch((List)ImmutableList.of(), false);
    private static final ConnectorSplitSource.ConnectorSplitBatch NO_MORE_SPLITS_BATCH = new ConnectorSplitSource.ConnectorSplitBatch((List)ImmutableList.of(), true);
    private final IcebergTableHandle tableHandle;
    private final TableScan tableScan;
    private final Optional<Long> maxScannedFileSizeInBytes;
    private final Map<Integer, Type.PrimitiveType> fieldIdToType;
    private final DynamicFilter dynamicFilter;
    private final long dynamicFilteringWaitTimeoutMillis;
    private final Stopwatch dynamicFilterWaitStopwatch;
    private final Constraint constraint;
    private final TypeManager typeManager;
    private CloseableIterable<CombinedScanTask> combinedScanIterable;
    private Iterator<FileScanTask> fileScanIterator;
    private TupleDomain<IcebergColumnHandle> pushedDownDynamicFilterPredicate;
    private final boolean recordScannedFiles;
    private final ImmutableSet.Builder<DataFile> scannedFiles = ImmutableSet.builder();

    public IcebergSplitSource(IcebergTableHandle tableHandle, TableScan tableScan, Optional<DataSize> maxScannedFileSize, DynamicFilter dynamicFilter, Duration dynamicFilteringWaitTimeout, Constraint constraint, TypeManager typeManager, boolean recordScannedFiles) {
        this.tableHandle = Objects.requireNonNull(tableHandle, "tableHandle is null");
        this.tableScan = Objects.requireNonNull(tableScan, "tableScan is null");
        this.maxScannedFileSizeInBytes = Objects.requireNonNull(maxScannedFileSize, "maxScannedFileSize is null").map(DataSize::toBytes);
        this.fieldIdToType = IcebergUtil.primitiveFieldTypes(tableScan.schema());
        this.dynamicFilter = Objects.requireNonNull(dynamicFilter, "dynamicFilter is null");
        this.dynamicFilteringWaitTimeoutMillis = Objects.requireNonNull(dynamicFilteringWaitTimeout, "dynamicFilteringWaitTimeout is null").toMillis();
        this.dynamicFilterWaitStopwatch = Stopwatch.createStarted();
        this.constraint = Objects.requireNonNull(constraint, "constraint is null");
        this.typeManager = Objects.requireNonNull(typeManager, "typeManager is null");
        this.recordScannedFiles = recordScannedFiles;
    }

    public CompletableFuture<ConnectorSplitSource.ConnectorSplitBatch> getNextBatch(ConnectorPartitionHandle partitionHandle, int maxSize) {
        TupleDomain dynamicFilterPredicate;
        long timeLeft = this.dynamicFilteringWaitTimeoutMillis - this.dynamicFilterWaitStopwatch.elapsed(TimeUnit.MILLISECONDS);
        if (this.dynamicFilter.isAwaitable() && timeLeft > 0L) {
            return ((CompletableFuture)this.dynamicFilter.isBlocked().thenApply(ignored -> EMPTY_BATCH)).completeOnTimeout(EMPTY_BATCH, timeLeft, TimeUnit.MILLISECONDS);
        }
        if (this.combinedScanIterable == null) {
            TupleDomain effectivePredicate;
            this.pushedDownDynamicFilterPredicate = this.dynamicFilter.getCurrentPredicate().transformKeys(IcebergColumnHandle.class::cast);
            TupleDomain fullPredicate = this.tableHandle.getUnenforcedPredicate().intersect(this.pushedDownDynamicFilterPredicate);
            TupleDomain simplifiedPredicate = fullPredicate.simplify(1000);
            if (!simplifiedPredicate.equals((Object)fullPredicate)) {
                this.pushedDownDynamicFilterPredicate = TupleDomain.all();
            }
            if ((effectivePredicate = this.tableHandle.getEnforcedPredicate().intersect(simplifiedPredicate)).isNone()) {
                this.finish();
                return CompletableFuture.completedFuture(NO_MORE_SPLITS_BATCH);
            }
            Expression filterExpression = ExpressionConverter.toIcebergExpression((TupleDomain<IcebergColumnHandle>)effectivePredicate);
            this.combinedScanIterable = this.tableScan.filter(filterExpression).includeColumnStats().planTasks();
            this.fileScanIterator = Streams.stream(this.combinedScanIterable).map(CombinedScanTask::files).flatMap(Collection::stream).iterator();
        }
        if ((dynamicFilterPredicate = this.dynamicFilter.getCurrentPredicate().transformKeys(IcebergColumnHandle.class::cast)).isNone()) {
            this.finish();
            return CompletableFuture.completedFuture(NO_MORE_SPLITS_BATCH);
        }
        Iterator fileScanTasks = Iterators.limit(this.fileScanIterator, (int)maxSize);
        ImmutableList.Builder splits = ImmutableList.builder();
        while (fileScanTasks.hasNext()) {
            FileScanTask scanTask = (FileScanTask)fileScanTasks.next();
            if (!scanTask.deletes().isEmpty()) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Iceberg tables with delete files are not supported: " + this.tableHandle.getSchemaTableName());
            }
            if (this.maxScannedFileSizeInBytes.isPresent() && scanTask.file().fileSizeInBytes() > this.maxScannedFileSizeInBytes.get()) continue;
            IcebergSplit icebergSplit = IcebergSplitSource.toIcebergSplit(scanTask);
            Schema fileSchema = scanTask.spec().schema();
            Set identityPartitionColumns = (Set)icebergSplit.getPartitionKeys().keySet().stream().map(fieldId -> IcebergUtil.getColumnHandle(fileSchema.findField(fieldId.intValue()), this.typeManager)).collect(ImmutableSet.toImmutableSet());
            com.google.common.base.Supplier partitionValues = Suppliers.memoize(() -> {
                HashMap<IcebergColumnHandle, NullableValue> bindings = new HashMap<IcebergColumnHandle, NullableValue>();
                for (IcebergColumnHandle partitionColumn : identityPartitionColumns) {
                    Object partitionValue = IcebergUtil.deserializePartitionValue(partitionColumn.getType(), icebergSplit.getPartitionKeys().get(partitionColumn.getId()).orElse(null), partitionColumn.getName());
                    NullableValue bindingValue = new NullableValue(partitionColumn.getType(), partitionValue);
                    bindings.put(partitionColumn, bindingValue);
                }
                return bindings;
            });
            if (!dynamicFilterPredicate.isAll() && !dynamicFilterPredicate.equals(this.pushedDownDynamicFilterPredicate) && (!IcebergSplitSource.partitionMatchesPredicate(identityPartitionColumns, (Supplier<Map<ColumnHandle, NullableValue>>)partitionValues, (TupleDomain<IcebergColumnHandle>)dynamicFilterPredicate) || !IcebergSplitSource.fileMatchesPredicate(this.fieldIdToType, (TupleDomain<IcebergColumnHandle>)dynamicFilterPredicate, scanTask.file().lowerBounds(), scanTask.file().upperBounds(), scanTask.file().nullValueCounts())) || !IcebergSplitSource.partitionMatchesConstraint(identityPartitionColumns, (Supplier<Map<ColumnHandle, NullableValue>>)partitionValues, this.constraint)) continue;
            if (this.recordScannedFiles) {
                this.scannedFiles.add((Object)scanTask.file());
            }
            splits.add((Object)icebergSplit);
        }
        return CompletableFuture.completedFuture(new ConnectorSplitSource.ConnectorSplitBatch((List)splits.build(), this.isFinished()));
    }

    private void finish() {
        this.close();
        this.combinedScanIterable = CloseableIterable.empty();
        this.fileScanIterator = Collections.emptyIterator();
    }

    public boolean isFinished() {
        return this.fileScanIterator != null && !this.fileScanIterator.hasNext();
    }

    public Optional<List<Object>> getTableExecuteSplitsInfo() {
        Preconditions.checkState((boolean)this.isFinished(), (Object)"Split source must be finished before TableExecuteSplitsInfo is read");
        if (!this.recordScannedFiles) {
            return Optional.empty();
        }
        return Optional.of(ImmutableList.copyOf((Collection)this.scannedFiles.build()));
    }

    public void close() {
        if (this.combinedScanIterable != null) {
            try {
                this.combinedScanIterable.close();
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
    }

    @VisibleForTesting
    static boolean fileMatchesPredicate(Map<Integer, Type.PrimitiveType> primitiveTypeForFieldId, TupleDomain<IcebergColumnHandle> dynamicFilterPredicate, @Nullable Map<Integer, ByteBuffer> lowerBounds, @Nullable Map<Integer, ByteBuffer> upperBounds, @Nullable Map<Integer, Long> nullValueCounts) {
        if (dynamicFilterPredicate.isNone()) {
            return false;
        }
        Map domains = (Map)dynamicFilterPredicate.getDomains().orElseThrow();
        for (Map.Entry domainEntry : domains.entrySet()) {
            Long nullValueCount;
            IcebergColumnHandle column = (IcebergColumnHandle)domainEntry.getKey();
            Domain domain = (Domain)domainEntry.getValue();
            int fieldId = column.getId();
            boolean mayContainNulls = nullValueCounts == null ? true : (nullValueCount = nullValueCounts.get(fieldId)) == null || nullValueCount > 0L;
            Type type = (Type)primitiveTypeForFieldId.get(fieldId);
            Domain statisticsDomain = IcebergSplitSource.domainForStatistics(column.getType(), lowerBounds == null ? null : Conversions.fromByteBuffer((Type)type, (ByteBuffer)lowerBounds.get(fieldId)), upperBounds == null ? null : Conversions.fromByteBuffer((Type)type, (ByteBuffer)upperBounds.get(fieldId)), mayContainNulls);
            if (domain.overlaps(statisticsDomain)) continue;
            return false;
        }
        return true;
    }

    private static Domain domainForStatistics(io.trino.spi.type.Type type, @Nullable Object lowerBound, @Nullable Object upperBound, boolean mayContainNulls) {
        Type icebergType = TypeConverter.toIcebergType(type);
        if (lowerBound == null && upperBound == null) {
            return Domain.create((ValueSet)ValueSet.all((io.trino.spi.type.Type)type), (boolean)mayContainNulls);
        }
        Range statisticsRange = lowerBound != null && upperBound != null ? Range.range((io.trino.spi.type.Type)type, (Object)IcebergTypes.convertIcebergValueToTrino(icebergType, lowerBound), (boolean)true, (Object)IcebergTypes.convertIcebergValueToTrino(icebergType, upperBound), (boolean)true) : (upperBound != null ? Range.lessThanOrEqual((io.trino.spi.type.Type)type, (Object)IcebergTypes.convertIcebergValueToTrino(icebergType, upperBound)) : Range.greaterThanOrEqual((io.trino.spi.type.Type)type, (Object)IcebergTypes.convertIcebergValueToTrino(icebergType, lowerBound)));
        return Domain.create((ValueSet)ValueSet.ofRanges((Range)statisticsRange, (Range[])new Range[0]), (boolean)mayContainNulls);
    }

    static boolean partitionMatchesConstraint(Set<IcebergColumnHandle> identityPartitionColumns, Supplier<Map<ColumnHandle, NullableValue>> partitionValues, Constraint constraint) {
        Verify.verify((boolean)constraint.getSummary().isAll());
        if (constraint.predicate().isEmpty() || Sets.intersection((Set)((Set)constraint.getPredicateColumns().orElseThrow()), identityPartitionColumns).isEmpty()) {
            return true;
        }
        return ((Predicate)constraint.predicate().get()).test(partitionValues.get());
    }

    @VisibleForTesting
    static boolean partitionMatchesPredicate(Set<IcebergColumnHandle> identityPartitionColumns, Supplier<Map<ColumnHandle, NullableValue>> partitionValues, TupleDomain<IcebergColumnHandle> dynamicFilterPredicate) {
        if (dynamicFilterPredicate.isNone()) {
            return false;
        }
        Map domains = (Map)dynamicFilterPredicate.getDomains().orElseThrow();
        for (IcebergColumnHandle partitionColumn : identityPartitionColumns) {
            Domain allowedDomain = (Domain)domains.get(partitionColumn);
            if (allowedDomain == null || allowedDomain.includesNullableValue(partitionValues.get().get(partitionColumn).getValue())) continue;
            return false;
        }
        return true;
    }

    private static IcebergSplit toIcebergSplit(FileScanTask task) {
        return new IcebergSplit(task.file().path().toString(), task.start(), task.length(), task.file().fileSizeInBytes(), IcebergFileFormat.fromIceberg(task.file().format()), (List<HostAddress>)ImmutableList.of(), IcebergUtil.getPartitionKeys(task));
    }
}

