/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.iceberg;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceUtf8;
import io.airlift.slice.Slices;
import io.trino.plugin.hive.metastore.HiveMetastore;
import io.trino.plugin.hive.metastore.Table;
import io.trino.plugin.iceberg.ColumnIdentity;
import io.trino.plugin.iceberg.IcebergColumnHandle;
import io.trino.plugin.iceberg.IcebergErrorCode;
import io.trino.plugin.iceberg.IcebergFileFormat;
import io.trino.plugin.iceberg.IcebergTableProperties;
import io.trino.plugin.iceberg.PartitionFields;
import io.trino.plugin.iceberg.TrinoCatalog;
import io.trino.plugin.iceberg.TypeConverter;
import io.trino.plugin.iceberg.catalog.IcebergTableOperations;
import io.trino.plugin.iceberg.catalog.IcebergTableOperationsProvider;
import io.trino.plugin.iceberg.util.Timestamps;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ColumnMetadata;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.connector.ConnectorTableMetadata;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.BooleanType;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.Decimals;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.Int128;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.RealType;
import io.trino.spi.type.TimeType;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.TimestampWithTimeZoneType;
import io.trino.spi.type.TypeManager;
import io.trino.spi.type.UuidType;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.apache.iceberg.BaseTable;
import org.apache.iceberg.FileFormat;
import org.apache.iceberg.FileScanTask;
import org.apache.iceberg.HistoryEntry;
import org.apache.iceberg.LocationProviders;
import org.apache.iceberg.PartitionField;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.Schema;
import org.apache.iceberg.StructLike;
import org.apache.iceberg.TableMetadata;
import org.apache.iceberg.TableOperations;
import org.apache.iceberg.Transaction;
import org.apache.iceberg.io.LocationProvider;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.TypeUtil;
import org.apache.iceberg.types.Types;

public final class IcebergUtil {
    private static final Pattern SIMPLE_NAME = Pattern.compile("[a-z][a-z0-9]*");

    private IcebergUtil() {
    }

    public static boolean isIcebergTable(Table table) {
        return "iceberg".equalsIgnoreCase((String)table.getParameters().get("table_type"));
    }

    public static org.apache.iceberg.Table loadIcebergTable(HiveMetastore metastore, IcebergTableOperationsProvider tableOperationsProvider, ConnectorSession session, SchemaTableName table) {
        IcebergTableOperations operations = tableOperationsProvider.createTableOperations(metastore, session, table.getSchemaName(), table.getTableName(), Optional.empty(), Optional.empty());
        return new BaseTable((TableOperations)operations, IcebergUtil.quotedTableName(table));
    }

    public static org.apache.iceberg.Table getIcebergTableWithMetadata(HiveMetastore metastore, IcebergTableOperationsProvider tableOperationsProvider, ConnectorSession session, SchemaTableName table, TableMetadata tableMetadata) {
        IcebergTableOperations operations = tableOperationsProvider.createTableOperations(metastore, session, table.getSchemaName(), table.getTableName(), Optional.empty(), Optional.empty());
        operations.initializeFromMetadata(tableMetadata);
        return new BaseTable((TableOperations)operations, IcebergUtil.quotedTableName(table));
    }

    public static long resolveSnapshotId(org.apache.iceberg.Table table, long snapshotId) {
        if (table.snapshot(snapshotId) != null) {
            return snapshotId;
        }
        return Lists.reverse((List)table.history()).stream().filter(entry -> entry.timestampMillis() <= snapshotId).map(HistoryEntry::snapshotId).findFirst().orElseThrow(() -> new TrinoException((ErrorCodeSupplier)IcebergErrorCode.ICEBERG_INVALID_SNAPSHOT_ID, String.format("Invalid snapshot [%s] for table: %s", snapshotId, table)));
    }

    public static List<IcebergColumnHandle> getColumns(Schema schema, TypeManager typeManager) {
        return (List)schema.columns().stream().map(column -> IcebergUtil.getColumnHandle(column, typeManager)).collect(ImmutableList.toImmutableList());
    }

    public static IcebergColumnHandle getColumnHandle(Types.NestedField column, TypeManager typeManager) {
        io.trino.spi.type.Type type = TypeConverter.toTrinoType(column.type(), typeManager);
        return new IcebergColumnHandle(ColumnIdentity.createColumnIdentity(column), type, (List<Integer>)ImmutableList.of(), type, Optional.ofNullable(column.doc()));
    }

    public static Map<PartitionField, Integer> getIdentityPartitions(PartitionSpec partitionSpec) {
        ImmutableMap.Builder columns = ImmutableMap.builder();
        for (int i = 0; i < partitionSpec.fields().size(); ++i) {
            PartitionField field = (PartitionField)partitionSpec.fields().get(i);
            if (!field.transform().toString().equals("identity")) continue;
            columns.put((Object)field, (Object)i);
        }
        return columns.buildOrThrow();
    }

    public static Map<Integer, Type.PrimitiveType> primitiveFieldTypes(Schema schema) {
        return (Map)IcebergUtil.primitiveFieldTypes(schema.columns()).collect(ImmutableMap.toImmutableMap(Map.Entry::getKey, Map.Entry::getValue));
    }

    private static Stream<Map.Entry<Integer, Type.PrimitiveType>> primitiveFieldTypes(List<Types.NestedField> nestedFields) {
        return nestedFields.stream().flatMap(IcebergUtil::primitiveFieldTypes);
    }

    private static Stream<Map.Entry<Integer, Type.PrimitiveType>> primitiveFieldTypes(Types.NestedField nestedField) {
        Type fieldType = nestedField.type();
        if (fieldType.isPrimitiveType()) {
            return Stream.of(Map.entry(nestedField.fieldId(), fieldType.asPrimitiveType()));
        }
        if (fieldType.isNestedType()) {
            return IcebergUtil.primitiveFieldTypes(fieldType.asNestedType().fields());
        }
        throw new IllegalStateException("Unsupported field type: " + nestedField);
    }

    public static IcebergFileFormat getFileFormat(org.apache.iceberg.Table table) {
        return IcebergFileFormat.fromIceberg(FileFormat.valueOf((String)table.properties().getOrDefault("write.format.default", "parquet").toUpperCase(Locale.ENGLISH)));
    }

    public static Optional<String> getTableComment(org.apache.iceberg.Table table) {
        return Optional.ofNullable((String)table.properties().get("comment"));
    }

    private static String quotedTableName(SchemaTableName name) {
        return IcebergUtil.quotedName(name.getSchemaName()) + "." + IcebergUtil.quotedName(name.getTableName());
    }

    private static String quotedName(String name) {
        if (SIMPLE_NAME.matcher(name).matches()) {
            return name;
        }
        return "\"" + name.replace("\"", "\"\"") + "\"";
    }

    public static Object deserializePartitionValue(io.trino.spi.type.Type type, String valueString, String name) {
        if (valueString == null) {
            return null;
        }
        try {
            if (type.equals(BooleanType.BOOLEAN)) {
                if (valueString.equalsIgnoreCase("true")) {
                    return true;
                }
                if (valueString.equalsIgnoreCase("false")) {
                    return false;
                }
                throw new IllegalArgumentException();
            }
            if (type.equals(IntegerType.INTEGER)) {
                return Long.parseLong(valueString);
            }
            if (type.equals(BigintType.BIGINT)) {
                return Long.parseLong(valueString);
            }
            if (type.equals(RealType.REAL)) {
                return (long)Float.floatToRawIntBits(Float.parseFloat(valueString));
            }
            if (type.equals(DoubleType.DOUBLE)) {
                return Double.parseDouble(valueString);
            }
            if (type.equals(DateType.DATE)) {
                return Long.parseLong(valueString);
            }
            if (type.equals(TimeType.TIME_MICROS)) {
                return Long.parseLong(valueString) * 1000000L;
            }
            if (type.equals(TimestampType.TIMESTAMP_MICROS)) {
                return Long.parseLong(valueString);
            }
            if (type.equals(TimestampWithTimeZoneType.TIMESTAMP_TZ_MICROS)) {
                return Timestamps.timestampTzFromMicros(Long.parseLong(valueString));
            }
            if (type instanceof VarcharType) {
                Slice value = Slices.utf8Slice((String)valueString);
                VarcharType varcharType = (VarcharType)type;
                if (!varcharType.isUnbounded() && SliceUtf8.countCodePoints((Slice)value) > varcharType.getBoundedLength()) {
                    throw new IllegalArgumentException();
                }
                return value;
            }
            if (type.equals(VarbinaryType.VARBINARY)) {
                return Slices.wrappedBuffer((byte[])Base64.getDecoder().decode(valueString));
            }
            if (type.equals(UuidType.UUID)) {
                return UuidType.javaUuidToTrinoUuid((UUID)UUID.fromString(valueString));
            }
            if (Decimals.isShortDecimal((io.trino.spi.type.Type)type) || Decimals.isLongDecimal((io.trino.spi.type.Type)type)) {
                DecimalType decimalType = (DecimalType)type;
                BigDecimal decimal = new BigDecimal(valueString);
                if ((decimal = decimal.setScale(decimalType.getScale(), 7)).precision() > decimalType.getPrecision()) {
                    throw new IllegalArgumentException();
                }
                BigInteger unscaledValue = decimal.unscaledValue();
                return Decimals.isShortDecimal((io.trino.spi.type.Type)type) ? Long.valueOf(unscaledValue.longValue()) : Int128.valueOf((BigInteger)unscaledValue);
            }
        }
        catch (IllegalArgumentException e) {
            throw new TrinoException((ErrorCodeSupplier)IcebergErrorCode.ICEBERG_INVALID_PARTITION_VALUE, String.format("Invalid partition value '%s' for %s partition key: %s", valueString, type.getDisplayName(), name));
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "Invalid partition type " + type.toString());
    }

    public static Map<Integer, Optional<String>> getPartitionKeys(FileScanTask scanTask) {
        return IcebergUtil.getPartitionKeys(scanTask.file().partition(), scanTask.spec());
    }

    public static Map<Integer, Optional<String>> getPartitionKeys(StructLike partition, PartitionSpec spec) {
        Map<PartitionField, Integer> fieldToIndex = IcebergUtil.getIdentityPartitions(spec);
        ImmutableMap.Builder partitionKeys = ImmutableMap.builder();
        fieldToIndex.forEach((field, index) -> {
            int id = field.sourceId();
            Type type = spec.schema().findType(id);
            Class javaClass = type.typeId().javaClass();
            Object value = partition.get(index.intValue(), javaClass);
            if (value == null) {
                partitionKeys.put((Object)id, Optional.empty());
            } else {
                String partitionValue = type.typeId() == Type.TypeID.FIXED || type.typeId() == Type.TypeID.BINARY ? Base64.getEncoder().encodeToString(((ByteBuffer)value).array()) : value.toString();
                partitionKeys.put((Object)id, Optional.of(partitionValue));
            }
        });
        return partitionKeys.buildOrThrow();
    }

    public static LocationProvider getLocationProvider(SchemaTableName schemaTableName, String tableLocation, Map<String, String> storageProperties) {
        if (storageProperties.containsKey("write.location-provider.impl")) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Table " + schemaTableName + " specifies " + storageProperties.get("write.location-provider.impl") + " as a location provider. Writing to Iceberg tables with custom location provider is not supported.");
        }
        return LocationProviders.locationsFor((String)tableLocation, storageProperties);
    }

    public static Schema toIcebergSchema(List<ColumnMetadata> columns) {
        ArrayList<Types.NestedField> icebergColumns = new ArrayList<Types.NestedField>();
        for (ColumnMetadata column : columns) {
            if (column.isHidden()) continue;
            int index = icebergColumns.size();
            Type type = TypeConverter.toIcebergType(column.getType());
            Types.NestedField field = Types.NestedField.of((int)index, (boolean)column.isNullable(), (String)column.getName(), (Type)type, (String)column.getComment());
            icebergColumns.add(field);
        }
        Types.StructType icebergSchema = Types.StructType.of(icebergColumns);
        AtomicInteger nextFieldId = new AtomicInteger(1);
        icebergSchema = TypeUtil.assignFreshIds((Type)icebergSchema, nextFieldId::getAndIncrement);
        return new Schema(icebergSchema.asStructType().fields());
    }

    public static Transaction newCreateTableTransaction(TrinoCatalog catalog, ConnectorTableMetadata tableMetadata, ConnectorSession session) {
        SchemaTableName schemaTableName = tableMetadata.getTable();
        Schema schema = IcebergUtil.toIcebergSchema(tableMetadata.getColumns());
        PartitionSpec partitionSpec = PartitionFields.parsePartitionFields(schema, IcebergTableProperties.getPartitioning(tableMetadata.getProperties()));
        String targetPath = IcebergTableProperties.getTableLocation(tableMetadata.getProperties()).orElseGet(() -> catalog.defaultTableLocation(session, schemaTableName));
        ImmutableMap.Builder propertiesBuilder = ImmutableMap.builderWithExpectedSize((int)2);
        IcebergFileFormat fileFormat = IcebergTableProperties.getFileFormat(tableMetadata.getProperties());
        propertiesBuilder.put((Object)"write.format.default", (Object)fileFormat.toIceberg().toString());
        if (tableMetadata.getComment().isPresent()) {
            propertiesBuilder.put((Object)"comment", (Object)((String)tableMetadata.getComment().get()));
        }
        return catalog.newCreateTableTransaction(session, schemaTableName, schema, partitionSpec, targetPath, (Map<String, String>)propertiesBuilder.buildOrThrow());
    }
}

