/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.iceberg;

import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import io.trino.plugin.iceberg.ExpressionConverter;
import io.trino.plugin.iceberg.IcebergColumnHandle;
import io.trino.plugin.iceberg.IcebergStatistics;
import io.trino.plugin.iceberg.IcebergTableHandle;
import io.trino.plugin.iceberg.IcebergTypes;
import io.trino.plugin.iceberg.IcebergUtil;
import io.trino.plugin.iceberg.TypeConverter;
import io.trino.spi.connector.Constraint;
import io.trino.spi.predicate.Domain;
import io.trino.spi.predicate.TupleDomain;
import io.trino.spi.statistics.ColumnStatistics;
import io.trino.spi.statistics.DoubleRange;
import io.trino.spi.statistics.Estimate;
import io.trino.spi.statistics.TableStatistics;
import io.trino.spi.type.TypeManager;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.iceberg.DataFile;
import org.apache.iceberg.FileScanTask;
import org.apache.iceberg.PartitionField;
import org.apache.iceberg.Schema;
import org.apache.iceberg.Table;
import org.apache.iceberg.TableScan;
import org.apache.iceberg.io.CloseableIterable;
import org.apache.iceberg.types.Type;

public class TableStatisticsMaker {
    private final TypeManager typeManager;
    private final Table icebergTable;

    private TableStatisticsMaker(TypeManager typeManager, Table icebergTable) {
        this.typeManager = typeManager;
        this.icebergTable = icebergTable;
    }

    public static TableStatistics getTableStatistics(TypeManager typeManager, Constraint constraint, IcebergTableHandle tableHandle, Table icebergTable) {
        return new TableStatisticsMaker(typeManager, icebergTable).makeTableStatistics(tableHandle, constraint);
    }

    private TableStatistics makeTableStatistics(IcebergTableHandle tableHandle, Constraint constraint) {
        if (tableHandle.getSnapshotId().isEmpty() || constraint.getSummary().isNone()) {
            return TableStatistics.empty();
        }
        TupleDomain intersection = constraint.getSummary().transformKeys(IcebergColumnHandle.class::cast).intersect(tableHandle.getEnforcedPredicate());
        if (intersection.isNone()) {
            return TableStatistics.empty();
        }
        Schema icebergTableSchema = this.icebergTable.schema();
        List columns = icebergTableSchema.columns();
        Map<Integer, Type.PrimitiveType> idToTypeMapping = IcebergUtil.primitiveFieldTypes(icebergTableSchema);
        List partitionFields = this.icebergTable.spec().fields();
        List<Type> icebergPartitionTypes = this.partitionTypes(partitionFields, idToTypeMapping);
        List<IcebergColumnHandle> columnHandles = IcebergUtil.getColumns(icebergTableSchema, this.typeManager);
        Map idToColumnHandle = columnHandles.stream().collect(Collectors.toUnmodifiableMap(IcebergColumnHandle::getId, Function.identity()));
        ImmutableMap.Builder idToDetailsBuilder = ImmutableMap.builder();
        for (int index = 0; index < partitionFields.size(); ++index) {
            PartitionField field = (PartitionField)partitionFields.get(index);
            Type type = icebergPartitionTypes.get(index);
            idToDetailsBuilder.put((Object)field.fieldId(), (Object)new ColumnFieldDetails(field, (IcebergColumnHandle)idToColumnHandle.get(field.sourceId()), type, TypeConverter.toTrinoType(type, this.typeManager), type.typeId().javaClass()));
        }
        ImmutableMap idToDetails = idToDetailsBuilder.buildOrThrow();
        TableScan tableScan = this.icebergTable.newScan().filter(ExpressionConverter.toIcebergExpression((TupleDomain<IcebergColumnHandle>)intersection)).useSnapshot(tableHandle.getSnapshotId().get().longValue()).includeColumnStats();
        IcebergStatistics.Builder icebergStatisticsBuilder = new IcebergStatistics.Builder(columns, this.typeManager);
        try (CloseableIterable fileScanTasks = tableScan.planFiles();){
            for (FileScanTask fileScanTask : fileScanTasks) {
                DataFile dataFile = fileScanTask.file();
                if (!this.dataFileMatches(dataFile, constraint, partitionFields, (Map<Integer, ColumnFieldDetails>)idToDetails)) continue;
                icebergStatisticsBuilder.acceptDataFile(dataFile, fileScanTask.spec());
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        IcebergStatistics summary = icebergStatisticsBuilder.build();
        if (summary.getFileCount() == 0L) {
            return TableStatistics.empty();
        }
        ImmutableMap.Builder columnHandleBuilder = ImmutableMap.builder();
        double recordCount = summary.getRecordCount();
        for (IcebergColumnHandle columnHandle : idToColumnHandle.values()) {
            Long columnSize;
            int fieldId = columnHandle.getId();
            ColumnStatistics.Builder columnBuilder = new ColumnStatistics.Builder();
            Long nullCount = summary.getNullCounts().get(fieldId);
            if (nullCount != null) {
                columnBuilder.setNullsFraction(Estimate.of((double)((double)nullCount.longValue() / recordCount)));
            }
            if (summary.getColumnSizes() != null && (columnSize = summary.getColumnSizes().get(fieldId)) != null) {
                columnBuilder.setDataSize(Estimate.of((double)columnSize.longValue()));
            }
            Object min = summary.getMinValues().get(fieldId);
            Object max = summary.getMaxValues().get(fieldId);
            if (min != null && max != null) {
                columnBuilder.setRange(DoubleRange.from((io.trino.spi.type.Type)columnHandle.getType(), (Object)min, (Object)max));
            }
            columnHandleBuilder.put((Object)columnHandle, (Object)columnBuilder.build());
        }
        return new TableStatistics(Estimate.of((double)recordCount), (Map)columnHandleBuilder.buildOrThrow());
    }

    private boolean dataFileMatches(DataFile dataFile, Constraint constraint, List<PartitionField> partitionFields, Map<Integer, ColumnFieldDetails> fieldDetails) {
        Verify.verify((boolean)constraint.predicate().isEmpty(), (String)"Unexpected Constraint predicate", (Object[])new Object[0]);
        TupleDomain constraintSummary = constraint.getSummary();
        Map domains = (Map)constraintSummary.getDomains().get();
        for (int index = 0; index < partitionFields.size(); ++index) {
            PartitionField field = partitionFields.get(index);
            int fieldId = field.fieldId();
            ColumnFieldDetails details = fieldDetails.get(fieldId);
            IcebergColumnHandle column = details.getColumnHandle();
            Object value = IcebergTypes.convertIcebergValueToTrino(details.getIcebergType(), dataFile.partition().get(index, details.getJavaClass()));
            Domain allowedDomain = (Domain)domains.get(column);
            if (allowedDomain == null || allowedDomain.includesNullableValue(value)) continue;
            return false;
        }
        return true;
    }

    public List<Type> partitionTypes(List<PartitionField> partitionFields, Map<Integer, Type.PrimitiveType> idToTypeMapping) {
        ImmutableList.Builder partitionTypeBuilder = ImmutableList.builder();
        for (PartitionField partitionField : partitionFields) {
            Type.PrimitiveType sourceType = idToTypeMapping.get(partitionField.sourceId());
            Type type = partitionField.transform().getResultType((Type)sourceType);
            partitionTypeBuilder.add((Object)type);
        }
        return partitionTypeBuilder.build();
    }

    private static class ColumnFieldDetails {
        private final PartitionField field;
        private final IcebergColumnHandle columnHandle;
        private final Type icebergType;
        private final io.trino.spi.type.Type trinoType;
        private final Class<?> javaClass;

        public ColumnFieldDetails(PartitionField field, IcebergColumnHandle columnHandle, Type icebergType, io.trino.spi.type.Type trinoType, Class<?> javaClass) {
            this.field = Objects.requireNonNull(field, "field is null");
            this.columnHandle = Objects.requireNonNull(columnHandle, "columnHandle is null");
            this.icebergType = Objects.requireNonNull(icebergType, "icebergType is null");
            this.trinoType = Objects.requireNonNull(trinoType, "trinoType is null");
            this.javaClass = Objects.requireNonNull(javaClass, "javaClass is null");
        }

        public PartitionField getField() {
            return this.field;
        }

        public IcebergColumnHandle getColumnHandle() {
            return this.columnHandle;
        }

        public Type getIcebergType() {
            return this.icebergType;
        }

        public io.trino.spi.type.Type getTrinoType() {
            return this.trinoType;
        }

        public Class<?> getJavaClass() {
            return this.javaClass;
        }
    }
}

