/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.kinesis.s3config;

import com.amazonaws.AmazonClientException;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.AmazonS3URI;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ListObjectsRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.google.common.collect.ImmutableMap;
import com.google.common.io.CharStreams;
import com.google.inject.Inject;
import io.airlift.json.JsonCodec;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import io.trino.plugin.kinesis.KinesisClientProvider;
import io.trino.plugin.kinesis.KinesisConfig;
import io.trino.plugin.kinesis.KinesisStreamDescription;
import io.trino.spi.connector.SchemaTableName;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import javax.annotation.PostConstruct;

public class S3TableConfigClient
implements Runnable {
    private static final Logger log = Logger.get(S3TableConfigClient.class);
    private final KinesisClientProvider clientManager;
    private final JsonCodec<KinesisStreamDescription> streamDescriptionCodec;
    private final Duration tableDescriptionRefreshInterval;
    private final Optional<String> bucketUrl;
    private volatile long lastCheck;
    private volatile ScheduledFuture<?> updateTaskHandle;
    private final Map<String, KinesisStreamDescription> descriptors = Collections.synchronizedMap(new HashMap());

    @Inject
    public S3TableConfigClient(KinesisConfig connectorConfig, KinesisClientProvider clientManager, JsonCodec<KinesisStreamDescription> jsonCodec) {
        Objects.requireNonNull(connectorConfig, "connectorConfig is null");
        this.tableDescriptionRefreshInterval = connectorConfig.getTableDescriptionRefreshInterval();
        this.clientManager = Objects.requireNonNull(clientManager, "clientManager is null");
        this.streamDescriptionCodec = Objects.requireNonNull(jsonCodec, "jsonCodec is null");
        this.bucketUrl = connectorConfig.getTableDescriptionLocation().startsWith("s3://") ? Optional.of(connectorConfig.getTableDescriptionLocation()) : Optional.empty();
    }

    @PostConstruct
    protected void startS3Updates() {
        if (this.bucketUrl.isPresent()) {
            ScheduledExecutorService scheduler = Executors.newSingleThreadScheduledExecutor();
            this.updateTaskHandle = scheduler.scheduleAtFixedRate(this::updateTablesFromS3, 5000L, this.tableDescriptionRefreshInterval.toMillis(), TimeUnit.MILLISECONDS);
        }
    }

    public boolean isUsingS3() {
        return this.bucketUrl.isPresent() && this.bucketUrl.get().startsWith("s3://");
    }

    public Map<SchemaTableName, KinesisStreamDescription> getTablesFromS3() {
        this.updateTablesFromS3();
        Collection<KinesisStreamDescription> streamValues = this.descriptors.values();
        ImmutableMap.Builder builder = ImmutableMap.builder();
        for (KinesisStreamDescription stream : streamValues) {
            builder.put((Object)new SchemaTableName(stream.getSchemaName(), stream.getTableName()), (Object)stream);
        }
        return builder.buildOrThrow();
    }

    @Override
    public void run() {
        if (this.isUsingS3() && this.updateTaskHandle != null) {
            this.updateTaskHandle.cancel(true);
        }
    }

    private List<S3ObjectSummary> getObjectSummaries() {
        AmazonS3Client s3client = this.clientManager.getS3Client();
        AmazonS3URI directoryURI = new AmazonS3URI(this.bucketUrl.get());
        ArrayList<S3ObjectSummary> result = new ArrayList<S3ObjectSummary>();
        try {
            ObjectListing response;
            log.info("Getting the listing of objects in the S3 table config directory: bucket %s prefix %s :", new Object[]{directoryURI.getBucket(), directoryURI.getKey()});
            ListObjectsRequest request = new ListObjectsRequest().withBucketName(directoryURI.getBucket()).withPrefix(directoryURI.getKey() + "/").withDelimiter("/").withMaxKeys(Integer.valueOf(25));
            do {
                response = s3client.listObjects(request);
                result.addAll(response.getObjectSummaries());
                request.setMarker(response.getNextMarker());
            } while (response.isTruncated());
            log.info("Completed getting S3 object listing.");
        }
        catch (AmazonClientException e) {
            log.error("Skipping update as faced error fetching table descriptions from S3 %s", new Object[]{e});
        }
        return result;
    }

    private void updateTablesFromS3() {
        long now = System.currentTimeMillis();
        AmazonS3Client s3client = this.clientManager.getS3Client();
        for (S3ObjectSummary summary : this.getObjectSummaries()) {
            if (this.descriptors.containsKey(summary.getKey()) && summary.getLastModified().getTime() < this.lastCheck || summary.getKey().endsWith("/")) continue;
            log.info("Getting : %s - %s", new Object[]{summary.getBucketName(), summary.getKey()});
            S3Object object = s3client.getObject(new GetObjectRequest(summary.getBucketName(), summary.getKey()));
            try (BufferedReader reader = new BufferedReader(new InputStreamReader((InputStream)object.getObjectContent(), StandardCharsets.UTF_8));){
                KinesisStreamDescription table = (KinesisStreamDescription)this.streamDescriptionCodec.fromJson(CharStreams.toString((Readable)reader));
                this.descriptors.put(summary.getKey(), table);
                log.info("Put table description into the map from %s", new Object[]{summary.getKey()});
            }
            catch (IOException iox) {
                log.error((Throwable)iox, "Problem reading input stream from object.");
                throw new RuntimeException(iox);
            }
        }
        log.info("Completed updating table definitions from S3.");
        this.lastCheck = now;
    }
}

