/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.exchange.s3;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.io.Closer;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import io.airlift.concurrent.MoreFutures;
import io.airlift.slice.InputStreamSliceInput;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceInput;
import io.trino.plugin.exchange.ExchangeStorageWriter;
import io.trino.plugin.exchange.FileStatus;
import io.trino.plugin.exchange.FileSystemExchangeStorage;
import io.trino.plugin.exchange.s3.ByteBufferAsyncRequestBody;
import io.trino.plugin.exchange.s3.ExchangeS3Config;
import io.trino.plugin.exchange.s3.S3RequestUtil;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import javax.annotation.PreDestroy;
import javax.annotation.concurrent.NotThreadSafe;
import javax.crypto.SecretKey;
import javax.inject.Inject;
import org.openjdk.jol.info.ClassLayout;
import software.amazon.awssdk.auth.credentials.AwsBasicCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProvider;
import software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider;
import software.amazon.awssdk.auth.credentials.StaticCredentialsProvider;
import software.amazon.awssdk.core.async.AsyncRequestBody;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkAdvancedClientOption;
import software.amazon.awssdk.core.retry.RetryPolicy;
import software.amazon.awssdk.core.sync.ResponseTransformer;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.s3.S3AsyncClient;
import software.amazon.awssdk.services.s3.S3AsyncClientBuilder;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.S3ClientBuilder;
import software.amazon.awssdk.services.s3.model.AbortMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.AbortMultipartUploadResponse;
import software.amazon.awssdk.services.s3.model.CommonPrefix;
import software.amazon.awssdk.services.s3.model.CompleteMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.CompleteMultipartUploadResponse;
import software.amazon.awssdk.services.s3.model.CompletedMultipartUpload;
import software.amazon.awssdk.services.s3.model.CompletedPart;
import software.amazon.awssdk.services.s3.model.CreateMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.CreateMultipartUploadResponse;
import software.amazon.awssdk.services.s3.model.Delete;
import software.amazon.awssdk.services.s3.model.DeleteObjectsRequest;
import software.amazon.awssdk.services.s3.model.DeleteObjectsResponse;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.HeadObjectRequest;
import software.amazon.awssdk.services.s3.model.HeadObjectResponse;
import software.amazon.awssdk.services.s3.model.ListObjectsV2Request;
import software.amazon.awssdk.services.s3.model.NoSuchKeyException;
import software.amazon.awssdk.services.s3.model.ObjectIdentifier;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.services.s3.model.S3Object;
import software.amazon.awssdk.services.s3.model.UploadPartRequest;
import software.amazon.awssdk.services.s3.paginators.ListObjectsV2Iterable;
import software.amazon.awssdk.services.s3.paginators.ListObjectsV2Publisher;

public class S3FileSystemExchangeStorage
implements FileSystemExchangeStorage {
    private static final String DIRECTORY_SUFFIX = "_$folder$";
    private final Optional<Region> region;
    private final Optional<String> endpoint;
    private final int multiUploadPartSize;
    private final S3Client s3Client;
    private final S3AsyncClient s3AsyncClient;

    @Inject
    public S3FileSystemExchangeStorage(ExchangeS3Config config) {
        Objects.requireNonNull(config, "config is null");
        this.region = config.getS3Region();
        this.endpoint = config.getS3Endpoint();
        this.multiUploadPartSize = Math.toIntExact(config.getS3UploadPartSize().toBytes());
        AwsCredentialsProvider credentialsProvider = S3FileSystemExchangeStorage.createAwsCredentialsProvider(config);
        RetryPolicy retryPolicy = RetryPolicy.builder().numRetries(Integer.valueOf(config.getS3MaxErrorRetries())).build();
        ClientOverrideConfiguration overrideConfig = (ClientOverrideConfiguration)ClientOverrideConfiguration.builder().retryPolicy(retryPolicy).putAdvancedOption(SdkAdvancedClientOption.USER_AGENT_PREFIX, (Object)"").putAdvancedOption(SdkAdvancedClientOption.USER_AGENT_SUFFIX, (Object)"Trino-exchange").build();
        this.s3Client = this.createS3Client(credentialsProvider, overrideConfig);
        this.s3AsyncClient = this.createS3AsyncClient(credentialsProvider, overrideConfig);
    }

    @Override
    public void createDirectories(URI dir) throws IOException {
    }

    @Override
    public SliceInput getSliceInput(URI file, Optional<SecretKey> secretKey) throws IOException {
        GetObjectRequest.Builder getObjectRequestBuilder = GetObjectRequest.builder().bucket(S3FileSystemExchangeStorage.getBucketName(file)).key(S3FileSystemExchangeStorage.keyFromUri(file));
        S3RequestUtil.configureEncryption(secretKey, getObjectRequestBuilder);
        try {
            return new InputStreamSliceInput((InputStream)this.s3Client.getObject((GetObjectRequest)getObjectRequestBuilder.build(), ResponseTransformer.toInputStream()));
        }
        catch (RuntimeException e) {
            throw new IOException(e);
        }
    }

    @Override
    public ExchangeStorageWriter createExchangeStorageWriter(URI file, Optional<SecretKey> secretKey) {
        String bucketName = S3FileSystemExchangeStorage.getBucketName(file);
        String key = S3FileSystemExchangeStorage.keyFromUri(file);
        return new S3ExchangeStorageWriter(this.s3AsyncClient, bucketName, key, this.multiUploadPartSize, secretKey);
    }

    @Override
    public boolean exists(URI file) throws IOException {
        return this.headObject(file, Optional.empty()) != null;
    }

    @Override
    public ListenableFuture<Void> createEmptyFile(URI file) {
        PutObjectRequest request = (PutObjectRequest)PutObjectRequest.builder().bucket(S3FileSystemExchangeStorage.getBucketName(file)).key(S3FileSystemExchangeStorage.keyFromUri(file)).build();
        return S3FileSystemExchangeStorage.transformFuture(MoreFutures.toListenableFuture((CompletableFuture)this.s3AsyncClient.putObject(request, AsyncRequestBody.empty())));
    }

    @Override
    public ListenableFuture<Void> deleteRecursively(URI uri) {
        Preconditions.checkArgument((boolean)S3FileSystemExchangeStorage.isDirectory(uri), (Object)"deleteRecursively called on file uri");
        ImmutableList.Builder keys = ImmutableList.builder();
        return S3FileSystemExchangeStorage.transformFuture(Futures.transformAsync((ListenableFuture)MoreFutures.toListenableFuture((CompletableFuture)this.listObjectsRecursively(uri).subscribe(listObjectsV2Response -> listObjectsV2Response.contents().stream().map(S3Object::key).forEach(arg_0 -> ((ImmutableList.Builder)keys).add(arg_0)))), ignored -> {
            keys.add((Object)(S3FileSystemExchangeStorage.keyFromUri(uri) + DIRECTORY_SUFFIX));
            return this.deleteObjects(S3FileSystemExchangeStorage.getBucketName(uri), (List<String>)keys.build());
        }, (Executor)MoreExecutors.directExecutor()));
    }

    @Override
    public List<FileStatus> listFiles(URI dir) throws IOException {
        ImmutableList.Builder builder = ImmutableList.builder();
        try {
            for (S3Object object : this.listObjects(dir).contents()) {
                builder.add((Object)new FileStatus(new URI(dir.getScheme(), dir.getHost(), "/" + object.key(), dir.getFragment()).toString(), object.size()));
            }
        }
        catch (RuntimeException e) {
            throw new IOException(e);
        }
        catch (URISyntaxException e) {
            throw new IllegalArgumentException(e);
        }
        return builder.build();
    }

    @Override
    public List<URI> listDirectories(URI dir) throws IOException {
        ImmutableList.Builder builder = ImmutableList.builder();
        try {
            for (CommonPrefix prefix : this.listObjects(dir).commonPrefixes()) {
                builder.add((Object)new URI(dir.getScheme(), dir.getHost(), "/" + prefix.prefix(), dir.getFragment()));
            }
        }
        catch (RuntimeException e) {
            throw new IOException(e);
        }
        catch (URISyntaxException e) {
            throw new IllegalArgumentException(e);
        }
        return builder.build();
    }

    @Override
    public int getWriteBufferSize() {
        return this.multiUploadPartSize;
    }

    @Override
    @PreDestroy
    public void close() throws IOException {
        try (Closer closer = Closer.create();){
            closer.register(() -> ((S3Client)this.s3Client).close());
            closer.register(() -> ((S3AsyncClient)this.s3AsyncClient).close());
        }
    }

    private HeadObjectResponse headObject(URI uri, Optional<SecretKey> secretKey) throws IOException {
        HeadObjectRequest.Builder headObjectRequestBuilder = HeadObjectRequest.builder().bucket(S3FileSystemExchangeStorage.getBucketName(uri)).key(S3FileSystemExchangeStorage.keyFromUri(uri));
        S3RequestUtil.configureEncryption(secretKey, headObjectRequestBuilder);
        try {
            return this.s3Client.headObject((HeadObjectRequest)headObjectRequestBuilder.build());
        }
        catch (RuntimeException e) {
            if (e instanceof NoSuchKeyException) {
                return null;
            }
            throw new IOException(e);
        }
    }

    private ListObjectsV2Iterable listObjects(URI dir) {
        Object key = S3FileSystemExchangeStorage.keyFromUri(dir);
        if (!((String)key).isEmpty()) {
            key = (String)key + "/";
        }
        ListObjectsV2Request request = (ListObjectsV2Request)ListObjectsV2Request.builder().bucket(S3FileSystemExchangeStorage.getBucketName(dir)).prefix((String)key).delimiter("/").build();
        return this.s3Client.listObjectsV2Paginator(request);
    }

    private ListObjectsV2Publisher listObjectsRecursively(URI dir) {
        ListObjectsV2Request request = (ListObjectsV2Request)ListObjectsV2Request.builder().bucket(S3FileSystemExchangeStorage.getBucketName(dir)).prefix(S3FileSystemExchangeStorage.keyFromUri(dir)).build();
        return this.s3AsyncClient.listObjectsV2Paginator(request);
    }

    private ListenableFuture<List<DeleteObjectsResponse>> deleteObjects(String bucketName, List<String> keys) {
        List subList = Lists.partition(keys, (int)1000);
        return Futures.allAsList((Iterable)((Iterable)subList.stream().map(list -> {
            DeleteObjectsRequest request = (DeleteObjectsRequest)DeleteObjectsRequest.builder().bucket(bucketName).delete((Delete)Delete.builder().objects((Collection)list.stream().map(key -> (ObjectIdentifier)ObjectIdentifier.builder().key(key).build()).collect(ImmutableList.toImmutableList())).build()).build();
            return MoreFutures.toListenableFuture((CompletableFuture)this.s3AsyncClient.deleteObjects(request));
        }).collect(ImmutableList.toImmutableList())));
    }

    private static String getBucketName(URI uri) {
        if (uri.getHost() != null) {
            return uri.getHost();
        }
        if (uri.getUserInfo() == null) {
            return uri.getAuthority();
        }
        throw new IllegalArgumentException("Unable to determine S3 bucket from URI.");
    }

    private static String keyFromUri(URI uri) {
        Preconditions.checkArgument((boolean)uri.isAbsolute(), (String)"Uri is not absolute: %s", (Object)uri);
        String key = Strings.nullToEmpty((String)uri.getPath());
        if (key.startsWith("/")) {
            key = key.substring("/".length());
        }
        if (key.endsWith("/")) {
            key = key.substring(0, key.length() - "/".length());
        }
        return key;
    }

    private static ListenableFuture<Void> transformFuture(ListenableFuture<?> listenableFuture) {
        return MoreFutures.asVoid((ListenableFuture)Futures.catchingAsync(listenableFuture, Throwable.class, throwable -> {
            if (throwable instanceof Error || throwable instanceof IOException) {
                return Futures.immediateFailedFuture((Throwable)throwable);
            }
            return Futures.immediateFailedFuture((Throwable)new IOException((Throwable)throwable));
        }, (Executor)MoreExecutors.directExecutor()));
    }

    private static boolean isDirectory(URI uri) {
        return uri.toString().endsWith("/");
    }

    private static AwsCredentialsProvider createAwsCredentialsProvider(ExchangeS3Config config) {
        if (config.getS3AwsAccessKey() != null && config.getS3AwsSecretKey() != null) {
            return StaticCredentialsProvider.create((AwsCredentials)AwsBasicCredentials.create((String)config.getS3AwsAccessKey(), (String)config.getS3AwsSecretKey()));
        }
        return DefaultCredentialsProvider.create();
    }

    private S3Client createS3Client(AwsCredentialsProvider credentialsProvider, ClientOverrideConfiguration overrideConfig) {
        S3ClientBuilder clientBuilder = (S3ClientBuilder)((S3ClientBuilder)S3Client.builder().credentialsProvider(credentialsProvider)).overrideConfiguration(overrideConfig);
        this.region.ifPresent(arg_0 -> ((S3ClientBuilder)clientBuilder).region(arg_0));
        this.endpoint.ifPresent(s3Endpoint -> clientBuilder.endpointOverride(URI.create(s3Endpoint)));
        return (S3Client)clientBuilder.build();
    }

    private S3AsyncClient createS3AsyncClient(AwsCredentialsProvider credentialsProvider, ClientOverrideConfiguration overrideConfig) {
        S3AsyncClientBuilder clientBuilder = (S3AsyncClientBuilder)((S3AsyncClientBuilder)S3AsyncClient.builder().credentialsProvider(credentialsProvider)).overrideConfiguration(overrideConfig);
        this.region.ifPresent(arg_0 -> ((S3AsyncClientBuilder)clientBuilder).region(arg_0));
        this.endpoint.ifPresent(s3Endpoint -> clientBuilder.endpointOverride(URI.create(s3Endpoint)));
        return (S3AsyncClient)clientBuilder.build();
    }

    @NotThreadSafe
    private static class S3ExchangeStorageWriter
    implements ExchangeStorageWriter {
        private static final int INSTANCE_SIZE = ClassLayout.parseClass(S3ExchangeStorageWriter.class).instanceSize();
        private final S3AsyncClient s3AsyncClient;
        private final String bucketName;
        private final String key;
        private final int partSize;
        private final Optional<SecretKey> secretKey;
        private int currentPartNumber;
        private ListenableFuture<Void> directUploadFuture;
        private ListenableFuture<String> multiPartUploadIdFuture;
        private final List<ListenableFuture<CompletedPart>> multiPartUploadFutures = new ArrayList<ListenableFuture<CompletedPart>>();
        private volatile boolean closed;

        public S3ExchangeStorageWriter(S3AsyncClient s3AsyncClient, String bucketName, String key, int partSize, Optional<SecretKey> secretKey) {
            this.s3AsyncClient = Objects.requireNonNull(s3AsyncClient, "s3AsyncClient is null");
            this.bucketName = Objects.requireNonNull(bucketName, "bucketName is null");
            this.key = Objects.requireNonNull(key, "key is null");
            this.partSize = partSize;
            this.secretKey = Objects.requireNonNull(secretKey, "secretKey is null");
        }

        @Override
        public ListenableFuture<Void> write(Slice slice) {
            Preconditions.checkState((this.directUploadFuture == null ? 1 : 0) != 0, (Object)"Direct upload already started");
            if (this.closed) {
                return Futures.immediateVoidFuture();
            }
            if (slice.length() < this.partSize && this.multiPartUploadIdFuture == null) {
                PutObjectRequest.Builder putObjectRequestBuilder = PutObjectRequest.builder().bucket(this.bucketName).key(this.key);
                S3RequestUtil.configureEncryption(this.secretKey, putObjectRequestBuilder);
                this.directUploadFuture = S3FileSystemExchangeStorage.transformFuture(MoreFutures.toListenableFuture((CompletableFuture)this.s3AsyncClient.putObject((PutObjectRequest)putObjectRequestBuilder.build(), ByteBufferAsyncRequestBody.fromByteBuffer(slice.toByteBuffer()))));
                return this.directUploadFuture;
            }
            if (this.multiPartUploadIdFuture == null) {
                this.multiPartUploadIdFuture = Futures.transform(this.createMultipartUpload(), CreateMultipartUploadResponse::uploadId, (Executor)MoreExecutors.directExecutor());
            }
            int partNum = ++this.currentPartNumber;
            ListenableFuture uploadFuture = Futures.transformAsync(this.multiPartUploadIdFuture, uploadId -> this.uploadPart((String)uploadId, slice, partNum), (Executor)MoreExecutors.directExecutor());
            this.multiPartUploadFutures.add((ListenableFuture<CompletedPart>)uploadFuture);
            return S3FileSystemExchangeStorage.transformFuture(uploadFuture);
        }

        @Override
        public ListenableFuture<Void> finish() {
            if (this.closed) {
                return Futures.immediateVoidFuture();
            }
            if (this.multiPartUploadIdFuture == null) {
                return Objects.requireNonNullElseGet(this.directUploadFuture, Futures::immediateVoidFuture);
            }
            ListenableFuture<Void> finishFuture = S3FileSystemExchangeStorage.transformFuture(Futures.transformAsync((ListenableFuture)Futures.allAsList(this.multiPartUploadFutures), completedParts -> this.completeMultipartUpload((String)MoreFutures.getFutureValue(this.multiPartUploadIdFuture), (List<CompletedPart>)completedParts), (Executor)MoreExecutors.directExecutor()));
            Futures.addCallback(finishFuture, (FutureCallback)new FutureCallback<Void>(){

                public void onSuccess(Void result) {
                    closed = true;
                }

                public void onFailure(Throwable ignored) {
                }
            }, (Executor)MoreExecutors.directExecutor());
            return finishFuture;
        }

        @Override
        public ListenableFuture<Void> abort() {
            if (this.closed) {
                return Futures.immediateVoidFuture();
            }
            this.closed = true;
            if (this.multiPartUploadIdFuture == null) {
                if (this.directUploadFuture != null) {
                    this.directUploadFuture.cancel(true);
                }
                return Futures.immediateVoidFuture();
            }
            Verify.verify((this.directUploadFuture == null ? 1 : 0) != 0);
            this.multiPartUploadFutures.forEach(future -> future.cancel(true));
            return S3FileSystemExchangeStorage.transformFuture(Futures.transformAsync(this.multiPartUploadIdFuture, this::abortMultipartUpload, (Executor)MoreExecutors.directExecutor()));
        }

        @Override
        public long getRetainedSize() {
            return INSTANCE_SIZE;
        }

        private ListenableFuture<CreateMultipartUploadResponse> createMultipartUpload() {
            CreateMultipartUploadRequest.Builder createMultipartUploadRequestBuilder = CreateMultipartUploadRequest.builder().bucket(this.bucketName).key(this.key);
            S3RequestUtil.configureEncryption(this.secretKey, createMultipartUploadRequestBuilder);
            return MoreFutures.toListenableFuture((CompletableFuture)this.s3AsyncClient.createMultipartUpload((CreateMultipartUploadRequest)createMultipartUploadRequestBuilder.build()));
        }

        private ListenableFuture<CompletedPart> uploadPart(String uploadId, Slice slice, int partNumber) {
            UploadPartRequest.Builder uploadPartRequestBuilder = UploadPartRequest.builder().bucket(this.bucketName).key(this.key).uploadId(uploadId).partNumber(Integer.valueOf(partNumber));
            S3RequestUtil.configureEncryption(this.secretKey, uploadPartRequestBuilder);
            UploadPartRequest uploadPartRequest = (UploadPartRequest)uploadPartRequestBuilder.build();
            return Futures.transform((ListenableFuture)MoreFutures.toListenableFuture((CompletableFuture)this.s3AsyncClient.uploadPart(uploadPartRequest, ByteBufferAsyncRequestBody.fromByteBuffer(slice.toByteBuffer()))), uploadPartResponse -> (CompletedPart)CompletedPart.builder().eTag(uploadPartResponse.eTag()).partNumber(Integer.valueOf(partNumber)).build(), (Executor)MoreExecutors.directExecutor());
        }

        private ListenableFuture<CompleteMultipartUploadResponse> completeMultipartUpload(String uploadId, List<CompletedPart> completedParts) {
            CompletedMultipartUpload completedMultipartUpload = (CompletedMultipartUpload)CompletedMultipartUpload.builder().parts(completedParts).build();
            CompleteMultipartUploadRequest completeMultipartUploadRequest = (CompleteMultipartUploadRequest)CompleteMultipartUploadRequest.builder().bucket(this.bucketName).key(this.key).uploadId(uploadId).multipartUpload(completedMultipartUpload).build();
            return MoreFutures.toListenableFuture((CompletableFuture)this.s3AsyncClient.completeMultipartUpload(completeMultipartUploadRequest));
        }

        private ListenableFuture<AbortMultipartUploadResponse> abortMultipartUpload(String uploadId) {
            AbortMultipartUploadRequest abortMultipartUploadRequest = (AbortMultipartUploadRequest)AbortMultipartUploadRequest.builder().bucket(this.bucketName).key(this.key).uploadId(uploadId).build();
            return MoreFutures.toListenableFuture((CompletableFuture)this.s3AsyncClient.abortMultipartUpload(abortMultipartUploadRequest));
        }
    }
}

