/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.bigquery;

import com.google.cloud.RetryOption;
import com.google.cloud.bigquery.BigQueryException;
import com.google.cloud.bigquery.Job;
import com.google.cloud.bigquery.JobConfiguration;
import com.google.cloud.bigquery.JobInfo;
import com.google.cloud.bigquery.QueryJobConfiguration;
import com.google.cloud.bigquery.Table;
import com.google.cloud.bigquery.TableId;
import com.google.cloud.bigquery.TableInfo;
import com.google.common.cache.CacheBuilder;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import io.trino.collect.cache.NonEvictableCache;
import io.trino.collect.cache.SafeCaches;
import io.trino.plugin.bigquery.BigQueryClient;
import io.trino.plugin.bigquery.BigQueryConfig;
import io.trino.plugin.bigquery.BigQueryErrorCode;
import io.trino.plugin.bigquery.BigQueryUtil;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.connector.TableNotFoundException;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import javax.inject.Inject;

public class ViewMaterializationCache {
    private static final Logger log = Logger.get(ViewMaterializationCache.class);
    private final NonEvictableCache<String, TableInfo> destinationTableCache;
    private final Optional<String> viewMaterializationProject;
    private final Optional<String> viewMaterializationDataset;

    @Inject
    public ViewMaterializationCache(BigQueryConfig config) {
        Objects.requireNonNull(config, "config is null");
        this.destinationTableCache = SafeCaches.buildNonEvictableCache((CacheBuilder)CacheBuilder.newBuilder().expireAfterWrite(config.getViewsCacheTtl().toMillis(), TimeUnit.MILLISECONDS).maximumSize(1000L));
        this.viewMaterializationProject = config.getViewMaterializationProject();
        this.viewMaterializationDataset = config.getViewMaterializationDataset();
    }

    public TableInfo getCachedTable(BigQueryClient client, String query, Duration viewExpiration, TableInfo remoteTableId) {
        try {
            return (TableInfo)this.destinationTableCache.get((Object)query, (Callable)new DestinationTableBuilder(client, viewExpiration, query, this.createDestinationTable(remoteTableId.getTableId())));
        }
        catch (ExecutionException e) {
            throw new TrinoException((ErrorCodeSupplier)BigQueryErrorCode.BIGQUERY_VIEW_DESTINATION_TABLE_CREATION_FAILED, "Error creating destination table", (Throwable)e);
        }
    }

    private TableId createDestinationTable(TableId remoteTableId) {
        String project = this.viewMaterializationProject.orElse(remoteTableId.getProject());
        String dataset = this.viewMaterializationDataset.orElse(remoteTableId.getDataset());
        String name = String.format("_pbc_%s", UUID.randomUUID().toString().toLowerCase(Locale.ENGLISH).replace("-", ""));
        return TableId.of((String)project, (String)dataset, (String)name);
    }

    private static class DestinationTableBuilder
    implements Callable<TableInfo> {
        private final BigQueryClient bigQueryClient;
        private final Duration viewExpiration;
        private final String query;
        private final TableId destinationTable;

        DestinationTableBuilder(BigQueryClient bigQueryClient, Duration viewExpiration, String query, TableId destinationTable) {
            this.bigQueryClient = Objects.requireNonNull(bigQueryClient, "bigQueryClient is null");
            this.viewExpiration = Objects.requireNonNull(viewExpiration, "viewExpiration is null");
            this.query = Objects.requireNonNull(query, "query is null");
            this.destinationTable = Objects.requireNonNull(destinationTable, "destinationTable is null");
        }

        @Override
        public TableInfo call() {
            return this.createTableFromQuery();
        }

        private TableInfo createTableFromQuery() {
            log.debug("destinationTable is %s", new Object[]{this.destinationTable});
            JobInfo jobInfo = JobInfo.of((JobConfiguration)QueryJobConfiguration.newBuilder((String)this.query).setDestinationTable(this.destinationTable).build());
            log.debug("running query %s", new Object[]{jobInfo});
            Job job = this.waitForJob(this.bigQueryClient.create(jobInfo));
            log.debug("job has finished. %s", new Object[]{job});
            if (job.getStatus().getError() != null) {
                throw BigQueryUtil.convertToBigQueryException(job.getStatus().getError());
            }
            TableInfo createdTable = this.bigQueryClient.getTable(this.destinationTable).orElseThrow(() -> new TableNotFoundException(new SchemaTableName(this.destinationTable.getDataset(), this.destinationTable.getTable())));
            long expirationTimeMillis = createdTable.getCreationTime() + this.viewExpiration.toMillis();
            Table updatedTable = this.bigQueryClient.update(createdTable.toBuilder().setExpirationTime(Long.valueOf(expirationTimeMillis)).build());
            return updatedTable;
        }

        private Job waitForJob(Job job) {
            try {
                return job.waitFor(new RetryOption[0]);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new BigQueryException(0, String.format("Job %s has been interrupted", job.getJobId()), (Throwable)e);
            }
        }
    }
}

