/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.segment.local.segment.store;

import com.google.common.base.Preconditions;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;
import org.apache.commons.configuration.Configuration;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.configuration.PropertiesConfiguration;
import org.apache.pinot.segment.local.segment.store.IndexEntry;
import org.apache.pinot.segment.local.segment.store.IndexKey;
import org.apache.pinot.segment.spi.index.metadata.SegmentMetadataImpl;
import org.apache.pinot.segment.spi.memory.PinotDataBuffer;
import org.apache.pinot.segment.spi.store.ColumnIndexDirectory;
import org.apache.pinot.segment.spi.store.ColumnIndexType;
import org.apache.pinot.spi.env.CommonsConfigurationUtils;
import org.apache.pinot.spi.utils.ReadMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class SingleFileIndexDirectory
extends ColumnIndexDirectory {
    private static final Logger LOGGER = LoggerFactory.getLogger(SingleFileIndexDirectory.class);
    private static final long MAGIC_MARKER = -2401053088876085587L;
    private static final int MAGIC_MARKER_SIZE_BYTES = 8;
    private static final String MAP_KEY_SEPARATOR = ".";
    private static final String MAP_KEY_NAME_START_OFFSET = "startOffset";
    private static final String MAP_KEY_NAME_SIZE = "size";
    private static final int MAX_ALLOCATION_SIZE = 0x7D000000;
    private final File _segmentDirectory;
    private SegmentMetadataImpl _segmentMetadata;
    private final ReadMode _readMode;
    private final File _indexFile;
    private final Map<IndexKey, IndexEntry> _columnEntries;
    private final List<PinotDataBuffer> _allocBuffers;

    public SingleFileIndexDirectory(File segmentDirectory, SegmentMetadataImpl segmentMetadata, ReadMode readMode) throws IOException, ConfigurationException {
        Preconditions.checkNotNull((Object)segmentDirectory);
        Preconditions.checkNotNull((Object)readMode);
        Preconditions.checkNotNull((Object)segmentMetadata);
        Preconditions.checkArgument((boolean)segmentDirectory.exists(), (Object)("SegmentDirectory: " + segmentDirectory.toString() + " does not exist"));
        Preconditions.checkArgument((boolean)segmentDirectory.isDirectory(), (Object)("SegmentDirectory: " + segmentDirectory.toString() + " is not a directory"));
        this._segmentDirectory = segmentDirectory;
        this._segmentMetadata = segmentMetadata;
        this._readMode = readMode;
        this._indexFile = new File(segmentDirectory, "columns.psf");
        if (!this._indexFile.exists()) {
            this._indexFile.createNewFile();
        }
        this._columnEntries = new HashMap<IndexKey, IndexEntry>(this._segmentMetadata.getAllColumns().size());
        this._allocBuffers = new ArrayList<PinotDataBuffer>();
        this.load();
    }

    public void setSegmentMetadata(SegmentMetadataImpl segmentMetadata) {
        this._segmentMetadata = segmentMetadata;
    }

    public PinotDataBuffer getBuffer(String column, ColumnIndexType type) throws IOException {
        return this.checkAndGetIndexBuffer(column, type);
    }

    public PinotDataBuffer newBuffer(String column, ColumnIndexType type, long sizeBytes) throws IOException {
        return this.allocNewBufferInternal(column, type, sizeBytes, type.name().toLowerCase() + ".create");
    }

    public boolean hasIndexFor(String column, ColumnIndexType type) {
        if (type == ColumnIndexType.TEXT_INDEX) {
            return this.hasTextIndex(column);
        }
        IndexKey key = new IndexKey(column, type);
        return this._columnEntries.containsKey(key);
    }

    private boolean hasTextIndex(final String column) {
        final String suffix = ".lucene.index";
        File[] textIndexFiles = this._segmentDirectory.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.equals(column + suffix);
            }
        });
        if (textIndexFiles.length > 0) {
            Preconditions.checkState((textIndexFiles.length == 1 ? 1 : 0) != 0, (Object)("Illegal number of text index directories for columns " + column + " segment directory " + this._segmentDirectory.getAbsolutePath()));
            return true;
        }
        return false;
    }

    private PinotDataBuffer checkAndGetIndexBuffer(String column, ColumnIndexType type) {
        IndexKey key = new IndexKey(column, type);
        IndexEntry entry = this._columnEntries.get(key);
        if (entry == null || entry.buffer == null) {
            throw new RuntimeException("Could not find index for column: " + column + ", type: " + type + ", segment: " + this._segmentDirectory.toString());
        }
        return entry.buffer;
    }

    private PinotDataBuffer allocNewBufferInternal(String column, ColumnIndexType indexType, long size, String context) throws IOException {
        IndexKey key = new IndexKey(column, indexType);
        this.checkKeyNotPresent(key);
        String allocContext = this.allocationContext(key) + context;
        IndexEntry entry = new IndexEntry(key);
        entry.startOffset = this._indexFile.length();
        entry.size = size + 8L;
        PinotDataBuffer appendBuffer = PinotDataBuffer.mapFile((File)this._indexFile, (boolean)false, (long)entry.startOffset, (long)entry.size, (ByteOrder)ByteOrder.BIG_ENDIAN, (String)allocContext);
        LOGGER.debug("Allotted buffer for key: {}, startOffset: {}, size: {}", new Object[]{key, entry.startOffset, entry.size});
        appendBuffer.putLong(0, -2401053088876085587L);
        this._allocBuffers.add(appendBuffer);
        entry.buffer = appendBuffer.view(8L, entry.size);
        this._columnEntries.put(key, entry);
        this.persistIndexMap(entry);
        return entry.buffer;
    }

    private void checkKeyNotPresent(IndexKey key) {
        if (this._columnEntries.containsKey(key)) {
            throw new RuntimeException("Attempt to re-create an existing index for key: " + key.toString() + ", for segmentDirectory: " + this._segmentDirectory.getAbsolutePath());
        }
    }

    private void validateMagicMarker(PinotDataBuffer buffer, long startOffset) {
        long actualMarkerValue = buffer.getLong(startOffset);
        if (actualMarkerValue != -2401053088876085587L) {
            LOGGER.error("Missing magic marker in index file: {} at position: {}", (Object)this._indexFile, (Object)startOffset);
            throw new RuntimeException("Inconsistent data read. Index data file " + this._indexFile.toString() + " is possibly corrupted");
        }
    }

    private void load() throws IOException, ConfigurationException {
        this.loadMap();
        this.mapBufferEntries();
    }

    private void loadMap() throws ConfigurationException {
        File mapFile = new File(this._segmentDirectory, "index_map");
        PropertiesConfiguration mapConfig = CommonsConfigurationUtils.fromFile((File)mapFile);
        for (String string : CommonsConfigurationUtils.getKeys((Configuration)mapConfig)) {
            int lastSeparatorPos = string.lastIndexOf(MAP_KEY_SEPARATOR);
            Preconditions.checkState((lastSeparatorPos != -1 ? 1 : 0) != 0, (Object)("Key separator not found: " + string + ", segment: " + this._segmentDirectory));
            String propertyName = string.substring(lastSeparatorPos + 1);
            int indexSeparatorPos = string.lastIndexOf(MAP_KEY_SEPARATOR, lastSeparatorPos - 1);
            Preconditions.checkState((indexSeparatorPos != -1 ? 1 : 0) != 0, (Object)("Index separator not found: " + string + " , segment: " + this._segmentDirectory));
            String indexName = string.substring(indexSeparatorPos + 1, lastSeparatorPos);
            String columnName = string.substring(0, indexSeparatorPos);
            IndexKey indexKey = new IndexKey(columnName, ColumnIndexType.getValue((String)indexName));
            IndexEntry entry = this._columnEntries.get(indexKey);
            if (entry == null) {
                entry = new IndexEntry(indexKey);
                this._columnEntries.put(indexKey, entry);
            }
            if (propertyName.equals(MAP_KEY_NAME_START_OFFSET)) {
                entry.startOffset = mapConfig.getLong(string);
                continue;
            }
            if (propertyName.equals(MAP_KEY_NAME_SIZE)) {
                entry.size = mapConfig.getLong(string);
                continue;
            }
            throw new ConfigurationException("Invalid map file key: " + string + ", segmentDirectory: " + this._segmentDirectory.toString());
        }
        for (Map.Entry entry : this._columnEntries.entrySet()) {
            IndexEntry entry2 = (IndexEntry)entry.getValue();
            if (entry2.size >= 0L && entry2.startOffset >= 0L) continue;
            throw new ConfigurationException("Invalid map entry for key: " + ((IndexKey)entry.getKey()).toString() + ", segment: " + this._segmentDirectory.toString());
        }
    }

    private void mapBufferEntries() throws IOException {
        TreeMap<Long, IndexEntry> indexStartMap = new TreeMap<Long, IndexEntry>();
        for (Map.Entry<IndexKey, IndexEntry> columnEntry : this._columnEntries.entrySet()) {
            long startOffset = columnEntry.getValue().startOffset;
            indexStartMap.put(startOffset, columnEntry.getValue());
        }
        long runningSize = 0L;
        ArrayList<Long> offsetAccum = new ArrayList<Long>();
        for (Map.Entry offsetEntry : indexStartMap.entrySet()) {
            IndexEntry entry = (IndexEntry)offsetEntry.getValue();
            if ((runningSize += entry.size) >= 0x7D000000L && !offsetAccum.isEmpty()) {
                this.mapAndSliceFile(indexStartMap, offsetAccum, (Long)offsetEntry.getKey());
                runningSize = entry.size;
                offsetAccum.clear();
            }
            offsetAccum.add((Long)offsetEntry.getKey());
        }
        if (!offsetAccum.isEmpty()) {
            this.mapAndSliceFile(indexStartMap, offsetAccum, (Long)offsetAccum.get(0) + runningSize);
        }
    }

    private void mapAndSliceFile(SortedMap<Long, IndexEntry> startOffsets, List<Long> offsetAccum, long endOffset) throws IOException {
        Preconditions.checkNotNull(startOffsets);
        Preconditions.checkNotNull(offsetAccum);
        Preconditions.checkArgument((!offsetAccum.isEmpty() ? 1 : 0) != 0);
        long fromFilePos = offsetAccum.get(0);
        long size = endOffset - fromFilePos;
        String context = this.allocationContext(this._indexFile, "single_file_index.rw.." + String.valueOf(fromFilePos) + MAP_KEY_SEPARATOR + String.valueOf(size));
        PinotDataBuffer buffer = this._readMode == ReadMode.heap ? PinotDataBuffer.loadFile((File)this._indexFile, (long)fromFilePos, (long)size, (ByteOrder)ByteOrder.BIG_ENDIAN, (String)context) : PinotDataBuffer.mapFile((File)this._indexFile, (boolean)true, (long)fromFilePos, (long)size, (ByteOrder)ByteOrder.BIG_ENDIAN, (String)context);
        this._allocBuffers.add(buffer);
        long prevSlicePoint = 0L;
        for (Long fileOffset : offsetAccum) {
            IndexEntry entry = (IndexEntry)startOffsets.get(fileOffset);
            long endSlicePoint = prevSlicePoint + entry.size;
            this.validateMagicMarker(buffer, prevSlicePoint);
            entry.buffer = buffer.view(prevSlicePoint + 8L, endSlicePoint);
            prevSlicePoint = endSlicePoint;
        }
    }

    private void persistIndexMap(IndexEntry entry) throws IOException {
        File mapFile = new File(this._segmentDirectory, "index_map");
        try (PrintWriter writer = new PrintWriter(new BufferedWriter(new FileWriter(mapFile, true)));){
            String startKey = this.getKey(entry.key.name, entry.key.type.getIndexName(), true);
            StringBuilder sb = new StringBuilder();
            sb.append(startKey).append(" = ").append(entry.startOffset);
            writer.println(sb.toString());
            String endKey = this.getKey(entry.key.name, entry.key.type.getIndexName(), false);
            sb = new StringBuilder();
            sb.append(endKey).append(" = ").append(entry.size);
            writer.println(sb.toString());
        }
    }

    private String getKey(String column, String indexName, boolean isStartOffset) {
        return column + MAP_KEY_SEPARATOR + indexName + MAP_KEY_SEPARATOR + (isStartOffset ? MAP_KEY_NAME_START_OFFSET : MAP_KEY_NAME_SIZE);
    }

    private String allocationContext(IndexKey key) {
        return ((Object)((Object)this)).getClass().getSimpleName() + key.toString();
    }

    public void close() throws IOException {
        for (PinotDataBuffer buf : this._allocBuffers) {
            buf.close();
        }
        this._columnEntries.clear();
        this._allocBuffers.clear();
    }

    public void removeIndex(String columnName, ColumnIndexType indexType) {
        throw new UnsupportedOperationException("Index removal is not supported for single file index format. Requested colum: " + columnName + " indexType: " + indexType);
    }

    public boolean isIndexRemovalSupported() {
        return false;
    }

    public String toString() {
        return this._segmentDirectory.toString() + "/" + this._indexFile.toString();
    }
}

