/*
 * Decompiled with CFR 0.152.
 */
package io.trino.rcfile;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.io.ByteStreams;
import io.airlift.slice.BasicSliceInput;
import io.airlift.slice.ChunkedSliceInput;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceInput;
import io.airlift.slice.Slices;
import io.airlift.units.DataSize;
import io.trino.rcfile.ColumnData;
import io.trino.rcfile.ColumnEncoding;
import io.trino.rcfile.RcFileCodecFactory;
import io.trino.rcfile.RcFileCorruptionException;
import io.trino.rcfile.RcFileDataSource;
import io.trino.rcfile.RcFileDataSourceId;
import io.trino.rcfile.RcFileDecoderUtils;
import io.trino.rcfile.RcFileDecompressor;
import io.trino.rcfile.RcFileEncoding;
import io.trino.rcfile.RcFileWriteValidation;
import io.trino.spi.Page;
import io.trino.spi.block.Block;
import io.trino.spi.block.RunLengthEncodedBlock;
import io.trino.spi.type.Type;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Predicate;

public class RcFileReader
implements Closeable {
    private static final Slice RCFILE_MAGIC = Slices.utf8Slice((String)"RCF");
    private static final int MAX_METADATA_ENTRIES = 500000;
    private static final int MAX_COLUMN_COUNT = 500000;
    private static final int MAX_METADATA_STRING_LENGTH = 0x100000;
    private static final int FIRST_VERSION = 0;
    private static final int CURRENT_VERSION = 1;
    private static final Slice SEQUENCE_FILE_MAGIC = Slices.utf8Slice((String)"SEQ");
    private static final byte SEQUENCE_FILE_VERSION = 6;
    private static final Slice RCFILE_KEY_BUFFER_NAME = Slices.utf8Slice((String)"org.apache.hadoop.hive.ql.io.RCFile$KeyBuffer");
    private static final Slice RCFILE_VALUE_BUFFER_NAME = Slices.utf8Slice((String)"org.apache.hadoop.hive.ql.io.RCFile$ValueBuffer");
    private static final String COLUMN_COUNT_METADATA_KEY = "hive.io.rcfile.column.number";
    private final RcFileDataSource dataSource;
    private final Map<Integer, Type> readColumns;
    private final ChunkedSliceInput input;
    private final long length;
    private final byte version;
    private final RcFileDecompressor decompressor;
    private final Map<String, String> metadata;
    private final int columnCount;
    private final long syncFirst;
    private final long syncSecond;
    private final Column[] columns;
    private final long end;
    private long rowsRead;
    private int rowGroupRowCount;
    private int rowGroupPosition;
    private int currentChunkRowCount;
    private Slice compressedHeaderBuffer = Slices.EMPTY_SLICE;
    private Slice headerBuffer = Slices.EMPTY_SLICE;
    private boolean closed;
    private final Optional<RcFileWriteValidation> writeValidation;
    private final Optional<RcFileWriteValidation.WriteChecksumBuilder> writeChecksumBuilder;

    public RcFileReader(RcFileDataSource dataSource, RcFileEncoding encoding, Map<Integer, Type> readColumns, RcFileCodecFactory codecFactory, long offset, long length, DataSize bufferSize) throws IOException {
        this(dataSource, encoding, readColumns, codecFactory, offset, length, bufferSize, Optional.empty());
    }

    private RcFileReader(RcFileDataSource dataSource, RcFileEncoding encoding, Map<Integer, Type> readColumns, RcFileCodecFactory codecFactory, long offset, long length, DataSize bufferSize, Optional<RcFileWriteValidation> writeValidation) throws IOException {
        boolean compressed;
        this.dataSource = Objects.requireNonNull(dataSource, "dataSource is null");
        this.readColumns = ImmutableMap.copyOf(Objects.requireNonNull(readColumns, "readColumns is null"));
        this.input = new ChunkedSliceInput((ChunkedSliceInput.SliceLoader)new DataSourceSliceLoader(dataSource), Math.toIntExact(bufferSize.toBytes()));
        this.writeValidation = Objects.requireNonNull(writeValidation, "writeValidation is null");
        this.writeChecksumBuilder = writeValidation.map(validation -> RcFileWriteValidation.WriteChecksumBuilder.createWriteChecksumBuilder(readColumns));
        this.verify(offset >= 0L, "offset is negative", new Object[0]);
        this.verify(offset < dataSource.getSize(), "offset is greater than data size", new Object[0]);
        this.verify(length >= 1L, "length must be at least 1", new Object[0]);
        this.length = length;
        this.end = offset + length;
        this.verify(this.end <= dataSource.getSize(), "offset plus length is greater than data size", new Object[0]);
        Slice magic = this.input.readSlice(RCFILE_MAGIC.length());
        if (RCFILE_MAGIC.equals((Object)magic)) {
            this.version = this.input.readByte();
            this.verify(this.version <= 1, "RCFile version %s not supported: %s", this.version, dataSource);
            this.validateWrite(validation -> validation.getVersion() == this.version, "Unexpected file version", new Object[0]);
            compressed = this.input.readBoolean();
        } else if (SEQUENCE_FILE_MAGIC.equals((Object)magic)) {
            this.validateWrite(validation -> false, "Expected file to start with RCFile magic", new Object[0]);
            byte sequenceFileVersion = this.input.readByte();
            this.verify(sequenceFileVersion == 6, "File %s is a SequenceFile not an RCFile", dataSource);
            this.version = 0;
            Slice keyClassName = this.readLengthPrefixedString((SliceInput)this.input);
            Slice valueClassName = this.readLengthPrefixedString((SliceInput)this.input);
            this.verify(RCFILE_KEY_BUFFER_NAME.equals((Object)keyClassName) && RCFILE_VALUE_BUFFER_NAME.equals((Object)valueClassName), "File %s is a SequenceFile not an RCFile", dataSource);
            compressed = this.input.readBoolean();
            if (this.input.readBoolean()) {
                throw this.corrupt("File %s is a SequenceFile not an RCFile", dataSource);
            }
        } else {
            throw this.corrupt("File %s is not an RCFile", dataSource);
        }
        if (compressed) {
            String codecClassName = this.readLengthPrefixedString((SliceInput)this.input).toStringUtf8();
            this.validateWrite(validation -> validation.getCodecClassName().equals(Optional.of(codecClassName)), "Unexpected compression codec", new Object[0]);
            this.decompressor = codecFactory.createDecompressor(codecClassName);
        } else {
            this.validateWrite(validation -> validation.getCodecClassName().equals(Optional.empty()), "Expected file to be compressed", new Object[0]);
            this.decompressor = null;
        }
        int metadataEntries = Integer.reverseBytes(this.input.readInt());
        this.verify(metadataEntries >= 0, "Invalid metadata entry count %s in RCFile %s", metadataEntries, dataSource);
        this.verify(metadataEntries <= 500000, "Too many metadata entries (%s) in RCFile %s", metadataEntries, dataSource);
        ImmutableMap.Builder metadataBuilder = ImmutableMap.builder();
        for (int i = 0; i < metadataEntries; ++i) {
            metadataBuilder.put((Object)this.readLengthPrefixedString((SliceInput)this.input).toStringUtf8(), (Object)this.readLengthPrefixedString((SliceInput)this.input).toStringUtf8());
        }
        this.metadata = metadataBuilder.buildOrThrow();
        this.validateWrite(validation -> validation.getMetadata().equals(this.metadata), "Unexpected metadata", new Object[0]);
        String columnCountString = this.metadata.get(COLUMN_COUNT_METADATA_KEY);
        try {
            this.columnCount = Integer.parseInt(columnCountString);
        }
        catch (NumberFormatException e) {
            throw this.corrupt("Invalid column count %s in RCFile %s", columnCountString, dataSource);
        }
        this.verify(this.columnCount <= 500000, "Too many columns (%s) in RCFile %s", columnCountString, dataSource);
        this.columns = new Column[this.columnCount];
        for (Map.Entry<Integer, Type> entry : readColumns.entrySet()) {
            if (entry.getKey() >= this.columnCount) continue;
            ColumnEncoding columnEncoding = encoding.getEncoding(entry.getValue());
            this.columns[entry.getKey().intValue()] = new Column(columnEncoding, this.decompressor);
        }
        this.syncFirst = this.input.readLong();
        this.validateWrite(validation -> validation.getSyncFirst() == this.syncFirst, "Unexpected sync sequence", new Object[0]);
        this.syncSecond = this.input.readLong();
        this.validateWrite(validation -> validation.getSyncSecond() == this.syncSecond, "Unexpected sync sequence", new Object[0]);
        if (offset != 0L) {
            this.seekToFirstRowGroupInRange(offset, length);
        }
    }

    public byte getVersion() {
        return this.version;
    }

    public Map<String, String> getMetadata() {
        return this.metadata;
    }

    public int getColumnCount() {
        return this.columnCount;
    }

    public long getLength() {
        return this.length;
    }

    public long getBytesRead() {
        return this.dataSource.getReadBytes();
    }

    public long getRowsRead() {
        return this.rowsRead;
    }

    public long getReadTimeNanos() {
        return this.dataSource.getReadTimeNanos();
    }

    public Slice getSync() {
        Slice sync = Slices.allocate((int)16);
        sync.setLong(0, this.syncFirst);
        sync.setLong(8, this.syncSecond);
        return sync;
    }

    @Override
    public void close() throws IOException {
        if (this.closed) {
            return;
        }
        this.closed = true;
        this.rowGroupPosition = 0;
        this.rowGroupRowCount = 0;
        this.currentChunkRowCount = 0;
        try {
            this.input.close();
        }
        finally {
            if (this.decompressor != null) {
                this.decompressor.destroy();
            }
        }
        if (this.writeChecksumBuilder.isPresent()) {
            RcFileWriteValidation.WriteChecksum actualChecksum = this.writeChecksumBuilder.get().build();
            this.validateWrite(validation -> validation.getChecksum().getTotalRowCount() == actualChecksum.getTotalRowCount(), "Invalid row count", new Object[0]);
            List<Long> columnHashes = actualChecksum.getColumnHashes();
            int i = 0;
            while (i < columnHashes.size()) {
                int columnIndex = i++;
                this.validateWrite(validation -> validation.getChecksum().getColumnHashes().get(columnIndex).equals(columnHashes.get(columnIndex)), "Invalid checksum for column %s", columnIndex);
            }
            this.validateWrite(validation -> validation.getChecksum().getRowGroupHash() == actualChecksum.getRowGroupHash(), "Invalid row group checksum", new Object[0]);
        }
    }

    public int advance() throws IOException {
        Slice header;
        if (this.closed) {
            return -1;
        }
        this.rowGroupPosition += 1024;
        this.currentChunkRowCount = Math.min(1024, this.rowGroupRowCount - this.rowGroupPosition);
        if (this.currentChunkRowCount > 0) {
            this.validateWritePageChecksum();
            return this.currentChunkRowCount;
        }
        if (this.input.remaining() == 0L) {
            this.close();
            return -1;
        }
        this.verify(this.input.remaining() >= 4L, "RCFile truncated %s", this.dataSource.getId());
        int unusedRowGroupSize = Integer.reverseBytes(this.input.readInt());
        if (unusedRowGroupSize == -1) {
            this.verify(this.input.remaining() >= 20L, "RCFile truncated %s", this.dataSource.getId());
            if (this.input.position() - 4L >= this.end) {
                this.close();
                return -1;
            }
            this.verify(this.syncFirst == this.input.readLong() && this.syncSecond == this.input.readLong(), "Invalid sync in RCFile %s", this.dataSource.getId());
            unusedRowGroupSize = Integer.reverseBytes(this.input.readInt());
        } else if (this.rowsRead > 0L) {
            this.validateWrite(writeValidation -> false, "Expected sync sequence for every row group except the first one", new Object[0]);
        }
        this.verify(unusedRowGroupSize > 0, "Invalid uncompressed row group length %s", unusedRowGroupSize);
        int uncompressedHeaderSize = Integer.reverseBytes(this.input.readInt());
        int compressedHeaderSize = Integer.reverseBytes(this.input.readInt());
        if (compressedHeaderSize > this.compressedHeaderBuffer.length()) {
            this.compressedHeaderBuffer = Slices.allocate((int)compressedHeaderSize);
        }
        this.input.readBytes(this.compressedHeaderBuffer, 0, compressedHeaderSize);
        if (this.decompressor != null) {
            if (this.headerBuffer.length() < uncompressedHeaderSize) {
                this.headerBuffer = Slices.allocate((int)uncompressedHeaderSize);
            }
            Slice buffer = this.headerBuffer.slice(0, uncompressedHeaderSize);
            this.decompressor.decompress(this.compressedHeaderBuffer, buffer);
            header = buffer;
        } else {
            this.verify(compressedHeaderSize == uncompressedHeaderSize, "Invalid RCFile %s", this.dataSource.getId());
            header = this.compressedHeaderBuffer;
        }
        BasicSliceInput headerInput = header.getInput();
        this.rowGroupRowCount = Math.toIntExact(RcFileDecoderUtils.readVInt((SliceInput)headerInput));
        this.rowsRead += (long)this.rowGroupRowCount;
        this.rowGroupPosition = 0;
        this.currentChunkRowCount = Math.min(1024, this.rowGroupRowCount);
        int totalCompressedDataSize = 0;
        for (int columnIndex = 0; columnIndex < this.columnCount; ++columnIndex) {
            int compressedDataSize = Math.toIntExact(RcFileDecoderUtils.readVInt((SliceInput)headerInput));
            totalCompressedDataSize += compressedDataSize;
            int uncompressedDataSize = Math.toIntExact(RcFileDecoderUtils.readVInt((SliceInput)headerInput));
            if (this.decompressor == null && compressedDataSize != uncompressedDataSize) {
                throw this.corrupt("Invalid RCFile %s", this.dataSource.getId());
            }
            int lengthsSize = Math.toIntExact(RcFileDecoderUtils.readVInt((SliceInput)headerInput));
            Slice lengthsBuffer = headerInput.readSlice(lengthsSize);
            if (this.readColumns.containsKey(columnIndex)) {
                Slice dataBuffer = this.input.readSlice(compressedDataSize);
                this.columns[columnIndex].setBuffers(lengthsBuffer, dataBuffer, uncompressedDataSize);
                continue;
            }
            ByteStreams.skipFully((InputStream)this.input, (long)compressedDataSize);
        }
        this.verify(unusedRowGroupSize == totalCompressedDataSize + uncompressedHeaderSize, "Invalid row group size", new Object[0]);
        this.validateWriteRowGroupChecksum();
        this.validateWritePageChecksum();
        return this.currentChunkRowCount;
    }

    public Block readBlock(int columnIndex) throws IOException {
        Preconditions.checkArgument((boolean)this.readColumns.containsKey(columnIndex), (String)"Column '%s' is not being read", (int)columnIndex);
        Preconditions.checkState((this.currentChunkRowCount > 0 ? 1 : 0) != 0, (Object)"No more data");
        if (columnIndex >= this.columns.length) {
            Type type = this.readColumns.get(columnIndex);
            Block nullBlock = type.createBlockBuilder(null, 1, 0).appendNull().build();
            return RunLengthEncodedBlock.create((Block)nullBlock, (int)this.currentChunkRowCount);
        }
        return this.columns[columnIndex].readBlock(this.rowGroupPosition, this.currentChunkRowCount);
    }

    public RcFileDataSourceId getId() {
        return this.dataSource.getId();
    }

    private void seekToFirstRowGroupInRange(long offset, long length) throws IOException {
        long startOfSyncSequence = RcFileDecoderUtils.findFirstSyncPosition(this.dataSource, offset, length, this.syncFirst, this.syncSecond);
        if (startOfSyncSequence < 0L) {
            this.closeQuietly();
            return;
        }
        this.input.setPosition(startOfSyncSequence);
    }

    private void closeQuietly() {
        try {
            this.close();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private Slice readLengthPrefixedString(SliceInput in) throws RcFileCorruptionException {
        int length = Math.toIntExact(RcFileDecoderUtils.readVInt(in));
        this.verify(length <= 0x100000, "Metadata string value is too long (%s) in RCFile %s", length, in);
        return in.readSlice(length);
    }

    private void verify(boolean expression, String messageFormat, Object ... args) throws RcFileCorruptionException {
        if (!expression) {
            throw this.corrupt(messageFormat, args);
        }
    }

    private RcFileCorruptionException corrupt(String messageFormat, Object ... args) {
        this.closeQuietly();
        return new RcFileCorruptionException(messageFormat, args);
    }

    private void validateWrite(Predicate<RcFileWriteValidation> test, String messageFormat, Object ... args) throws RcFileCorruptionException {
        if (this.writeValidation.isPresent() && !test.test(this.writeValidation.get())) {
            throw this.corrupt("Write validation failed: " + messageFormat, args);
        }
    }

    private void validateWriteRowGroupChecksum() {
        if (this.writeChecksumBuilder.isPresent()) {
            this.writeChecksumBuilder.get().addRowGroup(this.rowGroupRowCount);
        }
    }

    private void validateWritePageChecksum() throws IOException {
        if (this.writeChecksumBuilder.isPresent()) {
            Block[] blocks = new Block[this.columns.length];
            for (int columnIndex = 0; columnIndex < this.columns.length; ++columnIndex) {
                blocks[columnIndex] = this.readBlock(columnIndex);
            }
            this.writeChecksumBuilder.get().addPage(new Page(this.currentChunkRowCount, blocks));
        }
    }

    static void validateFile(RcFileWriteValidation writeValidation, RcFileDataSource input, RcFileEncoding encoding, List<Type> types, RcFileCodecFactory codecFactory) throws RcFileCorruptionException {
        ImmutableMap.Builder readTypes = ImmutableMap.builder();
        for (int columnIndex = 0; columnIndex < types.size(); ++columnIndex) {
            readTypes.put((Object)columnIndex, (Object)types.get(columnIndex));
        }
        try (RcFileReader rcFileReader = new RcFileReader(input, encoding, (Map<Integer, Type>)readTypes.buildOrThrow(), codecFactory, 0L, input.getSize(), DataSize.of((long)8L, (DataSize.Unit)DataSize.Unit.MEGABYTE), Optional.of(writeValidation));){
            while (rcFileReader.advance() >= 0) {
            }
        }
        catch (RcFileCorruptionException e) {
            throw e;
        }
        catch (IOException e) {
            throw new RcFileCorruptionException(e, "Validation failed", new Object[0]);
        }
    }

    private static class DataSourceSliceLoader
    implements ChunkedSliceInput.SliceLoader<ByteArrayBufferReference> {
        private final RcFileDataSource dataSource;

        public DataSourceSliceLoader(RcFileDataSource dataSource) {
            this.dataSource = dataSource;
        }

        public ByteArrayBufferReference createBuffer(int bufferSize) {
            return new ByteArrayBufferReference(bufferSize);
        }

        public long getSize() {
            return this.dataSource.getSize();
        }

        public void load(long position, ByteArrayBufferReference bufferReference, int length) {
            try {
                this.dataSource.readFully(position, bufferReference.getByteBuffer(), 0, length);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }

        public void close() {
            try {
                this.dataSource.close();
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
    }

    private static class Column {
        private final ColumnEncoding encoding;
        private final RcFileDecompressor decompressor;
        private BasicSliceInput lengthsInput;
        private Slice dataBuffer;
        private int uncompressedDataSize;
        private byte[] decompressedBuffer = new byte[0];
        private boolean compressed;
        private int currentPosition;
        private int currentOffset;
        private int runLength;
        private int lastValueLength = -1;

        public Column(ColumnEncoding encoding, RcFileDecompressor decompressor) {
            this.encoding = encoding;
            this.decompressor = decompressor;
        }

        public void setBuffers(Slice lengthsBuffer, Slice dataBuffer, int uncompressedDataSize) {
            this.lengthsInput = lengthsBuffer.getInput();
            this.dataBuffer = dataBuffer;
            this.uncompressedDataSize = uncompressedDataSize;
            this.compressed = this.decompressor != null;
            this.currentPosition = 0;
            this.currentOffset = 0;
            this.runLength = 0;
            this.lastValueLength = 0;
        }

        public Block readBlock(int position, int size) throws IOException {
            Preconditions.checkArgument((size > 0 && size <= 1024 ? 1 : 0) != 0, (Object)"Invalid size");
            Preconditions.checkArgument((this.currentPosition <= position ? 1 : 0) != 0, (Object)"Invalid position");
            if (this.currentPosition < position) {
                this.skipTo(position);
            }
            int[] offsets = this.readOffsets(size);
            ColumnData columnData = new ColumnData(offsets, this.getDataBuffer());
            Block block = this.encoding.decodeColumn(columnData);
            return block;
        }

        private int[] readOffsets(int batchSize) throws IOException {
            int[] offsets = new int[batchSize + 1];
            offsets[0] = this.currentOffset;
            for (int i = 0; i < batchSize; ++i) {
                int valueLength = this.readNextValueLength();
                offsets[i + 1] = offsets[i] + valueLength;
            }
            this.currentOffset = offsets[batchSize];
            this.currentPosition += batchSize;
            return offsets;
        }

        private void skipTo(int position) throws IOException {
            Preconditions.checkArgument((this.currentPosition <= position ? 1 : 0) != 0, (Object)"Invalid position");
            while (this.currentPosition < position) {
                int valueLength = this.readNextValueLength();
                this.currentOffset += valueLength;
                ++this.currentPosition;
            }
        }

        private int readNextValueLength() throws IOException {
            if (this.runLength > 0) {
                --this.runLength;
                return this.lastValueLength;
            }
            int valueLength = Math.toIntExact(RcFileDecoderUtils.readVInt((SliceInput)this.lengthsInput));
            if (valueLength < 0) {
                if (this.lastValueLength == -1) {
                    throw new RcFileCorruptionException("First column value length is negative");
                }
                this.runLength = ~valueLength - 1;
                return this.lastValueLength;
            }
            this.runLength = 0;
            this.lastValueLength = valueLength;
            return valueLength;
        }

        private Slice getDataBuffer() throws IOException {
            if (this.compressed) {
                if (this.decompressedBuffer.length < this.uncompressedDataSize) {
                    this.decompressedBuffer = new byte[this.uncompressedDataSize];
                }
                Slice buffer = Slices.wrappedBuffer((byte[])this.decompressedBuffer, (int)0, (int)this.uncompressedDataSize);
                this.decompressor.decompress(this.dataBuffer, buffer);
                this.dataBuffer = buffer;
                this.compressed = false;
            }
            return this.dataBuffer;
        }
    }

    private static class ByteArrayBufferReference
    implements ChunkedSliceInput.BufferReference {
        private final byte[] byteBuffer;
        private final Slice sliceBuffer;

        public ByteArrayBufferReference(int size) {
            this.byteBuffer = new byte[size];
            this.sliceBuffer = Slices.wrappedBuffer((byte[])this.byteBuffer);
        }

        public byte[] getByteBuffer() {
            return this.byteBuffer;
        }

        public Slice getSlice() {
            return this.sliceBuffer;
        }
    }
}

