/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.cassandra;

import com.datastax.oss.driver.api.core.AllNodesFailedException;
import com.datastax.oss.driver.api.core.CqlIdentifier;
import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.ProtocolVersion;
import com.datastax.oss.driver.api.core.Version;
import com.datastax.oss.driver.api.core.connection.ReconnectionPolicy;
import com.datastax.oss.driver.api.core.cql.PreparedStatement;
import com.datastax.oss.driver.api.core.cql.ResultSet;
import com.datastax.oss.driver.api.core.cql.Row;
import com.datastax.oss.driver.api.core.cql.SimpleStatement;
import com.datastax.oss.driver.api.core.cql.Statement;
import com.datastax.oss.driver.api.core.metadata.Node;
import com.datastax.oss.driver.api.core.metadata.TokenMap;
import com.datastax.oss.driver.api.core.metadata.schema.ColumnMetadata;
import com.datastax.oss.driver.api.core.metadata.schema.IndexMetadata;
import com.datastax.oss.driver.api.core.metadata.schema.KeyspaceMetadata;
import com.datastax.oss.driver.api.core.metadata.schema.RelationMetadata;
import com.datastax.oss.driver.api.core.metadata.schema.TableMetadata;
import com.datastax.oss.driver.api.core.metadata.schema.ViewMetadata;
import com.datastax.oss.driver.api.core.metadata.token.TokenRange;
import com.datastax.oss.driver.api.core.type.DataType;
import com.datastax.oss.driver.api.core.type.ListType;
import com.datastax.oss.driver.api.core.type.MapType;
import com.datastax.oss.driver.api.core.type.SetType;
import com.datastax.oss.driver.api.core.type.TupleType;
import com.datastax.oss.driver.api.core.type.UserDefinedType;
import com.datastax.oss.driver.api.querybuilder.QueryBuilder;
import com.datastax.oss.driver.api.querybuilder.relation.Relation;
import com.datastax.oss.driver.api.querybuilder.select.Select;
import com.datastax.oss.driver.api.querybuilder.term.Term;
import com.google.common.base.Preconditions;
import com.google.common.base.Suppliers;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Ordering;
import com.google.common.collect.Sets;
import io.airlift.json.JsonCodec;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import io.trino.plugin.cassandra.CassandraColumnHandle;
import io.trino.plugin.cassandra.CassandraErrorCode;
import io.trino.plugin.cassandra.CassandraPartition;
import io.trino.plugin.cassandra.CassandraTable;
import io.trino.plugin.cassandra.CassandraTableHandle;
import io.trino.plugin.cassandra.CassandraType;
import io.trino.plugin.cassandra.CassandraTypeManager;
import io.trino.plugin.cassandra.ExtraColumnMetadata;
import io.trino.plugin.cassandra.SizeEstimate;
import io.trino.plugin.cassandra.util.CassandraCqlUtils;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ColumnHandle;
import io.trino.spi.connector.SchemaNotFoundException;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.connector.TableNotFoundException;
import io.trino.spi.predicate.NullableValue;
import io.trino.spi.predicate.TupleDomain;
import java.io.Closeable;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

public class CassandraSession
implements Closeable {
    private static final Logger log = Logger.get(CassandraSession.class);
    private static final String SYSTEM = "system";
    private static final String SIZE_ESTIMATES = "size_estimates";
    private static final Version PARTITION_FETCH_WITH_IN_PREDICATE_VERSION = Version.parse((String)"2.2");
    private final CassandraTypeManager cassandraTypeManager;
    private final JsonCodec<List<ExtraColumnMetadata>> extraColumnMetadataCodec;
    private final Supplier<CqlSession> session;
    private final Duration noHostAvailableRetryTimeout;

    public CassandraSession(CassandraTypeManager cassandraTypeManager, JsonCodec<List<ExtraColumnMetadata>> extraColumnMetadataCodec, Supplier<CqlSession> sessionSupplier, Duration noHostAvailableRetryTimeout) {
        this.cassandraTypeManager = Objects.requireNonNull(cassandraTypeManager, "cassandraTypeManager is null");
        this.extraColumnMetadataCodec = Objects.requireNonNull(extraColumnMetadataCodec, "extraColumnMetadataCodec is null");
        this.noHostAvailableRetryTimeout = Objects.requireNonNull(noHostAvailableRetryTimeout, "noHostAvailableRetryTimeout is null");
        this.session = Suppliers.memoize(sessionSupplier::get);
    }

    public Version getCassandraVersion() {
        ResultSet result = this.executeWithSession(session -> session.execute("select release_version from system.local"));
        Row versionRow = (Row)result.one();
        if (versionRow == null) {
            throw new TrinoException((ErrorCodeSupplier)CassandraErrorCode.CASSANDRA_VERSION_ERROR, "The cluster version is not available. Please make sure that the Cassandra cluster is up and running, and that the contact points are specified correctly.");
        }
        return Version.parse((String)versionRow.getString("release_version"));
    }

    public ProtocolVersion getProtocolVersion() {
        return this.executeWithSession(session -> session.getContext().getProtocolVersion());
    }

    public String getPartitioner() {
        return this.executeWithSession(session -> ((TokenMap)session.getMetadata().getTokenMap().orElseThrow()).getPartitionerName());
    }

    public Set<TokenRange> getTokenRanges() {
        return this.executeWithSession(session -> ((TokenMap)session.getMetadata().getTokenMap().orElseThrow()).getTokenRanges());
    }

    public Set<Node> getReplicas(String caseSensitiveSchemaName, TokenRange tokenRange) {
        Objects.requireNonNull(caseSensitiveSchemaName, "caseSensitiveSchemaName is null");
        Objects.requireNonNull(tokenRange, "tokenRange is null");
        return this.executeWithSession(session -> session.getMetadata().getTokenMap().map(tokenMap -> tokenMap.getReplicas(CassandraCqlUtils.validSchemaName(caseSensitiveSchemaName), tokenRange)).orElse((Set)ImmutableSet.of()));
    }

    public Set<Node> getReplicas(String caseSensitiveSchemaName, ByteBuffer partitionKey) {
        Objects.requireNonNull(caseSensitiveSchemaName, "caseSensitiveSchemaName is null");
        Objects.requireNonNull(partitionKey, "partitionKey is null");
        return this.executeWithSession(session -> session.getMetadata().getTokenMap().map(tokenMap -> tokenMap.getReplicas(CassandraCqlUtils.validSchemaName(caseSensitiveSchemaName), partitionKey)).orElse((Set)ImmutableSet.of()));
    }

    public String getCaseSensitiveSchemaName(String caseInsensitiveSchemaName) {
        return this.getKeyspaceByCaseInsensitiveName(caseInsensitiveSchemaName).getName().asInternal();
    }

    public List<String> getCaseSensitiveSchemaNames() {
        ImmutableList.Builder builder = ImmutableList.builder();
        Map keyspaces = this.executeWithSession(session -> session.getMetadata().getKeyspaces());
        for (KeyspaceMetadata meta : keyspaces.values()) {
            builder.add((Object)meta.getName().asInternal());
        }
        return builder.build();
    }

    public List<String> getCaseSensitiveTableNames(String caseInsensitiveSchemaName) throws SchemaNotFoundException {
        KeyspaceMetadata keyspace = this.getKeyspaceByCaseInsensitiveName(caseInsensitiveSchemaName);
        ImmutableList.Builder builder = ImmutableList.builder();
        for (TableMetadata table : keyspace.getTables().values()) {
            builder.add((Object)table.getName().asInternal());
        }
        for (ViewMetadata materializedView : keyspace.getViews().values()) {
            builder.add((Object)materializedView.getName().asInternal());
        }
        return builder.build();
    }

    public CassandraTable getTable(SchemaTableName schemaTableName) throws TableNotFoundException {
        Optional<CassandraColumnHandle> columnHandle;
        boolean hidden;
        KeyspaceMetadata keyspace = this.getKeyspaceByCaseInsensitiveName(schemaTableName.getSchemaName());
        RelationMetadata tableMeta = CassandraSession.getTableMetadata(keyspace, schemaTableName.getTableName());
        List<String> columnNames = new ArrayList<String>();
        Collection<ColumnMetadata> columns = tableMeta.getColumns().values();
        CassandraSession.checkColumnNames(columns);
        for (ColumnMetadata columnMetadata : columns) {
            columnNames.add(columnMetadata.getName().asInternal());
        }
        Object comment = tableMeta.getOptions().get(CqlIdentifier.fromInternal((String)"comment"));
        Object hiddenColumns = ImmutableSet.of();
        if (comment instanceof String && ((String)comment).startsWith("Presto Metadata:")) {
            String columnOrderingString = ((String)comment).substring("Presto Metadata:".length());
            List extras = (List)this.extraColumnMetadataCodec.fromJson(columnOrderingString);
            ArrayList explicitColumnOrder = new ArrayList(ImmutableList.copyOf((Iterable)Iterables.transform((Iterable)extras, ExtraColumnMetadata::getName)));
            hiddenColumns = (Set)extras.stream().filter(ExtraColumnMetadata::isHidden).map(ExtraColumnMetadata::getName).collect(ImmutableSet.toImmutableSet());
            List remaining = columnNames.stream().filter(name -> !explicitColumnOrder.contains(name)).collect(Collectors.toList());
            explicitColumnOrder.addAll(remaining);
            columnNames = Ordering.explicit((List)explicitColumnOrder).sortedCopy(columnNames);
        }
        ImmutableList.Builder columnHandles = ImmutableList.builder();
        HashSet<CqlIdentifier> primaryKeySet = new HashSet<CqlIdentifier>();
        for (ColumnMetadata columnMeta : tableMeta.getPartitionKey()) {
            primaryKeySet.add(columnMeta.getName());
            hidden = hiddenColumns.contains(columnMeta.getName().asInternal());
            columnHandle = this.buildColumnHandle(tableMeta, columnMeta, true, false, columnNames.indexOf(columnMeta.getName().asInternal()), hidden).orElseThrow(() -> new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Unsupported partition key type: " + columnMeta.getType().asCql(false, false)));
            columnHandles.add((Object)columnHandle);
        }
        for (ColumnMetadata columnMeta : tableMeta.getClusteringColumns().keySet()) {
            primaryKeySet.add(columnMeta.getName());
            hidden = hiddenColumns.contains(columnMeta.getName().asInternal());
            columnHandle = this.buildColumnHandle(tableMeta, columnMeta, false, true, columnNames.indexOf(columnMeta.getName().asInternal()), hidden);
            columnHandle.ifPresent(arg_0 -> ((ImmutableList.Builder)columnHandles).add(arg_0));
        }
        for (ColumnMetadata columnMeta : columns) {
            if (primaryKeySet.contains(columnMeta.getName())) continue;
            hidden = hiddenColumns.contains(columnMeta.getName().asInternal());
            columnHandle = this.buildColumnHandle(tableMeta, columnMeta, false, false, columnNames.indexOf(columnMeta.getName().asInternal()), hidden);
            columnHandle.ifPresent(arg_0 -> ((ImmutableList.Builder)columnHandles).add(arg_0));
        }
        List<CassandraColumnHandle> sortedColumnHandles = columnHandles.build().stream().sorted(Comparator.comparing(CassandraColumnHandle::getOrdinalPosition)).collect(Collectors.toList());
        CassandraTableHandle tableHandle = new CassandraTableHandle(tableMeta.getKeyspace().asInternal(), tableMeta.getName().asInternal());
        return new CassandraTable(tableHandle, sortedColumnHandles);
    }

    private KeyspaceMetadata getKeyspaceByCaseInsensitiveName(String caseInsensitiveSchemaName) throws SchemaNotFoundException {
        Collection keyspaces = this.executeWithSession(session -> session.getMetadata().getKeyspaces()).values();
        KeyspaceMetadata result = null;
        List sortedKeyspaces = (List)keyspaces.stream().sorted(Comparator.comparing(keyspaceMetadata -> keyspaceMetadata.getName().asInternal())).collect(ImmutableList.toImmutableList());
        for (KeyspaceMetadata keyspace : sortedKeyspaces) {
            if (!keyspace.getName().asInternal().equalsIgnoreCase(caseInsensitiveSchemaName)) continue;
            if (result != null) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, String.format("More than one keyspace has been found for the case insensitive schema name: %s -> (%s, %s)", caseInsensitiveSchemaName, result.getName(), keyspace.getName()));
            }
            result = keyspace;
        }
        if (result == null) {
            throw new SchemaNotFoundException(caseInsensitiveSchemaName);
        }
        return result;
    }

    private static RelationMetadata getTableMetadata(KeyspaceMetadata keyspace, String caseInsensitiveTableName) {
        List tables = (List)Stream.concat(keyspace.getTables().values().stream(), keyspace.getViews().values().stream()).filter(table -> table.getName().asInternal().equalsIgnoreCase(caseInsensitiveTableName)).collect(ImmutableList.toImmutableList());
        if (tables.size() == 0) {
            throw new TableNotFoundException(new SchemaTableName(keyspace.getName().asInternal(), caseInsensitiveTableName));
        }
        if (tables.size() == 1) {
            return (RelationMetadata)tables.get(0);
        }
        String tableNames = tables.stream().map(metadata -> metadata.getName().asInternal()).sorted().collect(Collectors.joining(", "));
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, String.format("More than one table has been found for the case insensitive table name: %s -> (%s)", caseInsensitiveTableName, tableNames));
    }

    public boolean isMaterializedView(SchemaTableName schemaTableName) {
        KeyspaceMetadata keyspace = this.getKeyspaceByCaseInsensitiveName(schemaTableName.getSchemaName());
        return keyspace.getView(CassandraCqlUtils.validTableName(schemaTableName.getTableName())).isPresent();
    }

    private static void checkColumnNames(Collection<ColumnMetadata> columns) {
        HashMap<String, ColumnMetadata> lowercaseNameToColumnMap = new HashMap<String, ColumnMetadata>();
        for (ColumnMetadata column : columns) {
            String lowercaseName = column.getName().asInternal().toLowerCase(Locale.ENGLISH);
            if (lowercaseNameToColumnMap.containsKey(lowercaseName)) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, String.format("More than one column has been found for the case insensitive column name: %s -> (%s, %s)", lowercaseName, ((ColumnMetadata)lowercaseNameToColumnMap.get(lowercaseName)).getName(), column.getName()));
            }
            lowercaseNameToColumnMap.put(lowercaseName, column);
        }
    }

    private Optional<CassandraColumnHandle> buildColumnHandle(RelationMetadata tableMetadata, ColumnMetadata columnMeta, boolean partitionKey, boolean clusteringKey, int ordinalPosition, boolean hidden) {
        Optional<CassandraType> cassandraType = this.cassandraTypeManager.toCassandraType(columnMeta.getType());
        if (cassandraType.isEmpty()) {
            log.debug("Unsupported column type: %s", new Object[]{columnMeta.getType().asCql(false, false)});
            return Optional.empty();
        }
        List<DataType> typeArgs = this.getTypeArguments(columnMeta.getType());
        for (DataType typeArgument : typeArgs) {
            if (this.cassandraTypeManager.isFullySupported(typeArgument)) continue;
            log.debug("%s column has unsupported type: %s", new Object[]{columnMeta.getName(), typeArgument});
            return Optional.empty();
        }
        boolean indexed = false;
        SchemaTableName schemaTableName = new SchemaTableName(tableMetadata.getKeyspace().asInternal(), tableMetadata.getName().asInternal());
        if (!this.isMaterializedView(schemaTableName)) {
            TableMetadata table = (TableMetadata)tableMetadata;
            for (IndexMetadata idx : table.getIndexes().values()) {
                if (!idx.getTarget().equals(columnMeta.getName().asInternal())) continue;
                indexed = true;
                break;
            }
        }
        return Optional.of(new CassandraColumnHandle(columnMeta.getName().asInternal(), ordinalPosition, cassandraType.get(), partitionKey, clusteringKey, indexed, hidden));
    }

    public List<CassandraPartition> getPartitions(CassandraTable table, List<Set<Object>> filterPrefixes) {
        Iterable<Row> rows;
        List<CassandraColumnHandle> partitionKeyColumns = table.getPartitionKeyColumns();
        if (filterPrefixes.size() != partitionKeyColumns.size()) {
            return ImmutableList.of((Object)CassandraPartition.UNPARTITIONED);
        }
        if (this.getCassandraVersion().compareTo(PARTITION_FETCH_WITH_IN_PREDICATE_VERSION) > 0) {
            log.debug("Using IN predicate to fetch partitions.");
            rows = this.queryPartitionKeysWithInClauses(table, filterPrefixes);
        } else {
            log.debug("Using combination of partition values to fetch partitions.");
            rows = this.queryPartitionKeysLegacyWithMultipleQueries(table, filterPrefixes);
        }
        ByteBuffer buffer = ByteBuffer.allocate(1000);
        HashMap<CassandraColumnHandle, NullableValue> map = new HashMap<CassandraColumnHandle, NullableValue>();
        HashSet<String> uniquePartitionIds = new HashSet<String>();
        StringBuilder stringBuilder = new StringBuilder();
        boolean isComposite = partitionKeyColumns.size() > 1;
        ImmutableList.Builder partitions = ImmutableList.builder();
        for (Row row : rows) {
            buffer.clear();
            map.clear();
            stringBuilder.setLength(0);
            for (int i = 0; i < partitionKeyColumns.size(); ++i) {
                ByteBuffer component = row.getBytesUnsafe(i).duplicate();
                if (isComposite) {
                    short len = (short)component.limit();
                    buffer.putShort(len);
                    buffer.put(component);
                    buffer.put((byte)0);
                } else {
                    buffer.put(component);
                }
                CassandraColumnHandle columnHandle = partitionKeyColumns.get(i);
                NullableValue keyPart = this.cassandraTypeManager.getColumnValue(columnHandle.getCassandraType(), row, i);
                map.put(columnHandle, keyPart);
                if (i > 0) {
                    stringBuilder.append(" AND ");
                }
                stringBuilder.append(CassandraCqlUtils.validColumnName(columnHandle.getName()));
                stringBuilder.append(" = ");
                stringBuilder.append(this.cassandraTypeManager.getColumnValueForCql(columnHandle.getCassandraType(), row, i));
            }
            buffer.flip();
            byte[] key = new byte[buffer.limit()];
            buffer.get(key);
            TupleDomain tupleDomain = TupleDomain.fromFixedValues(map);
            String partitionId = stringBuilder.toString();
            if (!uniquePartitionIds.add(partitionId)) continue;
            partitions.add((Object)new CassandraPartition(key, partitionId, (TupleDomain<ColumnHandle>)tupleDomain, false));
        }
        return partitions.build();
    }

    public ResultSet execute(String cql) {
        log.debug("Execute cql: %s", new Object[]{cql});
        return this.executeWithSession(session -> session.execute(cql));
    }

    public PreparedStatement prepare(SimpleStatement statement) {
        log.debug("Execute SimpleStatement: %s", new Object[]{statement});
        return this.executeWithSession(session -> session.prepare(statement));
    }

    public ResultSet execute(Statement statement) {
        return this.executeWithSession(session -> session.execute(statement));
    }

    private Iterable<Row> queryPartitionKeysWithInClauses(CassandraTable table, List<Set<Object>> filterPrefixes) {
        CassandraTableHandle tableHandle = table.getTableHandle();
        List<CassandraColumnHandle> partitionKeyColumns = table.getPartitionKeyColumns();
        Select partitionKeys = (Select)CassandraCqlUtils.selectDistinctFrom(tableHandle, partitionKeyColumns).where(this.getInRelations(partitionKeyColumns, filterPrefixes));
        log.debug("Execute cql for partition keys with IN clauses: %s", new Object[]{partitionKeys});
        return this.execute((Statement)partitionKeys.build()).all();
    }

    private Iterable<Row> queryPartitionKeysLegacyWithMultipleQueries(CassandraTable table, List<Set<Object>> filterPrefixes) {
        CassandraTableHandle tableHandle = table.getTableHandle();
        List<CassandraColumnHandle> partitionKeyColumns = table.getPartitionKeyColumns();
        Set filterCombinations = Sets.cartesianProduct(filterPrefixes);
        ImmutableList.Builder rowList = ImmutableList.builder();
        for (List combination : filterCombinations) {
            Select partitionKeys = (Select)CassandraCqlUtils.selectDistinctFrom(tableHandle, partitionKeyColumns).where(this.getEqualityRelations(partitionKeyColumns, combination));
            log.debug("Execute cql for partition keys with multiple queries: %s", new Object[]{partitionKeys});
            List resultRows = this.execute((Statement)partitionKeys.build()).all();
            if (resultRows.isEmpty()) continue;
            rowList.addAll((Iterable)resultRows);
        }
        return rowList.build();
    }

    private List<Relation> getInRelations(List<CassandraColumnHandle> partitionKeyColumns, List<Set<Object>> filterPrefixes) {
        return (List)IntStream.range(0, Math.min(partitionKeyColumns.size(), filterPrefixes.size())).mapToObj(i -> this.getInRelation((CassandraColumnHandle)partitionKeyColumns.get(i), (Set)filterPrefixes.get(i))).collect(ImmutableList.toImmutableList());
    }

    private Relation getInRelation(CassandraColumnHandle column, Set<Object> filterPrefixes) {
        List values = filterPrefixes.stream().map(value -> this.cassandraTypeManager.getJavaValue(column.getCassandraType().getKind(), value)).map(QueryBuilder::literal).collect(Collectors.toList());
        return (Relation)Relation.column((String)CassandraCqlUtils.validColumnName(column.getName())).in(values);
    }

    private List<Relation> getEqualityRelations(List<CassandraColumnHandle> partitionKeyColumns, List<Object> filterPrefix) {
        return (List)IntStream.range(0, Math.min(partitionKeyColumns.size(), filterPrefix.size())).mapToObj(i -> {
            CassandraColumnHandle column = (CassandraColumnHandle)partitionKeyColumns.get(i);
            Object value = this.cassandraTypeManager.getJavaValue(column.getCassandraType().getKind(), filterPrefix.get(i));
            return (Relation)Relation.column((String)CassandraCqlUtils.validColumnName(column.getName())).isEqualTo((Term)QueryBuilder.literal((Object)value));
        }).collect(ImmutableList.toImmutableList());
    }

    public List<SizeEstimate> getSizeEstimates(String keyspaceName, String tableName) {
        this.checkSizeEstimatesTableExist();
        SimpleStatement statement = ((Select)QueryBuilder.selectFrom((String)SYSTEM, (String)SIZE_ESTIMATES).column("partitions_count").where(new Relation[]{(Relation)Relation.column((String)"keyspace_name").isEqualTo((Term)QueryBuilder.literal((Object)keyspaceName)), (Relation)Relation.column((String)"table_name").isEqualTo((Term)QueryBuilder.literal((Object)tableName))})).build();
        ResultSet result = this.executeWithSession(session -> session.execute((Statement)statement));
        ImmutableList.Builder estimates = ImmutableList.builder();
        for (Row row : result.all()) {
            SizeEstimate estimate = new SizeEstimate(row.getLong("partitions_count"));
            estimates.add((Object)estimate);
        }
        return estimates.build();
    }

    private void checkSizeEstimatesTableExist() {
        Optional keyspaceMetadata = this.executeWithSession(session -> session.getMetadata().getKeyspace(SYSTEM));
        Preconditions.checkState((boolean)keyspaceMetadata.isPresent(), (Object)"system keyspace metadata must not be null");
        Optional sizeEstimatesTableMetadata = keyspaceMetadata.flatMap(metadata -> metadata.getTable(SIZE_ESTIMATES));
        if (sizeEstimatesTableMetadata.isEmpty()) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Cassandra versions prior to 2.1.5 are not supported");
        }
    }

    private <T> T executeWithSession(SessionCallable<T> sessionCallable) {
        ReconnectionPolicy reconnectionPolicy = this.session.get().getContext().getReconnectionPolicy();
        ReconnectionPolicy.ReconnectionSchedule schedule = reconnectionPolicy.newControlConnectionSchedule(false);
        long deadline = System.currentTimeMillis() + this.noHostAvailableRetryTimeout.toMillis();
        while (true) {
            try {
                return sessionCallable.executeWithSession(this.session.get());
            }
            catch (AllNodesFailedException e) {
                long timeLeft = deadline - System.currentTimeMillis();
                if (timeLeft <= 0L) {
                    throw e;
                }
                long delay = Math.min(schedule.nextDelay().toMillis(), timeLeft);
                log.warn(e.getMessage());
                log.warn("Reconnecting in %dms", new Object[]{delay});
                try {
                    Thread.sleep(delay);
                }
                catch (InterruptedException interrupted) {
                    Thread.currentThread().interrupt();
                    throw new RuntimeException("interrupted", interrupted);
                }
            }
        }
    }

    private List<DataType> getTypeArguments(DataType dataType) {
        if (dataType instanceof UserDefinedType) {
            return ImmutableList.copyOf((Collection)((UserDefinedType)dataType).getFieldTypes());
        }
        if (dataType instanceof MapType) {
            MapType mapType = (MapType)dataType;
            return ImmutableList.of((Object)mapType.getKeyType(), (Object)mapType.getValueType());
        }
        if (dataType instanceof ListType) {
            return ImmutableList.of((Object)((ListType)dataType).getElementType());
        }
        if (dataType instanceof TupleType) {
            return ImmutableList.copyOf((Collection)((TupleType)dataType).getComponentTypes());
        }
        if (dataType instanceof SetType) {
            return ImmutableList.of((Object)((SetType)dataType).getElementType());
        }
        return ImmutableList.of();
    }

    @Override
    public void close() {
        this.session.get().close();
    }

    private static interface SessionCallable<T> {
        public T executeWithSession(CqlSession var1);
    }
}

