/*
 * Decompiled with CFR 0.152.
 */
package io.trino.execution.buffer;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ListenableFuture;
import io.airlift.slice.Slice;
import io.airlift.units.DataSize;
import io.trino.execution.StateMachine;
import io.trino.execution.buffer.BufferResult;
import io.trino.execution.buffer.BufferState;
import io.trino.execution.buffer.ClientBuffer;
import io.trino.execution.buffer.OutputBuffer;
import io.trino.execution.buffer.OutputBufferInfo;
import io.trino.execution.buffer.OutputBufferMemoryManager;
import io.trino.execution.buffer.OutputBufferStateMachine;
import io.trino.execution.buffer.OutputBufferStatus;
import io.trino.execution.buffer.OutputBuffers;
import io.trino.execution.buffer.PagesSerde;
import io.trino.execution.buffer.PipelinedBufferInfo;
import io.trino.execution.buffer.PipelinedOutputBuffers;
import io.trino.execution.buffer.SerializedPageReference;
import io.trino.memory.context.LocalMemoryContext;
import io.trino.plugin.base.metrics.TDigestHistogram;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;

public class ArbitraryOutputBuffer
implements OutputBuffer {
    private final OutputBufferMemoryManager memoryManager;
    private final SerializedPageReference.PagesReleasedListener onPagesReleased;
    @GuardedBy(value="this")
    private volatile PipelinedOutputBuffers outputBuffers = PipelinedOutputBuffers.createInitial(PipelinedOutputBuffers.BufferType.ARBITRARY);
    private final MasterBuffer masterBuffer;
    @GuardedBy(value="this")
    private final ConcurrentMap<PipelinedOutputBuffers.OutputBufferId, ClientBuffer> buffers = new ConcurrentHashMap<PipelinedOutputBuffers.OutputBufferId, ClientBuffer>();
    private final AtomicInteger nextClientBufferIndex = new AtomicInteger(0);
    private final OutputBufferStateMachine stateMachine;
    private final String taskInstanceId;
    private final AtomicLong totalPagesAdded = new AtomicLong();
    private final AtomicLong totalRowsAdded = new AtomicLong();

    public ArbitraryOutputBuffer(String taskInstanceId, OutputBufferStateMachine stateMachine, DataSize maxBufferSize, Supplier<LocalMemoryContext> memoryContextSupplier, Executor notificationExecutor) {
        this.taskInstanceId = Objects.requireNonNull(taskInstanceId, "taskInstanceId is null");
        this.stateMachine = Objects.requireNonNull(stateMachine, "stateMachine is null");
        Objects.requireNonNull(maxBufferSize, "maxBufferSize is null");
        Preconditions.checkArgument((maxBufferSize.toBytes() > 0L ? 1 : 0) != 0, (Object)"maxBufferSize must be at least 1");
        this.memoryManager = new OutputBufferMemoryManager(maxBufferSize.toBytes(), Objects.requireNonNull(memoryContextSupplier, "memoryContextSupplier is null"), Objects.requireNonNull(notificationExecutor, "notificationExecutor is null"));
        this.onPagesReleased = SerializedPageReference.PagesReleasedListener.forOutputBufferMemoryManager(this.memoryManager);
        this.masterBuffer = new MasterBuffer(this.onPagesReleased);
    }

    @Override
    public void addStateChangeListener(StateMachine.StateChangeListener<BufferState> stateChangeListener) {
        this.stateMachine.addStateChangeListener(stateChangeListener);
    }

    @Override
    public double getUtilization() {
        return this.memoryManager.getUtilization();
    }

    @Override
    public OutputBufferStatus getStatus() {
        return OutputBufferStatus.builder(this.outputBuffers.getVersion()).setOverutilized(this.memoryManager.getUtilization() >= 0.5 || !this.stateMachine.getState().canAddPages()).build();
    }

    @Override
    public OutputBufferInfo getInfo() {
        BufferState state = this.stateMachine.getState();
        Collection buffers = this.buffers.values();
        int totalBufferedPages = this.masterBuffer.getBufferedPages();
        ImmutableList.Builder infos = ImmutableList.builderWithExpectedSize((int)buffers.size());
        for (ClientBuffer buffer : buffers) {
            PipelinedBufferInfo bufferInfo = buffer.getInfo();
            infos.add((Object)bufferInfo);
            totalBufferedPages += bufferInfo.getBufferedPages();
        }
        return new OutputBufferInfo("ARBITRARY", state, state.canAddBuffers(), state.canAddPages(), this.memoryManager.getBufferedBytes(), totalBufferedPages, this.totalRowsAdded.get(), this.totalPagesAdded.get(), Optional.of(infos.build()), Optional.of(new TDigestHistogram(this.memoryManager.getUtilizationHistogram())), Optional.empty());
    }

    @Override
    public BufferState getState() {
        return this.stateMachine.getState();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void setOutputBuffers(OutputBuffers newOutputBuffers) {
        Preconditions.checkState((!Thread.holdsLock(this) ? 1 : 0) != 0, (Object)"Cannot set output buffers while holding a lock on this");
        Objects.requireNonNull(newOutputBuffers, "newOutputBuffers is null");
        Preconditions.checkArgument((boolean)(newOutputBuffers instanceof PipelinedOutputBuffers), (Object)"newOutputBuffers is expected to be an instance of PipelinedOutputBuffers");
        ArbitraryOutputBuffer arbitraryOutputBuffer = this;
        synchronized (arbitraryOutputBuffer) {
            BufferState state = this.stateMachine.getState();
            if (state.isTerminal() || this.outputBuffers.getVersion() >= newOutputBuffers.getVersion()) {
                return;
            }
            this.outputBuffers.checkValidTransition(newOutputBuffers);
            this.outputBuffers = (PipelinedOutputBuffers)newOutputBuffers;
            for (PipelinedOutputBuffers.OutputBufferId outputBufferId : this.outputBuffers.getBuffers().keySet()) {
                this.getBuffer(outputBufferId);
            }
            this.nextClientBufferIndex.set(0);
            if (this.outputBuffers.isNoMoreBufferIds()) {
                this.stateMachine.noMoreBuffers();
            }
        }
        if (!this.stateMachine.getState().canAddBuffers()) {
            this.noMoreBuffers();
        }
        this.checkFlushComplete();
    }

    @Override
    public ListenableFuture<Void> isFull() {
        return this.memoryManager.getBufferBlockedFuture();
    }

    @Override
    public void enqueue(List<Slice> pages) {
        Preconditions.checkState((!Thread.holdsLock(this) ? 1 : 0) != 0, (Object)"Cannot enqueue pages while holding a lock on this");
        Objects.requireNonNull(pages, "pages is null");
        if (!this.stateMachine.getState().canAddPages()) {
            return;
        }
        ImmutableList.Builder references = ImmutableList.builderWithExpectedSize((int)pages.size());
        long bytesAdded = 0L;
        long rowCount = 0L;
        for (Slice page : pages) {
            bytesAdded += page.getRetainedSize();
            int positionCount = PagesSerde.getSerializedPagePositionCount(page);
            rowCount += (long)positionCount;
            references.add((Object)new SerializedPageReference(page, positionCount, 1));
        }
        ImmutableList serializedPageReferences = references.build();
        this.totalRowsAdded.addAndGet(rowCount);
        this.totalPagesAdded.addAndGet(serializedPageReferences.size());
        this.memoryManager.updateMemoryUsage(bytesAdded);
        this.masterBuffer.addPages((List<SerializedPageReference>)serializedPageReferences);
        List<ClientBuffer> buffers = this.safeGetBuffersSnapshot();
        if (buffers.isEmpty()) {
            return;
        }
        int index = this.nextClientBufferIndex.get() % buffers.size();
        for (int i = 0; i < buffers.size(); ++i) {
            buffers.get(index).loadPagesIfNecessary(this.masterBuffer);
            index = (index + 1) % buffers.size();
            if (!this.masterBuffer.isEmpty()) continue;
            this.nextClientBufferIndex.set(index);
            break;
        }
    }

    @Override
    public void enqueue(int partition, List<Slice> pages) {
        Preconditions.checkState((partition == 0 ? 1 : 0) != 0, (Object)"Expected partition number to be zero");
        this.enqueue(pages);
    }

    @Override
    public ListenableFuture<BufferResult> get(PipelinedOutputBuffers.OutputBufferId bufferId, long startingSequenceId, DataSize maxSize) {
        Preconditions.checkState((!Thread.holdsLock(this) ? 1 : 0) != 0, (Object)"Cannot get pages while holding a lock on this");
        Objects.requireNonNull(bufferId, "bufferId is null");
        Preconditions.checkArgument((maxSize.toBytes() > 0L ? 1 : 0) != 0, (Object)"maxSize must be at least 1 byte");
        return this.getBuffer(bufferId).getPages(startingSequenceId, maxSize, Optional.of(this.masterBuffer));
    }

    @Override
    public void acknowledge(PipelinedOutputBuffers.OutputBufferId bufferId, long sequenceId) {
        Preconditions.checkState((!Thread.holdsLock(this) ? 1 : 0) != 0, (Object)"Cannot acknowledge pages while holding a lock on this");
        Objects.requireNonNull(bufferId, "bufferId is null");
        this.getBuffer(bufferId).acknowledgePages(sequenceId);
    }

    @Override
    public void destroy(PipelinedOutputBuffers.OutputBufferId bufferId) {
        Preconditions.checkState((!Thread.holdsLock(this) ? 1 : 0) != 0, (Object)"Cannot destroy while holding a lock on this");
        Objects.requireNonNull(bufferId, "bufferId is null");
        this.getBuffer(bufferId).destroy();
        this.checkFlushComplete();
    }

    @Override
    public void setNoMorePages() {
        Preconditions.checkState((!Thread.holdsLock(this) ? 1 : 0) != 0, (Object)"Cannot set no more pages while holding a lock on this");
        this.stateMachine.noMorePages();
        this.memoryManager.setNoBlockOnFull();
        this.masterBuffer.setNoMorePages();
        for (ClientBuffer clientBuffer : this.safeGetBuffersSnapshot()) {
            clientBuffer.loadPagesIfNecessary(this.masterBuffer);
        }
        this.checkFlushComplete();
    }

    @Override
    public void destroy() {
        Preconditions.checkState((!Thread.holdsLock(this) ? 1 : 0) != 0, (Object)"Cannot destroy while holding a lock on this");
        if (this.stateMachine.finish()) {
            this.noMoreBuffers();
            this.masterBuffer.destroy();
            this.safeGetBuffersSnapshot().forEach(ClientBuffer::destroy);
            this.memoryManager.setNoBlockOnFull();
            this.forceFreeMemory();
        }
    }

    @Override
    public void abort() {
        if (this.stateMachine.abort()) {
            this.memoryManager.setNoBlockOnFull();
            this.forceFreeMemory();
        }
    }

    @Override
    public long getPeakMemoryUsage() {
        return this.memoryManager.getPeakMemoryUsage();
    }

    @Override
    public Optional<Throwable> getFailureCause() {
        return this.stateMachine.getFailureCause();
    }

    @VisibleForTesting
    void forceFreeMemory() {
        this.memoryManager.close();
    }

    private synchronized ClientBuffer getBuffer(PipelinedOutputBuffers.OutputBufferId id) {
        ClientBuffer buffer = (ClientBuffer)this.buffers.get(id);
        if (buffer != null) {
            return buffer;
        }
        Preconditions.checkState((this.stateMachine.getState().canAddBuffers() || !this.outputBuffers.isNoMoreBufferIds() ? 1 : 0) != 0, (Object)"No more buffers already set");
        buffer = new ClientBuffer(this.taskInstanceId, id, this.onPagesReleased);
        if (this.stateMachine.getState() == BufferState.FINISHED) {
            buffer.destroy();
        }
        this.buffers.put(id, buffer);
        return buffer;
    }

    private synchronized List<ClientBuffer> safeGetBuffersSnapshot() {
        return ImmutableList.copyOf(this.buffers.values());
    }

    private synchronized void noMoreBuffers() {
        if (this.outputBuffers.isNoMoreBufferIds()) {
            Sets.SetView undeclaredCreatedBuffers = Sets.difference(this.buffers.keySet(), this.outputBuffers.getBuffers().keySet());
            Preconditions.checkState((boolean)undeclaredCreatedBuffers.isEmpty(), (String)"Final output buffers does not contain all created buffer ids: %s", (Object)undeclaredCreatedBuffers);
        }
    }

    private void checkFlushComplete() {
        BufferState state = this.stateMachine.getState();
        if ((state == BufferState.FLUSHING || state == BufferState.NO_MORE_PAGES && this.masterBuffer.isEmpty()) && this.safeGetBuffersSnapshot().stream().allMatch(ClientBuffer::isDestroyed)) {
            this.destroy();
        }
    }

    @VisibleForTesting
    OutputBufferMemoryManager getMemoryManager() {
        return this.memoryManager;
    }

    @ThreadSafe
    private static class MasterBuffer
    implements ClientBuffer.PagesSupplier {
        private final SerializedPageReference.PagesReleasedListener onPagesReleased;
        @GuardedBy(value="this")
        private final LinkedList<SerializedPageReference> masterBuffer = new LinkedList();
        @GuardedBy(value="this")
        private boolean noMorePages;
        private final AtomicInteger bufferedPages = new AtomicInteger();

        private MasterBuffer(SerializedPageReference.PagesReleasedListener onPagesReleased) {
            this.onPagesReleased = Objects.requireNonNull(onPagesReleased, "onPagesReleased is null");
        }

        public synchronized void addPages(List<SerializedPageReference> pages) {
            this.masterBuffer.addAll(pages);
            this.bufferedPages.set(this.masterBuffer.size());
        }

        public synchronized boolean isEmpty() {
            return this.masterBuffer.isEmpty();
        }

        @Override
        public synchronized boolean mayHaveMorePages() {
            return !this.noMorePages || !this.masterBuffer.isEmpty();
        }

        public synchronized void setNoMorePages() {
            this.noMorePages = true;
        }

        @Override
        public synchronized List<SerializedPageReference> getPages(DataSize maxSize) {
            SerializedPageReference page;
            long maxBytes = maxSize.toBytes();
            ArrayList<SerializedPageReference> pages = new ArrayList<SerializedPageReference>();
            long bytesRemoved = 0L;
            while ((page = this.masterBuffer.peek()) != null && (pages.isEmpty() || (bytesRemoved += page.getRetainedSizeInBytes()) <= maxBytes)) {
                Preconditions.checkState((this.masterBuffer.poll() == page ? 1 : 0) != 0, (Object)"Master buffer corrupted");
                pages.add(page);
            }
            this.bufferedPages.set(this.masterBuffer.size());
            return ImmutableList.copyOf(pages);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void destroy() {
            ImmutableList pages;
            Preconditions.checkState((!Thread.holdsLock(this) ? 1 : 0) != 0, (Object)"Cannot destroy master buffer while holding a lock on this");
            MasterBuffer masterBuffer = this;
            synchronized (masterBuffer) {
                pages = ImmutableList.copyOf(this.masterBuffer);
                this.masterBuffer.clear();
                this.bufferedPages.set(0);
            }
            SerializedPageReference.dereferencePages((List<SerializedPageReference>)pages, this.onPagesReleased);
        }

        public int getBufferedPages() {
            return this.bufferedPages.get();
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("bufferedPages", this.bufferedPages.get()).toString();
        }
    }
}

