/*
 * Decompiled with CFR 0.152.
 */
package io.trino.operator.output;

import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.primitives.Ints;
import com.google.common.util.concurrent.ListenableFuture;
import io.airlift.slice.Slice;
import io.airlift.units.DataSize;
import io.trino.execution.buffer.OutputBuffer;
import io.trino.execution.buffer.PageSplitterUtil;
import io.trino.execution.buffer.PagesSerde;
import io.trino.execution.buffer.PagesSerdeFactory;
import io.trino.operator.OperatorContext;
import io.trino.operator.PartitionFunction;
import io.trino.operator.output.PartitionedOutputOperator;
import io.trino.operator.output.PositionsAppenderFactory;
import io.trino.operator.output.PositionsAppenderPageBuilder;
import io.trino.spi.Page;
import io.trino.spi.PageBuilder;
import io.trino.spi.block.Block;
import io.trino.spi.block.DictionaryBlock;
import io.trino.spi.block.RunLengthEncodedBlock;
import io.trino.spi.predicate.NullableValue;
import io.trino.spi.type.Type;
import it.unimi.dsi.fastutil.ints.IntArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.IntUnaryOperator;
import java.util.function.Supplier;
import javax.annotation.Nullable;

public class PagePartitioner {
    private static final int COLUMNAR_STRATEGY_COEFFICIENT = 2;
    private final OutputBuffer outputBuffer;
    private final Type[] sourceTypes;
    private final PartitionFunction partitionFunction;
    private final int[] partitionChannels;
    @Nullable
    private final Block[] partitionConstantBlocks;
    private final PagesSerde serde;
    private final PageBuilder[] pageBuilders;
    private final PositionsAppenderPageBuilder[] positionsAppenders;
    private final boolean replicatesAnyRow;
    private final int nullChannel;
    private final AtomicLong rowsAdded = new AtomicLong();
    private final AtomicLong pagesAdded = new AtomicLong();
    private final OperatorContext operatorContext;
    private boolean hasAnyRowBeenReplicated;

    public PagePartitioner(PartitionFunction partitionFunction, List<Integer> partitionChannels, List<Optional<NullableValue>> partitionConstants, boolean replicatesAnyRow, OptionalInt nullChannel, OutputBuffer outputBuffer, PagesSerdeFactory serdeFactory, List<Type> sourceTypes, DataSize maxMemory, OperatorContext operatorContext, PositionsAppenderFactory positionsAppenderFactory) {
        int i;
        this.partitionFunction = Objects.requireNonNull(partitionFunction, "partitionFunction is null");
        this.partitionChannels = Ints.toArray((Collection)Objects.requireNonNull(partitionChannels, "partitionChannels is null"));
        Objects.requireNonNull(positionsAppenderFactory, "positionsAppenderFactory is null");
        Block[] partitionConstantBlocks = (Block[])partitionConstants.stream().map(constant -> constant.map(NullableValue::asBlock).orElse(null)).toArray(Block[]::new);
        this.partitionConstantBlocks = Arrays.stream(partitionConstantBlocks).anyMatch(Objects::nonNull) ? partitionConstantBlocks : null;
        this.replicatesAnyRow = replicatesAnyRow;
        this.nullChannel = nullChannel.orElse(-1);
        this.outputBuffer = Objects.requireNonNull(outputBuffer, "outputBuffer is null");
        this.sourceTypes = sourceTypes.toArray(new Type[0]);
        this.serde = serdeFactory.createPagesSerde();
        this.operatorContext = Objects.requireNonNull(operatorContext, "operatorContext is null");
        for (int i2 = 0; i2 < this.partitionChannels.length; ++i2) {
            if (this.partitionChannels[i2] >= 0) continue;
            Preconditions.checkArgument((this.partitionConstantBlocks != null && this.partitionConstantBlocks[i2] != null ? 1 : 0) != 0, (String)"Expected constant for partitioning channel %s, but none was found", (int)i2);
        }
        int partitionCount = partitionFunction.getPartitionCount();
        int pageSize = Math.toIntExact(Math.min(0x100000L, maxMemory.toBytes() / (long)partitionCount));
        pageSize = Math.max(1, pageSize);
        this.positionsAppenders = new PositionsAppenderPageBuilder[partitionCount];
        for (i = 0; i < partitionCount; ++i) {
            this.positionsAppenders[i] = PositionsAppenderPageBuilder.withMaxPageSize(pageSize, sourceTypes, positionsAppenderFactory);
        }
        this.pageBuilders = new PageBuilder[partitionCount];
        for (i = 0; i < partitionCount; ++i) {
            this.pageBuilders[i] = PageBuilder.withMaxPageSize((int)pageSize, sourceTypes);
        }
    }

    public ListenableFuture<Void> isFull() {
        return this.outputBuffer.isFull();
    }

    public long getSizeInBytes() {
        long sizeInBytes = 0L;
        for (PositionsAppenderPageBuilder positionsAppenderPageBuilder : this.positionsAppenders) {
            sizeInBytes += positionsAppenderPageBuilder.getSizeInBytes();
        }
        for (PositionsAppenderPageBuilder positionsAppenderPageBuilder : this.pageBuilders) {
            sizeInBytes += positionsAppenderPageBuilder.getSizeInBytes();
        }
        return sizeInBytes;
    }

    public long getRetainedSizeInBytes() {
        long sizeInBytes = 0L;
        for (PositionsAppenderPageBuilder positionsAppenderPageBuilder : this.positionsAppenders) {
            sizeInBytes += positionsAppenderPageBuilder.getRetainedSizeInBytes();
        }
        for (PositionsAppenderPageBuilder positionsAppenderPageBuilder : this.pageBuilders) {
            sizeInBytes += positionsAppenderPageBuilder.getRetainedSizeInBytes();
        }
        return sizeInBytes;
    }

    public Supplier<PartitionedOutputOperator.PartitionedOutputInfo> getOperatorInfoSupplier() {
        return PagePartitioner.createPartitionedOutputOperatorInfoSupplier(this.rowsAdded, this.pagesAdded, this.outputBuffer);
    }

    private static Supplier<PartitionedOutputOperator.PartitionedOutputInfo> createPartitionedOutputOperatorInfoSupplier(AtomicLong rowsAdded, AtomicLong pagesAdded, OutputBuffer outputBuffer) {
        Objects.requireNonNull(rowsAdded, "rowsAdded is null");
        Objects.requireNonNull(pagesAdded, "pagesAdded is null");
        Objects.requireNonNull(outputBuffer, "outputBuffer is null");
        return () -> new PartitionedOutputOperator.PartitionedOutputInfo(rowsAdded.get(), pagesAdded.get(), outputBuffer.getPeakMemoryUsage());
    }

    public void partitionPage(Page page) {
        if (page.getPositionCount() == 0) {
            return;
        }
        if (page.getPositionCount() < this.partitionFunction.getPartitionCount() * 2) {
            this.partitionPageByRow(page);
        } else {
            this.partitionPageByColumn(page);
        }
    }

    public void partitionPageByRow(Page page) {
        int position;
        Objects.requireNonNull(page, "page is null");
        if (page.getPositionCount() == 0) {
            return;
        }
        if (this.replicatesAnyRow && !this.hasAnyRowBeenReplicated) {
            for (PageBuilder pageBuilder : this.pageBuilders) {
                this.appendRow(pageBuilder, page, 0);
            }
            this.hasAnyRowBeenReplicated = true;
            position = 1;
        } else {
            position = 0;
        }
        Page partitionFunctionArgs = this.getPartitionFunctionArguments(page);
        if (this.nullChannel >= 0 && page.getBlock(this.nullChannel).mayHaveNull()) {
            Block nullsBlock = page.getBlock(this.nullChannel);
            while (position < page.getPositionCount()) {
                if (nullsBlock.isNull(position)) {
                    for (PageBuilder pageBuilder : this.pageBuilders) {
                        this.appendRow(pageBuilder, page, position);
                    }
                } else {
                    int partition = this.partitionFunction.getPartition(partitionFunctionArgs, position);
                    this.appendRow(this.pageBuilders[partition], page, position);
                }
                ++position;
            }
        } else {
            while (position < page.getPositionCount()) {
                int partition = this.partitionFunction.getPartition(partitionFunctionArgs, position);
                this.appendRow(this.pageBuilders[partition], page, position);
                ++position;
            }
        }
        this.flushPageBuilders(false);
    }

    private void appendRow(PageBuilder pageBuilder, Page page, int position) {
        pageBuilder.declarePosition();
        for (int channel = 0; channel < this.sourceTypes.length; ++channel) {
            Type type = this.sourceTypes[channel];
            type.appendTo(page.getBlock(channel), position, pageBuilder.getBlockBuilder(channel));
        }
    }

    public void partitionPageByColumn(Page page) {
        IntArrayList[] partitionedPositions = this.partitionPositions(page);
        for (int i = 0; i < this.partitionFunction.getPartitionCount(); ++i) {
            IntArrayList partitionPositions = partitionedPositions[i];
            if (partitionPositions.isEmpty()) continue;
            this.positionsAppenders[i].appendToOutputPartition(page, partitionPositions);
            partitionPositions.clear();
        }
        this.flushPositionsAppenders(false);
    }

    private IntArrayList[] partitionPositions(Page page) {
        int position;
        Verify.verify((page.getPositionCount() > 0 ? 1 : 0) != 0, (String)"position count is 0", (Object[])new Object[0]);
        IntArrayList[] partitionPositions = this.initPositions(page);
        if (this.replicatesAnyRow && !this.hasAnyRowBeenReplicated) {
            for (IntArrayList partitionPosition : partitionPositions) {
                partitionPosition.add(0);
            }
            this.hasAnyRowBeenReplicated = true;
            position = 1;
        } else {
            position = 0;
        }
        Page partitionFunctionArgs = this.getPartitionFunctionArguments(page);
        if (partitionFunctionArgs.getChannelCount() > 0 && this.onlyRleBlocks(partitionFunctionArgs)) {
            this.partitionBySingleRleValue(page, position, partitionFunctionArgs, partitionPositions);
        } else if (partitionFunctionArgs.getChannelCount() == 1 && this.isDictionaryProcessingFaster(partitionFunctionArgs.getBlock(0))) {
            this.partitionBySingleDictionary(page, position, partitionFunctionArgs, partitionPositions);
        } else {
            this.partitionGeneric(page, position, aPosition -> this.partitionFunction.getPartition(partitionFunctionArgs, aPosition), partitionPositions);
        }
        return partitionPositions;
    }

    private IntArrayList[] initPositions(Page page) {
        IntArrayList[] partitionPositions = new IntArrayList[this.partitionFunction.getPartitionCount()];
        for (int i = 0; i < partitionPositions.length; ++i) {
            partitionPositions[i] = new IntArrayList(PagePartitioner.initialPartitionSize(page.getPositionCount() / this.partitionFunction.getPartitionCount()));
        }
        return partitionPositions;
    }

    private static int initialPartitionSize(int averagePositionsPerPartition) {
        return (int)((double)averagePositionsPerPartition * 1.1) + 32;
    }

    private boolean onlyRleBlocks(Page page) {
        for (int i = 0; i < page.getChannelCount(); ++i) {
            if (page.getBlock(i) instanceof RunLengthEncodedBlock) continue;
            return false;
        }
        return true;
    }

    private void partitionBySingleRleValue(Page page, int position, Page partitionFunctionArgs, IntArrayList[] partitionPositions) {
        if (this.nullChannel != -1 && page.getBlock(this.nullChannel).isNull(0)) {
            Verify.verify((boolean)(page.getBlock(this.nullChannel) instanceof RunLengthEncodedBlock), (String)"null channel is not RunLengthEncodedBlock", (Object)page.getBlock(this.nullChannel));
            int[] allPositions = this.integersInRange(position, page.getPositionCount());
            for (IntArrayList partitionPosition : partitionPositions) {
                partitionPosition.addElements(position, allPositions);
            }
        } else {
            Page rlePage = this.extractRlePage(partitionFunctionArgs);
            int partition = this.partitionFunction.getPartition(rlePage, 0);
            IntArrayList positions = partitionPositions[partition];
            for (int i = position; i < page.getPositionCount(); ++i) {
                positions.add(i);
            }
        }
    }

    private Page extractRlePage(Page page) {
        Block[] valueBlocks = new Block[page.getChannelCount()];
        for (int channel = 0; channel < valueBlocks.length; ++channel) {
            valueBlocks[channel] = ((RunLengthEncodedBlock)page.getBlock(channel)).getValue();
        }
        return new Page(valueBlocks);
    }

    private int[] integersInRange(int start, int endExclusive) {
        int[] array = new int[endExclusive - start];
        int current = start;
        for (int i = 0; i < array.length; ++i) {
            array[i] = current++;
        }
        return array;
    }

    private boolean isDictionaryProcessingFaster(Block block) {
        if (!(block instanceof DictionaryBlock)) {
            return false;
        }
        DictionaryBlock dictionaryBlock = (DictionaryBlock)block;
        return dictionaryBlock.getPositionCount() > dictionaryBlock.getDictionary().getPositionCount();
    }

    private void partitionBySingleDictionary(Page page, int position, Page partitionFunctionArgs, IntArrayList[] partitionPositions) {
        DictionaryBlock dictionaryBlock = (DictionaryBlock)partitionFunctionArgs.getBlock(0);
        Block dictionary = dictionaryBlock.getDictionary();
        int[] dictionaryPartitions = new int[dictionary.getPositionCount()];
        Page dictionaryPage = new Page(new Block[]{dictionary});
        for (int i = 0; i < dictionary.getPositionCount(); ++i) {
            dictionaryPartitions[i] = this.partitionFunction.getPartition(dictionaryPage, i);
        }
        this.partitionGeneric(page, position, aPosition -> dictionaryPartitions[dictionaryBlock.getId(aPosition)], partitionPositions);
    }

    private void partitionGeneric(Page page, int position, IntUnaryOperator partitionFunction, IntArrayList[] partitionPositions) {
        if (this.nullChannel != -1 && page.getBlock(this.nullChannel).mayHaveNull()) {
            this.partitionNullablePositions(page, position, partitionPositions, partitionFunction);
        } else {
            this.partitionNotNullPositions(page, position, partitionPositions, partitionFunction);
        }
    }

    private IntArrayList[] partitionNullablePositions(Page page, int position, IntArrayList[] partitionPositions, IntUnaryOperator partitionFunction) {
        Block nullsBlock = page.getBlock(this.nullChannel);
        int[] nullPositions = new int[page.getPositionCount()];
        int[] nonNullPositions = new int[page.getPositionCount()];
        int nullCount = 0;
        int nonNullCount = 0;
        for (int i = position; i < page.getPositionCount(); ++i) {
            nullPositions[nullCount] = i;
            nonNullPositions[nonNullCount] = i;
            int isNull = nullsBlock.isNull(i) ? 1 : 0;
            nullCount += isNull;
            nonNullCount += isNull ^ 1;
        }
        for (IntArrayList positions : partitionPositions) {
            positions.addElements(position, nullPositions, 0, nullCount);
        }
        for (int i = 0; i < nonNullCount; ++i) {
            int nonNullPosition = nonNullPositions[i];
            int partition = partitionFunction.applyAsInt(nonNullPosition);
            partitionPositions[partition].add(nonNullPosition);
        }
        return partitionPositions;
    }

    private IntArrayList[] partitionNotNullPositions(Page page, int startingPosition, IntArrayList[] partitionPositions, IntUnaryOperator partitionFunction) {
        int position;
        int positionCount = page.getPositionCount();
        int[] partitionPerPosition = new int[positionCount];
        for (position = startingPosition; position < positionCount; ++position) {
            int partition;
            partitionPerPosition[position] = partition = partitionFunction.applyAsInt(position);
        }
        for (position = startingPosition; position < positionCount; ++position) {
            partitionPositions[partitionPerPosition[position]].add(position);
        }
        return partitionPositions;
    }

    private Page getPartitionFunctionArguments(Page page) {
        if (this.partitionConstantBlocks == null) {
            return page.getColumns(this.partitionChannels);
        }
        Block[] blocks = new Block[this.partitionChannels.length];
        for (int i = 0; i < blocks.length; ++i) {
            int channel = this.partitionChannels[i];
            blocks[i] = channel < 0 ? RunLengthEncodedBlock.create((Block)this.partitionConstantBlocks[i], (int)page.getPositionCount()) : page.getBlock(channel);
        }
        return new Page(page.getPositionCount(), blocks);
    }

    public void forceFlush() {
        this.flushPositionsAppenders(true);
        this.flushPageBuilders(true);
    }

    private void flushPageBuilders(boolean force) {
        try (PagesSerde.PagesSerdeContext context = this.serde.newContext();){
            for (int partition = 0; partition < this.pageBuilders.length; ++partition) {
                PageBuilder partitionPageBuilder = this.pageBuilders[partition];
                if (partitionPageBuilder.isEmpty() || !force && !partitionPageBuilder.isFull()) continue;
                Page pagePartition = partitionPageBuilder.build();
                partitionPageBuilder.reset();
                this.enqueuePage(pagePartition, partition, context);
            }
        }
    }

    private void flushPositionsAppenders(boolean force) {
        try (PagesSerde.PagesSerdeContext context = this.serde.newContext();){
            for (int partition = 0; partition < this.positionsAppenders.length; ++partition) {
                PositionsAppenderPageBuilder partitionPageBuilder = this.positionsAppenders[partition];
                if (partitionPageBuilder.isEmpty() || !force && !partitionPageBuilder.isFull()) continue;
                Page pagePartition = partitionPageBuilder.build();
                this.enqueuePage(pagePartition, partition, context);
            }
        }
    }

    private void enqueuePage(Page pagePartition, int partition, PagesSerde.PagesSerdeContext context) {
        this.operatorContext.recordOutput(pagePartition.getSizeInBytes(), pagePartition.getPositionCount());
        this.outputBuffer.enqueue(partition, this.splitAndSerializePage(context, pagePartition));
        this.pagesAdded.incrementAndGet();
        this.rowsAdded.addAndGet(pagePartition.getPositionCount());
    }

    private List<Slice> splitAndSerializePage(PagesSerde.PagesSerdeContext context, Page page) {
        List<Page> split = PageSplitterUtil.splitPage(page, 0x100000L);
        ImmutableList.Builder builder = ImmutableList.builderWithExpectedSize((int)split.size());
        for (Page chunk : split) {
            builder.add((Object)this.serde.serialize(context, chunk));
        }
        return builder.build();
    }
}

