/*
 * Decompiled with CFR 0.152.
 */
package io.trino.type;

import com.google.common.base.Preconditions;
import io.trino.spi.block.Block;
import io.trino.spi.type.DateTimeEncoding;
import io.trino.spi.type.LongTimeWithTimeZone;
import io.trino.spi.type.LongTimestamp;
import io.trino.spi.type.LongTimestampWithTimeZone;
import io.trino.spi.type.TimeZoneKey;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.TimestampWithTimeZoneType;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoField;
import java.util.List;
import java.util.function.Consumer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public final class DateTimes {
    public static final Pattern DATETIME_PATTERN = Pattern.compile("(?<year>[-+]?\\d{4,})-(?<month>\\d{1,2})-(?<day>\\d{1,2})(?: (?<hour>\\d{1,2}):(?<minute>\\d{1,2})(?::(?<second>\\d{1,2})(?:\\.(?<fraction>\\d+))?)?)?\\s*(?<timezone>.+)?");
    private static final DateTimeFormatter TIMESTAMP_FORMATTER = DateTimeFormatter.ofPattern("uuuu-MM-dd HH:mm:ss");
    public static final Pattern TIME_PATTERN = Pattern.compile("(?<hour>\\d{1,2}):(?<minute>\\d{1,2})(?::(?<second>\\d{1,2})(?:\\.(?<fraction>\\d+))?)?\\s*((?<sign>[+-])(?<offsetHour>\\d\\d):(?<offsetMinute>\\d\\d))?");
    private static final long[] POWERS_OF_TEN = new long[]{1L, 10L, 100L, 1000L, 10000L, 100000L, 1000000L, 10000000L, 100000000L, 1000000000L, 10000000000L, 100000000000L, 1000000000000L};
    public static final int MILLISECONDS_PER_SECOND = 1000;
    public static final long MILLISECONDS_PER_MINUTE = 60000L;
    public static final long MILLISECONDS_PER_DAY = 86400000L;
    public static final int MICROSECONDS_PER_SECOND = 1000000;
    public static final int MICROSECONDS_PER_MILLISECOND = 1000;
    public static final long MICROSECONDS_PER_DAY = 86400000000L;
    public static final long PICOSECONDS_PER_SECOND = 1000000000000L;
    public static final long NANOSECONDS_PER_SECOND = 1000000000L;
    public static final long NANOSECONDS_PER_MINUTE = 60000000000L;
    public static final long NANOSECONDS_PER_HOUR = 3600000000000L;
    public static final long NANOSECONDS_PER_DAY = 86400000000000L;
    public static final int NANOSECONDS_PER_MILLISECOND = 1000000;
    public static final int NANOSECONDS_PER_MICROSECOND = 1000;
    public static final int PICOSECONDS_PER_MILLISECOND = 1000000000;
    public static final int PICOSECONDS_PER_MICROSECOND = 1000000;
    public static final int PICOSECONDS_PER_NANOSECOND = 1000;
    public static final long SECONDS_PER_MINUTE = 60L;
    public static final long MINUTES_PER_HOUR = 60L;
    public static final long HOURS_PER_DAY = 24L;
    public static final long PICOSECONDS_PER_MINUTE = 60000000000000L;
    public static final long PICOSECONDS_PER_HOUR = 3600000000000000L;
    public static final long PICOSECONDS_PER_DAY = 86400000000000000L;
    public static final long SECONDS_PER_DAY = 86400L;

    private DateTimes() {
    }

    private static long roundDiv(long value, long factor) {
        Preconditions.checkArgument((factor > 0L ? 1 : 0) != 0, (Object)"factor must be positive");
        if (factor == 1L) {
            return value;
        }
        if (value >= 0L) {
            return (value + factor / 2L) / factor;
        }
        return (value + 1L - factor / 2L) / factor;
    }

    public static long scaleEpochMicrosToMillis(long value) {
        return Math.floorDiv(value, 1000);
    }

    public static long epochMicrosToMillisWithRounding(long epochMicros) {
        return DateTimes.roundDiv(epochMicros, 1000L);
    }

    public static long scaleEpochMillisToSeconds(long epochMillis) {
        return Math.floorDiv(epochMillis, 1000);
    }

    public static long scaleEpochMicrosToSeconds(long epochMicros) {
        return Math.floorDiv(epochMicros, 1000000);
    }

    public static long scaleEpochMillisToMicros(long epochMillis) {
        return Math.multiplyExact(epochMillis, 1000);
    }

    public static long epochSecondToMicrosWithRounding(long epochSecond, long picoOfSecond) {
        return epochSecond * 1000000L + DateTimes.roundDiv(picoOfSecond, 1000000L);
    }

    public static int getMicrosOfSecond(long epochMicros) {
        return Math.floorMod(epochMicros, 1000000);
    }

    public static int getMillisOfSecond(long epochMillis) {
        return Math.floorMod(epochMillis, 1000);
    }

    public static int getMicrosOfMilli(long epochMicros) {
        return Math.floorMod(epochMicros, 1000);
    }

    public static long toEpochMicros(long epochMillis, int picosOfMilli) {
        return DateTimes.scaleEpochMillisToMicros(epochMillis) + (long)(picosOfMilli / 1000000);
    }

    public static long round(long value, int magnitude) {
        return DateTimes.roundToNearest(value, POWERS_OF_TEN[magnitude]);
    }

    public static long roundToNearest(long value, long bound) {
        return DateTimes.roundDiv(value, bound) * bound;
    }

    public static long scaleFactor(int fromPrecision, int toPrecision) {
        if (fromPrecision > toPrecision) {
            throw new IllegalArgumentException("fromPrecision must be <= toPrecision");
        }
        return POWERS_OF_TEN[toPrecision - fromPrecision];
    }

    public static long rescale(long value, int fromPrecision, int toPrecision) {
        value = fromPrecision <= toPrecision ? (value *= DateTimes.scaleFactor(fromPrecision, toPrecision)) : (value /= DateTimes.scaleFactor(toPrecision, fromPrecision));
        return value;
    }

    public static long rescaleWithRounding(long value, int fromPrecision, int toPrecision) {
        value = DateTimes.round(value, fromPrecision - toPrecision);
        value = DateTimes.rescale(value, fromPrecision, toPrecision);
        return value;
    }

    public static boolean timestampHasTimeZone(String value) {
        Matcher matcher = DATETIME_PATTERN.matcher(value);
        if (!matcher.matches()) {
            throw new IllegalArgumentException(String.format("Invalid timestamp '%s'", value));
        }
        return matcher.group("timezone") != null;
    }

    public static int extractTimestampPrecision(String value) {
        Matcher matcher = DATETIME_PATTERN.matcher(value);
        if (!matcher.matches()) {
            throw new IllegalArgumentException(String.format("Invalid timestamp '%s'", value));
        }
        String fraction = matcher.group("fraction");
        if (fraction == null) {
            return 0;
        }
        return fraction.length();
    }

    public static LocalDateTime toLocalDateTime(TimestampType type, Block block, int position) {
        long epochMicros;
        int precision = type.getPrecision();
        int picosOfMicro = 0;
        if (precision <= 6) {
            epochMicros = type.getLong(block, position);
        } else {
            LongTimestamp timestamp = (LongTimestamp)type.getObject(block, position);
            epochMicros = timestamp.getEpochMicros();
            picosOfMicro = timestamp.getPicosOfMicro();
        }
        long epochSecond = DateTimes.scaleEpochMicrosToSeconds(epochMicros);
        int nanoFraction = DateTimes.getMicrosOfSecond(epochMicros) * 1000 + (int)(DateTimes.roundToNearest(picosOfMicro, 1000L) / 1000L);
        Instant instant = Instant.ofEpochSecond(epochSecond, nanoFraction);
        return LocalDateTime.ofInstant(instant, ZoneOffset.UTC);
    }

    public static ZonedDateTime toZonedDateTime(TimestampWithTimeZoneType type, Block block, int position) {
        ZoneId zoneId;
        long epochMillis;
        int precision = type.getPrecision();
        int picosOfMilli = 0;
        if (precision <= 3) {
            long packedEpochMillis = type.getLong(block, position);
            epochMillis = DateTimeEncoding.unpackMillisUtc((long)packedEpochMillis);
            zoneId = DateTimeEncoding.unpackZoneKey((long)packedEpochMillis).getZoneId();
        } else {
            LongTimestampWithTimeZone timestamp = (LongTimestampWithTimeZone)type.getObject(block, position);
            epochMillis = timestamp.getEpochMillis();
            picosOfMilli = timestamp.getPicosOfMilli();
            zoneId = TimeZoneKey.getTimeZoneKey((short)timestamp.getTimeZoneKey()).getZoneId();
        }
        long epochSecond = DateTimes.scaleEpochMillisToSeconds(epochMillis);
        int nanoFraction = DateTimes.getMillisOfSecond(epochMillis) * 1000000 + (int)(DateTimes.roundToNearest(picosOfMilli, 1000L) / 1000L);
        return Instant.ofEpochSecond(epochSecond, nanoFraction).atZone(zoneId);
    }

    public static String formatTimestamp(int precision, long epochMicros, int picosOfMicro, ZoneId zoneId) {
        return DateTimes.formatTimestamp(precision, epochMicros, picosOfMicro, zoneId, TIMESTAMP_FORMATTER);
    }

    public static String formatTimestamp(int precision, long epochMicros, int picosOfMicro, ZoneId zoneId, DateTimeFormatter yearToSecondFormatter) {
        Preconditions.checkArgument((picosOfMicro >= 0 && picosOfMicro < 1000000 ? 1 : 0) != 0, (Object)"picosOfMicro is out of range [0, 1_000_000]");
        Instant instant = Instant.ofEpochSecond(DateTimes.scaleEpochMicrosToSeconds(epochMicros));
        LocalDateTime dateTime = LocalDateTime.ofInstant(instant, zoneId);
        long picoFraction = (long)DateTimes.getMicrosOfSecond(epochMicros) * 1000000L + (long)picosOfMicro;
        return DateTimes.formatTimestamp(precision, dateTime, picoFraction, yearToSecondFormatter, (StringBuilder builder) -> {});
    }

    public static String formatTimestampWithTimeZone(int precision, long epochMillis, int picosOfMilli, ZoneId zoneId) {
        Instant instant = Instant.ofEpochMilli(epochMillis);
        LocalDateTime dateTime = LocalDateTime.ofInstant(instant, zoneId);
        long picoFraction = (long)DateTimes.getMillisOfSecond(epochMillis) * 1000000000L + (long)picosOfMilli;
        return DateTimes.formatTimestamp(precision, dateTime, picoFraction, TIMESTAMP_FORMATTER, (StringBuilder builder) -> builder.append(" ").append(zoneId));
    }

    public static String formatTimestamp(int precision, LocalDateTime dateTime, long picoFraction, DateTimeFormatter yearToSecondFormatter, Consumer<StringBuilder> zoneIdFormatter) {
        StringBuilder builder = new StringBuilder();
        builder.append(yearToSecondFormatter.format(dateTime));
        if (precision > 0) {
            builder.append(".");
            builder.append(String.format("%0" + precision + "d", DateTimes.rescale(picoFraction, 12, precision)));
        }
        zoneIdFormatter.accept(builder);
        return builder.toString();
    }

    public static Object parseTimestamp(int precision, String value) {
        if (precision <= 6) {
            return DateTimes.parseShortTimestamp(value);
        }
        return DateTimes.parseLongTimestamp(value);
    }

    public static Object parseTimestampWithTimeZone(int precision, String value) {
        if (precision <= 3) {
            return DateTimes.parseShortTimestampWithTimeZone(value);
        }
        return DateTimes.parseLongTimestampWithTimeZone(value);
    }

    private static long parseShortTimestamp(String value) {
        Matcher matcher = DATETIME_PATTERN.matcher(value);
        if (!matcher.matches() || matcher.group("timezone") != null) {
            throw new IllegalArgumentException("Invalid timestamp: " + value);
        }
        String year = matcher.group("year");
        String month = matcher.group("month");
        String day = matcher.group("day");
        String hour = matcher.group("hour");
        String minute = matcher.group("minute");
        String second = matcher.group("second");
        String fraction = matcher.group("fraction");
        long epochSecond = DateTimes.toEpochSecond(year, month, day, hour, minute, second, ZoneOffset.UTC);
        int precision = 0;
        long fractionValue = 0L;
        if (fraction != null) {
            precision = fraction.length();
            fractionValue = Long.parseLong(fraction);
        }
        if (precision > 6) {
            throw new IllegalArgumentException(String.format("Cannot parse '%s' as short timestamp. Max allowed precision = %s", value, 6));
        }
        return Math.multiplyExact(epochSecond, 1000000) + DateTimes.rescale(fractionValue, precision, 6);
    }

    private static LongTimestamp parseLongTimestamp(String value) {
        Matcher matcher = DATETIME_PATTERN.matcher(value);
        if (!matcher.matches() || matcher.group("timezone") != null) {
            throw new IllegalArgumentException("Invalid timestamp: " + value);
        }
        String year = matcher.group("year");
        String month = matcher.group("month");
        String day = matcher.group("day");
        String hour = matcher.group("hour");
        String minute = matcher.group("minute");
        String second = matcher.group("second");
        String fraction = matcher.group("fraction");
        if (fraction == null || fraction.length() <= 6) {
            throw new IllegalArgumentException(String.format("Cannot parse '%s' as long timestamp. Precision must be in the range [%s, %s]", value, 7, 12));
        }
        int precision = fraction.length();
        long epochSecond = DateTimes.toEpochSecond(year, month, day, hour, minute, second, ZoneOffset.UTC);
        long picoFraction = DateTimes.rescale(Long.parseLong(fraction), precision, 12);
        return DateTimes.longTimestamp(epochSecond, picoFraction);
    }

    private static long parseShortTimestampWithTimeZone(String value) {
        Matcher matcher = DATETIME_PATTERN.matcher(value);
        if (!matcher.matches() || matcher.group("timezone") == null) {
            throw new IllegalArgumentException("Invalid timestamp with time zone: " + value);
        }
        String year = matcher.group("year");
        String month = matcher.group("month");
        String day = matcher.group("day");
        String hour = matcher.group("hour");
        String minute = matcher.group("minute");
        String second = matcher.group("second");
        String fraction = matcher.group("fraction");
        String timezone = matcher.group("timezone");
        ZoneId zoneId = ZoneId.of(timezone);
        long epochSecond = DateTimes.toEpochSecond(year, month, day, hour, minute, second, zoneId);
        int precision = 0;
        long fractionValue = 0L;
        if (fraction != null) {
            precision = fraction.length();
            fractionValue = Long.parseLong(fraction);
        }
        if (precision > 6) {
            throw new IllegalArgumentException(String.format("Cannot parse '%s' as short timestamp. Max allowed precision = %s", value, 6));
        }
        long epochMillis = epochSecond * 1000L + DateTimes.rescale(fractionValue, precision, 3);
        return DateTimeEncoding.packDateTimeWithZone((long)epochMillis, (String)timezone);
    }

    private static LongTimestampWithTimeZone parseLongTimestampWithTimeZone(String value) {
        Matcher matcher = DATETIME_PATTERN.matcher(value);
        if (!matcher.matches() || matcher.group("timezone") == null) {
            throw new IllegalArgumentException("Invalid timestamp: " + value);
        }
        String year = matcher.group("year");
        String month = matcher.group("month");
        String day = matcher.group("day");
        String hour = matcher.group("hour");
        String minute = matcher.group("minute");
        String second = matcher.group("second");
        String fraction = matcher.group("fraction");
        String timezone = matcher.group("timezone");
        if (fraction == null || fraction.length() <= 3) {
            throw new IllegalArgumentException(String.format("Cannot parse '%s' as long timestamp. Precision must be in the range [%s, %s]", value, 4, 12));
        }
        ZoneId zoneId = ZoneId.of(timezone);
        long epochSecond = DateTimes.toEpochSecond(year, month, day, hour, minute, second, zoneId);
        return LongTimestampWithTimeZone.fromEpochSecondsAndFraction((long)epochSecond, (long)DateTimes.rescale(Long.parseLong(fraction), fraction.length(), 12), (TimeZoneKey)TimeZoneKey.getTimeZoneKey((String)timezone));
    }

    private static long toEpochSecond(String year, String month, String day, String hour, String minute, String second, ZoneId zoneId) {
        LocalDateTime timestamp = LocalDateTime.of(Integer.parseInt(year), Integer.parseInt(month), Integer.parseInt(day), hour == null ? 0 : Integer.parseInt(hour), minute == null ? 0 : Integer.parseInt(minute), second == null ? 0 : Integer.parseInt(second), 0);
        List<ZoneOffset> offsets = zoneId.getRules().getValidOffsets(timestamp);
        if (offsets.isEmpty()) {
            throw new IllegalArgumentException("Invalid timestamp due to daylight savings transition");
        }
        return timestamp.toEpochSecond(offsets.get(0));
    }

    public static boolean timeHasTimeZone(String value) {
        Matcher matcher = TIME_PATTERN.matcher(value);
        if (!matcher.matches()) {
            throw new IllegalArgumentException(String.format("Invalid time '%s'", value));
        }
        return matcher.group("offsetHour") != null && matcher.group("offsetMinute") != null;
    }

    public static int extractTimePrecision(String value) {
        Matcher matcher = TIME_PATTERN.matcher(value);
        if (!matcher.matches()) {
            throw new IllegalArgumentException(String.format("Invalid time '%s'", value));
        }
        String fraction = matcher.group("fraction");
        if (fraction == null) {
            return 0;
        }
        return fraction.length();
    }

    public static long parseTime(String value) {
        int second;
        Matcher matcher = TIME_PATTERN.matcher(value);
        if (!matcher.matches() || matcher.group("offsetHour") != null || matcher.group("offsetMinute") != null) {
            throw new IllegalArgumentException("Invalid time: " + value);
        }
        int hour = Integer.parseInt(matcher.group("hour"));
        int minute = Integer.parseInt(matcher.group("minute"));
        int n = second = matcher.group("second") == null ? 0 : Integer.parseInt(matcher.group("second"));
        if (hour > 23 || minute > 59 || second > 59) {
            throw new IllegalArgumentException("Invalid time: " + value);
        }
        int precision = 0;
        String fraction = matcher.group("fraction");
        long fractionValue = 0L;
        if (fraction != null) {
            precision = fraction.length();
            fractionValue = Long.parseLong(fraction);
        }
        if (precision > 12) {
            throw new IllegalArgumentException("Invalid time: " + value);
        }
        return (long)((hour * 60 + minute) * 60 + second) * 1000000000000L + DateTimes.rescale(fractionValue, precision, 12);
    }

    public static Object parseTimeWithTimeZone(int precision, String value) {
        if (precision <= 9) {
            return DateTimes.parseShortTimeWithTimeZone(value);
        }
        return DateTimes.parseLongTimeWithTimeZone(value);
    }

    public static long parseShortTimeWithTimeZone(String value) {
        Matcher matcher = TIME_PATTERN.matcher(value);
        if (!matcher.matches() || matcher.group("offsetHour") == null || matcher.group("offsetMinute") == null) {
            throw new IllegalArgumentException("Invalid time with time zone: " + value);
        }
        int hour = Integer.parseInt(matcher.group("hour"));
        int minute = Integer.parseInt(matcher.group("minute"));
        int second = matcher.group("second") == null ? 0 : Integer.parseInt(matcher.group("second"));
        int offsetSign = matcher.group("sign").equals("+") ? 1 : -1;
        int offsetHour = Integer.parseInt(matcher.group("offsetHour"));
        int offsetMinute = Integer.parseInt(matcher.group("offsetMinute"));
        if (hour > 23 || minute > 59 || second > 59 || !DateTimes.isValidOffset(offsetHour, offsetMinute)) {
            throw new IllegalArgumentException("Invalid time with time zone: " + value);
        }
        int precision = 0;
        String fraction = matcher.group("fraction");
        long fractionValue = 0L;
        if (fraction != null) {
            precision = fraction.length();
            fractionValue = Long.parseLong(fraction);
        }
        long nanos = (long)((hour * 60 + minute) * 60 + second) * 1000000000L + DateTimes.rescale(fractionValue, precision, 9);
        return DateTimeEncoding.packTimeWithTimeZone((long)nanos, (int)DateTimes.calculateOffsetMinutes(offsetSign, offsetHour, offsetMinute));
    }

    public static LongTimeWithTimeZone parseLongTimeWithTimeZone(String value) {
        Matcher matcher = TIME_PATTERN.matcher(value);
        if (!matcher.matches() || matcher.group("offsetHour") == null || matcher.group("offsetMinute") == null) {
            throw new IllegalArgumentException("Invalid time with time zone: " + value);
        }
        int hour = Integer.parseInt(matcher.group("hour"));
        int minute = Integer.parseInt(matcher.group("minute"));
        int second = matcher.group("second") == null ? 0 : Integer.parseInt(matcher.group("second"));
        int offsetSign = matcher.group("sign").equals("+") ? 1 : -1;
        int offsetHour = Integer.parseInt(matcher.group("offsetHour"));
        int offsetMinute = Integer.parseInt(matcher.group("offsetMinute"));
        if (hour > 23 || minute > 59 || second > 59 || !DateTimes.isValidOffset(offsetHour, offsetMinute)) {
            throw new IllegalArgumentException("Invalid time with time zone: " + value);
        }
        int precision = 0;
        String fraction = matcher.group("fraction");
        long fractionValue = 0L;
        if (fraction != null) {
            precision = fraction.length();
            fractionValue = Long.parseLong(fraction);
        }
        long picos = (long)((hour * 60 + minute) * 60 + second) * 1000000000000L + DateTimes.rescale(fractionValue, precision, 12);
        return new LongTimeWithTimeZone(picos, DateTimes.calculateOffsetMinutes(offsetSign, offsetHour, offsetMinute));
    }

    public static LongTimestamp longTimestamp(long precision, Instant start) {
        Preconditions.checkArgument((precision > 6L && precision <= 12L ? 1 : 0) != 0, (Object)"Precision is out of range");
        return new LongTimestamp(start.getEpochSecond() * 1000000L + start.getLong(ChronoField.MICRO_OF_SECOND), (int)DateTimes.round(start.getNano() % 1000 * 1000, (int)(12L - precision)));
    }

    public static LongTimestamp longTimestamp(long epochSecond, long fractionInPicos) {
        return new LongTimestamp(Math.multiplyExact(epochSecond, 1000000) + fractionInPicos / 1000000L, (int)(fractionInPicos % 1000000L));
    }

    public static LongTimestampWithTimeZone longTimestampWithTimeZone(long precision, Instant start, TimeZoneKey timeZoneKey) {
        Preconditions.checkArgument((precision <= 12L ? 1 : 0) != 0, (Object)"Precision is out of range");
        return LongTimestampWithTimeZone.fromEpochMillisAndFraction((long)start.toEpochMilli(), (int)((int)DateTimes.round(start.getNano() % 1000000 * 1000, (int)(12L - precision))), (TimeZoneKey)timeZoneKey);
    }

    public static LongTimestampWithTimeZone longTimestampWithTimeZone(long epochSecond, long fractionInPicos, ZoneId zoneId) {
        return LongTimestampWithTimeZone.fromEpochMillisAndFraction((long)(Math.multiplyExact(epochSecond, 1000) + fractionInPicos / 1000000000L), (int)((int)(fractionInPicos % 1000000000L)), (TimeZoneKey)TimeZoneKey.getTimeZoneKey((String)zoneId.getId()));
    }

    public static long roundToEpochMillis(LongTimestampWithTimeZone timestamp) {
        long epochMillis = timestamp.getEpochMillis();
        if (DateTimes.roundToNearest(timestamp.getPicosOfMilli(), 1000000000L) == 1000000000L) {
            ++epochMillis;
        }
        return epochMillis;
    }

    public static int calculateOffsetMinutes(int sign, int offsetHour, int offsetMinute) {
        return sign * (offsetHour * 60 + offsetMinute);
    }

    public static int getOffsetMinutes(Instant instant, TimeZoneKey zoneKey) {
        return zoneKey.getZoneId().getRules().getOffset(instant).getTotalSeconds() / 60;
    }

    public static boolean isValidOffset(int hour, int minute) {
        return hour == 14 && minute == 0 || hour >= 0 && hour < 14 && minute >= 0 && minute <= 59;
    }
}

