/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.jdbc;

import com.google.common.base.CharMatcher;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.io.BaseEncoding;
import com.google.common.primitives.Shorts;
import com.google.common.primitives.SignedBytes;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceUtf8;
import io.airlift.slice.Slices;
import io.trino.plugin.jdbc.BooleanWriteFunction;
import io.trino.plugin.jdbc.ColumnMapping;
import io.trino.plugin.jdbc.DoubleWriteFunction;
import io.trino.plugin.jdbc.JdbcErrorCode;
import io.trino.plugin.jdbc.LongReadFunction;
import io.trino.plugin.jdbc.LongWriteFunction;
import io.trino.plugin.jdbc.ObjectReadFunction;
import io.trino.plugin.jdbc.ObjectWriteFunction;
import io.trino.plugin.jdbc.PredicatePushdownController;
import io.trino.plugin.jdbc.SliceReadFunction;
import io.trino.plugin.jdbc.SliceWriteFunction;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.TrinoException;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.BooleanType;
import io.trino.spi.type.CharType;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.Decimals;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.Int128;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.LongTimestamp;
import io.trino.spi.type.RealType;
import io.trino.spi.type.SmallintType;
import io.trino.spi.type.TimeType;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.Timestamps;
import io.trino.spi.type.TinyintType;
import io.trino.spi.type.Type;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.MathContext;
import java.math.RoundingMode;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import org.joda.time.DateTimeZone;
import org.joda.time.chrono.ISOChronology;

public final class StandardColumnMappings {
    private static final int MAX_LOCAL_DATE_TIME_PRECISION = 9;

    private StandardColumnMappings() {
    }

    public static ColumnMapping booleanColumnMapping() {
        return ColumnMapping.booleanMapping((Type)BooleanType.BOOLEAN, ResultSet::getBoolean, StandardColumnMappings.booleanWriteFunction());
    }

    public static BooleanWriteFunction booleanWriteFunction() {
        return PreparedStatement::setBoolean;
    }

    public static ColumnMapping tinyintColumnMapping() {
        return ColumnMapping.longMapping((Type)TinyintType.TINYINT, ResultSet::getByte, StandardColumnMappings.tinyintWriteFunction());
    }

    public static LongWriteFunction tinyintWriteFunction() {
        return LongWriteFunction.of(-6, (statement, index, value) -> statement.setByte(index, SignedBytes.checkedCast((long)value)));
    }

    public static ColumnMapping smallintColumnMapping() {
        return ColumnMapping.longMapping((Type)SmallintType.SMALLINT, ResultSet::getShort, StandardColumnMappings.smallintWriteFunction());
    }

    public static LongWriteFunction smallintWriteFunction() {
        return LongWriteFunction.of(5, (statement, index, value) -> statement.setShort(index, Shorts.checkedCast((long)value)));
    }

    public static ColumnMapping integerColumnMapping() {
        return ColumnMapping.longMapping((Type)IntegerType.INTEGER, ResultSet::getInt, StandardColumnMappings.integerWriteFunction());
    }

    public static LongWriteFunction integerWriteFunction() {
        return LongWriteFunction.of(4, (statement, index, value) -> statement.setInt(index, Math.toIntExact(value)));
    }

    public static ColumnMapping bigintColumnMapping() {
        return ColumnMapping.longMapping((Type)BigintType.BIGINT, ResultSet::getLong, StandardColumnMappings.bigintWriteFunction());
    }

    public static LongWriteFunction bigintWriteFunction() {
        return LongWriteFunction.of(-5, PreparedStatement::setLong);
    }

    public static ColumnMapping realColumnMapping() {
        return ColumnMapping.longMapping((Type)RealType.REAL, (resultSet, columnIndex) -> Float.floatToRawIntBits(resultSet.getFloat(columnIndex)), StandardColumnMappings.realWriteFunction());
    }

    public static LongWriteFunction realWriteFunction() {
        return LongWriteFunction.of(7, (statement, index, value) -> statement.setFloat(index, Float.intBitsToFloat(Math.toIntExact(value))));
    }

    public static ColumnMapping doubleColumnMapping() {
        return ColumnMapping.doubleMapping((Type)DoubleType.DOUBLE, ResultSet::getDouble, StandardColumnMappings.doubleWriteFunction());
    }

    public static DoubleWriteFunction doubleWriteFunction() {
        return DoubleWriteFunction.of(8, PreparedStatement::setDouble);
    }

    public static ColumnMapping decimalColumnMapping(DecimalType decimalType) {
        return StandardColumnMappings.decimalColumnMapping(decimalType, RoundingMode.UNNECESSARY);
    }

    public static ColumnMapping decimalColumnMapping(DecimalType decimalType, RoundingMode roundingMode) {
        if (decimalType.isShort()) {
            Preconditions.checkArgument((roundingMode == RoundingMode.UNNECESSARY ? 1 : 0) != 0, (Object)"Round mode is not supported for short decimal, map the type to long decimal instead");
            return ColumnMapping.longMapping((Type)decimalType, StandardColumnMappings.shortDecimalReadFunction(decimalType), StandardColumnMappings.shortDecimalWriteFunction(decimalType));
        }
        return ColumnMapping.objectMapping((Type)decimalType, StandardColumnMappings.longDecimalReadFunction(decimalType, roundingMode), StandardColumnMappings.longDecimalWriteFunction(decimalType));
    }

    public static LongReadFunction shortDecimalReadFunction(DecimalType decimalType) {
        return StandardColumnMappings.shortDecimalReadFunction(decimalType, RoundingMode.UNNECESSARY);
    }

    public static LongReadFunction shortDecimalReadFunction(DecimalType decimalType, RoundingMode roundingMode) {
        int scale = decimalType.getScale();
        Objects.requireNonNull(roundingMode, "roundingMode is null");
        return (resultSet, columnIndex) -> Decimals.encodeShortScaledValue((BigDecimal)resultSet.getBigDecimal(columnIndex), (int)scale, (RoundingMode)roundingMode);
    }

    public static LongWriteFunction shortDecimalWriteFunction(DecimalType decimalType) {
        Objects.requireNonNull(decimalType, "decimalType is null");
        Preconditions.checkArgument((boolean)decimalType.isShort());
        return LongWriteFunction.of(3, (statement, index, value) -> {
            BigInteger unscaledValue = BigInteger.valueOf(value);
            BigDecimal bigDecimal = new BigDecimal(unscaledValue, decimalType.getScale(), new MathContext(decimalType.getPrecision()));
            statement.setBigDecimal(index, bigDecimal);
        });
    }

    public static ObjectReadFunction longDecimalReadFunction(DecimalType decimalType) {
        return StandardColumnMappings.longDecimalReadFunction(decimalType, RoundingMode.UNNECESSARY);
    }

    public static ObjectReadFunction longDecimalReadFunction(DecimalType decimalType, RoundingMode roundingMode) {
        int scale = decimalType.getScale();
        Objects.requireNonNull(roundingMode, "roundingMode is null");
        return ObjectReadFunction.of(Int128.class, (resultSet, columnIndex) -> Decimals.valueOf((BigDecimal)resultSet.getBigDecimal(columnIndex).setScale(scale, roundingMode)));
    }

    public static ObjectWriteFunction longDecimalWriteFunction(final DecimalType decimalType) {
        Objects.requireNonNull(decimalType, "decimalType is null");
        Preconditions.checkArgument((!decimalType.isShort() ? 1 : 0) != 0);
        return new ObjectWriteFunction(){

            public Class<Int128> getJavaType() {
                return Int128.class;
            }

            @Override
            public void set(PreparedStatement statement, int index, Object value) throws SQLException {
                BigInteger unscaledValue = ((Int128)value).toBigInteger();
                BigDecimal bigDecimal = new BigDecimal(unscaledValue, decimalType.getScale(), new MathContext(decimalType.getPrecision()));
                statement.setBigDecimal(index, bigDecimal);
            }

            @Override
            public void setNull(PreparedStatement statement, int index) throws SQLException {
                statement.setNull(index, 3);
            }
        };
    }

    public static ColumnMapping defaultCharColumnMapping(int columnSize, boolean isRemoteCaseSensitive) {
        if (columnSize > 65536) {
            return StandardColumnMappings.defaultVarcharColumnMapping(columnSize, isRemoteCaseSensitive);
        }
        return StandardColumnMappings.charColumnMapping(CharType.createCharType((long)columnSize), isRemoteCaseSensitive);
    }

    public static ColumnMapping charColumnMapping(CharType charType, boolean isRemoteCaseSensitive) {
        Objects.requireNonNull(charType, "charType is null");
        PredicatePushdownController pushdownController = isRemoteCaseSensitive ? PredicatePushdownController.FULL_PUSHDOWN : PredicatePushdownController.CASE_INSENSITIVE_CHARACTER_PUSHDOWN;
        return ColumnMapping.sliceMapping((Type)charType, StandardColumnMappings.charReadFunction(charType), StandardColumnMappings.charWriteFunction(), pushdownController);
    }

    public static SliceReadFunction charReadFunction(CharType charType) {
        Objects.requireNonNull(charType, "charType is null");
        return (resultSet, columnIndex) -> {
            Slice slice = Slices.utf8Slice((String)CharMatcher.is((char)' ').trimTrailingFrom((CharSequence)resultSet.getString(columnIndex)));
            StandardColumnMappings.checkLengthInCodePoints(slice, (Type)charType, charType.getLength());
            return slice;
        };
    }

    public static SliceWriteFunction charWriteFunction() {
        return SliceWriteFunction.of(1, (statement, index, value) -> statement.setString(index, value.toStringUtf8()));
    }

    public static ColumnMapping defaultVarcharColumnMapping(int columnSize, boolean isRemoteCaseSensitive) {
        if (columnSize > 0x7FFFFFFE) {
            return StandardColumnMappings.varcharColumnMapping(VarcharType.createUnboundedVarcharType(), isRemoteCaseSensitive);
        }
        return StandardColumnMappings.varcharColumnMapping(VarcharType.createVarcharType((int)columnSize), isRemoteCaseSensitive);
    }

    public static ColumnMapping varcharColumnMapping(VarcharType varcharType, boolean isRemoteCaseSensitive) {
        PredicatePushdownController pushdownController = isRemoteCaseSensitive ? PredicatePushdownController.FULL_PUSHDOWN : PredicatePushdownController.CASE_INSENSITIVE_CHARACTER_PUSHDOWN;
        return ColumnMapping.sliceMapping((Type)varcharType, StandardColumnMappings.varcharReadFunction(varcharType), StandardColumnMappings.varcharWriteFunction(), pushdownController);
    }

    public static SliceReadFunction varcharReadFunction(VarcharType varcharType) {
        Objects.requireNonNull(varcharType, "varcharType is null");
        if (varcharType.isUnbounded()) {
            return (resultSet, columnIndex) -> Slices.utf8Slice((String)resultSet.getString(columnIndex));
        }
        return (resultSet, columnIndex) -> {
            Slice slice = Slices.utf8Slice((String)resultSet.getString(columnIndex));
            StandardColumnMappings.checkLengthInCodePoints(slice, (Type)varcharType, varcharType.getBoundedLength());
            return slice;
        };
    }

    private static void checkLengthInCodePoints(Slice value, Type characterDataType, int lengthLimit) {
        if (value.length() <= lengthLimit) {
            return;
        }
        if (SliceUtf8.countCodePoints((Slice)value) <= lengthLimit) {
            return;
        }
        throw new IllegalStateException(String.format("Illegal value for type %s: '%s' [%s]", characterDataType, value.toStringUtf8(), BaseEncoding.base16().encode(value.getBytes())));
    }

    public static SliceWriteFunction varcharWriteFunction() {
        return SliceWriteFunction.of(12, (statement, index, value) -> statement.setString(index, value.toStringUtf8()));
    }

    public static ColumnMapping varbinaryColumnMapping() {
        return ColumnMapping.sliceMapping((Type)VarbinaryType.VARBINARY, StandardColumnMappings.varbinaryReadFunction(), StandardColumnMappings.varbinaryWriteFunction(), PredicatePushdownController.DISABLE_PUSHDOWN);
    }

    public static SliceReadFunction varbinaryReadFunction() {
        return (resultSet, columnIndex) -> Slices.wrappedBuffer((byte[])resultSet.getBytes(columnIndex));
    }

    public static SliceWriteFunction varbinaryWriteFunction() {
        return SliceWriteFunction.of(-3, (statement, index, value) -> statement.setBytes(index, value.getBytes()));
    }

    @Deprecated
    public static ColumnMapping dateColumnMappingUsingSqlDate() {
        return ColumnMapping.longMapping((Type)DateType.DATE, StandardColumnMappings.dateReadFunctionUsingSqlDate(), StandardColumnMappings.dateWriteFunctionUsingSqlDate());
    }

    @Deprecated
    public static LongReadFunction dateReadFunctionUsingSqlDate() {
        return (resultSet, columnIndex) -> {
            long localMillis = resultSet.getDate(columnIndex).getTime();
            long utcMillis = ISOChronology.getInstance().getZone().getMillisKeepLocal(DateTimeZone.UTC, localMillis);
            return TimeUnit.MILLISECONDS.toDays(utcMillis);
        };
    }

    @Deprecated
    public static LongWriteFunction dateWriteFunctionUsingSqlDate() {
        return LongWriteFunction.of(91, (statement, index, value) -> {
            long millis = TimeUnit.DAYS.toMillis(value);
            statement.setDate(index, new Date(DateTimeZone.UTC.getMillisKeepLocal(DateTimeZone.getDefault(), millis)));
        });
    }

    public static ColumnMapping dateColumnMappingUsingLocalDate() {
        return ColumnMapping.longMapping((Type)DateType.DATE, StandardColumnMappings.dateReadFunctionUsingLocalDate(), StandardColumnMappings.dateWriteFunctionUsingLocalDate());
    }

    public static LongReadFunction dateReadFunctionUsingLocalDate() {
        return new LongReadFunction(){

            @Override
            public boolean isNull(ResultSet resultSet, int columnIndex) throws SQLException {
                resultSet.getObject(columnIndex, LocalDate.class);
                return resultSet.wasNull();
            }

            @Override
            public long readLong(ResultSet resultSet, int columnIndex) throws SQLException {
                LocalDate value = resultSet.getObject(columnIndex, LocalDate.class);
                if (value == null) {
                    throw new TrinoException((ErrorCodeSupplier)JdbcErrorCode.JDBC_ERROR, "Driver returned null LocalDate for a non-null value");
                }
                return value.toEpochDay();
            }
        };
    }

    public static LongWriteFunction dateWriteFunctionUsingLocalDate() {
        return LongWriteFunction.of(91, (statement, index, value) -> statement.setObject(index, LocalDate.ofEpochDay(value)));
    }

    @Deprecated
    public static ColumnMapping timeColumnMappingUsingSqlTime() {
        return ColumnMapping.longMapping((Type)TimeType.TIME_MILLIS, (resultSet, columnIndex) -> {
            Time time = resultSet.getTime(columnIndex);
            return StandardColumnMappings.toLocalTime(time).toNanoOfDay() * 1000L % 86400000000000000L;
        }, StandardColumnMappings.timeWriteFunctionUsingSqlTime());
    }

    private static LocalTime toLocalTime(Time sqlTime) {
        return sqlTime.toLocalTime().withNano(Math.toIntExact(TimeUnit.MILLISECONDS.toNanos(Math.floorMod(sqlTime.getTime(), 1000L))));
    }

    @Deprecated
    public static LongWriteFunction timeWriteFunctionUsingSqlTime() {
        return LongWriteFunction.of(92, (statement, index, value) -> statement.setTime(index, StandardColumnMappings.toSqlTime(StandardColumnMappings.fromTrinoTime(value))));
    }

    private static Time toSqlTime(LocalTime localTime) {
        return new Time(Time.valueOf(localTime).getTime() + TimeUnit.NANOSECONDS.toMillis(localTime.getNano()));
    }

    public static ColumnMapping timeColumnMapping(TimeType timeType) {
        return ColumnMapping.longMapping((Type)timeType, StandardColumnMappings.timeReadFunction(timeType), StandardColumnMappings.timeWriteFunction(timeType.getPrecision()));
    }

    public static LongReadFunction timeReadFunction(TimeType timeType) {
        Objects.requireNonNull(timeType, "timeType is null");
        Preconditions.checkArgument((timeType.getPrecision() <= 9 ? 1 : 0) != 0, (String)"Unsupported type precision: %s", (Object)timeType);
        return (resultSet, columnIndex) -> {
            LocalTime time = resultSet.getObject(columnIndex, LocalTime.class);
            long nanosOfDay = time.toNanoOfDay();
            Verify.verify((nanosOfDay < 86400000000000L ? 1 : 0) != 0, (String)"Invalid value of nanosOfDay: %s", (long)nanosOfDay);
            long picosOfDay = nanosOfDay * 1000L;
            long rounded = Timestamps.round((long)picosOfDay, (int)(12 - timeType.getPrecision()));
            if (rounded == 86400000000000000L) {
                rounded = 0L;
            }
            return rounded;
        };
    }

    public static LongWriteFunction timeWriteFunction(int precision) {
        Preconditions.checkArgument((precision <= 9 ? 1 : 0) != 0, (String)"Unsupported precision: %s", (int)precision);
        return LongWriteFunction.of(92, (statement, index, picosOfDay) -> {
            if ((picosOfDay = Timestamps.round((long)picosOfDay, (int)(12 - precision))) == 86400000000000000L) {
                picosOfDay = 0L;
            }
            statement.setObject(index, StandardColumnMappings.fromTrinoTime(picosOfDay));
        });
    }

    @Deprecated
    public static ColumnMapping timestampColumnMappingUsingSqlTimestampWithRounding(TimestampType timestampType) {
        Preconditions.checkArgument((timestampType.getPrecision() <= 6 ? 1 : 0) != 0, (String)"Precision is out of range: %s", (int)timestampType.getPrecision());
        return ColumnMapping.longMapping((Type)timestampType, (resultSet, columnIndex) -> {
            LocalDateTime localDateTime = resultSet.getTimestamp(columnIndex).toLocalDateTime();
            int roundedNanos = Math.toIntExact(Timestamps.round((long)localDateTime.getNano(), (int)(9 - timestampType.getPrecision())));
            LocalDateTime rounded = localDateTime.withNano(0).plusNanos(roundedNanos);
            return StandardColumnMappings.toTrinoTimestamp(timestampType, rounded);
        }, StandardColumnMappings.timestampWriteFunctionUsingSqlTimestamp(timestampType), PredicatePushdownController.DISABLE_PUSHDOWN);
    }

    public static ColumnMapping timestampColumnMapping(TimestampType timestampType) {
        if (timestampType.getPrecision() <= 6) {
            return ColumnMapping.longMapping((Type)timestampType, StandardColumnMappings.timestampReadFunction(timestampType), StandardColumnMappings.timestampWriteFunction(timestampType));
        }
        Preconditions.checkArgument((timestampType.getPrecision() <= 9 ? 1 : 0) != 0, (String)"Precision is out of range: %s", (int)timestampType.getPrecision());
        return ColumnMapping.objectMapping((Type)timestampType, StandardColumnMappings.longTimestampReadFunction(timestampType), StandardColumnMappings.longTimestampWriteFunction(timestampType, timestampType.getPrecision()));
    }

    public static LongReadFunction timestampReadFunction(TimestampType timestampType) {
        Preconditions.checkArgument((timestampType.getPrecision() <= 6 ? 1 : 0) != 0, (String)"Precision is out of range: %s", (int)timestampType.getPrecision());
        return (resultSet, columnIndex) -> StandardColumnMappings.toTrinoTimestamp(timestampType, resultSet.getObject(columnIndex, LocalDateTime.class));
    }

    public static ObjectReadFunction longTimestampReadFunction(TimestampType timestampType) {
        Preconditions.checkArgument((timestampType.getPrecision() > 6 && timestampType.getPrecision() <= 9 ? 1 : 0) != 0, (String)"Precision is out of range: %s", (int)timestampType.getPrecision());
        return ObjectReadFunction.of(LongTimestamp.class, (resultSet, columnIndex) -> StandardColumnMappings.toLongTrinoTimestamp(timestampType, resultSet.getObject(columnIndex, LocalDateTime.class)));
    }

    @Deprecated
    public static LongWriteFunction timestampWriteFunctionUsingSqlTimestamp(TimestampType timestampType) {
        Preconditions.checkArgument((timestampType.getPrecision() <= 6 ? 1 : 0) != 0, (String)"Precision is out of range: %s", (int)timestampType.getPrecision());
        return LongWriteFunction.of(93, (statement, index, value) -> statement.setTimestamp(index, Timestamp.valueOf(StandardColumnMappings.fromTrinoTimestamp(value))));
    }

    public static LongWriteFunction timestampWriteFunction(TimestampType timestampType) {
        Preconditions.checkArgument((timestampType.getPrecision() <= 6 ? 1 : 0) != 0, (String)"Precision is out of range: %s", (int)timestampType.getPrecision());
        return LongWriteFunction.of(93, (statement, index, value) -> statement.setObject(index, StandardColumnMappings.fromTrinoTimestamp(value)));
    }

    public static ObjectWriteFunction longTimestampWriteFunction(TimestampType timestampType, int roundToPrecision) {
        Preconditions.checkArgument((timestampType.getPrecision() > 6 ? 1 : 0) != 0, (String)"Precision is out of range: %s", (int)timestampType.getPrecision());
        Preconditions.checkArgument((6 <= roundToPrecision && roundToPrecision <= 9 && roundToPrecision <= timestampType.getPrecision() ? 1 : 0) != 0, (String)"Invalid roundToPrecision for %s: %s", (Object)timestampType, (int)roundToPrecision);
        return ObjectWriteFunction.of(LongTimestamp.class, (statement, index, value) -> statement.setObject(index, StandardColumnMappings.fromLongTrinoTimestamp(value, roundToPrecision)));
    }

    public static long toTrinoTimestamp(TimestampType timestampType, LocalDateTime localDateTime) {
        long precision = timestampType.getPrecision();
        Preconditions.checkArgument((precision <= 6L ? 1 : 0) != 0, (String)"Precision is out of range: %s", (long)precision);
        long epochMicros = localDateTime.toEpochSecond(ZoneOffset.UTC) * 1000000L + (long)(localDateTime.getNano() / 1000);
        Verify.verify((epochMicros == Timestamps.round((long)epochMicros, (int)(6 - timestampType.getPrecision())) ? 1 : 0) != 0, (String)"Invalid value of epochMicros for precision %s: %s", (long)precision, (long)epochMicros);
        return epochMicros;
    }

    public static LongTimestamp toLongTrinoTimestamp(TimestampType timestampType, LocalDateTime localDateTime) {
        long precision = timestampType.getPrecision();
        Preconditions.checkArgument((precision > 6L ? 1 : 0) != 0, (String)"Precision is out of range: %s", (long)precision);
        long epochMicros = localDateTime.toEpochSecond(ZoneOffset.UTC) * 1000000L + (long)(localDateTime.getNano() / 1000);
        int picosOfMicro = localDateTime.getNano() % 1000 * 1000;
        Verify.verify(((long)picosOfMicro == Timestamps.round((long)picosOfMicro, (int)(12 - timestampType.getPrecision())) ? 1 : 0) != 0, (String)"Invalid value of picosOfMicro for precision %s: %s", (long)precision, (int)picosOfMicro);
        return new LongTimestamp(epochMicros, picosOfMicro);
    }

    public static LocalDateTime fromTrinoTimestamp(long epochMicros) {
        long epochSecond = Math.floorDiv(epochMicros, 1000000);
        int nanoFraction = Math.floorMod(epochMicros, 1000000) * 1000;
        Instant instant = Instant.ofEpochSecond(epochSecond, nanoFraction);
        return LocalDateTime.ofInstant(instant, ZoneOffset.UTC);
    }

    public static LocalDateTime fromLongTrinoTimestamp(LongTimestamp timestamp, int precision) {
        Preconditions.checkArgument((6 <= precision && precision <= 9 ? 1 : 0) != 0, (String)"Unsupported precision: %s", (int)precision);
        long epochSeconds = Math.floorDiv(timestamp.getEpochMicros(), 1000000);
        int microsOfSecond = Math.floorMod(timestamp.getEpochMicros(), 1000000);
        long picosOfMicro = Timestamps.round((long)timestamp.getPicosOfMicro(), (int)(12 - precision));
        int nanosOfSecond = microsOfSecond * 1000 + Math.toIntExact(picosOfMicro / 1000L);
        Instant instant = Instant.ofEpochSecond(epochSeconds, nanosOfSecond);
        return LocalDateTime.ofInstant(instant, ZoneOffset.UTC);
    }

    public static LocalTime fromTrinoTime(long value) {
        return LocalTime.ofNanoOfDay(Timestamps.roundDiv((long)value, (long)1000L) % 86400000000000L);
    }
}

