/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.redis;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import io.airlift.log.Logger;
import io.airlift.slice.Slice;
import io.trino.decoder.DecoderColumnHandle;
import io.trino.decoder.FieldValueProvider;
import io.trino.decoder.FieldValueProviders;
import io.trino.decoder.RowDecoder;
import io.trino.plugin.redis.RedisColumnHandle;
import io.trino.plugin.redis.RedisDataType;
import io.trino.plugin.redis.RedisInternalFieldDescription;
import io.trino.plugin.redis.RedisJedisManager;
import io.trino.plugin.redis.RedisSplit;
import io.trino.plugin.redis.decoder.RedisRowDecoder;
import io.trino.spi.connector.ColumnHandle;
import io.trino.spi.connector.RecordCursor;
import io.trino.spi.predicate.Domain;
import io.trino.spi.predicate.Range;
import io.trino.spi.predicate.Ranges;
import io.trino.spi.predicate.SortedRangeSet;
import io.trino.spi.predicate.TupleDomain;
import io.trino.spi.predicate.ValueSet;
import io.trino.spi.type.Type;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Queue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Pipeline;
import redis.clients.jedis.exceptions.JedisDataException;
import redis.clients.jedis.params.ScanParams;
import redis.clients.jedis.resps.ScanResult;

public class RedisRecordCursor
implements RecordCursor {
    private static final Logger log = Logger.get(RedisRecordCursor.class);
    private static final String EMPTY_STRING = "";
    private final RowDecoder keyDecoder;
    private final RowDecoder valueDecoder;
    private final RedisSplit split;
    private final List<RedisColumnHandle> columnHandles;
    private final JedisPool jedisPool;
    private final ScanParams scanParams;
    private final int maxKeysPerFetch;
    private final char redisKeyDelimiter;
    private final boolean isKeyPrefixSchemaTable;
    private final int redisScanCount;
    private ScanResult<String> redisCursor;
    private List<String> keys;
    private final AtomicBoolean reported = new AtomicBoolean();
    private List<String> stringValues;
    private List<Object> hashValues;
    private long totalBytes;
    private long totalValues;
    private final Queue<FieldValueProvider[]> currentRowGroup;

    RedisRecordCursor(RowDecoder keyDecoder, RowDecoder valueDecoder, RedisSplit split, List<RedisColumnHandle> columnHandles, RedisJedisManager redisJedisManager) {
        this.keyDecoder = keyDecoder;
        this.valueDecoder = valueDecoder;
        this.split = split;
        this.columnHandles = columnHandles;
        this.jedisPool = redisJedisManager.getJedisPool(split.getNodes().get(0));
        this.redisKeyDelimiter = redisJedisManager.getRedisKeyDelimiter();
        this.isKeyPrefixSchemaTable = redisJedisManager.isKeyPrefixSchemaTable();
        this.redisScanCount = redisJedisManager.getRedisScanCount();
        this.scanParams = this.setScanParams();
        this.maxKeysPerFetch = redisJedisManager.getRedisMaxKeysPerFetch();
        this.currentRowGroup = new LinkedList<FieldValueProvider[]>();
        if (split.getConstraint().isAll()) {
            this.fetchKeys();
        } else {
            this.setPushdownKeys();
        }
    }

    public long getCompletedBytes() {
        return this.totalBytes;
    }

    public long getReadTimeNanos() {
        return 0L;
    }

    public Type getType(int field) {
        Preconditions.checkArgument((field < this.columnHandles.size() ? 1 : 0) != 0, (Object)"Invalid field index");
        return this.columnHandles.get(field).getType();
    }

    public boolean hasUnscannedData() {
        if (this.redisCursor == null) {
            return false;
        }
        return !this.redisCursor.getCursor().equals("0");
    }

    public boolean advanceNextPosition() {
        this.currentRowGroup.poll();
        while (this.currentRowGroup.isEmpty()) {
            while (this.keys.isEmpty()) {
                if (!this.split.getConstraint().isAll()) {
                    return false;
                }
                if (!this.hasUnscannedData()) {
                    return this.endOfData();
                }
                this.fetchKeys();
            }
            this.fetchNextRowGroup();
        }
        return true;
    }

    private boolean endOfData() {
        if (!this.reported.getAndSet(true)) {
            log.debug("Read a total of %d values with %d bytes.", new Object[]{this.totalValues, this.totalBytes});
        }
        return false;
    }

    private void fetchNextRowGroup() {
        List<String> currentKeys = this.keys.size() > this.maxKeysPerFetch ? this.keys.subList(0, this.maxKeysPerFetch) : this.keys;
        this.fetchData(currentKeys);
        switch (this.split.getValueDataType()) {
            case STRING: {
                this.processStringValues(currentKeys);
                break;
            }
            case HASH: {
                this.processHashValues(currentKeys);
                break;
            }
            default: {
                log.warn("Redis value of type %s is unsupported", new Object[]{this.split.getValueDataType()});
            }
        }
        currentKeys.clear();
    }

    private void processStringValues(List<String> currentKeys) {
        for (int i = 0; i < currentKeys.size(); ++i) {
            String keyString = currentKeys.get(i);
            String valueString = this.stringValues.get(i);
            if (valueString == null) {
                log.warn("The string value at key %s does not exist", new Object[]{keyString});
                continue;
            }
            this.generateRowValues(keyString, valueString, null);
        }
    }

    private void processHashValues(List<String> currentKeys) {
        for (int i = 0; i < currentKeys.size(); ++i) {
            String keyString = currentKeys.get(i);
            Object object = this.hashValues.get(i);
            if (object instanceof JedisDataException) {
                throw (JedisDataException)object;
            }
            Map hashValueMap = (Map)object;
            if (hashValueMap.isEmpty()) {
                log.warn("The hash value at key %s does not exist", new Object[]{keyString});
                continue;
            }
            this.generateRowValues(keyString, EMPTY_STRING, hashValueMap);
        }
    }

    /*
     * WARNING - void declaration
     */
    private void generateRowValues(String keyString, String valueString, @Nullable Map<String, String> hashValueMap) {
        void var10_12;
        byte[] keyData = keyString.getBytes(StandardCharsets.UTF_8);
        byte[] stringValueData = valueString.getBytes(StandardCharsets.UTF_8);
        Optional decodedKey = this.keyDecoder.decodeRow(keyData);
        Optional<Map<DecoderColumnHandle, FieldValueProvider>> decodedValue = this.valueDecoder instanceof RedisRowDecoder ? ((RedisRowDecoder)this.valueDecoder).decodeRow(hashValueMap) : this.valueDecoder.decodeRow(stringValueData);
        this.totalBytes += (long)stringValueData.length;
        ++this.totalValues;
        HashMap<DecoderColumnHandle, FieldValueProvider> currentRowValuesMap = new HashMap<DecoderColumnHandle, FieldValueProvider>();
        block8: for (DecoderColumnHandle decoderColumnHandle : this.columnHandles) {
            if (!decoderColumnHandle.isInternal()) continue;
            RedisInternalFieldDescription fieldDescription = RedisInternalFieldDescription.forColumnName(decoderColumnHandle.getName());
            switch (fieldDescription) {
                case KEY_FIELD: {
                    currentRowValuesMap.put(decoderColumnHandle, FieldValueProviders.bytesValueProvider((byte[])keyData));
                    continue block8;
                }
                case VALUE_FIELD: {
                    currentRowValuesMap.put(decoderColumnHandle, FieldValueProviders.bytesValueProvider((byte[])stringValueData));
                    continue block8;
                }
                case KEY_LENGTH_FIELD: {
                    currentRowValuesMap.put(decoderColumnHandle, FieldValueProviders.longValueProvider((long)keyData.length));
                    continue block8;
                }
                case VALUE_LENGTH_FIELD: {
                    currentRowValuesMap.put(decoderColumnHandle, FieldValueProviders.longValueProvider((long)stringValueData.length));
                    continue block8;
                }
                case KEY_CORRUPT_FIELD: {
                    currentRowValuesMap.put(decoderColumnHandle, FieldValueProviders.booleanValueProvider((boolean)decodedKey.isEmpty()));
                    continue block8;
                }
                case VALUE_CORRUPT_FIELD: {
                    currentRowValuesMap.put(decoderColumnHandle, FieldValueProviders.booleanValueProvider((boolean)decodedValue.isEmpty()));
                    continue block8;
                }
            }
            throw new IllegalArgumentException("unknown internal field " + fieldDescription);
        }
        decodedKey.ifPresent(currentRowValuesMap::putAll);
        decodedValue.ifPresent(currentRowValuesMap::putAll);
        FieldValueProvider[] currentRowValues = new FieldValueProvider[this.columnHandles.size()];
        boolean bl = false;
        while (var10_12 < this.columnHandles.size()) {
            ColumnHandle columnHandle = (ColumnHandle)this.columnHandles.get((int)var10_12);
            currentRowValues[var10_12] = (FieldValueProvider)currentRowValuesMap.get(columnHandle);
            ++var10_12;
        }
        this.currentRowGroup.offer(currentRowValues);
    }

    public boolean getBoolean(int field) {
        return this.getFieldValueProvider(field, Boolean.TYPE).getBoolean();
    }

    public long getLong(int field) {
        return this.getFieldValueProvider(field, Long.TYPE).getLong();
    }

    public double getDouble(int field) {
        return this.getFieldValueProvider(field, Double.TYPE).getDouble();
    }

    public Slice getSlice(int field) {
        return this.getFieldValueProvider(field, Slice.class).getSlice();
    }

    public boolean isNull(int field) {
        Preconditions.checkArgument((field < this.columnHandles.size() ? 1 : 0) != 0, (Object)"Invalid field index");
        FieldValueProvider[] currentRowValues = this.currentRowGroup.peek();
        return currentRowValues == null || currentRowValues[field].isNull();
    }

    public Object getObject(int field) {
        Preconditions.checkArgument((field < this.columnHandles.size() ? 1 : 0) != 0, (Object)"Invalid field index");
        throw new IllegalArgumentException(String.format("Type %s is not supported", this.getType(field)));
    }

    private FieldValueProvider getFieldValueProvider(int field, Class<?> expectedType) {
        Preconditions.checkArgument((field < this.columnHandles.size() ? 1 : 0) != 0, (Object)"Invalid field index");
        this.checkFieldType(field, expectedType);
        FieldValueProvider[] currentRowValues = this.currentRowGroup.peek();
        return Objects.requireNonNull(currentRowValues)[field];
    }

    private void checkFieldType(int field, Class<?> expected) {
        Class actual = this.getType(field).getJavaType();
        Preconditions.checkArgument((actual == expected ? 1 : 0) != 0, (String)"Expected field %s to be type %s but is %s", (Object)field, expected, (Object)actual);
    }

    public void close() {
    }

    private ScanParams setScanParams() {
        if (this.split.getKeyDataType() == RedisDataType.STRING) {
            ScanParams scanParams = new ScanParams();
            scanParams.count(Integer.valueOf(this.redisScanCount));
            if (this.isKeyPrefixSchemaTable) {
                Object keyMatch = EMPTY_STRING;
                if (!this.split.getSchemaName().equals("default")) {
                    keyMatch = this.split.getSchemaName() + this.redisKeyDelimiter;
                }
                keyMatch = (String)keyMatch + this.split.getTableName() + this.redisKeyDelimiter + "*";
                scanParams.match((String)keyMatch);
            }
            return scanParams;
        }
        return null;
    }

    private void setPushdownKeys() {
        String keyStringPrefix = this.isKeyPrefixSchemaTable ? this.scanParams.match().substring(0, this.scanParams.match().length() - 1) : EMPTY_STRING;
        TupleDomain<ColumnHandle> constraint = this.split.getConstraint();
        Map domains = (Map)constraint.getDomains().orElseThrow();
        for (Map.Entry entry : domains.entrySet()) {
            Ranges ranges;
            List rangeList;
            if (!((RedisColumnHandle)entry.getKey()).isKeyDecoder()) continue;
            Domain domain = (Domain)entry.getValue();
            if (domain.isSingleValue()) {
                String value = ((Slice)domain.getSingleValue()).toStringUtf8();
                this.keys = keyStringPrefix.isEmpty() || value.contains(keyStringPrefix) ? Lists.newArrayList((Object[])new String[]{value}) : Collections.emptyList();
                log.debug("Set pushdown keys %s with single value", new Object[]{this.keys.toString()});
                return;
            }
            ValueSet valueSet = domain.getValues();
            if (!(valueSet instanceof SortedRangeSet) || !(rangeList = (ranges = ((SortedRangeSet)valueSet).getRanges()).getOrderedRanges()).stream().allMatch(Range::isSingleValue)) continue;
            this.keys = rangeList.stream().map(range -> ((Slice)range.getSingleValue()).toStringUtf8()).filter(str -> keyStringPrefix.isEmpty() || str.contains(keyStringPrefix)).collect(Collectors.toList());
            log.debug("Set pushdown keys %s with sorted range values", new Object[]{this.keys.toString()});
            return;
        }
        this.keys = ImmutableList.of();
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void fetchKeys() {
        try (Jedis jedis = this.jedisPool.getResource();){
            switch (this.split.getKeyDataType()) {
                case STRING: {
                    String cursor = ScanParams.SCAN_POINTER_START;
                    if (this.redisCursor != null) {
                        cursor = this.redisCursor.getCursor();
                    }
                    log.debug("Scanning new Redis keys from cursor %s . %d values read so far", new Object[]{cursor, this.totalValues});
                    this.redisCursor = jedis.scan(cursor, this.scanParams);
                    this.keys = this.redisCursor.getResult();
                    return;
                }
                case ZSET: {
                    this.keys = jedis.zrange(this.split.getKeyName(), this.split.getStart(), this.split.getEnd());
                    return;
                }
                default: {
                    log.warn("Redis key of type %s is unsupported", new Object[]{this.split.getKeyDataFormat()});
                    return;
                }
            }
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void fetchData(List<String> currentKeys) {
        this.stringValues = null;
        this.hashValues = null;
        try (Jedis jedis = this.jedisPool.getResource();){
            switch (this.split.getValueDataType()) {
                case STRING: {
                    this.stringValues = jedis.mget(currentKeys.toArray(new String[0]));
                    return;
                }
                case HASH: {
                    Pipeline pipeline = jedis.pipelined();
                    for (String key : currentKeys) {
                        pipeline.hgetAll(key);
                    }
                    this.hashValues = pipeline.syncAndReturnAll();
                    return;
                }
                default: {
                    log.warn("Redis value of type %s is unsupported", new Object[]{this.split.getValueDataType()});
                    return;
                }
            }
        }
    }
}

