/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive.fs;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.cache.Cache;
import com.google.common.collect.ImmutableList;
import io.airlift.units.Duration;
import io.trino.collect.cache.CacheUtils;
import io.trino.collect.cache.EvictableCacheBuilder;
import io.trino.plugin.hive.HiveConfig;
import io.trino.plugin.hive.fs.DirectoryLister;
import io.trino.plugin.hive.fs.DirectoryListingCacheKey;
import io.trino.plugin.hive.fs.SimpleRemoteIterator;
import io.trino.plugin.hive.fs.TrinoFileStatus;
import io.trino.plugin.hive.fs.TrinoFileStatusRemoteIterator;
import io.trino.plugin.hive.metastore.Partition;
import io.trino.plugin.hive.metastore.Storage;
import io.trino.plugin.hive.metastore.Table;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.connector.SchemaTablePrefix;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import javax.inject.Inject;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.weakref.jmx.Managed;

public class CachingDirectoryLister
implements DirectoryLister {
    private final Cache<DirectoryListingCacheKey, ValueHolder> cache;
    private final List<SchemaTablePrefix> tablePrefixes;

    @Inject
    public CachingDirectoryLister(HiveConfig hiveClientConfig) {
        this(hiveClientConfig.getFileStatusCacheExpireAfterWrite(), hiveClientConfig.getFileStatusCacheMaxSize(), hiveClientConfig.getFileStatusCacheTables());
    }

    public CachingDirectoryLister(Duration expireAfterWrite, long maxSize, List<String> tables) {
        this.cache = EvictableCacheBuilder.newBuilder().maximumWeight(maxSize).weigher((key, value) -> value.files.map(List::size).orElse(1)).expireAfterWrite(expireAfterWrite.toMillis(), TimeUnit.MILLISECONDS).shareNothingWhenDisabled().recordStats().build();
        this.tablePrefixes = (List)tables.stream().map(CachingDirectoryLister::parseTableName).collect(ImmutableList.toImmutableList());
    }

    private static SchemaTablePrefix parseTableName(String tableName) {
        if (tableName.equals("*")) {
            return new SchemaTablePrefix();
        }
        String[] parts = tableName.split("\\.");
        Preconditions.checkArgument((parts.length == 2 ? 1 : 0) != 0, (String)"Invalid schemaTableName: %s", (Object)tableName);
        String schema = parts[0];
        String table = parts[1];
        if (table.equals("*")) {
            return new SchemaTablePrefix(schema);
        }
        return new SchemaTablePrefix(schema, table);
    }

    @Override
    public RemoteIterator<TrinoFileStatus> list(FileSystem fs, Table table, Path path) throws IOException {
        if (!this.isCacheEnabledFor(table.getSchemaTableName())) {
            return new TrinoFileStatusRemoteIterator((RemoteIterator<LocatedFileStatus>)fs.listLocatedStatus(path));
        }
        return this.listInternal(fs, new DirectoryListingCacheKey(path, false));
    }

    @Override
    public RemoteIterator<TrinoFileStatus> listFilesRecursively(FileSystem fs, Table table, Path path) throws IOException {
        if (!this.isCacheEnabledFor(table.getSchemaTableName())) {
            return new TrinoFileStatusRemoteIterator((RemoteIterator<LocatedFileStatus>)fs.listFiles(path, true));
        }
        return this.listInternal(fs, new DirectoryListingCacheKey(path, true));
    }

    private RemoteIterator<TrinoFileStatus> listInternal(FileSystem fs, DirectoryListingCacheKey cacheKey) throws IOException {
        ValueHolder cachedValueHolder = (ValueHolder)CacheUtils.uncheckedCacheGet(this.cache, (Object)cacheKey, ValueHolder::new);
        if (cachedValueHolder.getFiles().isPresent()) {
            return new SimpleRemoteIterator(cachedValueHolder.getFiles().get().iterator());
        }
        return this.cachingRemoteIterator(cachedValueHolder, CachingDirectoryLister.createListingRemoteIterator(fs, cacheKey), cacheKey);
    }

    private static RemoteIterator<TrinoFileStatus> createListingRemoteIterator(FileSystem fs, DirectoryListingCacheKey cacheKey) throws IOException {
        if (cacheKey.isRecursiveFilesOnly()) {
            return new TrinoFileStatusRemoteIterator((RemoteIterator<LocatedFileStatus>)fs.listFiles(cacheKey.getPath(), true));
        }
        return new TrinoFileStatusRemoteIterator((RemoteIterator<LocatedFileStatus>)fs.listLocatedStatus(cacheKey.getPath()));
    }

    @Override
    public void invalidate(Table table) {
        if (this.isCacheEnabledFor(table.getSchemaTableName()) && CachingDirectoryLister.isLocationPresent(table.getStorage())) {
            if (table.getPartitionColumns().isEmpty()) {
                this.cache.invalidateAll(DirectoryListingCacheKey.allKeysWithPath(new Path(table.getStorage().getLocation())));
            } else {
                this.cache.invalidateAll();
            }
        }
    }

    @Override
    public void invalidate(Partition partition) {
        if (this.isCacheEnabledFor(partition.getSchemaTableName()) && CachingDirectoryLister.isLocationPresent(partition.getStorage())) {
            this.cache.invalidateAll(DirectoryListingCacheKey.allKeysWithPath(new Path(partition.getStorage().getLocation())));
        }
    }

    private RemoteIterator<TrinoFileStatus> cachingRemoteIterator(final ValueHolder cachedValueHolder, final RemoteIterator<TrinoFileStatus> iterator, final DirectoryListingCacheKey key) {
        return new RemoteIterator<TrinoFileStatus>(){
            private final List<TrinoFileStatus> files = new ArrayList<TrinoFileStatus>();

            public boolean hasNext() throws IOException {
                boolean hasNext = iterator.hasNext();
                if (!hasNext) {
                    CachingDirectoryLister.this.cache.asMap().replace(key, cachedValueHolder, new ValueHolder(this.files));
                }
                return hasNext;
            }

            public TrinoFileStatus next() throws IOException {
                TrinoFileStatus next = (TrinoFileStatus)iterator.next();
                this.files.add(next);
                return next;
            }
        };
    }

    @Managed
    public void flushCache() {
        this.cache.invalidateAll();
    }

    @Managed
    public Double getHitRate() {
        return this.cache.stats().hitRate();
    }

    @Managed
    public Double getMissRate() {
        return this.cache.stats().missRate();
    }

    @Managed
    public long getHitCount() {
        return this.cache.stats().hitCount();
    }

    @Managed
    public long getMissCount() {
        return this.cache.stats().missCount();
    }

    @Managed
    public long getRequestCount() {
        return this.cache.stats().requestCount();
    }

    @VisibleForTesting
    boolean isCached(Path path) {
        return this.isCached(new DirectoryListingCacheKey(path, false));
    }

    @VisibleForTesting
    boolean isCached(DirectoryListingCacheKey cacheKey) {
        ValueHolder cached = (ValueHolder)this.cache.getIfPresent((Object)cacheKey);
        return cached != null && cached.getFiles().isPresent();
    }

    private boolean isCacheEnabledFor(SchemaTableName schemaTableName) {
        return this.tablePrefixes.stream().anyMatch(prefix -> prefix.matches(schemaTableName));
    }

    private static boolean isLocationPresent(Storage storage) {
        return storage.getOptionalLocation().isPresent() && StringUtils.isNotEmpty((CharSequence)storage.getLocation());
    }

    private static class ValueHolder {
        private final Optional<List<TrinoFileStatus>> files;

        public ValueHolder() {
            this.files = Optional.empty();
        }

        public ValueHolder(List<TrinoFileStatus> files) {
            this.files = Optional.of(ImmutableList.copyOf((Collection)Objects.requireNonNull(files, "files is null")));
        }

        public Optional<List<TrinoFileStatus>> getFiles() {
            return this.files;
        }
    }
}

