/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import io.trino.hadoop.$internal.com.google.common.annotations.VisibleForTesting;
import io.trino.hadoop.$internal.com.google.common.base.Preconditions;
import io.trino.hadoop.$internal.com.google.common.collect.Lists;
import io.trino.hadoop.$internal.org.slf4j.Logger;
import io.trino.hadoop.$internal.org.slf4j.LoggerFactory;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletionService;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.crypto.key.KeyProviderCryptoExtension;
import org.apache.hadoop.fs.FileEncryptionInfo;
import org.apache.hadoop.fs.XAttr;
import org.apache.hadoop.fs.XAttrSetFlag;
import org.apache.hadoop.hdfs.protocol.ZoneReencryptionStatus;
import org.apache.hadoop.hdfs.server.namenode.EncryptionFaultInjector;
import org.apache.hadoop.hdfs.server.namenode.FSDirEncryptionZoneOp;
import org.apache.hadoop.hdfs.server.namenode.FSDirectory;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.INodesInPath;
import org.apache.hadoop.hdfs.server.namenode.ReencryptionHandler;
import org.apache.hadoop.hdfs.server.namenode.SafeModeException;
import org.apache.hadoop.ipc.RetriableException;
import org.apache.hadoop.util.StopWatch;

@InterfaceAudience.Private
public final class ReencryptionUpdater
implements Runnable {
    public static final Logger LOG = LoggerFactory.getLogger(ReencryptionUpdater.class);
    private volatile boolean shouldPauseForTesting = false;
    private volatile int pauseAfterNthCheckpoint = 0;
    private volatile long pauseZoneId = 0L;
    private double throttleLimitRatio;
    private final StopWatch throttleTimerAll = new StopWatch();
    private final StopWatch throttleTimerLocked = new StopWatch();
    private volatile long faultRetryInterval = 60000L;
    private volatile boolean isRunning = false;
    private final FSDirectory dir;
    private final CompletionService<ReencryptionTask> batchService;
    private final ReencryptionHandler handler;

    @VisibleForTesting
    synchronized void pauseForTesting() {
        this.shouldPauseForTesting = true;
        LOG.info("Pausing re-encrypt updater for testing.");
        this.notify();
    }

    @VisibleForTesting
    synchronized void resumeForTesting() {
        this.shouldPauseForTesting = false;
        LOG.info("Resuming re-encrypt updater for testing.");
        this.notify();
    }

    @VisibleForTesting
    void pauseForTestingAfterNthCheckpoint(long zoneId, int count) {
        assert (this.pauseAfterNthCheckpoint == 0);
        this.pauseAfterNthCheckpoint = count;
        this.pauseZoneId = zoneId;
    }

    @VisibleForTesting
    boolean isRunning() {
        return this.isRunning;
    }

    ReencryptionUpdater(FSDirectory fsd, CompletionService<ReencryptionTask> service, ReencryptionHandler rh, Configuration conf) {
        this.dir = fsd;
        this.batchService = service;
        this.handler = rh;
        this.throttleLimitRatio = conf.getDouble("dfs.namenode.reencrypt.throttle.limit.updater.ratio", 1.0);
        Preconditions.checkArgument(this.throttleLimitRatio > 0.0, "dfs.namenode.reencrypt.throttle.limit.updater.ratio is not positive.");
    }

    void markZoneSubmissionDone(long zoneId) throws IOException, InterruptedException {
        ZoneSubmissionTracker tracker = this.handler.getTracker(zoneId);
        if (tracker != null && !tracker.getTasks().isEmpty()) {
            tracker.submissionDone = true;
        } else {
            this.handler.addDummyTracker(zoneId, tracker);
        }
    }

    @Override
    public void run() {
        this.isRunning = true;
        this.throttleTimerAll.start();
        while (true) {
            try {
                while (true) {
                    this.takeAndProcessTasks();
                }
            }
            catch (InterruptedException ie) {
                LOG.warn("Re-encryption updater thread interrupted. Exiting.");
                Thread.currentThread().interrupt();
                this.isRunning = false;
                return;
            }
            catch (IOException | CancellationException e) {
                LOG.warn("Re-encryption updater thread exception.", e);
                continue;
            }
            catch (Throwable t) {
                LOG.error("Re-encryption updater thread exiting.", t);
                this.isRunning = false;
                return;
            }
            break;
        }
    }

    private void processTaskEntries(String zoneNodePath, ReencryptionTask task) throws IOException, InterruptedException {
        assert (this.dir.hasWriteLock());
        if (!task.batch.isEmpty() && task.numFailures == 0) {
            LOG.debug("Updating file xattrs for re-encrypting zone {}, starting at {}", (Object)zoneNodePath, (Object)task.batch.getFirstFilePath());
            int batchSize = task.batch.size();
            Iterator<FileEdekInfo> it = task.batch.getBatch().iterator();
            while (it.hasNext()) {
                FileEdekInfo entry = it.next();
                LOG.trace("Updating {} for re-encryption.", (Object)entry.getInodeId());
                INode inode = this.dir.getInode(entry.getInodeId());
                if (inode == null) {
                    LOG.debug("INode {} doesn't exist, skipping re-encrypt.", (Object)entry.getInodeId());
                    it.remove();
                    continue;
                }
                Preconditions.checkNotNull(entry.edek);
                FileEncryptionInfo fei = FSDirEncryptionZoneOp.getFileEncryptionInfo(this.dir, INodesInPath.fromINode(inode));
                if (!fei.getKeyName().equals(entry.edek.getEncryptionKeyName())) {
                    LOG.debug("Inode {} EZ key changed, skipping re-encryption.", (Object)entry.getInodeId());
                    it.remove();
                    continue;
                }
                if (fei.getEzKeyVersionName().equals(entry.edek.getEncryptionKeyVersionName())) {
                    LOG.debug("Inode {} EZ key version unchanged, skipping re-encryption.", (Object)entry.getInodeId());
                    it.remove();
                    continue;
                }
                if (!Arrays.equals(fei.getEncryptedDataEncryptionKey(), entry.existingEdek.getEncryptedKeyVersion().getMaterial())) {
                    LOG.debug("Inode {} existing edek changed, skipping re-encryption", (Object)entry.getInodeId());
                    it.remove();
                    continue;
                }
                FileEncryptionInfo newFei = new FileEncryptionInfo(fei.getCipherSuite(), fei.getCryptoProtocolVersion(), entry.edek.getEncryptedKeyVersion().getMaterial(), entry.edek.getEncryptedKeyIv(), fei.getKeyName(), entry.edek.getEncryptionKeyVersionName());
                INodesInPath iip = INodesInPath.fromINode(inode);
                FSDirEncryptionZoneOp.setFileEncryptionInfo(this.dir, iip, newFei, XAttrSetFlag.REPLACE);
                task.lastFile = iip.getPath();
                ++task.numFilesUpdated;
            }
            LOG.info("Updated xattrs on {}({}) files in zone {} for re-encryption, starting:{}.", task.numFilesUpdated, batchSize, zoneNodePath, task.batch.getFirstFilePath());
        }
        task.processed = true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<XAttr> processCheckpoints(INode zoneNode, ZoneSubmissionTracker tracker) throws ExecutionException, IOException, InterruptedException {
        assert (this.dir.hasWriteLock());
        long zoneId = zoneNode.getId();
        String zonePath = zoneNode.getFullPathName();
        ZoneReencryptionStatus status = this.handler.getReencryptionStatus().getZoneStatus(zoneId);
        assert (status != null);
        LinkedList<Future> tasks = tracker.getTasks();
        ArrayList<XAttr> xAttrs = Lists.newArrayListWithCapacity(1);
        ListIterator iter = tasks.listIterator();
        ReencryptionHandler reencryptionHandler = this.handler;
        synchronized (reencryptionHandler) {
            Future curr;
            while (iter.hasNext() && !(curr = (Future)iter.next()).isCancelled() && curr.isDone() && ((ReencryptionTask)curr.get()).processed) {
                ReencryptionTask task = (ReencryptionTask)curr.get();
                LOG.debug("Updating re-encryption checkpoint with completed task. last: {} size:{}.", (Object)task.lastFile, (Object)task.batch.size());
                assert (zoneId == task.zoneId);
                try {
                    XAttr xattr = FSDirEncryptionZoneOp.updateReencryptionProgress(this.dir, zoneNode, status, task.lastFile, task.numFilesUpdated, task.numFailures);
                    xAttrs.clear();
                    xAttrs.add(xattr);
                }
                catch (IOException ie) {
                    LOG.warn("Failed to update re-encrypted progress to xattr for zone {}", (Object)zonePath, (Object)ie);
                    ++task.numFailures;
                }
                ++tracker.numCheckpointed;
                iter.remove();
            }
        }
        if (tracker.isCompleted()) {
            LOG.debug("Removed re-encryption tracker for zone {} because it completed with {} tasks.", (Object)zonePath, (Object)tracker.numCheckpointed);
            return this.handler.completeReencryption(zoneNode);
        }
        return xAttrs;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void takeAndProcessTasks() throws Exception {
        boolean shouldRetry;
        Future<ReencryptionTask> completed = this.batchService.take();
        this.throttle();
        this.checkPauseForTesting();
        if (completed.isCancelled()) {
            LOG.debug("Skipped a canceled re-encryption task");
            return;
        }
        ReencryptionTask task = completed.get();
        do {
            this.dir.getFSNamesystem().writeLock();
            try {
                this.throttleTimerLocked.start();
                this.processTask(task);
                shouldRetry = false;
            }
            catch (SafeModeException | RetriableException re) {
                LOG.info("Exception when processing re-encryption task for zone {}, retrying...", (Object)task.zoneId, (Object)re);
                shouldRetry = true;
                Thread.sleep(this.faultRetryInterval);
            }
            catch (IOException ioe) {
                LOG.warn("Failure processing re-encryption task for zone {}", (Object)task.zoneId, (Object)ioe);
                ++task.numFailures;
                task.processed = true;
                shouldRetry = false;
            }
            finally {
                this.dir.getFSNamesystem().writeUnlock("reencryptUpdater");
                this.throttleTimerLocked.stop();
            }
            this.dir.getEditLog().logSync();
        } while (shouldRetry);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void processTask(ReencryptionTask task) throws InterruptedException, ExecutionException, IOException {
        List<XAttr> xAttrs;
        String zonePath;
        this.dir.writeLock();
        try {
            this.handler.getTraverser().checkINodeReady(task.zoneId);
            INode zoneNode = this.dir.getInode(task.zoneId);
            if (zoneNode == null) {
                return;
            }
            zonePath = zoneNode.getFullPathName();
            LOG.info("Processing returned re-encryption task for zone {}({}), batch size {}, start:{}", zonePath, task.zoneId, task.batch.size(), task.batch.getFirstFilePath());
            ZoneSubmissionTracker tracker = this.handler.getTracker(zoneNode.getId());
            if (tracker == null) {
                LOG.info("Re-encryption was canceled.");
                return;
            }
            tracker.numFutureDone++;
            EncryptionFaultInjector.getInstance().reencryptUpdaterProcessOneTask();
            this.processTaskEntries(zonePath, task);
            EncryptionFaultInjector.getInstance().reencryptUpdaterProcessCheckpoint();
            xAttrs = this.processCheckpoints(zoneNode, tracker);
        }
        finally {
            this.dir.writeUnlock();
        }
        FSDirEncryptionZoneOp.saveFileXAttrsForBatch(this.dir, task.batch.getBatch());
        if (!xAttrs.isEmpty()) {
            this.dir.getEditLog().logSetXAttrs(zonePath, xAttrs, false);
        }
    }

    private synchronized void checkPauseForTesting() throws InterruptedException {
        ZoneSubmissionTracker tracker;
        assert (!this.dir.hasWriteLock());
        assert (!this.dir.getFSNamesystem().hasWriteLock());
        if (this.pauseAfterNthCheckpoint != 0 && (tracker = this.handler.unprotectedGetTracker(this.pauseZoneId)) != null && tracker.numFutureDone == this.pauseAfterNthCheckpoint) {
            this.shouldPauseForTesting = true;
            this.pauseAfterNthCheckpoint = 0;
        }
        while (this.shouldPauseForTesting) {
            LOG.info("Sleeping in the re-encryption updater for unit test.");
            this.wait();
            LOG.info("Continuing re-encryption updater after pausing.");
        }
    }

    private void throttle() throws InterruptedException {
        if (this.throttleLimitRatio >= 1.0) {
            return;
        }
        long expect = (long)((double)this.throttleTimerAll.now(TimeUnit.MILLISECONDS) * this.throttleLimitRatio);
        long actual = this.throttleTimerLocked.now(TimeUnit.MILLISECONDS);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Re-encryption updater throttling expect: {}, actual: {}, throttleTimerAll:{}", expect, actual, this.throttleTimerAll.now(TimeUnit.MILLISECONDS));
        }
        if (expect - actual < 0L) {
            long sleepMs = (long)((double)actual / this.throttleLimitRatio) - this.throttleTimerAll.now(TimeUnit.MILLISECONDS);
            LOG.debug("Throttling re-encryption, sleeping for {} ms", (Object)sleepMs);
            Thread.sleep(sleepMs);
        }
        this.throttleTimerAll.reset().start();
        this.throttleTimerLocked.reset();
    }

    static final class FileEdekInfo {
        private final long inodeId;
        private final KeyProviderCryptoExtension.EncryptedKeyVersion existingEdek;
        private KeyProviderCryptoExtension.EncryptedKeyVersion edek = null;

        FileEdekInfo(FSDirectory dir, INodeFile inode) throws IOException {
            assert (dir.hasReadLock());
            Preconditions.checkNotNull(inode, "INodeFile is null");
            this.inodeId = inode.getId();
            FileEncryptionInfo fei = FSDirEncryptionZoneOp.getFileEncryptionInfo(dir, INodesInPath.fromINode(inode));
            Preconditions.checkNotNull(fei, "FileEncryptionInfo is null for " + this.inodeId);
            this.existingEdek = KeyProviderCryptoExtension.EncryptedKeyVersion.createForDecryption(fei.getKeyName(), fei.getEzKeyVersionName(), fei.getIV(), fei.getEncryptedDataEncryptionKey());
        }

        long getInodeId() {
            return this.inodeId;
        }

        KeyProviderCryptoExtension.EncryptedKeyVersion getExistingEdek() {
            return this.existingEdek;
        }

        void setEdek(KeyProviderCryptoExtension.EncryptedKeyVersion ekv) {
            assert (ekv != null);
            this.edek = ekv;
        }
    }

    static final class ReencryptionTask {
        private final long zoneId;
        private boolean processed = false;
        private int numFilesUpdated = 0;
        private int numFailures = 0;
        private String lastFile = null;
        private final ReencryptionHandler.ReencryptionBatch batch;

        ReencryptionTask(long id, int failures, ReencryptionHandler.ReencryptionBatch theBatch) {
            this.zoneId = id;
            this.numFailures = failures;
            this.batch = theBatch;
        }
    }

    static final class ZoneSubmissionTracker {
        private boolean submissionDone = false;
        private LinkedList<Future> tasks = new LinkedList();
        private int numCheckpointed = 0;
        private int numFutureDone = 0;

        ZoneSubmissionTracker() {
        }

        void reset() {
            this.submissionDone = false;
            this.tasks.clear();
            this.numCheckpointed = 0;
            this.numFutureDone = 0;
        }

        LinkedList<Future> getTasks() {
            return this.tasks;
        }

        void cancelAllTasks() {
            if (!this.tasks.isEmpty()) {
                LOG.info("Cancelling {} re-encryption tasks", (Object)this.tasks.size());
                for (Future f : this.tasks) {
                    f.cancel(true);
                }
            }
        }

        void addTask(Future task) {
            this.tasks.add(task);
        }

        private boolean isCompleted() {
            return this.submissionDone && this.tasks.isEmpty();
        }

        void setSubmissionDone() {
            this.submissionDone = true;
        }
    }
}

