/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive.metastore.file;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.google.common.io.ByteStreams;
import io.airlift.json.JsonCodec;
import io.trino.collect.cache.EvictableCacheBuilder;
import io.trino.hdfs.DynamicHdfsConfiguration;
import io.trino.hdfs.HdfsConfig;
import io.trino.hdfs.HdfsConfiguration;
import io.trino.hdfs.HdfsConfigurationInitializer;
import io.trino.hdfs.HdfsContext;
import io.trino.hdfs.HdfsEnvironment;
import io.trino.hdfs.authentication.HdfsAuthentication;
import io.trino.hdfs.authentication.NoHdfsAuthentication;
import io.trino.plugin.hive.HiveBasicStatistics;
import io.trino.plugin.hive.HiveColumnStatisticType;
import io.trino.plugin.hive.HiveErrorCode;
import io.trino.plugin.hive.HivePartitionManager;
import io.trino.plugin.hive.HiveType;
import io.trino.plugin.hive.NodeVersion;
import io.trino.plugin.hive.PartitionNotFoundException;
import io.trino.plugin.hive.PartitionStatistics;
import io.trino.plugin.hive.SchemaAlreadyExistsException;
import io.trino.plugin.hive.TableAlreadyExistsException;
import io.trino.plugin.hive.acid.AcidTransaction;
import io.trino.plugin.hive.metastore.Column;
import io.trino.plugin.hive.metastore.Database;
import io.trino.plugin.hive.metastore.HiveColumnStatistics;
import io.trino.plugin.hive.metastore.HiveMetastore;
import io.trino.plugin.hive.metastore.HiveMetastoreConfig;
import io.trino.plugin.hive.metastore.HivePrincipal;
import io.trino.plugin.hive.metastore.HivePrivilegeInfo;
import io.trino.plugin.hive.metastore.MetastoreUtil;
import io.trino.plugin.hive.metastore.Partition;
import io.trino.plugin.hive.metastore.PartitionWithStatistics;
import io.trino.plugin.hive.metastore.PrincipalPrivileges;
import io.trino.plugin.hive.metastore.Table;
import io.trino.plugin.hive.metastore.file.DatabaseMetadata;
import io.trino.plugin.hive.metastore.file.FileHiveMetastoreConfig;
import io.trino.plugin.hive.metastore.file.PartitionMetadata;
import io.trino.plugin.hive.metastore.file.PermissionMetadata;
import io.trino.plugin.hive.metastore.file.TableMetadata;
import io.trino.plugin.hive.metastore.thrift.ThriftMetastoreUtil;
import io.trino.plugin.hive.util.HiveUtil;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ColumnNotFoundException;
import io.trino.spi.connector.SchemaNotFoundException;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.connector.TableNotFoundException;
import io.trino.spi.predicate.TupleDomain;
import io.trino.spi.security.ConnectorIdentity;
import io.trino.spi.security.PrincipalType;
import io.trino.spi.security.RoleGrant;
import io.trino.spi.type.Type;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.common.FileUtils;
import org.apache.hadoop.hive.metastore.TableType;

@ThreadSafe
public class FileHiveMetastore
implements HiveMetastore {
    private static final String PUBLIC_ROLE_NAME = "public";
    private static final String ADMIN_ROLE_NAME = "admin";
    private static final String TRINO_SCHEMA_FILE_NAME_SUFFIX = ".trinoSchema";
    private static final String TRINO_PERMISSIONS_DIRECTORY_NAME = ".trinoPermissions";
    public static final String ROLES_FILE_NAME = ".roles";
    public static final String ROLE_GRANTS_FILE_NAME = ".roleGrants";
    private static final Set<String> ADMIN_USERS = ImmutableSet.of((Object)"admin", (Object)"hive", (Object)"hdfs");
    private static final int MAX_DATABASE_NAME_LENGTH = 128;
    private final String currentVersion;
    private final FileHiveMetastoreConfig.VersionCompatibility versionCompatibility;
    private final HdfsEnvironment hdfsEnvironment;
    private final Path catalogDirectory;
    private final HdfsContext hdfsContext;
    private final boolean hideDeltaLakeTables;
    private final FileSystem metadataFileSystem;
    private final JsonCodec<DatabaseMetadata> databaseCodec = JsonCodec.jsonCodec(DatabaseMetadata.class);
    private final JsonCodec<TableMetadata> tableCodec = JsonCodec.jsonCodec(TableMetadata.class);
    private final JsonCodec<PartitionMetadata> partitionCodec = JsonCodec.jsonCodec(PartitionMetadata.class);
    private final JsonCodec<List<PermissionMetadata>> permissionsCodec = JsonCodec.listJsonCodec(PermissionMetadata.class);
    private final JsonCodec<List<String>> rolesCodec = JsonCodec.listJsonCodec(String.class);
    private final JsonCodec<List<RoleGrant>> roleGrantsCodec = JsonCodec.listJsonCodec(RoleGrant.class);
    private final LoadingCache<String, List<String>> listTablesCache;

    @VisibleForTesting
    public static FileHiveMetastore createTestingFileHiveMetastore(File catalogDirectory) {
        HdfsConfig hdfsConfig = new HdfsConfig();
        DynamicHdfsConfiguration hdfsConfiguration = new DynamicHdfsConfiguration(new HdfsConfigurationInitializer(hdfsConfig), (Set)ImmutableSet.of());
        HdfsEnvironment hdfsEnvironment = new HdfsEnvironment((HdfsConfiguration)hdfsConfiguration, hdfsConfig, (HdfsAuthentication)new NoHdfsAuthentication());
        return new FileHiveMetastore(new NodeVersion("testversion"), hdfsEnvironment, new HiveMetastoreConfig().isHideDeltaLakeTables(), new FileHiveMetastoreConfig().setCatalogDirectory(catalogDirectory.toURI().toString()).setMetastoreUser("test"));
    }

    public FileHiveMetastore(NodeVersion nodeVersion, HdfsEnvironment hdfsEnvironment, boolean hideDeltaLakeTables, FileHiveMetastoreConfig config) {
        this.currentVersion = nodeVersion.toString();
        this.versionCompatibility = Objects.requireNonNull(config.getVersionCompatibility(), "config.getVersionCompatibility() is null");
        this.hdfsEnvironment = Objects.requireNonNull(hdfsEnvironment, "hdfsEnvironment is null");
        this.catalogDirectory = new Path(Objects.requireNonNull(config.getCatalogDirectory(), "catalogDirectory is null"));
        this.hdfsContext = new HdfsContext(ConnectorIdentity.ofUser((String)config.getMetastoreUser()));
        this.hideDeltaLakeTables = hideDeltaLakeTables;
        try {
            this.metadataFileSystem = hdfsEnvironment.getFileSystem(this.hdfsContext, this.catalogDirectory);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, (Throwable)e);
        }
        this.listTablesCache = EvictableCacheBuilder.newBuilder().expireAfterWrite(10L, TimeUnit.SECONDS).build(CacheLoader.from(this::doListAllTables));
    }

    @Override
    public synchronized void createDatabase(Database database) {
        Objects.requireNonNull(database, "database is null");
        database = new Database(database.getDatabaseName().toLowerCase(Locale.ENGLISH), database.getLocation(), database.getOwnerName(), database.getOwnerType(), database.getComment(), database.getParameters());
        if (database.getLocation().isPresent()) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Database cannot be created with a location set");
        }
        this.verifyDatabaseNameLength(database.getDatabaseName());
        this.verifyDatabaseNotExists(database.getDatabaseName());
        Path databaseMetadataDirectory = this.getDatabaseMetadataDirectory(database.getDatabaseName());
        this.writeSchemaFile(SchemaType.DATABASE, databaseMetadataDirectory, this.databaseCodec, new DatabaseMetadata(this.currentVersion, database), false);
        try {
            this.metadataFileSystem.mkdirs(databaseMetadataDirectory);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not write database", (Throwable)e);
        }
    }

    @Override
    public synchronized void dropDatabase(String databaseName, boolean deleteData) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        databaseName = databaseName.toLowerCase(Locale.ENGLISH);
        this.getRequiredDatabase(databaseName);
        if (!this.getAllTables(databaseName).isEmpty()) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Database " + databaseName + " is not empty");
        }
        if (deleteData) {
            this.deleteDirectoryAndSchema(SchemaType.DATABASE, this.getDatabaseMetadataDirectory(databaseName));
        } else {
            this.deleteSchemaFile(SchemaType.DATABASE, this.getDatabaseMetadataDirectory(databaseName));
        }
    }

    @Override
    public synchronized void renameDatabase(String databaseName, String newDatabaseName) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        Objects.requireNonNull(newDatabaseName, "newDatabaseName is null");
        this.verifyDatabaseNameLength(newDatabaseName);
        this.getRequiredDatabase(databaseName);
        this.verifyDatabaseNotExists(newDatabaseName);
        Path oldDatabaseMetadataDirectory = this.getDatabaseMetadataDirectory(databaseName);
        Path newDatabaseMetadataDirectory = this.getDatabaseMetadataDirectory(newDatabaseName);
        try {
            this.renameSchemaFile(SchemaType.DATABASE, oldDatabaseMetadataDirectory, newDatabaseMetadataDirectory);
            if (!this.metadataFileSystem.rename(oldDatabaseMetadataDirectory, newDatabaseMetadataDirectory)) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not rename database metadata directory");
            }
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, (Throwable)e);
        }
    }

    @Override
    public synchronized void setDatabaseOwner(String databaseName, HivePrincipal principal) {
        Database database = this.getRequiredDatabase(databaseName);
        Path databaseMetadataDirectory = this.getDatabaseMetadataDirectory(database.getDatabaseName());
        Database newDatabase = Database.builder(database).setOwnerName(Optional.of(principal.getName())).setOwnerType(Optional.of(principal.getType())).build();
        this.writeSchemaFile(SchemaType.DATABASE, databaseMetadataDirectory, this.databaseCodec, new DatabaseMetadata(this.currentVersion, newDatabase), true);
    }

    @Override
    public synchronized Optional<Database> getDatabase(String databaseName) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        String normalizedName = databaseName.toLowerCase(Locale.ENGLISH);
        Path databaseMetadataDirectory = this.getDatabaseMetadataDirectory(normalizedName);
        return this.readSchemaFile(SchemaType.DATABASE, databaseMetadataDirectory, this.databaseCodec).map(databaseMetadata -> {
            this.checkVersion(databaseMetadata.getWriterVersion());
            return databaseMetadata.toDatabase(normalizedName, databaseMetadataDirectory.toString());
        });
    }

    private Database getRequiredDatabase(String databaseName) {
        return this.getDatabase(databaseName).orElseThrow(() -> new SchemaNotFoundException(databaseName));
    }

    private void verifyDatabaseNameLength(String databaseName) {
        if (databaseName.length() > 128) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, String.format("Schema name must be shorter than or equal to '%s' characters but got '%s'", 128, databaseName.length()));
        }
    }

    private void verifyDatabaseNotExists(String databaseName) {
        if (this.getDatabase(databaseName).isPresent()) {
            throw new SchemaAlreadyExistsException(databaseName);
        }
    }

    @Override
    public synchronized List<String> getAllDatabases() {
        List databases = this.getChildSchemaDirectories(SchemaType.DATABASE, this.catalogDirectory).stream().map(Path::getName).collect(Collectors.toList());
        return ImmutableList.copyOf(databases);
    }

    @Override
    public synchronized void createTable(Table table, PrincipalPrivileges principalPrivileges) {
        Path tableMetadataDirectory;
        block12: {
            this.verifyDatabaseExists(table.getDatabaseName());
            this.verifyTableNotExists(table.getDatabaseName(), table.getTableName());
            tableMetadataDirectory = this.getTableMetadataDirectory(table);
            if (table.getTableType().equals(TableType.VIRTUAL_VIEW.name())) {
                Preconditions.checkArgument((boolean)table.getStorage().getLocation().isEmpty(), (Object)"Storage location for view must be empty");
            } else if (table.getTableType().equals(TableType.MANAGED_TABLE.name())) {
                if (!new Path(table.getStorage().getLocation()).toString().contains(tableMetadataDirectory.toString())) {
                    throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Table directory must be " + tableMetadataDirectory);
                }
            } else {
                if (table.getTableType().equals(TableType.EXTERNAL_TABLE.name())) {
                    try {
                        Path externalLocation = new Path(table.getStorage().getLocation());
                        FileSystem externalFileSystem = this.hdfsEnvironment.getFileSystem(this.hdfsContext, externalLocation);
                        if (!externalFileSystem.isDirectory(externalLocation)) {
                            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "External table location does not exist");
                        }
                        break block12;
                    }
                    catch (IOException e) {
                        throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not validate external location", (Throwable)e);
                    }
                }
                if (!table.getTableType().equals(TableType.MATERIALIZED_VIEW.name())) {
                    throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Table type not supported: " + table.getTableType());
                }
            }
        }
        this.writeSchemaFile(SchemaType.TABLE, tableMetadataDirectory, this.tableCodec, new TableMetadata(this.currentVersion, table), false);
        for (Map.Entry entry : principalPrivileges.getUserPrivileges().asMap().entrySet()) {
            this.setTablePrivileges(new HivePrincipal(PrincipalType.USER, (String)entry.getKey()), table.getDatabaseName(), table.getTableName(), (Collection)entry.getValue());
        }
        for (Map.Entry entry : principalPrivileges.getRolePrivileges().asMap().entrySet()) {
            this.setTablePrivileges(new HivePrincipal(PrincipalType.ROLE, (String)entry.getKey()), table.getDatabaseName(), table.getTableName(), (Collection)entry.getValue());
        }
    }

    @Override
    public synchronized Optional<Table> getTable(String databaseName, String tableName) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        Objects.requireNonNull(tableName, "tableName is null");
        Path tableMetadataDirectory = this.getTableMetadataDirectory(databaseName, tableName);
        return this.readSchemaFile(SchemaType.TABLE, tableMetadataDirectory, this.tableCodec).map(tableMetadata -> {
            this.checkVersion(tableMetadata.getWriterVersion());
            return tableMetadata.toTable(databaseName, tableName, tableMetadataDirectory.toString());
        });
    }

    @Override
    public synchronized void setTableOwner(String databaseName, String tableName, HivePrincipal principal) {
        if (principal.getType() != PrincipalType.USER) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Setting table owner type as a role is not supported");
        }
        Table table = this.getRequiredTable(databaseName, tableName);
        Path tableMetadataDirectory = this.getTableMetadataDirectory(table);
        Table newTable = Table.builder(table).setOwner(Optional.of(principal.getName())).build();
        this.writeSchemaFile(SchemaType.TABLE, tableMetadataDirectory, this.tableCodec, new TableMetadata(this.currentVersion, newTable), true);
    }

    @Override
    public Set<HiveColumnStatisticType> getSupportedColumnStatistics(Type type) {
        return ThriftMetastoreUtil.getSupportedColumnStatistics(type);
    }

    @Override
    public synchronized PartitionStatistics getTableStatistics(Table table) {
        return this.getTableStatistics(table.getDatabaseName(), table.getTableName());
    }

    private synchronized PartitionStatistics getTableStatistics(String databaseName, String tableName) {
        Path tableMetadataDirectory = this.getTableMetadataDirectory(databaseName, tableName);
        TableMetadata tableMetadata = this.readSchemaFile(SchemaType.TABLE, tableMetadataDirectory, this.tableCodec).orElseThrow(() -> new TableNotFoundException(new SchemaTableName(databaseName, tableName)));
        this.checkVersion(tableMetadata.getWriterVersion());
        HiveBasicStatistics basicStatistics = ThriftMetastoreUtil.getHiveBasicStatistics(tableMetadata.getParameters());
        Map<String, HiveColumnStatistics> columnStatistics = tableMetadata.getColumnStatistics();
        return new PartitionStatistics(basicStatistics, columnStatistics);
    }

    @Override
    public synchronized Map<String, PartitionStatistics> getPartitionStatistics(Table table, List<Partition> partitions) {
        return (Map)partitions.stream().collect(ImmutableMap.toImmutableMap(partition -> MetastoreUtil.makePartitionName(table, partition), partition -> this.getPartitionStatisticsInternal(table, partition.getValues())));
    }

    private synchronized PartitionStatistics getPartitionStatisticsInternal(Table table, List<String> partitionValues) {
        Path partitionDirectory = this.getPartitionMetadataDirectory(table, (List<String>)ImmutableList.copyOf(partitionValues));
        PartitionMetadata partitionMetadata = this.readSchemaFile(SchemaType.PARTITION, partitionDirectory, this.partitionCodec).orElseThrow(() -> new PartitionNotFoundException(table.getSchemaTableName(), partitionValues));
        HiveBasicStatistics basicStatistics = ThriftMetastoreUtil.getHiveBasicStatistics(partitionMetadata.getParameters());
        return new PartitionStatistics(basicStatistics, partitionMetadata.getColumnStatistics());
    }

    private Table getRequiredTable(String databaseName, String tableName) {
        return this.getTable(databaseName, tableName).orElseThrow(() -> new TableNotFoundException(new SchemaTableName(databaseName, tableName)));
    }

    private void verifyDatabaseExists(String databaseName) {
        if (this.getDatabase(databaseName).isEmpty()) {
            throw new SchemaNotFoundException(databaseName);
        }
    }

    private void verifyTableNotExists(String newDatabaseName, String newTableName) {
        if (this.getTable(newDatabaseName, newTableName).isPresent()) {
            throw new TableAlreadyExistsException(new SchemaTableName(newDatabaseName, newTableName));
        }
    }

    @Override
    public synchronized void updateTableStatistics(String databaseName, String tableName, AcidTransaction transaction, Function<PartitionStatistics, PartitionStatistics> update) {
        PartitionStatistics originalStatistics = this.getTableStatistics(databaseName, tableName);
        PartitionStatistics updatedStatistics = update.apply(originalStatistics);
        Path tableMetadataDirectory = this.getTableMetadataDirectory(databaseName, tableName);
        TableMetadata tableMetadata = this.readSchemaFile(SchemaType.TABLE, tableMetadataDirectory, this.tableCodec).orElseThrow(() -> new TableNotFoundException(new SchemaTableName(databaseName, tableName)));
        this.checkVersion(tableMetadata.getWriterVersion());
        TableMetadata updatedMetadata = tableMetadata.withParameters(this.currentVersion, ThriftMetastoreUtil.updateStatisticsParameters(tableMetadata.getParameters(), updatedStatistics.getBasicStatistics())).withColumnStatistics(this.currentVersion, updatedStatistics.getColumnStatistics());
        this.writeSchemaFile(SchemaType.TABLE, tableMetadataDirectory, this.tableCodec, updatedMetadata, true);
    }

    @Override
    public synchronized void updatePartitionStatistics(Table table, Map<String, Function<PartitionStatistics, PartitionStatistics>> updates) {
        updates.forEach((partitionName, update) -> {
            PartitionStatistics originalStatistics = this.getPartitionStatisticsInternal(table, HivePartitionManager.extractPartitionValues(partitionName));
            PartitionStatistics updatedStatistics = (PartitionStatistics)update.apply(originalStatistics);
            List<String> partitionValues = HivePartitionManager.extractPartitionValues(partitionName);
            Path partitionDirectory = this.getPartitionMetadataDirectory(table, partitionValues);
            PartitionMetadata partitionMetadata = this.readSchemaFile(SchemaType.PARTITION, partitionDirectory, this.partitionCodec).orElseThrow(() -> new PartitionNotFoundException(new SchemaTableName(table.getDatabaseName(), table.getTableName()), partitionValues));
            PartitionMetadata updatedMetadata = partitionMetadata.withParameters(ThriftMetastoreUtil.updateStatisticsParameters(partitionMetadata.getParameters(), updatedStatistics.getBasicStatistics())).withColumnStatistics(updatedStatistics.getColumnStatistics());
            this.writeSchemaFile(SchemaType.PARTITION, partitionDirectory, this.partitionCodec, updatedMetadata, true);
        });
    }

    @Override
    public synchronized List<String> getAllTables(String databaseName) {
        return (List)this.listAllTables(databaseName).stream().filter(this.hideDeltaLakeTables ? Predicate.not(arg_0 -> ((ImmutableSet)ImmutableSet.copyOf(this.getTablesWithParameter(databaseName, "spark.sql.sources.provider", "delta"))).contains(arg_0)) : tableName -> true).collect(ImmutableList.toImmutableList());
    }

    @Override
    public synchronized List<String> getTablesWithParameter(String databaseName, String parameterKey, String parameterValue) {
        Objects.requireNonNull(parameterKey, "parameterKey is null");
        Objects.requireNonNull(parameterValue, "parameterValue is null");
        List<String> tables = this.listAllTables(databaseName);
        return (List)tables.stream().map(tableName -> this.getTable(databaseName, (String)tableName)).filter(Optional::isPresent).map(Optional::get).filter(table -> parameterValue.equals(table.getParameters().get(parameterKey))).map(Table::getTableName).collect(ImmutableList.toImmutableList());
    }

    @GuardedBy(value="this")
    private List<String> listAllTables(String databaseName) {
        return (List)this.listTablesCache.getUnchecked((Object)databaseName);
    }

    @GuardedBy(value="this")
    private List<String> doListAllTables(String databaseName) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        Optional<Database> database = this.getDatabase(databaseName);
        if (database.isEmpty()) {
            return ImmutableList.of();
        }
        Path databaseMetadataDirectory = this.getDatabaseMetadataDirectory(databaseName);
        List tables = (List)this.getChildSchemaDirectories(SchemaType.TABLE, databaseMetadataDirectory).stream().map(Path::getName).collect(ImmutableList.toImmutableList());
        return tables;
    }

    @Override
    public synchronized List<String> getAllViews(String databaseName) {
        return (List)this.getAllTables(databaseName).stream().map(tableName -> this.getTable(databaseName, (String)tableName)).filter(Optional::isPresent).map(Optional::get).filter(table -> TableType.valueOf((String)table.getTableType()).equals((Object)TableType.VIRTUAL_VIEW)).map(Table::getTableName).collect(ImmutableList.toImmutableList());
    }

    @Override
    public synchronized void dropTable(String databaseName, String tableName, boolean deleteData) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        Objects.requireNonNull(tableName, "tableName is null");
        Table table = this.getRequiredTable(databaseName, tableName);
        Path tableMetadataDirectory = this.getTableMetadataDirectory(databaseName, tableName);
        if (deleteData) {
            this.deleteDirectoryAndSchema(SchemaType.TABLE, tableMetadataDirectory);
        } else {
            this.deleteSchemaFile(SchemaType.TABLE, tableMetadataDirectory);
            this.deleteTablePrivileges(table);
        }
    }

    @Override
    public synchronized void replaceTable(String databaseName, String tableName, Table newTable, PrincipalPrivileges principalPrivileges) {
        Table table = this.getRequiredTable(databaseName, tableName);
        if (!table.getDatabaseName().equals(databaseName) || !table.getTableName().equals(tableName)) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Replacement table must have same name");
        }
        if (HiveUtil.isIcebergTable(table) && !Objects.equals(table.getParameters().get("metadata_location"), newTable.getParameters().get("previous_metadata_location"))) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_CONCURRENT_MODIFICATION_DETECTED, "Cannot update Iceberg table: supplied previous location does not match current location");
        }
        Path tableMetadataDirectory = this.getTableMetadataDirectory(table);
        this.writeSchemaFile(SchemaType.TABLE, tableMetadataDirectory, this.tableCodec, new TableMetadata(this.currentVersion, newTable), true);
        this.deleteTablePrivileges(table);
        for (Map.Entry entry : principalPrivileges.getUserPrivileges().asMap().entrySet()) {
            this.setTablePrivileges(new HivePrincipal(PrincipalType.USER, (String)entry.getKey()), table.getDatabaseName(), table.getTableName(), (Collection)entry.getValue());
        }
        for (Map.Entry entry : principalPrivileges.getRolePrivileges().asMap().entrySet()) {
            this.setTablePrivileges(new HivePrincipal(PrincipalType.ROLE, (String)entry.getKey()), table.getDatabaseName(), table.getTableName(), (Collection)entry.getValue());
        }
    }

    @Override
    public synchronized void renameTable(String databaseName, String tableName, String newDatabaseName, String newTableName) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        Objects.requireNonNull(tableName, "tableName is null");
        Objects.requireNonNull(newDatabaseName, "newDatabaseName is null");
        Objects.requireNonNull(newTableName, "newTableName is null");
        Table table = this.getRequiredTable(databaseName, tableName);
        this.getRequiredDatabase(newDatabaseName);
        this.verifyTableNotExists(newDatabaseName, newTableName);
        Path oldPath = this.getTableMetadataDirectory(databaseName, tableName);
        Path newPath = this.getTableMetadataDirectory(newDatabaseName, newTableName);
        try {
            if (HiveUtil.isIcebergTable(table)) {
                if (!this.metadataFileSystem.mkdirs(newPath)) {
                    throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not create new table directory");
                }
                if (!this.metadataFileSystem.rename(FileHiveMetastore.getSchemaPath(SchemaType.TABLE, oldPath), FileHiveMetastore.getSchemaPath(SchemaType.TABLE, newPath))) {
                    throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not rename table schema file");
                }
            } else if (!this.metadataFileSystem.rename(oldPath, newPath)) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not rename table directory");
            }
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, (Throwable)e);
        }
        finally {
            this.listTablesCache.invalidateAll();
        }
    }

    @Override
    public synchronized void commentTable(String databaseName, String tableName, Optional<String> comment) {
        this.alterTable(databaseName, tableName, oldTable -> {
            Map<String, String> parameters = oldTable.getParameters().entrySet().stream().filter(entry -> !((String)entry.getKey()).equals("comment")).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
            comment.ifPresent(value -> parameters.put("comment", (String)value));
            return oldTable.withParameters(this.currentVersion, parameters);
        });
    }

    @Override
    public synchronized void commentColumn(String databaseName, String tableName, String columnName, Optional<String> comment) {
        this.alterTable(databaseName, tableName, oldTable -> {
            if (oldTable.getColumn(columnName).isEmpty()) {
                SchemaTableName name = new SchemaTableName(databaseName, tableName);
                throw new ColumnNotFoundException(name, columnName);
            }
            ImmutableList.Builder newDataColumns = ImmutableList.builder();
            for (Column fieldSchema : oldTable.getDataColumns()) {
                if (fieldSchema.getName().equals(columnName)) {
                    newDataColumns.add((Object)new Column(columnName, fieldSchema.getType(), comment));
                    continue;
                }
                newDataColumns.add((Object)fieldSchema);
            }
            return oldTable.withDataColumns(this.currentVersion, (List<Column>)newDataColumns.build());
        });
    }

    @Override
    public synchronized void addColumn(String databaseName, String tableName, String columnName, HiveType columnType, String columnComment) {
        this.alterTable(databaseName, tableName, oldTable -> {
            if (oldTable.getColumn(columnName).isPresent()) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.ALREADY_EXISTS, "Column already exists: " + columnName);
            }
            return oldTable.withDataColumns(this.currentVersion, (List<Column>)ImmutableList.builder().addAll(oldTable.getDataColumns()).add((Object)new Column(columnName, columnType, Optional.ofNullable(columnComment))).build());
        });
    }

    @Override
    public synchronized void renameColumn(String databaseName, String tableName, String oldColumnName, String newColumnName) {
        this.alterTable(databaseName, tableName, oldTable -> {
            if (oldTable.getColumn(newColumnName).isPresent()) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.ALREADY_EXISTS, "Column already exists: " + newColumnName);
            }
            if (oldTable.getColumn(oldColumnName).isEmpty()) {
                SchemaTableName name = new SchemaTableName(databaseName, tableName);
                throw new ColumnNotFoundException(name, oldColumnName);
            }
            for (Column column : oldTable.getPartitionColumns()) {
                if (!column.getName().equals(oldColumnName)) continue;
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Renaming partition columns is not supported");
            }
            ImmutableList.Builder newDataColumns = ImmutableList.builder();
            for (Column fieldSchema : oldTable.getDataColumns()) {
                if (fieldSchema.getName().equals(oldColumnName)) {
                    newDataColumns.add((Object)new Column(newColumnName, fieldSchema.getType(), fieldSchema.getComment()));
                    continue;
                }
                newDataColumns.add((Object)fieldSchema);
            }
            return oldTable.withDataColumns(this.currentVersion, (List<Column>)newDataColumns.build());
        });
    }

    @Override
    public synchronized void dropColumn(String databaseName, String tableName, String columnName) {
        this.alterTable(databaseName, tableName, oldTable -> {
            MetastoreUtil.verifyCanDropColumn(this, databaseName, tableName, columnName);
            if (oldTable.getColumn(columnName).isEmpty()) {
                SchemaTableName name = new SchemaTableName(databaseName, tableName);
                throw new ColumnNotFoundException(name, columnName);
            }
            ImmutableList.Builder newDataColumns = ImmutableList.builder();
            for (Column fieldSchema : oldTable.getDataColumns()) {
                if (fieldSchema.getName().equals(columnName)) continue;
                newDataColumns.add((Object)fieldSchema);
            }
            return oldTable.withDataColumns(this.currentVersion, (List<Column>)newDataColumns.build());
        });
    }

    private void alterTable(String databaseName, String tableName, Function<TableMetadata, TableMetadata> alterFunction) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        Objects.requireNonNull(tableName, "tableName is null");
        Path tableMetadataDirectory = this.getTableMetadataDirectory(databaseName, tableName);
        TableMetadata oldTableSchema = this.readSchemaFile(SchemaType.TABLE, tableMetadataDirectory, this.tableCodec).orElseThrow(() -> new TableNotFoundException(new SchemaTableName(databaseName, tableName)));
        this.checkVersion(oldTableSchema.getWriterVersion());
        TableMetadata newTableSchema = alterFunction.apply(oldTableSchema);
        if (oldTableSchema == newTableSchema) {
            return;
        }
        this.writeSchemaFile(SchemaType.TABLE, tableMetadataDirectory, this.tableCodec, newTableSchema, true);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public synchronized void addPartitions(String databaseName, String tableName, List<PartitionWithStatistics> partitions) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        Objects.requireNonNull(tableName, "tableName is null");
        Objects.requireNonNull(partitions, "partitions is null");
        Table table = this.getRequiredTable(databaseName, tableName);
        TableType tableType = TableType.valueOf((String)table.getTableType());
        Preconditions.checkArgument((boolean)EnumSet.of(TableType.MANAGED_TABLE, TableType.EXTERNAL_TABLE).contains(tableType), (String)"Invalid table type: %s", (Object)tableType);
        try {
            LinkedHashMap<Path, byte[]> schemaFiles = new LinkedHashMap<Path, byte[]>();
            for (PartitionWithStatistics partitionWithStatistics : partitions) {
                Partition partition = partitionWithStatistics.getPartition();
                this.verifiedPartition(table, partition);
                Path partitionMetadataDirectory = this.getPartitionMetadataDirectory(table, partition.getValues());
                Path schemaPath = FileHiveMetastore.getSchemaPath(SchemaType.PARTITION, partitionMetadataDirectory);
                if (this.metadataFileSystem.exists(schemaPath)) {
                    throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Partition already exists");
                }
                byte[] schemaJson = this.partitionCodec.toJsonBytes((Object)new PartitionMetadata(table, partitionWithStatistics));
                schemaFiles.put(schemaPath, schemaJson);
            }
            LinkedHashSet<Path> createdFiles = new LinkedHashSet<Path>();
            try {
                for (Map.Entry entry : schemaFiles.entrySet()) {
                    try {
                        FSDataOutputStream outputStream = this.metadataFileSystem.create((Path)entry.getKey());
                        try {
                            createdFiles.add((Path)entry.getKey());
                            outputStream.write((byte[])entry.getValue());
                        }
                        finally {
                            if (outputStream == null) continue;
                            outputStream.close();
                        }
                    }
                    catch (IOException e) {
                        throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not write partition schema", (Throwable)e);
                        return;
                    }
                }
            }
            catch (Throwable throwable) {
                for (Path createdFile : createdFiles) {
                    try {
                        this.metadataFileSystem.delete(createdFile, false);
                    }
                    catch (IOException iOException) {}
                }
                throw throwable;
            }
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, (Throwable)e);
        }
    }

    private void verifiedPartition(Table table, Partition partition) {
        block8: {
            Path partitionMetadataDirectory = this.getPartitionMetadataDirectory(table, partition.getValues());
            if (table.getTableType().equals(TableType.MANAGED_TABLE.name())) {
                if (!partitionMetadataDirectory.equals((Object)new Path(partition.getStorage().getLocation()))) {
                    throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Partition directory must be " + partitionMetadataDirectory);
                }
            } else {
                if (table.getTableType().equals(TableType.EXTERNAL_TABLE.name())) {
                    try {
                        Path externalLocation = new Path(partition.getStorage().getLocation());
                        FileSystem externalFileSystem = this.hdfsEnvironment.getFileSystem(this.hdfsContext, externalLocation);
                        if (!externalFileSystem.isDirectory(externalLocation)) {
                            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "External partition location does not exist");
                        }
                        if (FileHiveMetastore.isChildDirectory(this.catalogDirectory, externalLocation)) {
                            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "External partition location cannot be inside the system metadata directory");
                        }
                        break block8;
                    }
                    catch (IOException e) {
                        throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not validate external partition location", (Throwable)e);
                    }
                }
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Partitions cannot be added to " + table.getTableType());
            }
        }
    }

    @Override
    public synchronized void dropPartition(String databaseName, String tableName, List<String> partitionValues, boolean deleteData) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        Objects.requireNonNull(tableName, "tableName is null");
        Objects.requireNonNull(partitionValues, "partitionValues is null");
        Optional<Table> tableReference = this.getTable(databaseName, tableName);
        if (tableReference.isEmpty()) {
            return;
        }
        Table table = tableReference.get();
        Path partitionMetadataDirectory = this.getPartitionMetadataDirectory(table, partitionValues);
        if (deleteData) {
            this.deleteDirectoryAndSchema(SchemaType.PARTITION, partitionMetadataDirectory);
        } else {
            this.deleteSchemaFile(SchemaType.PARTITION, partitionMetadataDirectory);
        }
    }

    @Override
    public synchronized void alterPartition(String databaseName, String tableName, PartitionWithStatistics partitionWithStatistics) {
        Table table = this.getRequiredTable(databaseName, tableName);
        Partition partition = partitionWithStatistics.getPartition();
        this.verifiedPartition(table, partition);
        Path partitionMetadataDirectory = this.getPartitionMetadataDirectory(table, partition.getValues());
        this.writeSchemaFile(SchemaType.PARTITION, partitionMetadataDirectory, this.partitionCodec, new PartitionMetadata(table, partitionWithStatistics), true);
    }

    @Override
    public synchronized void createRole(String role, String grantor) {
        HashSet<String> roles = new HashSet<String>(this.listRoles());
        roles.add(role);
        this.writeFile("roles", this.getRolesFile(), this.rolesCodec, ImmutableList.copyOf(roles), true);
    }

    @Override
    public synchronized void dropRole(String role) {
        HashSet<String> roles = new HashSet<String>(this.listRoles());
        roles.remove(role);
        this.writeFile("roles", this.getRolesFile(), this.rolesCodec, ImmutableList.copyOf(roles), true);
        Set<RoleGrant> grants = this.listRoleGrantsSanitized();
        this.writeRoleGrantsFile(grants);
    }

    @Override
    public synchronized Set<String> listRoles() {
        HashSet<String> roles = new HashSet<String>();
        roles.add(ADMIN_ROLE_NAME);
        this.readFile("roles", this.getRolesFile(), this.rolesCodec).ifPresent(roles::addAll);
        return ImmutableSet.copyOf(roles);
    }

    @Override
    public synchronized void grantRoles(Set<String> roles, Set<HivePrincipal> grantees, boolean adminOption, HivePrincipal grantor) {
        Set<String> existingRoles = this.listRoles();
        Set<RoleGrant> existingGrants = this.listRoleGrantsSanitized();
        Set<RoleGrant> modifiedGrants = new HashSet<RoleGrant>(existingGrants);
        for (HivePrincipal grantee : grantees) {
            for (String role : roles) {
                Preconditions.checkArgument((boolean)existingRoles.contains(role), (String)"Role does not exist: %s", (Object)role);
                if (grantee.getType() == PrincipalType.ROLE) {
                    Preconditions.checkArgument((boolean)existingRoles.contains(grantee.getName()), (String)"Role does not exist: %s", (Object)grantee.getName());
                }
                RoleGrant grantWithAdminOption = new RoleGrant(grantee.toTrinoPrincipal(), role, true);
                RoleGrant grantWithoutAdminOption = new RoleGrant(grantee.toTrinoPrincipal(), role, false);
                if (adminOption) {
                    modifiedGrants.remove(grantWithoutAdminOption);
                    modifiedGrants.add(grantWithAdminOption);
                    continue;
                }
                modifiedGrants.remove(grantWithAdminOption);
                modifiedGrants.add(grantWithoutAdminOption);
            }
        }
        if (!existingGrants.equals(modifiedGrants = this.removeDuplicatedEntries(modifiedGrants))) {
            this.writeRoleGrantsFile(modifiedGrants);
        }
    }

    @Override
    public synchronized void revokeRoles(Set<String> roles, Set<HivePrincipal> grantees, boolean adminOption, HivePrincipal grantor) {
        Set<RoleGrant> existingGrants = this.listRoleGrantsSanitized();
        Set<RoleGrant> modifiedGrants = new HashSet<RoleGrant>(existingGrants);
        for (HivePrincipal grantee : grantees) {
            for (String role : roles) {
                RoleGrant grantWithAdminOption = new RoleGrant(grantee.toTrinoPrincipal(), role, true);
                RoleGrant grantWithoutAdminOption = new RoleGrant(grantee.toTrinoPrincipal(), role, false);
                if (!modifiedGrants.contains(grantWithAdminOption) && !modifiedGrants.contains(grantWithoutAdminOption)) continue;
                if (adminOption) {
                    modifiedGrants.remove(grantWithAdminOption);
                    modifiedGrants.add(grantWithoutAdminOption);
                    continue;
                }
                modifiedGrants.remove(grantWithAdminOption);
                modifiedGrants.remove(grantWithoutAdminOption);
            }
        }
        if (!existingGrants.equals(modifiedGrants = this.removeDuplicatedEntries(modifiedGrants))) {
            this.writeRoleGrantsFile(modifiedGrants);
        }
    }

    @Override
    public synchronized Set<RoleGrant> listGrantedPrincipals(String role) {
        return (Set)this.listRoleGrantsSanitized().stream().filter(grant -> grant.getRoleName().equals(role)).collect(ImmutableSet.toImmutableSet());
    }

    @Override
    public synchronized Set<RoleGrant> listRoleGrants(HivePrincipal principal) {
        ImmutableSet.Builder result = ImmutableSet.builder();
        if (principal.getType() == PrincipalType.USER) {
            result.add((Object)new RoleGrant(principal.toTrinoPrincipal(), PUBLIC_ROLE_NAME, false));
            if (ADMIN_USERS.contains(principal.getName())) {
                result.add((Object)new RoleGrant(principal.toTrinoPrincipal(), ADMIN_ROLE_NAME, true));
            }
        }
        result.addAll((Iterable)this.listRoleGrantsSanitized().stream().filter(grant -> HivePrincipal.from(grant.getGrantee()).equals(principal)).collect(Collectors.toSet()));
        return result.build();
    }

    private synchronized Set<RoleGrant> listRoleGrantsSanitized() {
        Set<RoleGrant> grants = this.readRoleGrantsFile();
        Set<String> existingRoles = this.listRoles();
        return this.removeDuplicatedEntries(FileHiveMetastore.removeNonExistingRoles(grants, existingRoles));
    }

    private Set<RoleGrant> removeDuplicatedEntries(Set<RoleGrant> grants) {
        HashMap<RoleGranteeTuple, RoleGrant> map = new HashMap<RoleGranteeTuple, RoleGrant>();
        for (RoleGrant grant : grants) {
            RoleGranteeTuple tuple = new RoleGranteeTuple(grant.getRoleName(), HivePrincipal.from(grant.getGrantee()));
            map.merge(tuple, grant, (first, second) -> first.isGrantable() ? first : second);
        }
        return ImmutableSet.copyOf(map.values());
    }

    private static Set<RoleGrant> removeNonExistingRoles(Set<RoleGrant> grants, Set<String> existingRoles) {
        ImmutableSet.Builder result = ImmutableSet.builder();
        for (RoleGrant grant : grants) {
            HivePrincipal grantee;
            if (!existingRoles.contains(grant.getRoleName()) || (grantee = HivePrincipal.from(grant.getGrantee())).getType() == PrincipalType.ROLE && !existingRoles.contains(grantee.getName())) continue;
            result.add((Object)grant);
        }
        return result.build();
    }

    private Set<RoleGrant> readRoleGrantsFile() {
        return ImmutableSet.copyOf((Collection)this.readFile("roleGrants", this.getRoleGrantsFile(), this.roleGrantsCodec).orElse((List<RoleGrant>)ImmutableList.of()));
    }

    private void writeRoleGrantsFile(Set<RoleGrant> roleGrants) {
        this.writeFile("roleGrants", this.getRoleGrantsFile(), this.roleGrantsCodec, ImmutableList.copyOf(roleGrants), true);
    }

    private synchronized Optional<List<String>> getAllPartitionNames(String databaseName, String tableName) {
        Objects.requireNonNull(databaseName, "databaseName is null");
        Objects.requireNonNull(tableName, "tableName is null");
        Optional<Table> tableReference = this.getTable(databaseName, tableName);
        if (tableReference.isEmpty()) {
            return Optional.empty();
        }
        Table table = tableReference.get();
        Path tableMetadataDirectory = this.getTableMetadataDirectory(table);
        List<ArrayDeque<String>> partitions = this.listPartitions(tableMetadataDirectory, table.getPartitionColumns());
        List partitionNames = partitions.stream().map(partitionValues -> MetastoreUtil.makePartitionName(table.getPartitionColumns(), (List<String>)ImmutableList.copyOf((Collection)partitionValues))).filter(partitionName -> this.isValidPartition(table, (String)partitionName)).collect(Collectors.toList());
        return Optional.of(ImmutableList.copyOf(partitionNames));
    }

    private boolean isValidPartition(Table table, String partitionName) {
        try {
            return this.metadataFileSystem.exists(FileHiveMetastore.getSchemaPath(SchemaType.PARTITION, this.getPartitionMetadataDirectory(table, partitionName)));
        }
        catch (IOException e) {
            return false;
        }
    }

    private List<ArrayDeque<String>> listPartitions(Path director, List<Column> partitionColumns) {
        if (partitionColumns.isEmpty()) {
            return ImmutableList.of();
        }
        try {
            String directoryPrefix = partitionColumns.get(0).getName() + "=";
            ArrayList<ArrayDeque<String>> partitionValues = new ArrayList<ArrayDeque<String>>();
            for (FileStatus fileStatus : this.metadataFileSystem.listStatus(director)) {
                if (!fileStatus.isDirectory() || !fileStatus.getPath().getName().startsWith(directoryPrefix)) continue;
                Object childPartitionValues = partitionColumns.size() == 1 ? ImmutableList.of(new ArrayDeque()) : this.listPartitions(fileStatus.getPath(), partitionColumns.subList(1, partitionColumns.size()));
                String value = FileUtils.unescapePathName((String)fileStatus.getPath().getName().substring(directoryPrefix.length()));
                Iterator iterator = childPartitionValues.iterator();
                while (iterator.hasNext()) {
                    ArrayDeque childPartition = (ArrayDeque)iterator.next();
                    childPartition.addFirst(value);
                    partitionValues.add(childPartition);
                }
            }
            return partitionValues;
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Error listing partition directories", (Throwable)e);
        }
    }

    @Override
    public synchronized Optional<Partition> getPartition(Table table, List<String> partitionValues) {
        Objects.requireNonNull(table, "table is null");
        Objects.requireNonNull(partitionValues, "partitionValues is null");
        Path partitionDirectory = this.getPartitionMetadataDirectory(table, partitionValues);
        return this.readSchemaFile(SchemaType.PARTITION, partitionDirectory, this.partitionCodec).map(partitionMetadata -> partitionMetadata.toPartition(table.getDatabaseName(), table.getTableName(), partitionValues, partitionDirectory.toString()));
    }

    @Override
    public Optional<List<String>> getPartitionNamesByFilter(String databaseName, String tableName, List<String> columnNames, TupleDomain<String> partitionKeysFilter) {
        return this.getAllPartitionNames(databaseName, tableName);
    }

    @Override
    public synchronized Map<String, Optional<Partition>> getPartitionsByNames(Table table, List<String> partitionNames) {
        ImmutableMap.Builder builder = ImmutableMap.builder();
        for (String partitionName : partitionNames) {
            List<String> partitionValues = HiveUtil.toPartitionValues(partitionName);
            builder.put((Object)partitionName, this.getPartition(table, partitionValues));
        }
        return builder.buildOrThrow();
    }

    @Override
    public synchronized Set<HivePrivilegeInfo> listTablePrivileges(String databaseName, String tableName, Optional<String> tableOwner, Optional<HivePrincipal> principal) {
        Table table = this.getRequiredTable(databaseName, tableName);
        Path permissionsDirectory = this.getPermissionsDirectory(table);
        if (principal.isEmpty()) {
            ImmutableSet.Builder privileges = ImmutableSet.builder().addAll(this.readAllPermissions(permissionsDirectory));
            tableOwner.ifPresent(owner -> privileges.add((Object)new HivePrivilegeInfo(HivePrivilegeInfo.HivePrivilege.OWNERSHIP, true, new HivePrincipal(PrincipalType.USER, (String)owner), new HivePrincipal(PrincipalType.USER, (String)owner))));
            return privileges.build();
        }
        ImmutableSet.Builder result = ImmutableSet.builder();
        if (principal.get().getType() == PrincipalType.USER && table.getOwner().orElseThrow().equals(principal.get().getName())) {
            result.add((Object)new HivePrivilegeInfo(HivePrivilegeInfo.HivePrivilege.OWNERSHIP, true, principal.get(), principal.get()));
        }
        result.addAll(this.readPermissionsFile(FileHiveMetastore.getPermissionsPath(permissionsDirectory, principal.get())));
        return result.build();
    }

    @Override
    public synchronized void grantTablePrivileges(String databaseName, String tableName, String tableOwner, HivePrincipal grantee, HivePrincipal grantor, Set<HivePrivilegeInfo.HivePrivilege> privileges, boolean grantOption) {
        this.setTablePrivileges(grantee, databaseName, tableName, (Collection)privileges.stream().map(privilege -> new HivePrivilegeInfo((HivePrivilegeInfo.HivePrivilege)((Object)privilege), grantOption, grantor, grantee)).collect(ImmutableList.toImmutableList()));
    }

    @Override
    public synchronized void revokeTablePrivileges(String databaseName, String tableName, String tableOwner, HivePrincipal grantee, HivePrincipal grantor, Set<HivePrivilegeInfo.HivePrivilege> privileges, boolean grantOption) {
        Set<HivePrivilegeInfo> currentPrivileges = this.listTablePrivileges(databaseName, tableName, Optional.of(tableOwner), Optional.of(grantee));
        Set privilegesToRemove = (Set)privileges.stream().map(p -> new HivePrivilegeInfo((HivePrivilegeInfo.HivePrivilege)((Object)p), grantOption, grantor, grantee)).collect(ImmutableSet.toImmutableSet());
        this.setTablePrivileges(grantee, databaseName, tableName, (Collection<HivePrivilegeInfo>)Sets.difference(currentPrivileges, (Set)privilegesToRemove));
    }

    private synchronized void setTablePrivileges(HivePrincipal grantee, String databaseName, String tableName, Collection<HivePrivilegeInfo> privileges) {
        Objects.requireNonNull(grantee, "grantee is null");
        Objects.requireNonNull(databaseName, "databaseName is null");
        Objects.requireNonNull(tableName, "tableName is null");
        Objects.requireNonNull(privileges, "privileges is null");
        try {
            Table table = this.getRequiredTable(databaseName, tableName);
            Path permissionsDirectory = this.getPermissionsDirectory(table);
            boolean created = this.metadataFileSystem.mkdirs(permissionsDirectory);
            if (!created && !this.metadataFileSystem.isDirectory(permissionsDirectory)) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not create permissions directory");
            }
            Path permissionFilePath = FileHiveMetastore.getPermissionsPath(permissionsDirectory, grantee);
            List permissions = privileges.stream().map(hivePrivilegeInfo -> new PermissionMetadata(hivePrivilegeInfo.getHivePrivilege(), hivePrivilegeInfo.isGrantOption(), grantee)).collect(Collectors.toList());
            this.writeFile("permissions", permissionFilePath, this.permissionsCodec, permissions, true);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, (Throwable)e);
        }
    }

    private synchronized void deleteTablePrivileges(Table table) {
        try {
            Path permissionsDirectory = this.getPermissionsDirectory(table);
            this.metadataFileSystem.delete(permissionsDirectory, true);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not delete table permissions", (Throwable)e);
        }
    }

    private List<Path> getChildSchemaDirectories(SchemaType type, Path metadataDirectory) {
        try {
            if (!this.metadataFileSystem.isDirectory(metadataDirectory)) {
                return ImmutableList.of();
            }
            ImmutableList.Builder childSchemaDirectories = ImmutableList.builder();
            for (FileStatus child : this.metadataFileSystem.listStatus(metadataDirectory)) {
                Path childPath;
                if (!child.isDirectory() || (childPath = child.getPath()).getName().startsWith(".") || !this.metadataFileSystem.isFile(FileHiveMetastore.getSchemaPath(type, childPath))) continue;
                childSchemaDirectories.add((Object)childPath);
            }
            return childSchemaDirectories.build();
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, (Throwable)e);
        }
    }

    private Set<HivePrivilegeInfo> readPermissionsFile(Path permissionFilePath) {
        return (Set)this.readFile("permissions", permissionFilePath, this.permissionsCodec).orElse((List<PermissionMetadata>)ImmutableList.of()).stream().map(PermissionMetadata::toHivePrivilegeInfo).collect(ImmutableSet.toImmutableSet());
    }

    private Set<HivePrivilegeInfo> readAllPermissions(Path permissionsDirectory) {
        try {
            return (Set)Arrays.stream(this.metadataFileSystem.listStatus(permissionsDirectory)).filter(FileStatus::isFile).filter(file -> !file.getPath().getName().startsWith(".")).flatMap(file -> this.readPermissionsFile(file.getPath()).stream()).collect(ImmutableSet.toImmutableSet());
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, (Throwable)e);
        }
    }

    private void deleteDirectoryAndSchema(SchemaType type, Path metadataDirectory) {
        try {
            Path schemaPath = FileHiveMetastore.getSchemaPath(type, metadataDirectory);
            if (!this.metadataFileSystem.isFile(schemaPath)) {
                return;
            }
            this.deleteSchemaFile(type, metadataDirectory);
            if (!this.metadataFileSystem.delete(metadataDirectory, true)) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not delete metadata directory");
            }
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, (Throwable)e);
        }
    }

    private void checkVersion(Optional<String> writerVersion) {
        if (writerVersion.isPresent() && writerVersion.get().equals(this.currentVersion)) {
            return;
        }
        if (this.versionCompatibility == FileHiveMetastoreConfig.VersionCompatibility.UNSAFE_ASSUME_COMPATIBILITY) {
            return;
        }
        throw new RuntimeException(String.format("The metadata file was written with %s while current version is %s. File metastore provides no compatibility for metadata written with a different version. You can disable this check by setting '%s=%s' configuration property.", new Object[]{writerVersion.map(version -> "version " + version).orElse("unknown version"), this.currentVersion, "hive.metastore.version-compatibility", FileHiveMetastoreConfig.VersionCompatibility.UNSAFE_ASSUME_COMPATIBILITY}));
    }

    private <T> Optional<T> readSchemaFile(SchemaType type, Path metadataDirectory, JsonCodec<T> codec) {
        return this.readFile(type + " schema", FileHiveMetastore.getSchemaPath(type, metadataDirectory), codec);
    }

    private <T> Optional<T> readFile(String type, Path path, JsonCodec<T> codec) {
        Optional<Object> optional;
        block9: {
            if (!this.metadataFileSystem.isFile(path)) {
                return Optional.empty();
            }
            FSDataInputStream inputStream = this.metadataFileSystem.open(path);
            try {
                byte[] json = ByteStreams.toByteArray((InputStream)inputStream);
                optional = Optional.of(codec.fromJson(json));
                if (inputStream == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (inputStream != null) {
                        try {
                            inputStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (Exception e) {
                    throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not read " + type, (Throwable)e);
                }
            }
            inputStream.close();
        }
        return optional;
    }

    private <T> void writeSchemaFile(SchemaType type, Path directory, JsonCodec<T> codec, T value, boolean overwrite) {
        this.writeFile(type + " schema", FileHiveMetastore.getSchemaPath(type, directory), codec, value, overwrite);
    }

    private <T> void writeFile(String type, Path path, JsonCodec<T> codec, T value, boolean overwrite) {
        try {
            byte[] json = codec.toJsonBytes(value);
            if (!overwrite && this.metadataFileSystem.exists(path)) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, type + " file already exists");
            }
            this.metadataFileSystem.mkdirs(path.getParent());
            try (FSDataOutputStream outputStream = this.metadataFileSystem.create(path, overwrite);){
                outputStream.write(json);
            }
        }
        catch (Exception e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not write " + type, (Throwable)e);
        }
        finally {
            this.listTablesCache.invalidateAll();
        }
    }

    private void renameSchemaFile(SchemaType type, Path oldMetadataDirectory, Path newMetadataDirectory) {
        try {
            if (!this.metadataFileSystem.rename(FileHiveMetastore.getSchemaPath(type, oldMetadataDirectory), FileHiveMetastore.getSchemaPath(type, newMetadataDirectory))) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not rename " + type + " schema");
            }
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not rename " + type + " schema", (Throwable)e);
        }
        finally {
            this.listTablesCache.invalidateAll();
        }
    }

    private void deleteSchemaFile(SchemaType type, Path metadataDirectory) {
        try {
            if (!this.metadataFileSystem.delete(FileHiveMetastore.getSchemaPath(type, metadataDirectory), false)) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not delete " + type + " schema");
            }
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_METASTORE_ERROR, "Could not delete " + type + " schema", (Throwable)e);
        }
        finally {
            this.listTablesCache.invalidateAll();
        }
    }

    private Path getDatabaseMetadataDirectory(String databaseName) {
        return new Path(this.catalogDirectory, databaseName);
    }

    private Path getTableMetadataDirectory(Table table) {
        return this.getTableMetadataDirectory(table.getDatabaseName(), table.getTableName());
    }

    private Path getTableMetadataDirectory(String databaseName, String tableName) {
        return new Path(this.getDatabaseMetadataDirectory(databaseName), tableName);
    }

    private Path getPartitionMetadataDirectory(Table table, List<String> values) {
        String partitionName = MetastoreUtil.makePartitionName(table.getPartitionColumns(), values);
        return this.getPartitionMetadataDirectory(table, partitionName);
    }

    private Path getPartitionMetadataDirectory(Table table, String partitionName) {
        Path tableMetadataDirectory = this.getTableMetadataDirectory(table);
        return new Path(tableMetadataDirectory, partitionName);
    }

    private Path getPermissionsDirectory(Table table) {
        return new Path(this.getTableMetadataDirectory(table), TRINO_PERMISSIONS_DIRECTORY_NAME);
    }

    private static Path getPermissionsPath(Path permissionsDirectory, HivePrincipal grantee) {
        return new Path(permissionsDirectory, grantee.getType().toString().toLowerCase(Locale.US) + "_" + grantee.getName());
    }

    private Path getRolesFile() {
        return new Path(this.catalogDirectory, ROLES_FILE_NAME);
    }

    private Path getRoleGrantsFile() {
        return new Path(this.catalogDirectory, ROLE_GRANTS_FILE_NAME);
    }

    private static Path getSchemaPath(SchemaType type, Path metadataDirectory) {
        if (type == SchemaType.DATABASE) {
            return new Path(Objects.requireNonNull(metadataDirectory.getParent(), "Can't use root directory as database path"), String.format(".%s%s", metadataDirectory.getName(), TRINO_SCHEMA_FILE_NAME_SUFFIX));
        }
        return new Path(metadataDirectory, TRINO_SCHEMA_FILE_NAME_SUFFIX);
    }

    private static boolean isChildDirectory(Path parentDirectory, Path childDirectory) {
        if (parentDirectory.equals((Object)childDirectory)) {
            return true;
        }
        if (childDirectory.isRoot()) {
            return false;
        }
        return FileHiveMetastore.isChildDirectory(parentDirectory, childDirectory.getParent());
    }

    static enum SchemaType {
        DATABASE,
        TABLE,
        PARTITION;


        public String toString() {
            return this.name().toLowerCase(Locale.ENGLISH);
        }
    }

    private static class RoleGranteeTuple {
        private final String role;
        private final HivePrincipal grantee;

        private RoleGranteeTuple(String role, HivePrincipal grantee) {
            this.role = Objects.requireNonNull(role, "role is null");
            this.grantee = Objects.requireNonNull(grantee, "grantee is null");
        }

        public String getRole() {
            return this.role;
        }

        public HivePrincipal getGrantee() {
            return this.grantee;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            RoleGranteeTuple that = (RoleGranteeTuple)o;
            return Objects.equals(this.role, that.role) && Objects.equals(this.grantee, that.grantee);
        }

        public int hashCode() {
            return Objects.hash(this.role, this.grantee);
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("role", (Object)this.role).add("grantee", (Object)this.grantee).toString();
        }
    }
}

