/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.google.sheets;

import com.google.api.client.auth.oauth2.Credential;
import com.google.api.client.googleapis.auth.oauth2.GoogleCredential;
import com.google.api.client.googleapis.javanet.GoogleNetHttpTransport;
import com.google.api.client.http.HttpRequestInitializer;
import com.google.api.client.http.HttpTransport;
import com.google.api.client.json.JsonFactory;
import com.google.api.client.json.jackson2.JacksonFactory;
import com.google.api.services.sheets.v4.Sheets;
import com.google.api.services.sheets.v4.model.ValueRange;
import com.google.common.base.Throwables;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.util.concurrent.UncheckedExecutionException;
import io.airlift.json.JsonCodec;
import io.airlift.log.Logger;
import io.trino.collect.cache.NonEvictableLoadingCache;
import io.trino.collect.cache.SafeCaches;
import io.trino.plugin.google.sheets.SheetsColumn;
import io.trino.plugin.google.sheets.SheetsConfig;
import io.trino.plugin.google.sheets.SheetsErrorCode;
import io.trino.plugin.google.sheets.SheetsTable;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.TrinoException;
import io.trino.spi.type.Type;
import io.trino.spi.type.VarcharType;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.GeneralSecurityException;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import javax.inject.Inject;

public class SheetsClient {
    private static final Logger log = Logger.get(SheetsClient.class);
    private static final String APPLICATION_NAME = "trino google sheets integration";
    private static final JsonFactory JSON_FACTORY = JacksonFactory.getDefaultInstance();
    private static final List<String> SCOPES = ImmutableList.of((Object)"https://www.googleapis.com/auth/spreadsheets.readonly");
    private final NonEvictableLoadingCache<String, Optional<String>> tableSheetMappingCache;
    private final NonEvictableLoadingCache<String, List<List<Object>>> sheetDataCache;
    private final String metadataSheetId;
    private final String credentialsFilePath;
    private final Sheets sheetsService;

    @Inject
    public SheetsClient(SheetsConfig config, JsonCodec<Map<String, List<SheetsTable>>> catalogCodec) {
        Objects.requireNonNull(catalogCodec, "catalogCodec is null");
        this.metadataSheetId = config.getMetadataSheetId();
        this.credentialsFilePath = config.getCredentialsFilePath();
        try {
            this.sheetsService = new Sheets.Builder((HttpTransport)GoogleNetHttpTransport.newTrustedTransport(), JSON_FACTORY, (HttpRequestInitializer)this.getCredentials()).setApplicationName(APPLICATION_NAME).build();
        }
        catch (IOException | GeneralSecurityException e) {
            throw new TrinoException((ErrorCodeSupplier)SheetsErrorCode.SHEETS_BAD_CREDENTIALS_ERROR, (Throwable)e);
        }
        long expiresAfterWriteMillis = config.getSheetsDataExpireAfterWrite().toMillis();
        long maxCacheSize = config.getSheetsDataMaxCacheSize();
        this.tableSheetMappingCache = SafeCaches.buildNonEvictableCache(SheetsClient.newCacheBuilder(expiresAfterWriteMillis, maxCacheSize), (CacheLoader)new CacheLoader<String, Optional<String>>(){

            public Optional<String> load(String tableName) {
                return SheetsClient.this.getSheetExpressionForTable(tableName);
            }

            public Map<String, Optional<String>> loadAll(Iterable<? extends String> tableList) {
                return SheetsClient.this.getAllTableSheetExpressionMapping();
            }
        });
        this.sheetDataCache = SafeCaches.buildNonEvictableCache(SheetsClient.newCacheBuilder(expiresAfterWriteMillis, maxCacheSize), (CacheLoader)CacheLoader.from(this::readAllValuesFromSheetExpression));
    }

    public Optional<SheetsTable> getTable(String tableName) {
        List<List<String>> values = SheetsClient.convertToStringValues(this.readAllValues(tableName));
        if (values.size() > 0) {
            ImmutableList.Builder columns = ImmutableList.builder();
            HashSet<Object> columnNames = new HashSet<Object>();
            List<String> header = values.get(0);
            int count = 0;
            for (String column : header) {
                Object columnValue = column.toLowerCase(Locale.ENGLISH);
                if (((String)columnValue).isEmpty() || columnNames.contains(columnValue)) {
                    columnValue = "column_" + ++count;
                }
                columnNames.add(columnValue);
                columns.add((Object)new SheetsColumn((String)columnValue, (Type)VarcharType.VARCHAR));
            }
            List<List<String>> dataValues = values.subList(1, values.size());
            return Optional.of(new SheetsTable(tableName, (List<SheetsColumn>)columns.build(), dataValues));
        }
        return Optional.empty();
    }

    public Set<String> getTableNames() {
        ImmutableSet.Builder tables = ImmutableSet.builder();
        try {
            List tableMetadata = (List)this.sheetDataCache.getUnchecked((Object)this.metadataSheetId);
            for (int i = 1; i < tableMetadata.size(); ++i) {
                if (((List)tableMetadata.get(i)).size() <= 0) continue;
                tables.add((Object)String.valueOf(((List)tableMetadata.get(i)).get(0)));
            }
            return tables.build();
        }
        catch (UncheckedExecutionException e) {
            Throwables.throwIfInstanceOf((Throwable)e.getCause(), TrinoException.class);
            throw new TrinoException((ErrorCodeSupplier)SheetsErrorCode.SHEETS_METASTORE_ERROR, (Throwable)e);
        }
    }

    public List<List<Object>> readAllValues(String tableName) {
        try {
            String sheetExpression = (String)((Optional)this.tableSheetMappingCache.getUnchecked((Object)tableName)).orElseThrow(() -> new TrinoException((ErrorCodeSupplier)SheetsErrorCode.SHEETS_UNKNOWN_TABLE_ERROR, "Sheet expression not found for table " + tableName));
            return (List)this.sheetDataCache.getUnchecked((Object)sheetExpression);
        }
        catch (UncheckedExecutionException e) {
            Throwables.throwIfInstanceOf((Throwable)e.getCause(), TrinoException.class);
            throw new TrinoException((ErrorCodeSupplier)SheetsErrorCode.SHEETS_TABLE_LOAD_ERROR, "Error loading data for table: " + tableName, (Throwable)e);
        }
    }

    public static List<List<String>> convertToStringValues(List<List<Object>> values) {
        return (List)values.stream().map(columns -> (ImmutableList)columns.stream().map(String::valueOf).collect(ImmutableList.toImmutableList())).collect(ImmutableList.toImmutableList());
    }

    private Optional<String> getSheetExpressionForTable(String tableName) {
        Map<String, Optional<String>> tableSheetMap = this.getAllTableSheetExpressionMapping();
        if (!tableSheetMap.containsKey(tableName)) {
            return Optional.empty();
        }
        return tableSheetMap.get(tableName);
    }

    private Map<String, Optional<String>> getAllTableSheetExpressionMapping() {
        ImmutableMap.Builder tableSheetMap = ImmutableMap.builder();
        List<List<Object>> data = this.readAllValuesFromSheetExpression(this.metadataSheetId);
        for (int i = 1; i < data.size(); ++i) {
            if (data.get(i).size() < 2) continue;
            String tableId = String.valueOf(data.get(i).get(0));
            String sheetId = String.valueOf(data.get(i).get(1));
            tableSheetMap.put((Object)tableId.toLowerCase(Locale.ENGLISH), Optional.of(sheetId));
        }
        return tableSheetMap.buildOrThrow();
    }

    private Credential getCredentials() {
        GoogleCredential googleCredential;
        FileInputStream in = new FileInputStream(this.credentialsFilePath);
        try {
            googleCredential = GoogleCredential.fromStream((InputStream)in).createScoped(SCOPES);
        }
        catch (Throwable throwable) {
            try {
                try {
                    ((InputStream)in).close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new TrinoException((ErrorCodeSupplier)SheetsErrorCode.SHEETS_BAD_CREDENTIALS_ERROR, (Throwable)e);
            }
        }
        ((InputStream)in).close();
        return googleCredential;
    }

    private List<List<Object>> readAllValuesFromSheetExpression(String sheetExpression) {
        try {
            String defaultRange = "$1:$10000";
            String[] tableOptions = sheetExpression.split("#");
            String sheetId = tableOptions[0];
            if (tableOptions.length > 1) {
                defaultRange = tableOptions[1];
            }
            log.debug("Accessing sheet id [%s] with range [%s]", new Object[]{sheetId, defaultRange});
            return ((ValueRange)this.sheetsService.spreadsheets().values().get(sheetId, defaultRange).execute()).getValues();
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)SheetsErrorCode.SHEETS_UNKNOWN_TABLE_ERROR, "Failed reading data from sheet: " + sheetExpression, (Throwable)e);
        }
    }

    private static CacheBuilder<Object, Object> newCacheBuilder(long expiresAfterWriteMillis, long maximumSize) {
        return CacheBuilder.newBuilder().expireAfterWrite(expiresAfterWriteMillis, TimeUnit.MILLISECONDS).maximumSize(maximumSize);
    }
}

