/*
 * Decompiled with CFR 0.152.
 */
package io.trino.orc.writer;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import io.airlift.slice.Slice;
import io.trino.orc.checkpoint.BooleanStreamCheckpoint;
import io.trino.orc.checkpoint.LongStreamCheckpoint;
import io.trino.orc.metadata.ColumnEncoding;
import io.trino.orc.metadata.CompressedMetadataWriter;
import io.trino.orc.metadata.CompressionKind;
import io.trino.orc.metadata.OrcColumnId;
import io.trino.orc.metadata.RowGroupIndex;
import io.trino.orc.metadata.Stream;
import io.trino.orc.metadata.statistics.ColumnStatistics;
import io.trino.orc.metadata.statistics.LongValueStatisticsBuilder;
import io.trino.orc.metadata.statistics.TimestampStatisticsBuilder;
import io.trino.orc.stream.LongOutputStream;
import io.trino.orc.stream.LongOutputStreamV2;
import io.trino.orc.stream.PresentOutputStream;
import io.trino.orc.stream.StreamDataOutput;
import io.trino.orc.writer.ColumnWriter;
import io.trino.spi.block.Block;
import io.trino.spi.type.DateTimeEncoding;
import io.trino.spi.type.LongTimestamp;
import io.trino.spi.type.LongTimestampWithTimeZone;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.TimestampWithTimeZoneType;
import io.trino.spi.type.Type;
import java.io.IOException;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Supplier;
import org.openjdk.jol.info.ClassLayout;

public class TimestampColumnWriter
implements ColumnWriter {
    private static final int INSTANCE_SIZE = Math.toIntExact(ClassLayout.parseClass(TimestampColumnWriter.class).instanceSize());
    private static final long ORC_EPOCH_IN_SECONDS = OffsetDateTime.of(2015, 1, 1, 0, 0, 0, 0, ZoneOffset.UTC).toEpochSecond();
    private final OrcColumnId columnId;
    private final Type type;
    private final TimestampKind timestampKind;
    private final boolean compressed;
    private final ColumnEncoding columnEncoding;
    private final LongOutputStream secondsStream;
    private final LongOutputStream nanosStream;
    private final PresentOutputStream presentStream;
    private final List<ColumnStatistics> rowGroupColumnStatistics = new ArrayList<ColumnStatistics>();
    private final Supplier<TimestampStatisticsBuilder> statisticsBuilderSupplier;
    private LongValueStatisticsBuilder statisticsBuilder;
    private boolean closed;

    public TimestampColumnWriter(OrcColumnId columnId, Type type, CompressionKind compression, int bufferSize, Supplier<TimestampStatisticsBuilder> statisticsBuilderSupplier) {
        this.columnId = Objects.requireNonNull(columnId, "columnId is null");
        this.type = Objects.requireNonNull(type, "type is null");
        this.timestampKind = TimestampColumnWriter.timestampKindForType(type);
        this.compressed = Objects.requireNonNull(compression, "compression is null") != CompressionKind.NONE;
        this.columnEncoding = new ColumnEncoding(ColumnEncoding.ColumnEncodingKind.DIRECT_V2, 0);
        this.secondsStream = new LongOutputStreamV2(compression, bufferSize, true, Stream.StreamKind.DATA);
        this.nanosStream = new LongOutputStreamV2(compression, bufferSize, false, Stream.StreamKind.SECONDARY);
        this.presentStream = new PresentOutputStream(compression, bufferSize);
        this.statisticsBuilderSupplier = Objects.requireNonNull(statisticsBuilderSupplier, "statisticsBuilderSupplier is null");
        this.statisticsBuilder = statisticsBuilderSupplier.get();
    }

    private static TimestampKind timestampKindForType(Type type) {
        if (type.equals(TimestampType.TIMESTAMP_MILLIS)) {
            return TimestampKind.TIMESTAMP_MILLIS;
        }
        if (type.equals(TimestampType.TIMESTAMP_MICROS)) {
            return TimestampKind.TIMESTAMP_MICROS;
        }
        if (type.equals(TimestampType.TIMESTAMP_NANOS)) {
            return TimestampKind.TIMESTAMP_NANOS;
        }
        if (type.equals(TimestampWithTimeZoneType.TIMESTAMP_TZ_MILLIS)) {
            return TimestampKind.INSTANT_MILLIS;
        }
        if (type.equals(TimestampWithTimeZoneType.TIMESTAMP_TZ_MICROS)) {
            return TimestampKind.INSTANT_MICROS;
        }
        if (type.equals(TimestampWithTimeZoneType.TIMESTAMP_TZ_NANOS)) {
            return TimestampKind.INSTANT_NANOS;
        }
        throw new IllegalArgumentException("Unsupported type for ORC timestamp writer: " + type);
    }

    @Override
    public Map<OrcColumnId, ColumnEncoding> getColumnEncodings() {
        return ImmutableMap.of((Object)this.columnId, (Object)this.columnEncoding);
    }

    @Override
    public void beginRowGroup() {
        this.presentStream.recordCheckpoint();
        this.secondsStream.recordCheckpoint();
        this.nanosStream.recordCheckpoint();
    }

    @Override
    public void writeBlock(Block block) {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0);
        Preconditions.checkArgument((block.getPositionCount() > 0 ? 1 : 0) != 0, (Object)"Block is empty");
        for (int position = 0; position < block.getPositionCount(); ++position) {
            this.presentStream.writeBoolean(!block.isNull(position));
        }
        switch (this.timestampKind) {
            case TIMESTAMP_MILLIS: 
            case TIMESTAMP_MICROS: {
                this.writeTimestampMicros(block);
                break;
            }
            case TIMESTAMP_NANOS: {
                this.writeTimestampNanos(block);
                break;
            }
            case INSTANT_MILLIS: {
                this.writeInstantShort(block);
                break;
            }
            case INSTANT_MICROS: 
            case INSTANT_NANOS: {
                this.writeInstantLong(block);
            }
        }
    }

    @Override
    public Map<OrcColumnId, ColumnStatistics> finishRowGroup() {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0);
        ColumnStatistics statistics = this.statisticsBuilder.buildColumnStatistics();
        this.rowGroupColumnStatistics.add(statistics);
        this.statisticsBuilder = this.statisticsBuilderSupplier.get();
        return ImmutableMap.of((Object)this.columnId, (Object)statistics);
    }

    @Override
    public void close() {
        this.closed = true;
        this.secondsStream.close();
        this.nanosStream.close();
        this.presentStream.close();
    }

    @Override
    public Map<OrcColumnId, ColumnStatistics> getColumnStripeStatistics() {
        Preconditions.checkState((boolean)this.closed);
        return ImmutableMap.of((Object)this.columnId, (Object)ColumnStatistics.mergeColumnStatistics(this.rowGroupColumnStatistics));
    }

    @Override
    public List<StreamDataOutput> getIndexStreams(CompressedMetadataWriter metadataWriter) throws IOException {
        Preconditions.checkState((boolean)this.closed);
        ImmutableList.Builder rowGroupIndexes = ImmutableList.builder();
        List secondsCheckpoints = this.secondsStream.getCheckpoints();
        List nanosCheckpoints = this.nanosStream.getCheckpoints();
        Optional<List<BooleanStreamCheckpoint>> presentCheckpoints = this.presentStream.getCheckpoints();
        int i = 0;
        while (i < this.rowGroupColumnStatistics.size()) {
            int groupId = i++;
            ColumnStatistics columnStatistics = this.rowGroupColumnStatistics.get(groupId);
            LongStreamCheckpoint secondsCheckpoint = (LongStreamCheckpoint)secondsCheckpoints.get(groupId);
            LongStreamCheckpoint nanosCheckpoint = (LongStreamCheckpoint)nanosCheckpoints.get(groupId);
            Optional<BooleanStreamCheckpoint> presentCheckpoint = presentCheckpoints.map(checkpoints -> (BooleanStreamCheckpoint)checkpoints.get(groupId));
            List<Integer> positions = TimestampColumnWriter.createTimestampColumnPositionList(this.compressed, secondsCheckpoint, nanosCheckpoint, presentCheckpoint);
            rowGroupIndexes.add((Object)new RowGroupIndex(positions, columnStatistics));
        }
        Slice slice = metadataWriter.writeRowIndexes((List<RowGroupIndex>)rowGroupIndexes.build());
        Stream stream = new Stream(this.columnId, Stream.StreamKind.ROW_INDEX, slice.length(), false);
        return ImmutableList.of((Object)new StreamDataOutput(slice, stream));
    }

    private static List<Integer> createTimestampColumnPositionList(boolean compressed, LongStreamCheckpoint secondsCheckpoint, LongStreamCheckpoint nanosCheckpoint, Optional<BooleanStreamCheckpoint> presentCheckpoint) {
        ImmutableList.Builder positionList = ImmutableList.builder();
        presentCheckpoint.ifPresent(booleanStreamCheckpoint -> positionList.addAll(booleanStreamCheckpoint.toPositionList(compressed)));
        positionList.addAll(secondsCheckpoint.toPositionList(compressed));
        positionList.addAll(nanosCheckpoint.toPositionList(compressed));
        return positionList.build();
    }

    @Override
    public List<StreamDataOutput> getBloomFilters(CompressedMetadataWriter metadataWriter) {
        return ImmutableList.of();
    }

    @Override
    public List<StreamDataOutput> getDataStreams() {
        Preconditions.checkState((boolean)this.closed);
        ImmutableList.Builder outputDataStreams = ImmutableList.builder();
        this.presentStream.getStreamDataOutput(this.columnId).ifPresent(arg_0 -> ((ImmutableList.Builder)outputDataStreams).add(arg_0));
        outputDataStreams.add((Object)this.secondsStream.getStreamDataOutput(this.columnId));
        outputDataStreams.add((Object)this.nanosStream.getStreamDataOutput(this.columnId));
        return outputDataStreams.build();
    }

    @Override
    public long getBufferedBytes() {
        return this.secondsStream.getBufferedBytes() + this.nanosStream.getBufferedBytes() + this.presentStream.getBufferedBytes();
    }

    @Override
    public long getRetainedBytes() {
        long retainedBytes = (long)INSTANCE_SIZE + this.secondsStream.getRetainedBytes() + this.nanosStream.getRetainedBytes() + this.presentStream.getRetainedBytes();
        for (ColumnStatistics statistics : this.rowGroupColumnStatistics) {
            retainedBytes += statistics.getRetainedSizeInBytes();
        }
        return retainedBytes;
    }

    @Override
    public void reset() {
        this.closed = false;
        this.secondsStream.reset();
        this.nanosStream.reset();
        this.presentStream.reset();
        this.rowGroupColumnStatistics.clear();
        this.statisticsBuilder = this.statisticsBuilderSupplier.get();
    }

    private void writeTimestampMicros(Block block) {
        for (int i = 0; i < block.getPositionCount(); ++i) {
            if (block.isNull(i)) continue;
            long micros = this.type.getLong(block, i);
            long seconds = micros / 1000000L;
            long microsFraction = Math.floorMod(micros, 1000000);
            long nanosFraction = microsFraction * 1000L;
            long millis = Math.floorDiv(micros, 1000);
            this.writeValues(seconds, nanosFraction);
            this.statisticsBuilder.addValue(millis);
        }
    }

    private void writeTimestampNanos(Block block) {
        for (int i = 0; i < block.getPositionCount(); ++i) {
            if (block.isNull(i)) continue;
            LongTimestamp timestamp = (LongTimestamp)this.type.getObject(block, i);
            long seconds = timestamp.getEpochMicros() / 1000000L;
            long microsFraction = Math.floorMod(timestamp.getEpochMicros(), 1000000);
            long nanosFraction = microsFraction * 1000L + (long)(timestamp.getPicosOfMicro() / 1000);
            long millis = Math.floorDiv(timestamp.getEpochMicros(), 1000);
            this.writeValues(seconds, nanosFraction);
            this.statisticsBuilder.addValue(millis);
        }
    }

    private void writeInstantShort(Block block) {
        for (int i = 0; i < block.getPositionCount(); ++i) {
            if (block.isNull(i)) continue;
            this.writeMillis(DateTimeEncoding.unpackMillisUtc((long)this.type.getLong(block, i)));
        }
    }

    private void writeInstantLong(Block block) {
        for (int i = 0; i < block.getPositionCount(); ++i) {
            if (block.isNull(i)) continue;
            LongTimestampWithTimeZone timestamp = (LongTimestampWithTimeZone)this.type.getObject(block, i);
            long millis = timestamp.getEpochMillis();
            long seconds = millis / 1000L;
            long millisFraction = Math.floorMod(millis, 1000);
            long nanosFraction = millisFraction * 1000000L + (long)(timestamp.getPicosOfMilli() / 1000);
            this.writeValues(seconds, nanosFraction);
            this.statisticsBuilder.addValue(millis);
        }
    }

    private void writeMillis(long millis) {
        long seconds = millis / 1000L;
        long millisFraction = Math.floorMod(millis, 1000);
        long nanosFraction = millisFraction * 1000000L;
        this.writeValues(seconds, nanosFraction);
        this.statisticsBuilder.addValue(millis);
    }

    private void writeValues(long seconds, long nanosFraction) {
        this.secondsStream.writeLong(seconds - ORC_EPOCH_IN_SECONDS);
        this.nanosStream.writeLong(TimestampColumnWriter.encodeNanos(nanosFraction));
    }

    private static long encodeNanos(long nanos) {
        int trailingZeros;
        if (nanos == 0L) {
            return 0L;
        }
        if (nanos % 100L != 0L) {
            return nanos << 3;
        }
        nanos /= 100L;
        for (trailingZeros = 1; nanos % 10L == 0L && trailingZeros < 7; ++trailingZeros) {
            nanos /= 10L;
        }
        return nanos << 3 | (long)trailingZeros;
    }

    private static enum TimestampKind {
        TIMESTAMP_MILLIS,
        TIMESTAMP_MICROS,
        TIMESTAMP_NANOS,
        INSTANT_MILLIS,
        INSTANT_MICROS,
        INSTANT_NANOS;

    }
}

