/*
 * Decompiled with CFR 0.152.
 */
package alluxio.client.file.cache.store;

import alluxio.client.file.cache.PageId;
import alluxio.client.file.cache.PageStore;
import alluxio.client.file.cache.store.LocalPageStoreOptions;
import alluxio.client.file.cache.store.PageStoreDir;
import alluxio.exception.PageNotFoundException;
import alluxio.exception.status.ResourceExhaustedException;
import alluxio.shaded.client.com.google.common.annotations.VisibleForTesting;
import alluxio.shaded.client.com.google.common.base.Preconditions;
import alluxio.shaded.client.javax.annotation.concurrent.NotThreadSafe;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;

@NotThreadSafe
public class LocalPageStore
implements PageStore {
    private static final String ERROR_NO_SPACE_LEFT = "No space left on device";
    public static final String TEMP_DIR = "TEMP";
    private final Path mRoot;
    private final long mPageSize;
    private final long mCapacity;
    private final int mFileBuckets;

    public LocalPageStore(LocalPageStoreOptions options) {
        this.mRoot = options.getRootDir();
        this.mPageSize = options.getPageSize();
        this.mCapacity = (long)((double)options.getCacheSize() / (1.0 + options.getOverheadRatio()));
        this.mFileBuckets = options.getFileBuckets();
    }

    @Override
    public void put(PageId pageId, byte[] page, boolean isTemporary) throws ResourceExhaustedException, IOException {
        Path p = this.getFilePath(pageId, isTemporary);
        try {
            if (!Files.exists(p, new LinkOption[0])) {
                Path parent = Preconditions.checkNotNull(p.getParent(), "parent of cache file should not be null");
                Files.createDirectories(parent, new FileAttribute[0]);
                Files.createFile(p, new FileAttribute[0]);
            }
            try (FileOutputStream fos = new FileOutputStream(p.toFile(), false);){
                fos.write(page);
            }
        }
        catch (Exception e) {
            Files.deleteIfExists(p);
            if (e.getMessage().contains(ERROR_NO_SPACE_LEFT)) {
                throw new ResourceExhaustedException(String.format("%s is full, configured with %d bytes", this.mRoot, this.mCapacity), e);
            }
            throw new IOException("Failed to write file " + p + " for page " + pageId);
        }
    }

    @Override
    public int get(PageId pageId, int pageOffset, int bytesToRead, byte[] buffer, int bufferOffset, boolean isTemporary) throws IOException, PageNotFoundException {
        Preconditions.checkArgument(pageOffset >= 0, "page offset should be non-negative");
        Preconditions.checkArgument(buffer.length >= bufferOffset, "page offset %s should be less or equal than buffer length %s", bufferOffset, buffer.length);
        Path p = this.getFilePath(pageId, isTemporary);
        if (!Files.exists(p, new LinkOption[0])) {
            throw new PageNotFoundException(p.toString());
        }
        long pageLength = p.toFile().length();
        Preconditions.checkArgument((long)pageOffset <= pageLength, "page offset %s exceeded page size %s", pageOffset, pageLength);
        try (RandomAccessFile localFile = new RandomAccessFile(p.toString(), "r");){
            int bytes;
            int bytesSkipped = localFile.skipBytes(pageOffset);
            if (pageOffset != bytesSkipped) {
                throw new IOException(String.format("Failed to read page %s (%s) from offset %s: %s bytes skipped", pageId, p, pageOffset, bytesSkipped));
            }
            int bytesRead = 0;
            int bytesLeft = (int)Math.min(pageLength - (long)pageOffset, (long)(buffer.length - bufferOffset));
            for (bytesLeft = Math.min(bytesLeft, bytesToRead); bytesLeft >= 0 && (bytes = localFile.read(buffer, bufferOffset + bytesRead, bytesLeft)) > 0; bytesLeft -= bytes) {
                bytesRead += bytes;
            }
            int n = bytesRead;
            return n;
        }
    }

    @Override
    public void delete(PageId pageId) throws IOException, PageNotFoundException {
        Path p = this.getFilePath(pageId, false);
        if (!Files.exists(p, new LinkOption[0])) {
            throw new PageNotFoundException(p.toString());
        }
        Files.delete(p);
        Path parent = Preconditions.checkNotNull(p.getParent(), "parent of cache file should not be null");
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(parent);){
            if (!stream.iterator().hasNext()) {
                Files.delete(parent);
            }
        }
    }

    @VisibleForTesting
    public Path getFilePath(PageId pageId, boolean isTemporary) {
        String bucketName = isTemporary ? TEMP_DIR : PageStoreDir.getFileBucket(this.mFileBuckets, pageId.getFileId());
        return Paths.get(this.mRoot.toString(), Long.toString(this.mPageSize), bucketName, pageId.getFileId(), Long.toString(pageId.getPageIndex()));
    }

    @Override
    public void close() {
    }
}

